using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.clusters;
using org.ovirt.engine.ui.uicommon.models.hosts;
using org.ovirt.engine.ui.uicommon.models.storage;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using VdcCommon.BusinessEntities;
using VdcCommon.Interfaces;
using VdcFrontend;
using org.ovirt.engine.ui.uicompat;

namespace org.ovirt.engine.ui.uicommon.models.datacenters
{
	public class DataCenterGuideModel : GuideModel, ITaskTarget
	{
		#region Action Constants

		public readonly string DataCenterConfigureClustersAction = "Configure Cluster";
		public readonly string DataCenterAddAnotherClusterAction = "Add another Cluster";
		public readonly string DataCenterConfigureHostsAction = "Configure Host";
		public readonly string DataCenterSelectHostsAction = "Select Hosts";
		public readonly string DataCenterConfigureStorageAction = "Configure Storage";
		public readonly string DataCenterAddMoreStorageAction = "Add more Storage";
		public readonly string DataCenterAttachStorageAction = "Attach Storage";
		public readonly string DataCenterAttachMoreStorageAction = "Attach more Storage";
		public readonly string DataCenterConfigureISOLibraryAction = "Configure ISO Library";
		public readonly string DataCenterAttachISOLibraryAction = "Attach ISO Library";

		public readonly string NoUpHostReason = "There should be at least one active Host in the Data Center";
		public readonly string NoDataDomainAttachedReason = "Cannot create an ISO domain in a non-active Data Center";

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		public new storage_pool Entity
		{
			get { return (storage_pool)base.Entity; }
			set { base.Entity = value; }
		}

		#endregion


		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();
			UpdateOptions();
		}

		private void UpdateOptions()
		{
			CompulsoryActions.Clear();
			OptionalActions.Clear();

			if (Entity != null)
			{
				if (Entity.storage_pool_type != StorageType.LOCALFS)
				{
					//Add cluster action.
					List<VDSGroup> clusters = DataProvider.GetClusterList(Entity.Id);

					UICommand addClusterAction = new UICommand("AddCluster", this);
					if (clusters.Count == 0)
					{
						addClusterAction.Title = DataCenterConfigureClustersAction;
						CompulsoryActions.Add(addClusterAction);
					}
					else
					{
						addClusterAction.Title = DataCenterAddAnotherClusterAction;
						OptionalActions.Add(addClusterAction);
					}

					//Add host action.
					//Version minimalClusterVersion = clusters.Min(a => a.compatibility_version);
					Version minimalClusterVersion = Linq.GetMinVersionByClusters(clusters);

					if (minimalClusterVersion == null)
					{
						minimalClusterVersion = new Version();
					}

					List<VDS> hosts = new List<VDS>();
					List<VDS> availableHosts = new List<VDS>();
					List<VDS> upHosts = new List<VDS>();
					foreach (VDS vds in DataProvider.GetHostList())
					{
						//var hosts = DataProvider.GetHostList()
						//    .Where(a => clusters.Any(b => b.ID == a.vds_group_id)
						//           && (a.Version.FullVersion == null || a.Version.FullVersion.GetFriendlyVersion() >= minimalClusterVersion))
						//    .ToList();
						if (Linq.IsClusterItemExistInList(clusters, vds.vds_group_id)
							&&
							(vds.Version == null || vds.Version.FullVersion == null ||
							 Extensions.GetFriendlyVersion(vds.Version.FullVersion).CompareTo(minimalClusterVersion) >= 0))
							hosts.Add(vds);

						//var availableHosts = DataProvider.GetHostList()
						//    .Where(a => clusters.All(b => b.ID != a.vds_group_id)
						//        && (a.status == VDSStatus.Maintenance || a.status == VDSStatus.PendingApproval)
						//        && (a.Version.FullVersion == null || a.Version.FullVersion.GetFriendlyVersion() >= minimalClusterVersion))
						//    .ToList();
						if ((!Linq.IsHostBelongsToAnyOfClusters(clusters, vds))
							&& (vds.status == VDSStatus.Maintenance || vds.status == VDSStatus.PendingApproval)
							&&
							(vds.Version.FullVersion == null ||
							 Extensions.GetFriendlyVersion(vds.Version.FullVersion).CompareTo(minimalClusterVersion) >= 0))
							availableHosts.Add(vds);

						//var upHosts = DataProvider.GetHostList().Where(a => a.status == VDSStatus.Up).ToList();
						if (vds.status == VDSStatus.Up && Linq.IsClusterItemExistInList(clusters, vds.vds_group_id))
						{
							upHosts.Add(vds);
						}
					}

					UICommand addHostAction =
						new UICommand("AddHost", this)
							{
								IsExecutionAllowed = clusters.Count > 0
							};

					addHostAction.Title = DataCenterConfigureHostsAction;
					CompulsoryActions.Add(addHostAction);


					//Select host action.
					UICommand selectHostAction = new UICommand("SelectHost", this);

					if (availableHosts.Count > 0 && clusters.Count > 0)
					{
						if (hosts.Count == 0)
						{
							selectHostAction.Title = DataCenterSelectHostsAction;
							CompulsoryActions.Add(selectHostAction);
						}
						else
						{
							selectHostAction.Title = DataCenterSelectHostsAction;
							OptionalActions.Add(selectHostAction);
						}
					}

					List<storage_domains> allStorage = DataProvider.GetStorageDomainList();
					List<storage_domains> unattachedStorage = new List<storage_domains>();
					bool addToList;
					Version version3_0 = new Version(3, 0);
					foreach (storage_domains item in allStorage)
					{
						addToList = false;
						if (item.storage_domain_type == StorageDomainType.Data &&
							item.storage_type == Entity.storage_pool_type &&
							  item.storage_domain_shared_status == StorageDomainSharedStatus.Unattached)
						{
							if (Entity.StoragePoolFormatType == null)
							{
								//compat logic: in case its not v1 and the version is less than 3.0 break.
								if (item.StorageStaticData.StorageFormat != StorageFormatType.V1 && Entity.compatibility_version.compareTo(version3_0) < 0)
								{
									continue;
								}
								addToList = true;
							}
							else if ((StorageFormatType)Entity.StoragePoolFormatType == item.StorageStaticData.StorageFormat)
							{
								addToList = true;
							}
						}

						if (addToList)
						{
							unattachedStorage.Add(item);
						}
					}

					List<storage_domains> attachedStorage = DataProvider.GetStorageDomainList(Entity.Id);

					List<storage_domains> attachedDataStorages = new List<storage_domains>();
					foreach (storage_domains a in attachedStorage)
					{
						if (a.storage_domain_type == StorageDomainType.Data || a.storage_domain_type == StorageDomainType.Master)
						{
							attachedDataStorages.Add(a);
						}
					}

					UICommand addDataStorageAction =
						new UICommand("AddDataStorage", this)
							{
								IsExecutionAllowed = upHosts.Count > 0,
							};
					addDataStorageAction.ExecuteProhibitionReasons.Add(NoUpHostReason);

					if (unattachedStorage.Count == 0 && attachedDataStorages.Count == 0)
					{
						addDataStorageAction.Title = DataCenterConfigureStorageAction;
						CompulsoryActions.Add(addDataStorageAction);
					}
					else
					{
						addDataStorageAction.Title = DataCenterAddMoreStorageAction;
						OptionalActions.Add(addDataStorageAction);
					}

					//Attach data storage action.
					UICommand attachDataStorageAction =
						new UICommand("AttachDataStorage", this)
							{
								IsExecutionAllowed = unattachedStorage.Count > 0 && upHosts.Count > 0,
							};
					if (upHosts.Count == 0)
					{
						attachDataStorageAction.ExecuteProhibitionReasons.Add(NoUpHostReason);
					}
					if (attachedDataStorages.Count == 0)
					{
						attachDataStorageAction.Title = DataCenterAttachStorageAction;
						CompulsoryActions.Add(attachDataStorageAction);
					}
					else
					{
						attachDataStorageAction.Title = DataCenterAttachMoreStorageAction;
						OptionalActions.Add(attachDataStorageAction);
					}

					List<storage_domains> isoStorages = DataProvider.GetISOStorageDomainList();

					UICommand addIsoStorageAction = new UICommand("AddIsoStorage", this)
														{
															IsExecutionAllowed = Entity.status == StoragePoolStatus.Up
														};
					addIsoStorageAction.ExecuteProhibitionReasons.Add(NoDataDomainAttachedReason);

					if (isoStorages.Count == 0)
					{
						addIsoStorageAction.Title = DataCenterConfigureISOLibraryAction;
						OptionalActions.Add(addIsoStorageAction);
					}

					//Attach ISO storage action.
					// Allow to attach ISO domain only when there are Data storages attached
					// and there ISO storages to attach and ther are no ISO storages actually 
					// attached.
					//var attachedIsoStorages = attachedStorage
					//    .Where(a => a.storage_domain_type == StorageDomainType.ISO)
					//    .ToList();
					List<storage_domains> attachedIsoStorages = new List<storage_domains>();
					foreach (storage_domains sd in attachedStorage)
						if (sd.storage_domain_type == StorageDomainType.ISO)
							attachedIsoStorages.Add(sd);

					//bool attachIsoAllowed =
					//    (attachedDataStorages.Count > 0
					//     && attachedDataStorages.Any(a => a.storage_domain_type == StorageDomainType.Master && a.status.HasValue && a.status.Value == StorageDomainStatus.Active)
					//     && isoStorages.Count() > 0
					//     && attachedIsoStorages.Count == 0);
					bool attachIsoAllowed =
						(attachedDataStorages.Count > 0
						 && Linq.IsAnyStorageDomainIsMatserAndActive(attachedDataStorages)
						 && isoStorages.Count > 0
						 && attachedIsoStorages.Count == 0
						 && upHosts.Count > 0);

					// The action is available if there are no storages attached to the 
					// Data Center. It will not always be allowed.
					bool attachIsoAvailable = attachedIsoStorages.Count == 0;

					UICommand attachIsoStorageAction =
						new UICommand("AttachIsoStorage", this)
							{
								IsExecutionAllowed = attachIsoAllowed,
								IsAvailable = attachIsoAvailable,
							};
					if (upHosts.Count == 0)
					{
						attachIsoStorageAction.ExecuteProhibitionReasons.Add(NoUpHostReason);
					}

					if (attachIsoAvailable)
					{
						attachIsoStorageAction.Title = DataCenterAttachISOLibraryAction;
						OptionalActions.Add(attachIsoStorageAction);
					}
				}
				else
				{
					List<VDSGroup> clusters = DataProvider.GetClusterList(Entity.Id);

					UICommand addClusterAction = new UICommand("AddCluster", this);
					if (clusters.Count == 0)
					{
						addClusterAction.Title = DataCenterConfigureClustersAction;
						CompulsoryActions.Add(addClusterAction);
					}
					else
					{
						UICommand addHostAction = new UICommand("AddHost", this) { Title = DataCenterConfigureHostsAction };
						UICommand selectHost = new UICommand("SelectHost", this) { Title = DataCenterSelectHostsAction };
						VdcQueryReturnValue retVal = Frontend.RunQuery(VdcQueryType.Search, new SearchParameters("Hosts: datacenter!= " + Entity.name + " status=maintenance or status=pendingapproval ", SearchType.VDS));
						bool hasMaintenance3_0Host = false;
						if (retVal != null && retVal.Succeeded)
						{
							List<VDS> list = (List<VDS>)retVal.ReturnValue;
							Version version3_0 = new Version(3, 0);
							foreach (VDS vds in list)
							{
								string[] hostVersions = vds.supported_cluster_levels.Split(',');
								foreach (string hostVersion in hostVersions)
								{
									if (version3_0.CompareTo(new Version(hostVersion)) <= 0)
									{
										hasMaintenance3_0Host = true;
										break;
									}
								}
								if (hasMaintenance3_0Host)
								{
									break;
								}
							}
						}
						
						VDS host = DataProvider.GetLocalStorageHost(Entity.name);
						if (host != null)
						{
							addHostAction.IsExecutionAllowed = false;
							selectHost.IsExecutionAllowed = false;
							string hasHostReason = "Local Data Center already contains a Host";
							addHostAction.ExecuteProhibitionReasons.Add(hasHostReason);
							selectHost.ExecuteProhibitionReasons.Add(hasHostReason);
							if (host.status == VDSStatus.Up)
							{
								UICommand addLocalStorageAction = new UICommand("AddLocalStorage", this) { Title = "Add Local Storage" };
								OptionalActions.Add(addLocalStorageAction);
							}
						}
						else if(Entity.status != StoragePoolStatus.Uninitialized)
						{
							addHostAction.IsExecutionAllowed = false;
							selectHost.IsExecutionAllowed = false;
							string dataCenterInitializeReason = "Data Center was already initialized";
							addHostAction.ExecuteProhibitionReasons.Add(dataCenterInitializeReason);
							selectHost.ExecuteProhibitionReasons.Add(dataCenterInitializeReason);
						}

						if (hasMaintenance3_0Host)
						{
							OptionalActions.Add(selectHost);
						}
						CompulsoryActions.Add(addHostAction);
					}
				}
			}
		}

		private void AddLocalStorage()
		{
			StorageModel model = new StorageModel(new NewEditStorageModelBehavior());
			Window = model;
			model.Title = "New Local Domain";
			model.HashName = "new_local_domain";
			LocalStorageModel localStorageModel = new LocalStorageModel();
			localStorageModel.Role = StorageDomainType.Data;

			List<IStorageModel> list = new List<IStorageModel>();
			list.Add(localStorageModel);
			model.Items = list;
			model.SelectedItem = list[0];

			VDS localHost = DataProvider.GetLocalStorageHost(Entity.name);
			model.Host.Items = new List<VDS>() { localHost };
			model.Host.SelectedItem = localHost;
			model.DataCenter.Items = new List<storage_pool>() { Entity };
			model.DataCenter.SelectedItem = Entity;
			model.Commands.Add(
				new UICommand("OnAddStorage", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void AddIsoStorage()
		{
			AddStorageInternal("New ISO Library", StorageDomainType.ISO);
		}

		public void AddDataStorage()
		{
			AddStorageInternal("New Storage", StorageDomainType.Data);
		}

		private void AddStorageInternal(string title, StorageDomainType type)
		{
			StorageModel model = new StorageModel(new NewEditStorageModelBehavior());
			Window = model;
			model.Title = title;
			model.HashName = "new_domain";
			model.DataCenter.SelectedItem = Entity;
			model.DataCenter.IsChangable = false;

			List<IStorageModel> items = new List<IStorageModel>();

			if (type == StorageDomainType.Data)
			{
				items.Add(new NfsStorageModel { Role = StorageDomainType.Data });
				items.Add(new IscsiStorageModel { Role = StorageDomainType.Data, IsGrouppedByTarget = true });
				items.Add(new FcpStorageModel { Role = StorageDomainType.Data });
				items.Add(new LocalStorageModel { Role = StorageDomainType.Data });
			}
			else if (type == StorageDomainType.ISO)
			{
				items.Add(new NfsStorageModel { Role = StorageDomainType.ISO });
			}

			model.Items = items;

			model.Initialize();


			model.Commands.Add(
				new UICommand("OnAddStorage", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnAddStorage()
		{
			StorageModel model = (StorageModel)Window;

			if (!model.Validate())
			{
				return;
			}

			//Save changes.
			if (model.SelectedItem is NfsStorageModel)
			{
				SaveNfsStorage();
			}
			else if (model.SelectedItem is LocalStorageModel)
			{
				SaveLocalStorage();
			}
			else
			{
				SaveSanStorage();
			}
		}

		private void SaveLocalStorage()
		{
			if (Window.Progress != null)
			{
				return;
			}

			Window.StartProgress(null);

			Task.Create(this, new List<object> { "SaveLocal" }).Run();
		}

		private void SaveLocalStorage(TaskContext context)
		{
			StorageModel model = (StorageModel)Window;
			LocalStorageModel localModel = (LocalStorageModel)model.SelectedItem;
			VDS host = (VDS)model.Host.SelectedItem;


			string path = (string)localModel.Path.Entity;
			string storageName;
			if (DataProvider.IsDomainAlreadyExist(host.storage_pool_id, path, out storageName))
			{
				context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					false,
					localModel,
					"Create Operation failed. Domain " + storageName + " already exists in the system."
				});

				return;
			}


			//Create storage connection.
			storage_server_connections connection =
				new storage_server_connections
				{
					connection = path,
					storage_type = localModel.Type
				};

			storage_domain_static storageDomain =
				new storage_domain_static
				{
					storage_name = (string)model.Name.Entity,
					storage_type = localModel.Type,
					storage_domain_type = localModel.Role
				};

			bool removeConnection = false;

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.AddStorageServerConnection,
				new StorageServerConnectionParametersBase(
					connection,
					host.vds_id
				)
			);

			if (returnValue != null && returnValue.Succeeded)
			{
				storageDomain.storage = (string)returnValue.ActionReturnValue;

				//Add storage domain.
				returnValue = Frontend.RunAction(VdcActionType.AddLocalStorageDomain,
					new StorageDomainManagementParameter(storageDomain)
					{
						VdsId = host.vds_id
					}
				);

				if (returnValue == null || !returnValue.Succeeded)
				{
					removeConnection = true;
				}
			}

			//Clean up connection in case of storage creation failure.
			if (removeConnection)
			{
				Frontend.RunAction(VdcActionType.RemoveStorageServerConnection,
					new StorageServerConnectionParametersBase(connection, host.vds_id));
			}


			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					returnValue != null && returnValue.Succeeded,
					localModel,
					null
				});
		}

		private void SaveNfsStorage()
		{
			if (Window.Progress != null)
			{
				return;
			}

			Window.StartProgress(null);

			Task.Create(this, new List<object> { "SaveNfs" }).Run();
		}

		private void SaveNfsStorage(TaskContext context)
		{
			StorageModel model = (StorageModel)Window;
			NfsStorageModel nfsModel = (NfsStorageModel)model.SelectedItem;
			VDS host = (VDS)model.Host.SelectedItem;


			string path = (string)nfsModel.Path.Entity;
			string storageName;
			if (DataProvider.IsDomainAlreadyExist(path, out storageName))
			{
				context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					false,
					nfsModel,
					"Create Operation failed. Domain " + storageName + " already exists in the system."
				});

				return;
			}


			//Create storage connection.
			storage_server_connections connection =
				new storage_server_connections
				{
					connection = path,
					storage_type = nfsModel.Type
				};

			storage_domain_static storageDomain =
				new storage_domain_static
				{
					storage_type = nfsModel.Type,
					storage_domain_type = nfsModel.Role,
					storage_name = (string)model.Name.Entity,
					StorageFormat = (StorageFormatType)model.Format.SelectedItem
				};

			bool attach = false;

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.AddStorageServerConnection,
				new StorageServerConnectionParametersBase(
					connection,
					host.vds_id
				)
			);

			if (returnValue != null && returnValue.Succeeded)
			{
				storageDomain.storage = (string)returnValue.ActionReturnValue;

				//Add storage domain.
				returnValue = Frontend.RunAction(VdcActionType.AddNFSStorageDomain,
					new StorageDomainManagementParameter(storageDomain)
					{
						VdsId = host.vds_id
					}
				);

				attach = returnValue != null && returnValue.Succeeded;
			}

			//Clean up connection.
			Frontend.RunAction(VdcActionType.RemoveStorageServerConnection,
				new StorageServerConnectionParametersBase(connection, host.vds_id));

			if (attach)
			{
				//Attach storage to data center as neccessary.
				storage_pool dataCenter = (storage_pool)model.DataCenter.SelectedItem;
				if (!dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId))
				{
					nguid storageId = (nguid)returnValue.ActionReturnValue;
					AttachStorageToDataCenter((Guid)storageId, dataCenter.Id);
				}
			}


			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					returnValue != null && returnValue.Succeeded,
					nfsModel,
					null
				});
		}

		private void SaveSanStorage()
		{
			if (Window.Progress != null)
			{
				return;
			}

			Window.StartProgress(null);

			Task.Create(this, new List<object> { "SaveSan" }).Run();
		}

		private void SaveSanStorage(TaskContext context)
		{
			StorageModel model = (StorageModel)Window;
			SanStorageModel sanModel = (SanStorageModel)model.SelectedItem;
			VDS host = (VDS)model.Host.SelectedItem;

			List<string> lunIds = new List<string>();
			foreach (LunModel lun in sanModel.AddedLuns)
			{
				lunIds.Add(lun.LunId);
			}

			storage_domain_static storageDomain =
				new storage_domain_static
				{
					storage_name = (string)model.Name.Entity,
					storage_type = sanModel.Type,
					storage_domain_type = sanModel.Role,
					StorageFormat = (StorageFormatType)sanModel.Container.Format.SelectedItem
				};

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.AddSANStorageDomain,
				new AddSANStorageDomainParameters(storageDomain)
				{
					VdsId = host.vds_id,
					LunIds = lunIds
				});

			if (returnValue != null && returnValue.Succeeded)
			{
				//Attach storage to data center as neccessary.
				storage_pool dataCenter = (storage_pool)model.DataCenter.SelectedItem;
				if (!dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId))
				{
					nguid storageId = (nguid)returnValue.ActionReturnValue;
					AttachStorageToDataCenter((Guid)storageId, dataCenter.Id);
				}
			}


			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					returnValue != null && returnValue.Succeeded,
					sanModel,
					null
				});
		}

		private void AttachStorageInternal(IList<storage_domains> storages, string title)
		{
			ListModel model = new ListModel();
			model.Title = title;
			Window = model;
			//var items = storages.Select(a => new EntityModel() { Entity = a }).ToList();
			List<EntityModel> items = new List<EntityModel>();
			foreach (storage_domains sd in storages)
			{
				items.Add(new EntityModel { Entity = sd });
			}

			model.Items = items;


			model.Commands.Add(
				new UICommand("OnAttachStorage", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void AttachStorageToDataCenter(Guid storageId, Guid dataCenterId)
		{
			Frontend.RunActionAsyncroniousely(VdcActionType.AttachStorageDomainToPool,
				new StorageDomainPoolParametersBase(storageId, dataCenterId));
		}

		public void OnAttachStorage()
		{
			ListModel model = (ListModel)Window;

			//var items = model.Items
			//    .Cast<EntityModel>()
			//    .Where(Selector.GetIsSelected)
			//    .Select(a => (storage_domains)a.Entity)
			//    .ToList();
			List<storage_domains> items = new List<storage_domains>();
			foreach (EntityModel a in Linq.Cast<EntityModel>(model.Items))
			{
				if (a.IsSelected)
				{
					items.Add((storage_domains)a.Entity);
				}
			}

			if (items.Count > 0)
			{
				//items.Select(a => (VdcActionParametersBase)new StorageDomainPoolParametersBase(a.id, Entity.id))
				//    .Each(a => Frontend.RunAction(VdcActionType.AttachStorageDomainToPool, a));
				foreach (storage_domains sd in items)
				{
					Frontend.RunAction(VdcActionType.AttachStorageDomainToPool,
						new StorageDomainPoolParametersBase(sd.id, Entity.Id));
				}
			}

			Cancel();
			PostAction();
		}

		public void AttachIsoStorage()
		{
			List<storage_domains> attachedStorage = DataProvider.GetStorageDomainList(Entity.Id);

			//AttachStorageInternal(DataProvider.GetISOStorageDomainList(Entity.id)
			//    .Where(a => attachedStorage.All(b => b.id != a.id))
			//    .ToList(),
			//    "Attach ISO Library");
			List<storage_domains> sdl = new List<storage_domains>();
			foreach (storage_domains a in DataProvider.GetISOStorageDomainList())
			{
				bool isContains = false;
				foreach (storage_domains b in attachedStorage)
					if (b.id == a.id)
					{
						isContains = true;
						break;
					}
				if (!isContains) sdl.Add(a);
			}
			AttachStorageInternal(sdl, "Attach ISO Library");
		}

		public void AttachDataStorage()
		{
			List<storage_domains> unattachedStorage = new List<storage_domains>();
			bool addToList;
			Version version3_0 = new Version(3, 0);
			foreach (storage_domains item in DataProvider.GetStorageDomainList())
			{
				addToList = false;
				if (item.storage_domain_type == StorageDomainType.Data &&
					item.storage_type == Entity.storage_pool_type &&
					  item.storage_domain_shared_status == StorageDomainSharedStatus.Unattached)
				{
					if (Entity.StoragePoolFormatType == null)
					{
						//compat logic: in case its not v1 and the version is less than 3.0 continue.
						if (item.StorageStaticData.StorageFormat != StorageFormatType.V1 && Entity.compatibility_version.compareTo(version3_0) < 0)
						{
							continue;
						}
						addToList = true;
					}
					else if ((StorageFormatType)Entity.StoragePoolFormatType == item.StorageStaticData.StorageFormat)
					{
						addToList = true;
					}
				}

				if (addToList)
				{
					unattachedStorage.Add(item);
				}
			}				
			AttachStorageInternal(unattachedStorage, "Attach Storage");
		}

		public void AddCluster()
		{
			ClusterModel model = new ClusterModel();
			Window = model;
			model.Title = "New Cluster";
			model.HashName = "new_cluster";
			model.IsNew = true;

			List<storage_pool> dataCenters = new List<storage_pool>();
			dataCenters.Add(Entity);
			model.DataCenter.Items = dataCenters;
			model.DataCenter.SelectedItem = Entity;
			model.DataCenter.IsChangable = false;


			model.Commands.Add(
				new UICommand("OnAddCluster", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnAddCluster()
		{
			ClusterModel model = (ClusterModel)Window;
			VDSGroup cluster = new VDSGroup();

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			//Save changes.
			Version version = (Version)model.Version.SelectedItem;

			cluster.name = (string)model.Name.Entity;
			cluster.description = (string)model.Description.Entity;
			cluster.storage_pool_id = ((storage_pool)model.DataCenter.SelectedItem).Id;
			cluster.cpu_name = ((ServerCpu)model.CPU.SelectedItem).CpuName;
			cluster.max_vds_memory_over_commit = model.MemoryOverCommit;
			cluster.TransparentHugepages = version.CompareTo(new Version("3.0")) >= 0;
			cluster.compatibility_version = version;


			model.StartProgress(null);

			Frontend.RunAction(VdcActionType.AddVdsGroup,
				new VdsGroupOperationParameters(cluster),
				result =>
				{
					DataCenterGuideModel localModel = (DataCenterGuideModel)result.State;

					localModel.PostOnAddCluster(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnAddCluster(VdcReturnValueBase returnValue)
		{
			ClusterModel model = (ClusterModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
				PostAction();
			}
		}

		public void SelectHost()
		{
			List<VDSGroup> clusters = DataProvider.GetClusterList(Entity.Id);

			MoveHost model = new MoveHost();
			model.Title = "Select Host";
			model.HashName = "select_host";
			Window = model;
			model.Cluster.Items = clusters;
			model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters);


			model.Commands.Add(
				new UICommand("OnSelectHost", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSelectHost()
		{
			MoveHost model = (MoveHost)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			model.SelectedHosts = new List<VDS>();
			foreach (EntityModel a in Linq.Cast<EntityModel>(model.Items))
			{
				if (a.IsSelected)
				{
					model.SelectedHosts.Add((VDS)a.Entity);
				}
			}

			VDSGroup cluster = (VDSGroup)model.Cluster.SelectedItem;
			List<VdcActionParametersBase> paramerterList = new List<VdcActionParametersBase>();
			foreach (VDS host in model.SelectedHosts)
			{
				//Try to change host's cluster as neccessary.
				if (host.vds_group_id != null && !host.vds_group_id.Equals(cluster.ID))
				{
					paramerterList.Add(new ChangeVDSClusterParameters(cluster.ID, host.vds_id));

				}
			}

			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.ChangeVDSCluster, paramerterList,
									   result =>
									   {
										   DataCenterGuideModel dataCenterGuideModel = (DataCenterGuideModel)result.State;
										   List<VDS> hosts = ((MoveHost)dataCenterGuideModel.Window).SelectedHosts;
										   List<VdcReturnValueBase> retVals = (List<VdcReturnValueBase>)result.ReturnValue;
										   if (retVals != null && hosts.Count == retVals.Count)
										   {
											   int i = 0;
											   foreach (VDS selectedHost in hosts)
											   {
												   if (selectedHost.status == VDSStatus.PendingApproval && retVals[i] != null && retVals[i].Succeeded)
												   {
													   Frontend.RunAction(VdcActionType.ApproveVds,
																		  new ApproveVdsParameters(selectedHost.vds_id));
												   }
											   }
											   i++;
										   }
										   dataCenterGuideModel.Window.StopProgress();
										   dataCenterGuideModel.Cancel();
										   dataCenterGuideModel.PostAction();
									   }
									   , this);
		}

		public void AddHost()
		{
			HostModel model = new HostModel();
			Window = model;
			model.Title = "New Host";
			model.HashName = "new_host_guide_me";
			model.Port.Entity = 54321;
			model.OverrideIpTables.Entity = true;

			model.DataCenter.Items = new List<storage_pool> { Entity };
			model.DataCenter.SelectedItem = Entity;
			model.DataCenter.IsChangable = false;


			model.Commands.Add(
				new UICommand("OnConfirmPMHost", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnConfirmPMHost()
		{
			HostModel model = (HostModel)Window;

			if (!model.Validate())
			{
				return;
			}

			if (!((bool)model.IsPm.Entity))
			{
				ConfirmationModel confirmModel = new ConfirmationModel();
				ConfirmWindow = confirmModel;
				confirmModel.Title = "Power Management Configuration";
				confirmModel.HashName = "power_management_configuration";
				confirmModel.Message = "You haven't configured Power Management for this Host. Are you sure you want to continue?";

				confirmModel.Commands.Add(
					new UICommand("OnAddHost", this)
					{
						Title = "OK",
						IsDefault = true
					});
				confirmModel.Commands.Add(
					new UICommand("CancelConfirmWithFocus", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
			else
			{
				OnAddHost();
			}
		}

		public void OnAddHost()
		{
			CancelConfirm();

			HostModel model = (HostModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			//Save changes.
			VDS host = new VDS();
			host.vds_name = (string)model.Name.Entity;
			host.host_name = (string)model.Host.Entity;
			host.ManagmentIp = (string)model.ManagementIp.Entity;
			host.port = Convert.ToInt32(model.Port.Entity.ToString());
			host.vds_group_id = ((VDSGroup)model.Cluster.SelectedItem).ID;
			host.pm_enabled = (bool)model.IsPm.Entity;
			host.pm_user = (bool)model.IsPm.Entity ? (string)model.PmUserName.Entity : null;
			host.pm_password = (bool)model.IsPm.Entity ? (string)model.PmPassword.Entity : null;
			host.pm_type = (bool)model.IsPm.Entity ? (string)model.PmType.SelectedItem : null;
			host.PmOptionsMap = (bool)model.IsPm.Entity ? new valueObjectMap(model.PmOptionsMap, false) : null;

			AddVdsActionParameters addVdsParams = new AddVdsActionParameters();
			addVdsParams.VdsId = host.vds_id;
			addVdsParams.vds = host;
			addVdsParams.RootPassword = (string)model.RootPassword.Entity;


			model.StartProgress(null);

			Frontend.RunAction(VdcActionType.AddVds, addVdsParams,
				result =>
				{
					DataCenterGuideModel localModel = (DataCenterGuideModel)result.State;

					localModel.PostOnAddHost(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnAddHost(VdcReturnValueBase returnValue)
		{
			HostModel model = (HostModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
				PostAction();
			}
		}

		private void PostAction()
		{
			UpdateOptions();
		}

		public void Cancel()
		{
			Window = null;
		}

		public void CancelConfirm()
		{
			ConfirmWindow = null;
		}

		public void CancelConfirmWithFocus()
		{
			ConfirmWindow = null;

			HostModel hostModel = (HostModel)Window;
			hostModel.IsPowerManagementSelected = true;
			hostModel.IsPm.Entity = true;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command.Name == "AddCluster")
			{
				AddCluster();
			}

			if (command.Name == "AddHost")
			{
				AddHost();
			}

			if (command.Name == "SelectHost")
			{
				SelectHost();
			}
			if (command.Name == "AddDataStorage")
			{
				AddDataStorage();
			}
			if (command.Name == "AttachDataStorage")
			{
				AttachDataStorage();
			}
			if (command.Name == "AddIsoStorage")
			{
				AddIsoStorage();
			}
			if (command.Name == "AttachIsoStorage")
			{
				AttachIsoStorage();
			}
			if (command.Name == "OnAddCluster")
			{
				OnAddCluster();
			}
			if (command.Name == "OnSelectHost")
			{
				OnSelectHost();
			}
			if (command.Name == "OnAddHost")
			{
				OnAddHost();
			}
			if (command.Name == "OnAddStorage")
			{
				OnAddStorage();
			}

			if (command.Name == "OnAttachStorage")
			{
				OnAttachStorage();
			}

			if (command.Name == "AddLocalStorage")
			{
				AddLocalStorage();
			}

			if (command.Name == "OnConfirmPMHost")
			{
				OnConfirmPMHost();
			}

			if (command.Name == "CancelConfirm")
			{
				CancelConfirm();
			}
			if (command.Name == "CancelConfirmWithFocus")
			{
				CancelConfirmWithFocus();
			}

			if (command.Name == "Cancel")
			{
				Cancel();
			}
		}

		public void run(TaskContext context)
		{
			List<object> data = (List<object>)context.State;
			string key = (string)data[0];

			switch (key)
			{
				case "SaveNfs":
					SaveNfsStorage(context);
					break;

				case "SaveLocal":
					SaveLocalStorage(context);
					break;

				case "SaveSan":
					SaveSanStorage(context);
					break;

				case "Finish":
					Window.StopProgress();

					if ((bool)data[1])
					{
						Cancel();
						PostAction();
					}
					else
					{
						((Model)data[2]).Message = (string)data[3];
					}
					break;
			}
		}
	}
}
