using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.hosts
{
	public class HostInterfaceListModel : SearchableListModel
	{
		#region Commands

		public UICommand EditCommand { get; private set; }
		public UICommand EditManagementNetworkCommand { get; private set; }
		public UICommand BondCommand { get; private set; }
		public UICommand DetachCommand { get; private set; }
		public UICommand SaveNetworkConfigCommand { get; private set; }

		#endregion

		#region Properties

		public new VDS Entity
		{
			get { return (VDS)base.Entity; }
			set { base.Entity = value; }
		}

		private List<VdsNetworkInterface> GetSelectedItems(bool withVlans)
		{
			List<VdsNetworkInterface> list = new List<VdsNetworkInterface>();
			if (Items != null)
			{
				bool isModelSelected;
				foreach (object item in Items)
				{
					isModelSelected = false;
					HostInterfaceLineModel model = (HostInterfaceLineModel)item;
					if (model.IsBonded)
					{
						if (model.IsSelected)
						{
							isModelSelected = true;
							list.Add(model.Interface);
						}
					}
					else
					{
						foreach (HostInterface hostInterface in model.Interfaces)
						{
							if (hostInterface.IsSelected)
							{
								isModelSelected = true;
								list.Add(hostInterface.Interface);
							}
						}
					}

					foreach (HostVLan vLan in model.VLans)
					{
						if (vLan.IsSelected || (withVlans && isModelSelected))
						{
							list.Add(vLan.Interface);
						}
					}
				}
			}

			return list;
		}

		public new List<VdsNetworkInterface> SelectedItems
		{
			get { return GetSelectedItems(false); }
		}
		public new List<VdsNetworkInterface> SelectedItemsWithVlans
		{
			get { return GetSelectedItems(true); }
		}

		private List<VdsNetworkInterface> InterfaceItems
		{
			get
			{
				List<VdsNetworkInterface> list = new List<VdsNetworkInterface>();
				if (Items != null)
				{
					foreach (object item in Items)
					{
						HostInterfaceLineModel model = (HostInterfaceLineModel)item;
						foreach (HostInterface hostInterface in model.Interfaces)
						{
							list.Add(hostInterface.Interface);
						}
					}
				}

				return list;
			}
		}

		private List<VdsNetworkInterface> AllItems
		{
			get
			{
				List<VdsNetworkInterface> list = new List<VdsNetworkInterface>();
				foreach (object a in Items)
				{
					HostInterfaceLineModel item = (HostInterfaceLineModel)a;
					if (item.IsBonded)
					{
						list.Add(item.Interface);
					}

					foreach (HostInterface hostInterface in item.Interfaces)
					{
						list.Add(hostInterface.Interface);
					}

					foreach (HostVLan vLan in item.VLans)
					{
						list.Add(vLan.Interface);
					}
				}

				return list;
			}
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		/// <summary>
		/// Gets a boolean value indicating whether a detach confirmation
		/// is needed (cuurently happens when attempting to change the Management
		/// Interface's network to 'None').
		/// </summary>
		//TODO: Suspect this property is obsolete.
		private bool detachConfirmationNeeded;
		public bool DetachConfirmationNeeded
		{
			get { return detachConfirmationNeeded; }

			private set
			{
				if (detachConfirmationNeeded != value)
				{
					detachConfirmationNeeded = value;
					OnPropertyChanged(new PropertyChangedEventArgs("DetachConfirmationNeeded"));
				}
			}
		}

		public Model currentModel { get; set; }

		#endregion

		public HostInterfaceListModel()
		{
			Title = "Network Interfaces";

			EditCommand = new UICommand("Edit", this);
			EditManagementNetworkCommand = new UICommand("EditManagementNetwork", this);
			BondCommand = new UICommand("Bond", this);
			DetachCommand = new UICommand("Detach", this);
			SaveNetworkConfigCommand = new UICommand("SaveNetworkConfig", this);

			UpdateActionAvailability();
		}

		public override void Search()
		{
			if (Entity != null)
			{
				base.Search();
			}
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			SearchCommand.Execute();
			UpdateActionAvailability();
		}

		//protected override void OnSelectedItemChanged()
		//{
		//    base.OnSelectedItemChanged();
		//    UpdateActionAvailability();
		//}

		//protected override void SelectedItemsChanged()
		//{
		//    base.SelectedItemsChanged();
		//    UpdateActionAvailability();
		//}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "status" || e.PropertyName == "net_config_dirty")
			{
				UpdateActionAvailability();
			}
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVdsInterfacesByVdsId,
				new GetVdsByVdsIdParameters(Entity.vds_id));

			if (returnValue != null && returnValue.Succeeded)
			{
				List<VdsNetworkInterface> items = new List<VdsNetworkInterface>();
				foreach (IVdcQueryable item in ((List<IVdcQueryable>)returnValue.ReturnValue))
				{
					VdsNetworkInterface i = (VdsNetworkInterface)item;
					items.Add(i);
				}

				UpdateItems(items);
			}
			else
			{
				UpdateItems(new List<VdsNetworkInterface>());
			}
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			SyncSearch();
		}

		private void UpdateItems(IEnumerable<VdsNetworkInterface> source)
		{
			List<HostInterfaceLineModel> items = new List<HostInterfaceLineModel>();

			//Add bonded interfaces.
			foreach (VdsNetworkInterface nic in source)
			{
				if (nic.Bonded.GetValueOrDefault())
				{
					HostInterfaceLineModel model = new HostInterfaceLineModel();
					model.Interfaces = new List<HostInterface>();
					model.Interface = nic;
					model.VLans = new List<HostVLan>();
					model.IsBonded = true;
					model.BondName = nic.Name;
					model.Address = nic.Address;
					model.NetworkName = nic.NetworkName;
					model.IsManagement = nic.IsManagement;

					items.Add(model);
				}
			}

			//Find for each bond containing interfaces.
			foreach (HostInterfaceLineModel model in items)
			{
				if (model.IsBonded)
				{
					foreach (VdsNetworkInterface nic in source)
					{
						if (nic.BondName == model.BondName)
						{
							HostInterface hi = new HostInterface();
							hi.Interface = nic;
							hi.Name = nic.Name;
							hi.Address = nic.Address;
							hi.MAC = nic.MacAddress;
							hi.Speed = nic.Speed;
							hi.RxRate = nic.Statistics.ReceiveRate;
							hi.RxDrop = nic.Statistics.ReceiveDropRate;
							hi.TxRate = nic.Statistics.TransmitRate;
							hi.TxDrop = nic.Statistics.TransmitDropRate;
							hi.Status = nic.Statistics.Status;
							hi.PropertyChangedEvent.addListener(this);

							model.Interfaces.Add(hi);
						}
					}
				}
			}

			//Add not bonded interfaces with no vlan.
			foreach (VdsNetworkInterface nic in source)
			{
				if (!nic.Bonded.GetValueOrDefault() && String.IsNullOrEmpty(nic.BondName) && !nic.VlanId.HasValue)
				{
					HostInterfaceLineModel model = new HostInterfaceLineModel();
					model.Interfaces = new List<HostInterface>();
					model.VLans = new List<HostVLan>();
					model.NetworkName = nic.NetworkName;
					model.IsManagement = nic.IsManagement;

					//There is only one interface.
					HostInterface hi = new HostInterface();
					hi.Interface = nic;
					hi.Name = nic.Name;
					hi.Address = nic.Address;
					hi.MAC = nic.MacAddress;
					hi.Speed = nic.Speed;
					hi.RxRate = nic.Statistics.ReceiveRate;
					hi.RxDrop = nic.Statistics.ReceiveDropRate;
					hi.TxRate = nic.Statistics.TransmitRate;
					hi.TxDrop = nic.Statistics.TransmitDropRate;
					hi.Status = nic.Statistics.Status;
					hi.PropertyChangedEvent.addListener(this);

					model.Interfaces.Add(hi);

					items.Add(model);
				}
			}

			//Find vlans.
			foreach (HostInterfaceLineModel model in items)
			{
				string nicName = model.IsBonded ? model.BondName : model.Interfaces[0].Name;

				foreach (VdsNetworkInterface nic in source)
				{
					if (nic.VlanId.HasValue && String.Format("{0}.{1}", nicName, nic.VlanId.Value) == nic.Name)
					{
						HostVLan hv = new HostVLan();
						hv.Interface = nic;
						hv.Name = nic.Name;
						hv.NetworkName = nic.NetworkName;
						hv.Address = nic.Address;
						hv.PropertyChangedEvent.addListener(this);

						model.VLans.Add(hv);
					}
				}

				List<HostVLan> list = model.VLans;
				list.Sort(new HostVLanByNameComparer());
			}


			Items = items;
			UpdateActionAvailability();
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ProvidePropertyChangedEvent.Definition))
			{
				Model_PropertyChanged(sender, (PropertyChangedEventArgs)args);
			}
			else if (sender is Model && ((Model)sender).Title == "InterfaceList")
			{
				HostManagementNetworkModel managementNetworkModel = ((HostManagementNetworkModel)Window);
				VdsNetworkInterface vdsNetworkInterface = (VdsNetworkInterface)managementNetworkModel.Interface.SelectedItem;
				if (vdsNetworkInterface.Bonded != null && vdsNetworkInterface.Bonded == true)
				{
					managementNetworkModel.BondingOptions.IsChangable = true;
				}
				else
				{
					managementNetworkModel.BondingOptions.IsChangable = false;
				}
			}
		}

		private void Model_PropertyChanged(object sender, PropertyChangedEventArgs args)
		{
			if (args.PropertyName != "IsSelected")
			{
				return;
			}


			if (sender is HostInterfaceLineModel)
			{
				HostInterfaceLineModel model = (HostInterfaceLineModel)sender;
				foreach (HostInterface hostInterface in model.Interfaces)
				{
					hostInterface.PropertyChangedEvent.removeListener(this);
					hostInterface.IsSelected = model.IsSelected;
					hostInterface.PropertyChangedEvent.addListener(this);
				}

				foreach (HostVLan vLan in model.VLans)
				{
					vLan.PropertyChangedEvent.removeListener(this);
					vLan.IsSelected = false;
					vLan.PropertyChangedEvent.addListener(this);
				}

				if (model.IsSelected)
				{
					if (model.IsBonded)
					{
						SelectedItem = model.Interface;
					}
					else
					{
						SelectedItem = model.Interfaces[0].Interface;
					}
				}
			}
			else if (sender is HostInterface)
			{
				HostInterface model = (HostInterface)sender;
				object selectItem = null;

				if (model.IsSelected)
				{
					selectItem = model.Interface;
				}

				foreach (object a in Items)
				{
					HostInterfaceLineModel item = (HostInterfaceLineModel)a;
					bool found = false;

					foreach (HostInterface hostInterface in item.Interfaces)
					{
						if (hostInterface == model)
						{
							item.PropertyChangedEvent.removeListener(this);
							item.IsSelected = model.IsSelected;
							item.PropertyChangedEvent.addListener(this);

							if (item.IsBonded && item.IsSelected)
							{
								selectItem = item.Interface;
							}

							foreach (HostVLan vLan in item.VLans)
							{
								vLan.PropertyChangedEvent.removeListener(this);
								vLan.IsSelected = false;
								vLan.PropertyChangedEvent.addListener(this);
							}

							found = true;

							break;
						}
					}

					if (found)
					{
						foreach (HostInterface hostInterface in item.Interfaces)
						{
							hostInterface.PropertyChangedEvent.removeListener(this);
							hostInterface.IsSelected = model.IsSelected;
							hostInterface.PropertyChangedEvent.addListener(this);
						}
					}
				}

				if (selectItem != null)
				{
					SelectedItem = selectItem;
				}
			}
			else if (sender is HostVLan)
			{
				HostVLan model = (HostVLan)sender;

				foreach (object a in Items)
				{
					HostInterfaceLineModel item = (HostInterfaceLineModel)a;
					foreach (HostVLan vLan in item.VLans)
					{
						if (vLan == model)
						{
							foreach (HostInterface hostInterface in item.Interfaces)
							{
								hostInterface.PropertyChangedEvent.removeListener(this);
								hostInterface.IsSelected = false;
								hostInterface.PropertyChangedEvent.addListener(this);
							}

							item.PropertyChangedEvent.removeListener(this);
							item.IsSelected = false;
							item.PropertyChangedEvent.addListener(this);

							break;
						}
					}
				}

				if (model.IsSelected)
				{
					SelectedItem = model.Interface;
				}
			}

			if (SelectedItems.Count == 0)
			{
				SelectedItem = null;
			}
			else
			{
				//Check whether the SelectedItem is still a one from SelectedItems. If not, choose the first one.
				bool found = false;
				foreach (VdsNetworkInterface item in SelectedItems)
				{
					if (item == SelectedItem)
					{
						found = true;
						break;
					}
				}

				if (!found)
				{
					SelectedItem = Linq.FirstOrDefault(SelectedItems);
				}
			}


			UpdateActionAvailability();
		}

		private List<string> GetSelectedNicsNetworks(out bool isVlanSelected, out bool isManagementSelected)
		{
			List<VdsNetworkInterface> selectedItems = SelectedItemsWithVlans;
			List<string> list = new List<string>();
			isVlanSelected = false;
			isManagementSelected = false;
			foreach (VdsNetworkInterface nic in selectedItems)
			{
				if (!String.IsNullOrEmpty(nic.NetworkName))
				{
					if (nic.IsManagement)
					{
						isManagementSelected = true;
					}
					list.Add(nic.NetworkName);
					if (!isVlanSelected && nic.VlanId.HasValue)
					{
						isVlanSelected = true;
					}
				}
			}
			return list;
		}

		public void Edit()
		{
			VdsNetworkInterface item = (VdsNetworkInterface)SelectedItem;

			if (Window != null)
			{
				return;
			}

			List<network> networksToAdd = new List<network>();
			network selectedNetwork = null;

			if (item.VlanId.HasValue)
			{
				List<network> networksByCluster = DataProvider.GetClusterNetworkList(Entity.vds_group_id);
				foreach (network network in networksByCluster)
				{
					if (network.name == item.NetworkName)
					{
						networksToAdd.Add(network);
						if (selectedNetwork == null)
						{
							selectedNetwork = network;
						}
					}
				}
			}
			else
			{
				networksToAdd = GetNetworksList(out selectedNetwork);
			}
			networksToAdd.Sort(new Linq.NetworkByNameComparer());

			//Add a 'none' option to networks.
			if (!String.IsNullOrEmpty(item.NetworkName))
			{
				networksToAdd.Insert(0, new network { Id = Guid.Empty, name = "None" });
			}

			HostInterfaceModel model = new HostInterfaceModel();
			Window = model;
			model.Entity = item.Name;
			model.Title = "Edit Network Interface";
			model.HashName = "edit_network_interface_hosts";

			model.Networks = SelectedItemsWithVlans;

			model.NoneBootProtocolAvailable = !item.IsManagement;
			model.BootProtocol =
				!model.NoneBootProtocolAvailable && item.BootProtocol == NetworkBootProtocol.None ?
				NetworkBootProtocol.Dhcp :
				item.BootProtocol;

			model.Address.Entity = item.Address;
			model.Subnet.Entity = item.Subnet;

			model.Network.Items = networksToAdd;
			model.Name = item.Name;

			model.BondingOptions.IsAvailable = false;
			if (item.Bonded != null && item.Bonded == true)
			{
				model.BondingOptions.IsAvailable = true;
				KeyValuePair<string, EntityModel> defaultItem;
				List<KeyValuePair<string, EntityModel>> list = DataProvider.GetBondingOptionList(out defaultItem);
				KeyValuePair<string, EntityModel> selectBondingOpt = new KeyValuePair<string, EntityModel>();
				bool containsSelectBondingOpt = false;
				model.BondingOptions.Items = list;
				for (int i = 0; i < list.Count; i++)
				{
					if (list[i].Key == item.BondOptions)
					{
						selectBondingOpt = list[i];
						containsSelectBondingOpt = true;
						break;
					}
				}
				if (containsSelectBondingOpt == false)
				{
					if (item.BondOptions == DataProvider.GetDefaultBondingOption())
					{
						selectBondingOpt = defaultItem;
					}
					else
					{
						selectBondingOpt = list[list.Count - 1];
						EntityModel entityModel = selectBondingOpt.Value;
						entityModel.Entity = item.BondOptions;
					}
				}
				model.BondingOptions.SelectedItem = selectBondingOpt;
			}
			if (selectedNetwork == null && networksToAdd.Count > 0)
			{
				selectedNetwork = networksToAdd[0];
			}
			model.Network.SelectedItem = selectedNetwork;

			model.CheckConnectivity.IsAvailable = !string.IsNullOrEmpty(item.NetworkName) && item.IsManagement;
			model.CheckConnectivity.IsChangable = !string.IsNullOrEmpty(item.NetworkName) && item.IsManagement;
			model.CheckConnectivity.Entity = item.IsManagement;


			if (networksToAdd.Count == 0)
			{
				model.Message = "There are no networks available. Please add additional networks.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnSave", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		public void EditManagementNetwork()
		{
			VdsNetworkInterface item = (VdsNetworkInterface)SelectedItem;

			if (Window != null)
			{
				return;
			}

			List<network> clusterNetworks = DataProvider.GetClusterNetworkList(Entity.vds_group_id);
			network networkToEdit = Linq.FindNetworkByName(clusterNetworks, item.NetworkName);
			HostManagementNetworkModel model = new HostManagementNetworkModel();
			Window = model;
			model.Entity = networkToEdit;
			model.Title = "Edit Management Network";
			model.HashName = "edit_management_network";

			model.NoneBootProtocolAvailable = !item.IsManagement;
			model.BootProtocol =
				!model.NoneBootProtocolAvailable && item.BootProtocol == NetworkBootProtocol.None ?
				NetworkBootProtocol.Dhcp :
				item.BootProtocol;

			model.Address.Entity = item.Address;
			model.Subnet.Entity = item.Subnet;
			model.Gateway.Entity = item.Gateway;

			string defaultInterfaceName;
			List<VdsNetworkInterface> interfaces = DataProvider.GetInterfaceOptionsForEditNetwork(item, networkToEdit, Entity.vds_id, out defaultInterfaceName);
			model.Interface.Items = interfaces;
			model.Interface.SelectedItem = Linq.FindInterfaceByName(Linq.VdsNetworkInterfaceListToBase(interfaces), defaultInterfaceName);
			if (item.Bonded != null && item.Bonded == true)
			{
				model.Interface.Title = "InterfaceList";
				model.Interface.SelectedItemChangedEvent.addListener(this);
			}
			model.CheckConnectivity.IsAvailable = true;
			model.CheckConnectivity.IsChangable = true;
			model.CheckConnectivity.Entity = item.IsManagement;		// currently, always should be true

			model.BondingOptions.IsAvailable = false;
			if (item.Bonded != null && item.Bonded == true)
			{
				model.BondingOptions.IsAvailable = true;
				KeyValuePair<string, EntityModel> defaultItem;
				List<KeyValuePair<string, EntityModel>> list = DataProvider.GetBondingOptionList(out defaultItem);
				KeyValuePair<string, EntityModel> selectBondingOpt = new KeyValuePair<string, EntityModel>();
				bool containsSelectBondingOpt = false;
				model.BondingOptions.Items = list;
				for (int i = 0; i < list.Count; i++)
				{
					if (list[i].Key == item.BondOptions)
					{
						selectBondingOpt = list[i];
						containsSelectBondingOpt = true;
						break;
					}
				}
				if (containsSelectBondingOpt == false)
				{
					if (item.BondOptions == DataProvider.GetDefaultBondingOption())
					{
						selectBondingOpt = defaultItem;
					}
					else
					{
						selectBondingOpt = list[list.Count - 1];
						EntityModel entityModel = selectBondingOpt.Value;
						entityModel.Entity = item.BondOptions;
					}
				}
				model.BondingOptions.SelectedItem = selectBondingOpt;
			}

			model.Commands.Add(
				new UICommand("OnEditManagementNetworkConfirmation", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnEditManagementNetworkConfirmation(bool isBond)
		{
			if (!isBond)
			{
				HostManagementNetworkModel model = (HostManagementNetworkModel)Window;
				if (!model.Validate())
				{
					return;
				}
				if ((bool)model.CheckConnectivity.Entity == true)
				{
					OnEditManagementNetwork();
					return;
				}
			}
			else
			{
				HostBondInterfaceModel model = (HostBondInterfaceModel)Window;
				if (!model.Validate())
				{
					return;
				}
				if ((bool)model.CheckConnectivity.Entity == true)
				{
					OnBond();
					return;
				}
			}
			ConfirmationModel confirmModel = new ConfirmationModel();
			ConfirmWindow = confirmModel;
			confirmModel.Title = "Confirm";
			confirmModel.Latch.Entity = true;

			if (!isBond)
			{
				confirmModel.Commands.Add(
				   new UICommand("OnEditManagementNetwork", this)
					   {
						   Title = "OK",
						   IsDefault = true
					   });
			}
			else
			{
				confirmModel.Commands.Add(
				   new UICommand("OnBond", this)
				   {
					   Title = "OK",
					   IsDefault = true
				   });
			}
			confirmModel.Commands.Add(
				new UICommand("CancelConfirm", this)
					{
						Title = "Cancel",
						IsCancel = true
					});

		}

		public void OnEditManagementNetwork()
		{
			HostManagementNetworkModel model = (HostManagementNetworkModel)Window;
			if (ConfirmWindow != null)
			{
				ConfirmationModel confirmModel = (ConfirmationModel)ConfirmWindow;
				if ((bool)confirmModel.Latch.Entity == true)
				{
					model.CheckConnectivity.Entity = true;
				}
			}

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			VdsNetworkInterface nic = (VdsNetworkInterface)model.Interface.SelectedItem;
			network network = model.Entity;

			VdcActionType actionType = VdcActionType.UpdateNetworkToVdsInterface;
			UpdateNetworkToVdsParameters parameters =
					new UpdateNetworkToVdsParameters
						(Entity.vds_id, network, new List<VdsNetworkInterface> { nic });

			KeyValuePair<string, EntityModel> bondingOption;
			if (model.BondingOptions.SelectedItem != null)
			{
				bondingOption = (KeyValuePair<string, EntityModel>)model.BondingOptions.SelectedItem;

				if (bondingOption.Key != "custom")
				{
					parameters.BondingOptions = (String.IsNullOrEmpty(bondingOption.Key) ? null : bondingOption.Key);
				}
				else
				{
					EntityModel entityModel = (EntityModel)bondingOption.Value;
					if (entityModel.Entity != null)
					{
						parameters.BondingOptions = entityModel.Entity.ToString();
					}
				}
			}
			VdsNetworkInterface selectedItem = (VdsNetworkInterface)SelectedItem;
			if (!nic.Bonded.HasValue || nic.Bonded.Value == false)
			{
				parameters.BondingOptions = null;
			}

			if (network != null)
			{
				parameters.OldNetworkName = network.name;
			}
			parameters.CheckConnectivity = (bool)model.CheckConnectivity.Entity;
			parameters.BootProtocol = model.BootProtocol;

			if (model.IsStaticAddress)
			{
				parameters.Address = (string)model.Address.Entity;
				parameters.Subnet = (string)model.Subnet.Entity;
				parameters.Gateway = (string)model.Gateway.Entity;
			}

			model.StartProgress(null);
			currentModel = model;

			Frontend.RunAction(actionType, parameters,
				result =>
				{
					HostInterfaceListModel hostInterfaceListModel = (HostInterfaceListModel)result.State;

					VdcReturnValueBase returnValueBase = result.ReturnValue;
					if (returnValueBase != null && returnValueBase.Succeeded)
					{
						if (((HostManagementNetworkModel)hostInterfaceListModel.currentModel).CommitChanges)
						{
							SaveNetworkConfig(hostInterfaceListModel.Entity.vds_id, hostInterfaceListModel);
						}
						else
						{
							hostInterfaceListModel.currentModel.StopProgress();
							hostInterfaceListModel.Cancel();
							hostInterfaceListModel.Search();
						}
					}
					else
					{
						hostInterfaceListModel.currentModel.StopProgress();
					}
				},
				this
			);
			CancelConfirm();
		}

		private List<network> GetNetworksList(out network selectedNetwork)
		{
			selectedNetwork = null;
			List<network> networksByCluster = DataProvider.GetClusterNetworkList(Entity.vds_group_id);
			List<network> networksToAdd = new List<network>();
			//creating dictionary of networks by name
			Dictionary<string, network> networkDictionary = new Dictionary<string, network>();
			foreach (network network in networksByCluster)
			{
				networkDictionary.Add(network.name, network);
			}
			//creating list of attached networks.
			List<network> attachedNetworks = new List<network>();
			foreach (VdsNetworkInterface nic in AllItems)
			{
				if (nic.NetworkName != null && networkDictionary.ContainsKey(nic.NetworkName))
				{
					attachedNetworks.Add(networkDictionary[nic.NetworkName]);
				}
			}

			List<network> unAttachedNetworks = Linq.Except(networksByCluster, attachedNetworks);

			//adding selected network names to list.
			bool isVlanSelected;
			bool isManagement;
			List<string> selectedNicsNetworks = GetSelectedNicsNetworks(out isVlanSelected, out isManagement);

			foreach (string selectedNetworkName in selectedNicsNetworks)
			{
				if (networkDictionary.ContainsKey(selectedNetworkName))
				{
					network network = networkDictionary[selectedNetworkName];
					networksToAdd.Add(network);
					attachedNetworks.Remove(network);

					if (selectedNetwork == null)
					{
						selectedNetwork = network;
					}
				}
			}

			if (!(isManagement && !isVlanSelected))
			{
				foreach (network unAttachedNetwork in unAttachedNetworks)
				{
					if (isVlanSelected)
					{
						if (unAttachedNetwork.vlan_id.HasValue)
						{
							networksToAdd.Add(unAttachedNetwork);
						}
					}
					else
					{
						networksToAdd.Add(unAttachedNetwork);
					}
				}
			}
			return networksToAdd;
		}

		public void Bond()
		{
			if (Window != null)
			{
				return;
			}

			HostBondInterfaceModel model = new HostBondInterfaceModel();
			Window = model;
			model.Title = "Bond Network Interfaces";
			model.HashName = "bond_network_interfaces";

			VDS host = Entity;
			//Allow change gateway if there one of the selected interfaces connected to rhevm network.
			bool isAnyManagement = false;
			foreach (VdsNetworkInterface item in SelectedItems)
			{
				if (item.IsManagement)
				{
					isAnyManagement = true;
					break;
				}
			}
			network selectedNetwork;
			List<network> networksToAdd = GetNetworksList(out selectedNetwork);
			model.Network.Items = networksToAdd;

			if (selectedNetwork == null && networksToAdd.Count > 0)
			{
				selectedNetwork = networksToAdd[0];
			}
			model.Network.SelectedItem = selectedNetwork;

			//Interface bond = selectedItems.FirstOrDefault(a => a.is_bond.HasValue && a.is_bond.Value);
			VdsNetworkInterface bond = Linq.FindInterfaceByIsBond(SelectedItems);
			if (bond != null)
			// one of the bond items is a bond itself -> don't 
			// allocate a new bond name, edit the existing one:
			{
				model.Bond.Items = new List<VdsNetworkInterface> { bond };
				model.Bond.SelectedItem = bond;
				model.Bond.IsChangable = false;
			}
			else
			{
				List<VdsNetworkInterface> bonds = DataProvider.GetFreeBondList(host.vds_id);
				model.Bond.Items = bonds;
				//((List<Interface>)model.Bond.Options).Sort(a => a.name);
				model.Bond.SelectedItem = Linq.FirstOrDefault(bonds);
			}
			List<NetworkInterface> baseSelectedItems = Linq.VdsNetworkInterfaceListToBase(SelectedItems);
			VdsNetworkInterface interfaceWithNetwork = (VdsNetworkInterface)Linq.FindInterfaceNetworkNameNotEmpty(baseSelectedItems);

			model.CheckConnectivity.IsChangable = interfaceWithNetwork != null;
			model.CheckConnectivity.IsAvailable = interfaceWithNetwork != null && interfaceWithNetwork.IsManagement;

			model.CheckConnectivity.Entity = interfaceWithNetwork != null && interfaceWithNetwork.IsManagement;
			model.NoneBootProtocolAvailable = !(interfaceWithNetwork != null && interfaceWithNetwork.IsManagement);

			if (interfaceWithNetwork != null)
			{
				model.BootProtocol =
					!model.NoneBootProtocolAvailable && interfaceWithNetwork.BootProtocol == NetworkBootProtocol.None ?
					NetworkBootProtocol.Dhcp :
					interfaceWithNetwork.BootProtocol;

				model.Address.Entity = interfaceWithNetwork.Address;
				model.Subnet.Entity = interfaceWithNetwork.Subnet;
				model.Gateway.Entity = interfaceWithNetwork.Gateway;
			}
			else
			{
				model.BootProtocol = NetworkBootProtocol.Dhcp;
			}

			model.Gateway.IsAvailable = isAnyManagement;

			if (networksToAdd.Count == 0)
			{
				model.Message = "There are no networks available. Please add additional networks.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				if (interfaceWithNetwork != null && interfaceWithNetwork.IsManagement)
				{
					model.Commands.Add(
						new UICommand("OnEditManagementNetworkConfirmation_Bond", this)
						{
							Title = "OK",
							IsDefault = true
						});
					model.Commands.Add(
						new UICommand("Cancel", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
				}
				else
				{
					model.Commands.Add(
						new UICommand("OnBond", this)
						{
							Title = "OK",
							IsDefault = true
						});
					model.Commands.Add(
						new UICommand("Cancel", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
				}
			}
		}

		public void OnBond()
		{
			HostBondInterfaceModel model = (HostBondInterfaceModel)Window;

			if (ConfirmWindow != null)
			{
				ConfirmationModel confirmModel = (ConfirmationModel)ConfirmWindow;
				if ((bool)confirmModel.Latch.Entity == true)
				{
					model.CheckConnectivity.Entity = true;
				}
				CancelConfirm();
			}

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			VDS host = Entity;
			List<VdsNetworkInterface> selectedItems = SelectedItems;
			network net = (network)model.Network.SelectedItem;

			//Interface interfaceWithNetwork = items.FirstOrDefault(a => !string.IsNullOrEmpty(a.network_name));
			VdsNetworkInterface interfaceWithNetwork = (VdsNetworkInterface)Linq.FindInterfaceNetworkNameNotEmpty(Linq.VdsNetworkInterfaceListToBase(selectedItems));

			// Look for lines with vlans.
			List<HostInterfaceLineModel> itemList = (List<HostInterfaceLineModel>)Items;
			foreach (HostInterfaceLineModel lineModel in itemList)
			{
				if (lineModel.IsSelected && lineModel.VLansCount == 1)
				{
					interfaceWithNetwork = lineModel.VLans[0].Interface;
					// Bond action is only enabled if there is one vlaned interface.
					break;
				}
			}

			if (interfaceWithNetwork != null)
			{
				UpdateNetworkToVdsParameters parameters = new UpdateNetworkToVdsParameters(host.vds_id, net, selectedItems);
				parameters.CheckConnectivity = (bool)model.CheckConnectivity.Entity;
				parameters.OldNetworkName = interfaceWithNetwork.NetworkName;

				KeyValuePair<string, EntityModel> bondingOption;
				if (model.BondingOptions.SelectedItem != null)
				{
					bondingOption = (KeyValuePair<string, EntityModel>)model.BondingOptions.SelectedItem;

					if (bondingOption.Key != "custom")
					{
						parameters.BondingOptions = (String.IsNullOrEmpty(bondingOption.Key) ? null : bondingOption.Key);
					}
					else
					{
						EntityModel entityModel = (EntityModel)bondingOption.Value;
						if (entityModel.Entity != null)
						{
							parameters.BondingOptions = entityModel.Entity.ToString();
						}
					}
				}

				parameters.BootProtocol = model.BootProtocol;
				parameters.BondName = ((VdsNetworkInterface)model.Bond.SelectedItem).Name;

				if (model.IsStaticAddress)
				{
					parameters.Address = (string)model.Address.Entity;
					parameters.Subnet = (string)model.Subnet.Entity;
					if (interfaceWithNetwork.IsManagement)
					{
						parameters.Gateway = (string)model.Gateway.Entity;
					}
				}

				model.StartProgress(null);
				currentModel = model;

				Frontend.RunAction(VdcActionType.UpdateNetworkToVdsInterface, parameters,
					result =>
					{
						HostInterfaceListModel hostInterfaceListModel = (HostInterfaceListModel)result.State;
						VdcReturnValueBase returnValueBase = result.ReturnValue;
						if (returnValueBase != null && returnValueBase.Succeeded)
						{
							if (((HostBondInterfaceModel)hostInterfaceListModel.currentModel).CommitChanges)
							{
								SaveNetworkConfig(hostInterfaceListModel.Entity.vds_id, hostInterfaceListModel);
							}
							else
							{
								hostInterfaceListModel.currentModel.StopProgress();
								hostInterfaceListModel.Cancel();
								hostInterfaceListModel.Search();
							}
						}
						else
						{
							hostInterfaceListModel.currentModel.StopProgress();
						}
					},
					this
				);
			}
			else
			{
				string[] nics = new string[selectedItems.Count];
				for (int i = 0; i < selectedItems.Count; i++)
				{
					nics[i] = selectedItems[i].Name;
				}
				//var parameters = new AddBondParameters(
				//    host.vds_id,
				//    model.Bond.ValueAs<Interface>().name,
				//    net,
				//    items.Select(a => a.name).ToArray())
				//        {
				//            BondingOptions = model.BondingOptions.ValueAs<string>(),
				//            BootProtocol = model.BootProtocol
				//        };
				AddBondParameters parameters = new AddBondParameters(
					host.vds_id,
					((VdsNetworkInterface)model.Bond.SelectedItem).Name,
					net,
					nics);
				KeyValuePair<string, EntityModel> bondingOption;
				if (model.BondingOptions.SelectedItem != null)
				{
					bondingOption = (KeyValuePair<string, EntityModel>)model.BondingOptions.SelectedItem;

					if (bondingOption.Key != "custom")
					{
						parameters.BondingOptions = (String.IsNullOrEmpty(bondingOption.Key) ? null : bondingOption.Key);
					}
					else
					{
						EntityModel entityModel = (EntityModel)bondingOption.Value;
						if (entityModel.Entity != null)
						{
							parameters.BondingOptions = entityModel.Entity.ToString();
						}
					}
				}
				parameters.BootProtocol = model.BootProtocol;

				if (model.IsStaticAddress)
				{
					parameters.Address = (string)model.Address.Entity;
					parameters.Subnet = (string)model.Subnet.Entity;
					parameters.Gateway = (string)model.Gateway.Entity;
				}

				model.StartProgress(null);
				currentModel = model;

				Frontend.RunAction(VdcActionType.AddBond, parameters,
					result =>
					{
						HostInterfaceListModel hostInterfaceListModel = (HostInterfaceListModel)result.State;
						VdcReturnValueBase returnValueBase = result.ReturnValue;
						if (returnValueBase != null && returnValueBase.Succeeded)
						{
							if (((HostBondInterfaceModel)hostInterfaceListModel.currentModel).CommitChanges)
							{
								SaveNetworkConfig(hostInterfaceListModel.Entity.vds_id, hostInterfaceListModel);
							}
							else
							{
								hostInterfaceListModel.currentModel.StopProgress();
								hostInterfaceListModel.Cancel();
								hostInterfaceListModel.Search();
							}
						}
						else
						{
							hostInterfaceListModel.currentModel.StopProgress();
						}
					},
					this
				);
			}
		}

		public void Detach()
		{
			if (Window != null)
			{
				return;
			}

			HostInterfaceModel model = new HostInterfaceModel();
			Window = model;
			model.Title = "Detach Network Interfaces";
			model.HashName = "detach_network_interfaces";

			VdsNetworkInterface nic = (VdsNetworkInterface)SelectedItem;
			model.Name = nic.Name;


			model.Commands.Add(
				new UICommand("OnDetach", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnDetach()
		{
			HostInterfaceModel model = (HostInterfaceModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			VdsNetworkInterface nic = (VdsNetworkInterface)SelectedItem;
			List<network> networks = DataProvider.GetClusterNetworkList(Entity.vds_group_id);

			network defaultNetwork = new network();
			defaultNetwork.name = nic.NetworkName;
			network net = Linq.FindNetworkByName(networks, nic.NetworkName)
				?? defaultNetwork;

			model.StartProgress(null);
			currentModel = model;

			Frontend.RunAction(VdcActionType.DetachNetworkFromVdsInterface, new AttachNetworkToVdsParameters(Entity.vds_id, net, nic),
					result =>
					{
						HostInterfaceListModel hostInterfaceListModel = (HostInterfaceListModel)result.State;
						VdcReturnValueBase returnValueBase = result.ReturnValue;
						if (returnValueBase != null && returnValueBase.Succeeded)
						{
							if (((HostInterfaceModel)hostInterfaceListModel.currentModel).CommitChanges)
							{
								SaveNetworkConfig(hostInterfaceListModel.Entity.vds_id, hostInterfaceListModel);
							}
							else
							{
								hostInterfaceListModel.currentModel.StopProgress();
								hostInterfaceListModel.Cancel();
								hostInterfaceListModel.Search();
							}
						}
						else
						{
							hostInterfaceListModel.currentModel.StopProgress();
						}
					},
					this
				);
		}

		public void OnSave()
		{
			HostInterfaceModel model = (HostInterfaceModel)Window;

			if (!model.Validate())
			{
				return;
			}

			string nicName = (string)model.Entity;
			VdsNetworkInterface nic = (VdsNetworkInterface)Linq.FindInterfaceByName(Linq.VdsNetworkInterfaceListToBase(AllItems), nicName);

			if (nic == null)
			{
				Cancel();
				return;
			}

			network network = (network)model.Network.SelectedItem;

			//Save changes.
			if (network.Id.Equals(Guid.Empty))
			{
				if (nic.IsManagement)
				{
					// We are trying to disconnect the management interface from its
					// network -> ask for the user's confirmation before doing that.
					ConfirmationModel confirmModel = new ConfirmationModel();
					ConfirmWindow = confirmModel;
					confirmModel.Title = "Edit Management Network Interface";
					confirmModel.HashName = "edit_management_network_interface";
					confirmModel.Message =
						String.Format(
							@"You are about to disconnect the Management Interface ({0}).\nAs a result, the Host might become unreachable.\n\nAre you sure you want to disconnect the Management Interface?",
							nic.Name);

					confirmModel.Commands.Add(
						new UICommand("OnConfirmManagementDetach", this)
						{
							Title = "OK",
							IsDefault = true
						});
					confirmModel.Commands.Add(
						new UICommand("Cancel", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
				}
				else
				{
					if (model.Progress != null)
					{
						return;
					}

					List<network> networks = DataProvider.GetClusterNetworkList(Entity.vds_group_id);
					network defaultNetwork = new network();
					defaultNetwork.name = nic.NetworkName;
					network net = Linq.FindNetworkByName(networks, nic.NetworkName)
						?? defaultNetwork;

					model.StartProgress(null);
					currentModel = model;

					Frontend.RunAction(VdcActionType.DetachNetworkFromVdsInterface, new AttachNetworkToVdsParameters(Entity.vds_id, net, nic),
							result =>
							{
								HostInterfaceListModel hostInterfaceListModel = (HostInterfaceListModel)result.State;
								VdcReturnValueBase returnValueBase = result.ReturnValue;
								if (returnValueBase != null && returnValueBase.Succeeded)
								{
									if (((HostInterfaceModel)hostInterfaceListModel.currentModel).CommitChanges)
									{
										SaveNetworkConfig(hostInterfaceListModel.Entity.vds_id, hostInterfaceListModel);
									}
									else
									{
										hostInterfaceListModel.currentModel.StopProgress();
										hostInterfaceListModel.Cancel();
										hostInterfaceListModel.Search();
									}
								}
								else
								{
									hostInterfaceListModel.currentModel.StopProgress();
								}
							},
							this
						);
				}
			}
			else
			{
				if (model.Progress != null)
				{
					return;
				}

				AttachNetworkToVdsParameters parameters;
				VdcActionType actionType;
				bool vLanAttached = false;
				bool bondWithVlans = false;
				bool isUpdateVlan = false;
				if (nic.Bonded.HasValue && nic.Bonded.Value)
				{
					foreach (HostInterfaceLineModel item in (List<HostInterfaceLineModel>)Items)
					{
						if (item.Interface != null && item.Interface.Id.Value.Equals(nic.Id))
						{
							if (item.VLans != null && item.VLans.Count > 0)
							{
								bondWithVlans = true;
								foreach (HostVLan vLan in item.VLans)
								{
									if (network.name == vLan.NetworkName)
									{
										vLanAttached = true;
										break;
									}
								}
							}
							break;
						}
					}
				}
				else
				{
					foreach (VdsNetworkInterface item in SelectedItemsWithVlans)
					{
						if (item.VlanId.HasValue && item.NetworkName == network.name)
						{
							isUpdateVlan = true;
							break;
						}
					}
				}
				//if the selected item is a non-attached or attached to vlan eth, or if the selected item is a bond that has vlans attached to it, 
				//and the selected network in the dialog is a new vlan, attach selected network.
				if (((String.IsNullOrEmpty(nic.NetworkName) && (!nic.Bonded.HasValue || !nic.Bonded.Value)) && !isUpdateVlan) || (bondWithVlans && (!vLanAttached && network.vlan_id.HasValue)))
				{
					parameters = new AttachNetworkToVdsParameters(Entity.vds_id, network, nic);
					actionType = VdcActionType.AttachNetworkToVdsInterface;
				}
				else
				{
					parameters = new UpdateNetworkToVdsParameters(Entity.vds_id, network, new List<VdsNetworkInterface> { nic });
					parameters.OldNetworkName = (nic.NetworkName != null ? nic.NetworkName : network.name);
					parameters.CheckConnectivity = (bool)model.CheckConnectivity.Entity;

					actionType = VdcActionType.UpdateNetworkToVdsInterface;
				}
				KeyValuePair<string, EntityModel> bondingOption;
				if (model.BondingOptions.SelectedItem != null)
				{
					bondingOption = (KeyValuePair<string, EntityModel>)model.BondingOptions.SelectedItem;

					if (bondingOption.Key != "custom")
					{
						parameters.BondingOptions = (String.IsNullOrEmpty(bondingOption.Key) ? null : bondingOption.Key);
					}
					else
					{
						EntityModel entityModel = (EntityModel)bondingOption.Value;
						if (entityModel.Entity != null)
						{
							parameters.BondingOptions = entityModel.Entity.ToString();
						}
					}
				}

				parameters.BootProtocol = model.BootProtocol;

				if (model.IsStaticAddress)
				{
					parameters.Address = (string)model.Address.Entity;
					parameters.Subnet = (string)model.Subnet.Entity;
				}

				model.StartProgress(null);
				currentModel = model;

				Frontend.RunAction(actionType, parameters,
				result =>
				{
					HostInterfaceListModel hostInterfaceListModel = (HostInterfaceListModel)result.State;
					VdcReturnValueBase returnValueBase = result.ReturnValue;
					if (returnValueBase != null && returnValueBase.Succeeded)
					{
						if (((HostInterfaceModel)hostInterfaceListModel.currentModel).CommitChanges)
						{
							SaveNetworkConfig(hostInterfaceListModel.Entity.vds_id, hostInterfaceListModel);
						}
						else
						{
							hostInterfaceListModel.currentModel.StopProgress();
							hostInterfaceListModel.Cancel();
							hostInterfaceListModel.Search();
						}
					}
					else
					{
						hostInterfaceListModel.currentModel.StopProgress();
					}
				},
				this
				);
			}
		}

		public void SaveNetworkConfig(guid vdsId, HostInterfaceListModel hostInterfaceListModel)
		{
			Frontend.RunAction(VdcActionType.CommitNetworkChanges,
				new VdsActionParameters(vdsId),
				result =>
				{
					VdcReturnValueBase returnValueBase = result.ReturnValue;
					if (returnValueBase != null && returnValueBase.Succeeded)
					{
						HostInterfaceListModel interfaceListModel = (HostInterfaceListModel)result.State;
						if (interfaceListModel.currentModel != null)
						{
							interfaceListModel.currentModel.StopProgress();
							interfaceListModel.Cancel();
							interfaceListModel.Search();
						}
					}
				},
				hostInterfaceListModel
			);
		}

		public void OnConfirmManagementDetach()
		{
			HostInterfaceModel model = (HostInterfaceModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			string nicName = (string)model.Entity;
			VdsNetworkInterface nic = (VdsNetworkInterface)Linq.FindInterfaceByName(Linq.Cast<NetworkInterface>(InterfaceItems), nicName);
			List<network> networks = DataProvider.GetClusterNetworkList(Entity.vds_group_id);

			network defaultNetwork = new network();
			defaultNetwork.name = nic.NetworkName;
			network net = Linq.FindNetworkByName(networks, nic.NetworkName)
				?? defaultNetwork;

			model.StartProgress(null);
			currentModel = model;

			Frontend.RunAction(VdcActionType.DetachNetworkFromVdsInterface, new AttachNetworkToVdsParameters(Entity.vds_id, net, nic),
					result =>
					{
						HostInterfaceListModel hostInterfaceListModel = (HostInterfaceListModel)result.State;
						VdcReturnValueBase returnValueBase = result.ReturnValue;
						if (returnValueBase != null && returnValueBase.Succeeded)
						{
							if (((HostInterfaceModel)hostInterfaceListModel.currentModel).CommitChanges)
							{
								SaveNetworkConfig(hostInterfaceListModel.Entity.vds_id, hostInterfaceListModel);
							}
							else
							{
								hostInterfaceListModel.currentModel.StopProgress();
								hostInterfaceListModel.Cancel();
								hostInterfaceListModel.Search();
							}
						}
						else
						{
							hostInterfaceListModel.currentModel.StopProgress();
						}
					},
					this
				);

		}

		public void Cancel()
		{
			ConfirmWindow = null;
			Window = null;
		}

		public void CancelConfirm()
		{
			ConfirmWindow = null;
		}

		private void UpdateActionAvailability()
		{
			VdsNetworkInterface selectedItem = (VdsNetworkInterface)SelectedItem;
			List<VdsNetworkInterface> selectedItems = SelectedItems;

			EditCommand.IsExecutionAllowed = Entity != null
				&& Entity.status != VDSStatus.NonResponsive
				&& selectedItem != null
				&& selectedItems.Count == 1
				&& String.IsNullOrEmpty(selectedItem.BondName)
				&& !selectedItem.IsManagement;

			BondCommand.IsExecutionAllowed = Entity != null
					  && Entity.status != VDSStatus.NonResponsive
					  && selectedItems.Count >= 2
					  && !IsAnyBond(selectedItems)
					  && Linq.FindAllInterfaceNetworkNameNotEmpty(Linq.VdsNetworkInterfaceListToBase(selectedItems)).Count <= 1
					  && Linq.FindAllInterfaceBondNameIsEmpty(selectedItems).Count == selectedItems.Count
					  && Linq.FindAllInterfaceVlanIdIsEmpty(selectedItems).Count == selectedItems.Count;

			// To bond, selected lines must not have more that 1 networks (vlan or not).
			if (Items != null && BondCommand.IsExecutionAllowed)
			{
				// Total network count cannot be more than 1.
				int totalNetworkCount = 0;

				List<HostInterfaceLineModel> itemList = (List<HostInterfaceLineModel>)Items;
				foreach (HostInterfaceLineModel lineModel in itemList)
				{
					if (lineModel.IsSelected)
					{
						int lineNetworkCount = lineModel.VLansCount + (lineModel.NetworkName != null ? 1 : 0);
						if (lineNetworkCount > 1)
						{
							// Bailout
							BondCommand.IsExecutionAllowed = false;
							break;
						}

						totalNetworkCount += lineNetworkCount;
						if (totalNetworkCount > 1)
						{
							// Bailout
							BondCommand.IsExecutionAllowed = false;
							break;
						}
					}
				}
			}


			DetachCommand.IsExecutionAllowed = Entity != null
				&& Entity.status != VDSStatus.NonResponsive
				&& selectedItems.Count == 1
				&& selectedItem != null
				&& !string.IsNullOrEmpty(selectedItem.NetworkName)
				&& !selectedItem.IsManagement;

			SaveNetworkConfigCommand.IsExecutionAllowed = Entity != null
				&& Entity.net_config_dirty.GetValueOrDefault();

			EditManagementNetworkCommand.IsExecutionAllowed = Entity != null
				&& Entity.status != VDSStatus.NonResponsive
				&& selectedItems.Count == 1
				&& selectedItem != null
				&& selectedItem.IsManagement;
		}

		private bool IsAnyBond(IEnumerable<VdsNetworkInterface> items)
		{
			foreach (VdsNetworkInterface item in items)
			{
				if (item.Bonded.GetValueOrDefault())
				{
					return true;
				}
			}

			return false;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == EditCommand)
			{
				Edit();
			}
			else if (command == EditManagementNetworkCommand)
			{
				EditManagementNetwork();
			}
			else if (command == BondCommand)
			{
				Bond();
			}
			else if (command == DetachCommand)
			{
				Detach();
			}
			else if (command == SaveNetworkConfigCommand)
			{
				SaveNetworkConfig(Entity.vds_id, this);
			}

			else if (command.Name == "OnSave")
			{
				OnSave();
			}

			else if (command.Name == "OnEditManagementNetwork")
			{
				OnEditManagementNetwork();
			}

			else if (command.Name == "OnEditManagementNetworkConfirmation")
			{
				OnEditManagementNetworkConfirmation(false);
			}

			else if (command.Name == "OnEditManagementNetworkConfirmation_Bond")
			{
				OnEditManagementNetworkConfirmation(true);
			}

			else if (command.Name == "OnBond")
			{
				OnBond();
			}

			else if (command.Name == "OnDetach")
			{
				OnDetach();
			}

			else if (command.Name == "OnConfirmManagementDetach")
			{
				OnConfirmManagementDetach();
			}

			else if (command.Name == "Cancel")
			{
				Cancel();
			}

			else if (command.Name == "CancelConfirm")
			{
				CancelConfirm();
			}
		}
	}


	public class HostVLanByNameComparer : IComparer<HostVLan>
	{
		public int Compare(HostVLan x, HostVLan y)
		{
			return x.Name.CompareTo(y.Name);
		}
	}
}
