using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.hosts;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using VdcCommon.BusinessEntities;
using VdcFrontend;

namespace org.ovirt.engine.ui.uicommon.models.clusters
{
	public class ClusterGuideModel : GuideModel
	{
		#region Action Constants

		public readonly string ClusterConfigureHostsAction = "Configure Host";
		public readonly string ClusterAddAnotherHostAction = "Add another Host";
		public readonly string SelectHostsAction = "Select Hosts";
		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		public new VDSGroup Entity
		{
			get { return base.Entity as VDSGroup; }
			set { base.Entity = value; }
		}

		#endregion


		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();
			UpdateOptions();
		}

		private void UpdateOptions()
		{
			CompulsoryActions.Clear();
			OptionalActions.Clear();

			if (Entity != null)
			{
				storage_pool dataCenter = null;
				if (Entity.storage_pool_id != null)
				{
					dataCenter = DataProvider.GetDataCenterById(Entity.storage_pool_id.Value);
				}
				if (dataCenter == null || dataCenter.storage_pool_type != StorageType.LOCALFS)
				{
					//Add host action.
					UICommand addHostAction = new UICommand("AddHost", this);

					// 				var hosts = DataProvider.GetHostListByCluster(Entity.name)
					// 					.Skip(1)
					// 					.ToList();
					List<VDS> hosts = DataProvider.GetHostListByCluster(Entity.name);
					if (hosts.Count > 1)
					{
						hosts.RemoveAt(0);
					}

					if (hosts.Count == 0)
					{
						addHostAction.Title = ClusterConfigureHostsAction;
						CompulsoryActions.Add(addHostAction);
					}
					else
					{
						addHostAction.Title = ClusterAddAnotherHostAction;
						OptionalActions.Add(addHostAction);
					}
					if (Entity.storage_pool_id == null)
					{
						addHostAction.IsExecutionAllowed = false;
						addHostAction.ExecuteProhibitionReasons.Add("The Cluster isn't attached to a Data Center");
						return;
					}
					List<VDSGroup> clusters = DataProvider.GetClusterList((Guid)Entity.storage_pool_id);
					Version minimalClusterVersion = Linq.GetMinVersionByClusters(clusters);
					List<VDS> availableHosts = new List<VDS>();
					foreach (VDS vds in DataProvider.GetHostList())
					{
						if ((!Linq.IsHostBelongsToAnyOfClusters(clusters, vds))
								&& (vds.status == VDSStatus.Maintenance || vds.status == VDSStatus.PendingApproval)
								&& (vds.Version.FullVersion == null || Extensions.GetFriendlyVersion(vds.Version.FullVersion).CompareTo(minimalClusterVersion) >= 0))
							availableHosts.Add(vds);
					}
					//Select host action.
					UICommand selectHostAction = new UICommand("SelectHost", this);

					if (availableHosts.Count > 0 && clusters.Count > 0)
					{
						if (hosts.Count == 0)
						{
							selectHostAction.Title = SelectHostsAction;
							CompulsoryActions.Add(selectHostAction);
						}
						else
						{
							selectHostAction.Title = SelectHostsAction;
							OptionalActions.Add(selectHostAction);
						}
					}
				}
				else
				{
					UICommand addHostAction = new UICommand("AddHost", this) { Title = ClusterAddAnotherHostAction };
					UICommand selectHost = new UICommand("SelectHost", this) { Title = SelectHostsAction };
					VDS host = DataProvider.GetLocalStorageHost(dataCenter.name);
					if (host != null)
					{
						addHostAction.IsExecutionAllowed = false;
						selectHost.IsExecutionAllowed = false;
						string hasHostReason = "This Cluster belongs to a Local Data Center which already contain a Host";
						addHostAction.ExecuteProhibitionReasons.Add(hasHostReason);
						selectHost.ExecuteProhibitionReasons.Add(hasHostReason);
					}
					CompulsoryActions.Add(addHostAction);
					OptionalActions.Add(selectHost);
				}
			}
		}

		public void SelectHost()
		{
			List<VDSGroup> clusters = new List<VDSGroup>();
			clusters.Add(Entity);

			MoveHost model = new MoveHost();
			model.Title = "Select Host";
			model.HashName = "select_host";
			Window = model;
			model.Cluster.Items = clusters;
			model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters);
			model.Cluster.IsAvailable = false;

			model.Commands.Add(
				new UICommand("OnSelectHost", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSelectHost()
		{
			MoveHost model = (MoveHost)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			model.SelectedHosts = new List<VDS>();
			foreach (EntityModel a in Linq.Cast<EntityModel>(model.Items))
			{
				if (a.IsSelected)
				{
					model.SelectedHosts.Add((VDS)a.Entity);
				}
			}

			VDSGroup cluster = (VDSGroup)model.Cluster.SelectedItem;

			List<VdcActionParametersBase> paramerterList = new List<VdcActionParametersBase>();
			foreach (VDS host in model.SelectedHosts)
			{
				//Try to change host's cluster as neccessary.
				if (host.vds_group_id != null && !host.vds_group_id.Equals(cluster.ID))
				{
					paramerterList.Add(new ChangeVDSClusterParameters(cluster.ID, host.vds_id));

				}
			}
			model.StartProgress(null);
			Frontend.RunMultipleAction(VdcActionType.ChangeVDSCluster, paramerterList,
									   result =>
									   {
										   ClusterGuideModel clusterGuideModel = (ClusterGuideModel)result.State;
									   		List<VDS> hosts = ((MoveHost) clusterGuideModel.Window).SelectedHosts;
									   		List<VdcReturnValueBase> retVals = (List<VdcReturnValueBase>) result.ReturnValue;
											if (retVals != null && hosts.Count == retVals.Count)
											{
												int i = 0;
												foreach (VDS selectedHost in hosts)
												{
													if (selectedHost.status == VDSStatus.PendingApproval && retVals[i] != null && retVals[i].Succeeded)
													{
														Frontend.RunAction(VdcActionType.ApproveVds,
														                   new ApproveVdsParameters(selectedHost.vds_id));
													}
												}
												i++;
											}
									   		clusterGuideModel.Window.StopProgress();
										    clusterGuideModel.Cancel();
										    clusterGuideModel.PostAction();
									   }
									   , this);
		}

		public void AddHost()
		{
			HostModel model = new HostModel();
			Window = model;
			model.Title = "New Host";
			model.HashName = "new_host";
			model.Port.Entity = 54321;
			model.OverrideIpTables.Entity = true;

			model.Cluster.SelectedItem = Entity;
			model.Cluster.IsChangable = false;

			List<storage_pool> dataCenters = DataProvider.GetDataCenterList();
			model.DataCenter.Items = dataCenters;
			if (Entity.storage_pool_id != null)
			{
				model.DataCenter.SelectedItem = Linq.FirstOrDefault(dataCenters, new Linq.DataCenterPredicate((Guid)Entity.storage_pool_id));
			}
			model.DataCenter.IsChangable = false;


			model.Commands.Add(
				new UICommand("OnConfirmPMHost", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnConfirmPMHost()
		{
			HostModel model = (HostModel)Window;

			if (!model.Validate())
			{
				return;
			}

			if (!((bool)model.IsPm.Entity))
			{
				ConfirmationModel confirmModel = new ConfirmationModel();
				ConfirmWindow = confirmModel;
				confirmModel.Title = "Power Management Configuration";
				confirmModel.HashName = "power_management_configuration";
				confirmModel.Message = "You haven't configured Power Management for this Host. Are you sure you want to continue?";

				confirmModel.Commands.Add(
					new UICommand("OnAddHost", this)
					{
						Title = "OK",
						IsDefault = true
					});
				confirmModel.Commands.Add(
					new UICommand("CancelConfirmWithFocus", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
			else
			{
				OnAddHost();
			}
		}

		public void OnAddHost()
		{
			CancelConfirm();

			HostModel model = (HostModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			//Save changes.
			VDS host = new VDS();
			host.vds_name = (string)model.Name.Entity;
			host.host_name = (string)model.Host.Entity;
			host.ManagmentIp = (string)model.ManagementIp.Entity;
			host.port = (int)model.Port.Entity;
			host.vds_group_id = ((VDSGroup)model.Cluster.SelectedItem).ID;
			host.pm_enabled = (bool)model.IsPm.Entity;
			host.pm_user = (bool)model.IsPm.Entity ? (string)model.PmUserName.Entity : null;
			host.pm_password = (bool)model.IsPm.Entity ? (string)model.PmPassword.Entity : null;
			host.pm_type = (bool)model.IsPm.Entity ? (string)model.PmType.SelectedItem : null;
			host.PmOptionsMap = (bool)model.IsPm.Entity ? new valueObjectMap(model.PmOptionsMap, false) : null;

            AddVdsActionParameters vdsActionParams = new AddVdsActionParameters();
            vdsActionParams.vds = host;
            vdsActionParams.VdsId = host.vds_id;
            vdsActionParams.RootPassword = (string)model.RootPassword.Entity;

			model.StartProgress(null);

			Frontend.RunAction(VdcActionType.AddVds, vdsActionParams,
				result =>
				{
					ClusterGuideModel localModel = (ClusterGuideModel)result.State;

					localModel.PostOnAddHost(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnAddHost(VdcReturnValueBase returnValue)
		{
			HostModel model = (HostModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
				PostAction();
			}
		}

		private void PostAction()
		{
			UpdateOptions();
		}

		public void Cancel()
		{
			Window = null;
		}

		public void CancelConfirm()
		{
			ConfirmWindow = null;
		}

		public void CancelConfirmWithFocus()
		{
			ConfirmWindow = null;

			HostModel hostModel = (HostModel)Window;
			hostModel.IsPowerManagementSelected = true;
			hostModel.IsPm.Entity = true;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command.Name == "AddHost")
			{
				AddHost();
			}
			if (command.Name == "OnConfirmPMHost")
			{
				OnConfirmPMHost();
			}
			if (command.Name == "OnAddHost")
			{
				OnAddHost();
			}
			if (command.Name == "SelectHost")
			{
				SelectHost();
			}
			if (command.Name == "OnSelectHost")
			{
				OnSelectHost();
			}
			if (command.Name == "Cancel")
			{
				Cancel();
			}
			if (command.Name == "CancelConfirm")
			{
				CancelConfirm();
			}
			if (command.Name == "CancelConfirmWithFocus")
			{
				CancelConfirmWithFocus();
			}
		}
	}
}
