using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Reflection;
using System.Resources;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicommon.models.tags;
using org.ovirt.engine.ui.uicommon.models.userportal;
using VdcCommon;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.dataprovider;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public class VmListModel : ListWithDetailsModel, ISupportSystemTreeContext
	{
		#region Commands

		public UICommand NewServerCommand { get; private set; }
		public UICommand NewDesktopCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand RunCommand { get; private set; }
		public UICommand PauseCommand { get; private set; }
		public UICommand StopCommand { get; private set; }
		public UICommand ShutdownCommand { get; private set; }
		public UICommand MigrateCommand { get; private set; }
		public UICommand NewTemplateCommand { get; private set; }
		public UICommand RunOnceCommand { get; private set; }
		public UICommand ExportCommand { get; private set; }
		public UICommand MoveCommand { get; private set; }
		public UICommand RetrieveIsoImagesCommand { get; private set; }
		public UICommand GuideCommand { get; private set; }
		public UICommand AssignTagsCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		private Model errorWindow;
		public Model ErrorWindow
		{
			get { return errorWindow; }
			set
			{
				if (errorWindow != value)
				{
					errorWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ErrorWindow"));
				}
			}
		}

		private ConsoleModel defaultConsoleModel;
		public ConsoleModel DefaultConsoleModel
		{
			get { return defaultConsoleModel; }
			set
			{
				if (defaultConsoleModel != value)
				{
					defaultConsoleModel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("DefaultConsoleModel"));
				}
			}
		}

		private ConsoleModel additionalConsoleModel;
		public ConsoleModel AdditionalConsoleModel
		{
			get { return additionalConsoleModel; }
			set
			{
				if (additionalConsoleModel != value)
				{
					additionalConsoleModel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("AdditionalConsoleModel"));
				}
			}
		}

		private bool hasAdditionalConsoleModel;
		public bool HasAdditionalConsoleModel
		{
			get { return hasAdditionalConsoleModel; }
			set
			{
				if (hasAdditionalConsoleModel != value)
				{
					hasAdditionalConsoleModel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasAdditionalConsoleModel"));
				}
			}
		}

		public ObservableCollection<ChangeCDModel> isoImages;
		public ObservableCollection<ChangeCDModel> IsoImages
		{
			get { return isoImages; }
			private set
			{
				if ((isoImages == null && value != null) || (isoImages != null && !isoImages.Equals(value)))
				{
					isoImages = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsoImages"));
				}
			}
		}

		protected object[] SelectedKeys
		{
			//get { return SelectedItems == null ? new object[0] : SelectedItems.Cast<VM>().Select(a => a.vm_guid).Cast<object>().ToArray(); }
			get
			{
				if (SelectedItems == null)
				{
					return new object[0];
				}

				object[] keys = new object[SelectedItems.Count];
				for (int i = 0; i < SelectedItems.Count; i++)
				{
					keys[i] = ((VM)SelectedItems[i]).vm_guid;
				}

				return keys;
			}
		}

		public object GuideContext { get; set; }
		public VM currentVm { get; set; }
		#endregion

		private readonly Dictionary<Guid, List<ConsoleModel>> cachedConsoleModels;

		private List<string> CustomPropertiesKeysList { get; set; }


		public VmListModel()
		{
			Title = "Virtual Machines";

			DefaultSearchString = "Vms:";
			SearchString = DefaultSearchString;

			cachedConsoleModels = new Dictionary<Guid, List<ConsoleModel>>();

			NewServerCommand = new UICommand("NewServer", this);
			NewDesktopCommand = new UICommand("NewDesktop", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			RunCommand = new UICommand("Run", this);
			PauseCommand = new UICommand("Pause", this);
			StopCommand = new UICommand("Stop", this);
			ShutdownCommand = new UICommand("Shutdown", this);
			MigrateCommand = new UICommand("Migrate", this);
			NewTemplateCommand = new UICommand("NewTemplate", this);
			RunOnceCommand = new UICommand("RunOnce", this);
			ExportCommand = new UICommand("Export", this);
			MoveCommand = new UICommand("Move", this);
			GuideCommand = new UICommand("Guide", this);
			RetrieveIsoImagesCommand = new UICommand("RetrieveIsoImages", this);
			AssignTagsCommand = new UICommand("AssignTags", this);

			IsoImages = new ObservableCollection<ChangeCDModel>();
			IsoImages.Add(new ChangeCDModel { Title = "Retrieving CDs..." });

			UpdateActionAvailability();

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;


			AsyncDataProvider.GetCustomPropertiesList(new AsyncQuery(this,
				(target, returnValue) =>
			{
					VmListModel model = (VmListModel)target;

					if (returnValue != null)
				{
						string[] array = ((string)returnValue).Split(';');
						model.CustomPropertiesKeysList = new List<string>();

						foreach (string s in array)
					{
							model.CustomPropertiesKeysList.Add(s);
					}
				}
				})
			);
		}

		private void AssignTags()
		{
			if (Window != null)
			{
				return;
			}

			TagListModel model = new TagListModel();
			Window = model;
			model.Title = "Assign Tags";
			model.HashName = "assign_tags_vms";

			model.AttachedTagsToEntities = GetAttachedTagsToSelectedVMs();
			List<TagModel> tags = (List<TagModel>)model.Items;

			model.Commands.Add(
				new UICommand("OnAssignTags", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private IDictionary<Guid, bool> GetAttachedTagsToSelectedVMs()
		{
			Dictionary<Guid, bool> tags = new Dictionary<Guid, bool>();

			//var vmIds = SelectedItems
			//   .Cast<VM>()
			//   .Select(a => a.vm_guid)
			//   .ToList();
			List<guid> vmIds = new List<guid>();
			foreach (object item in SelectedItems)
			{
				VM vm = (VM)item;
				vmIds.Add(vm.vm_guid);
			}


			//var allAttachedTags = vmIds.SelectMany(a => DataProvider.GetAttachedTagsToVm(a)).ToList();
			List<VdcCommon.BusinessEntities.tags> allAttachedTags = new List<VdcCommon.BusinessEntities.tags>();
			foreach (guid id in vmIds)
			{
				allAttachedTags.AddRange(DataProvider.GetAttachedTagsToVm(id));
			}

			//var attachedTags = allAttachedTags
			//    .Distinct(new TagsEqualityComparer())
			//    .ToList();
			List<VdcCommon.BusinessEntities.tags> attachedTags = Linq.Distinct(allAttachedTags, new TagsEqualityComparer());

			//attachedTags.Each(a => { tags.Add(a.tag_id, allAttachedTags.Count(b => b.tag_id == a.tag_id) == vmIds.Count() ? true : false); });
			foreach (VdcCommon.BusinessEntities.tags tag in attachedTags)
			{
				int count = 0;
				foreach (VdcCommon.BusinessEntities.tags tag2 in allAttachedTags)
				{
					if (tag2.tag_id.Equals(tag.tag_id))
						count++;
				}
				tags.Add(tag.tag_id, count == vmIds.Count ? true : false);
			}

			return tags;
		}

		private void OnAssignTags()
		{
			TagListModel model = (TagListModel)Window;

			//var vmIds = SelectedItems
			//    .Cast<VM>()
			//    .Select(a => a.vm_guid)
			//    .ToList();
			List<Guid> vmIds = new List<Guid>();
			foreach (object item in SelectedItems)
			{
				VM vm = (VM)item;
				vmIds.Add(vm.vm_guid);
			}

			IDictionary<Guid, bool> attachedTags = GetAttachedTagsToSelectedVMs();

			//prepare attach/detach lists
			List<Guid> tagsToAttach = new List<Guid>();
			List<Guid> tagsToDetach = new List<Guid>();

			//model.Items
			//    .Cast<TagModel>()
			//    .First()
			//    .EachRecursive(a => a.Children, (a, b) =>
			//    {
			//        if (a.Selection == true && (!attachedTags.ContainsKey(a.Id) || attachedTags[a.Id] == false))
			//        {
			//            tagsToAttach.Add(a.Id);
			//        }
			//        else if (a.Selection == false && attachedTags.ContainsKey(a.Id))
			//        {
			//            tagsToDetach.Add(a.Id);
			//        }
			//    });
			if (model.Items != null && ((List<TagModel>)model.Items).Count > 0)
			{
				List<TagModel> tags = (List<TagModel>)model.Items;
				TagModel rootTag = tags[0];
				TagModel.RecursiveEditAttachDetachLists(rootTag, attachedTags, tagsToAttach, tagsToDetach);
			}


			//Frontend.RunMultipleActions(VdcActionType.AttachVmsToTag,
			//    tagsToAttach.Select(a =>
			//        (VdcActionParametersBase)new AttachEntityToTagParameters(a, vmIds)
			//    )
			//    .ToList()
			//);

			List<VdcActionParametersBase> parameters = new List<VdcActionParametersBase>();
			foreach (Guid a in tagsToAttach)
			{
				parameters.Add(new AttachEntityToTagParameters(a, vmIds));
			}
			Frontend.RunMultipleAction(VdcActionType.AttachVmsToTag, parameters);

			//Detach tags.
			//Frontend.RunMultipleActions(VdcActionType.DetachVmFromTag,
			//    tagsToDetach.Select(a =>
			//        (VdcActionParametersBase)new AttachEntityToTagParameters(a, vmIds)
			//    )
			//    .ToList()
			//);

			parameters = new List<VdcActionParametersBase>();
			foreach (Guid a in tagsToDetach)
			{
				parameters.Add(new AttachEntityToTagParameters(a, vmIds));
			}
			Frontend.RunMultipleAction(VdcActionType.DetachVmFromTag, parameters);


			Cancel();
		}

		private void Guide()
		{
			VmGuideModel model = new VmGuideModel();
			Window = model;
			model.Title = "New Virtual Machine - Guide Me";
			model.HashName = "new_virtual_machine_-_guide_me";

			model.Entity = GuideContext != null
				? DataProvider.GetVmById((guid)GuideContext)
				: null;


			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Configure Later",
					IsDefault = true,
					IsCancel = true
				});
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new VmGeneralModel());
			list.Add(new VmInterfaceListModel());
			list.Add(new VmDiskListModel());
			list.Add(new VmSnapshotListModel());
			list.Add(new VmEventListModel());
			list.Add(new VmAppListModel());
			list.Add(new PermissionListModel());
			DetailModels = list;
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("vm");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.Search, new SearchParameters(SearchString, SearchType.VM)
			{
				MaxCount = SearchPageSize
			});
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterSearch(SearchString, SearchType.VM, SearchPageSize);
			Items = AsyncResult.Data;
		}

		private void UpdateConsoleModels()
		{
			IList selectedItems = SelectedItems ?? new ArrayList();
			VM vm = SelectedItem as VM;

			if (vm == null || selectedItems.Count > 1)
			{
				DefaultConsoleModel = null;
				AdditionalConsoleModel = null;
				HasAdditionalConsoleModel = false;
			}
			else
			{
				if (!cachedConsoleModels.ContainsKey(vm.vm_guid))
				{
					SpiceConsoleModel spiceConsoleModel = new SpiceConsoleModel();
					spiceConsoleModel.ErrorEvent.addListener(this);
					VncConsoleModel vncConsoleModel = new VncConsoleModel();
					RdpConsoleModel rdpConsoleModel = new RdpConsoleModel();

					cachedConsoleModels.Add(vm.vm_guid,
						new List<ConsoleModel>
					    {
					    	spiceConsoleModel,
					    	vncConsoleModel,
					    	rdpConsoleModel
					    });
				}


				List<ConsoleModel> cachedModels = cachedConsoleModels[vm.vm_guid];
				foreach (ConsoleModel a in cachedModels)
				{
					a.Entity = null;
					a.Entity = vm;
				}

				DefaultConsoleModel = vm.display_type == DisplayType.vnc
										? cachedModels[1]
										: cachedModels[0];

				if (DataProvider.IsWindowsOsType(vm.vm_os))
				{
					foreach (ConsoleModel a in cachedModels)
					{
						if (a is RdpConsoleModel)
						{
							AdditionalConsoleModel = a;
							break;
						}
					}
					HasAdditionalConsoleModel = true;
				}
				else
				{
					AdditionalConsoleModel = null;
					HasAdditionalConsoleModel = false;
				}
			}
		}

		public List<ConsoleModel> GetConsoleModelsByVmGuid(Guid vmGuid)
		{
			if (cachedConsoleModels != null && cachedConsoleModels.ContainsKey(vmGuid))
			{
				return cachedConsoleModels[vmGuid];
			}

			return null;
		}

		private void NewDesktop()
		{
			NewInternal(VmType.Desktop);
		}

		private void NewServer()
		{
			NewInternal(VmType.Server);
		}

		private void NewInternal(VmType vmType)
		{
			if (Window != null)
			{
				return;
			}

			UnitVmModel model = new UnitVmModel(new NewVmModelBehavior());
			Window = model;
			model.Title = String.Format("New {0} Virtual Machine", vmType == VmType.Server ? "Server" : "Desktop");
			model.HashName = "new_" + (vmType == VmType.Server ? "server" : "desktop");
			model.IsNew = true;
			model.VmType = vmType;
			model.CustomPropertiesKeysList = CustomPropertiesKeysList;

			model.Initialize(SystemTreeSelectedItem);

			// Ensures that the default provisioning is "Clone" for a new server and "Thin" for a new desktop.
			EntityModel selectedItem = null;
			bool selectValue = model.VmType == VmType.Server;

			foreach (object item in model.Provisioning.Items)
			{
				EntityModel a = (EntityModel)item;
				if ((bool)a.Entity == selectValue)
				{
					selectedItem = a;
					break;
			}
			}
			model.Provisioning.SelectedItem = selectedItem;


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Edit()
		{
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				return;
			}

			if (Window != null)
			{
				return;
			}

			UnitVmModel model = new UnitVmModel(new ExistingVmModelBehavior(vm));
			model.VmType = vm.vm_type;
			Window = model;
			model.Title = String.Format("Edit {0} Virtual Machine", vm.vm_type == VmType.Server ? "Server" : "Desktop");
			model.HashName = "edit_" + (vm.vm_type == VmType.Server ? "server" : "desktop");
			model.CustomPropertiesKeysList = CustomPropertiesKeysList;

			model.Initialize(this.SystemTreeSelectedItem);

			//TODO:
			//VDSGroup cluster = null;
			//if (model.Cluster.Items == null)
			//{
			//    model.Commands.Add(
			//        new UICommand("Cancel", this)
			//        {
			//            Title = "Cancel",
			//            IsCancel = true
			//        });
			//    return;
			//}


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Virtual Machine(s)";
			model.HashName = "remove_virtual_machine";
			model.Message = "Virtual Machine(s)";

			//model.Items = SelectedItems.Cast<VM>().Select(a => a.vm_name);
			List<string> list = new List<string>();
			foreach (object selectedItem in SelectedItems)
			{
				VM a = (VM)selectedItem;
				list.Add(a.vm_name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Move()
		{
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				return;
			}

			if (Window != null)
			{
				return;
			}

			ListModel model = new ListModel();
			Window = model;
			model.Title = "Move Virtual Machine";
			model.HashName = "move_virtual_machine";

			//var storageDomains = (vm.vmt_guid != Guid.Empty
			//    ? DataProvider.GetStorageDomainListByTemplate(vm.vmt_guid)
			//    : DataProvider.GetStorageDomainList(vm.storage_pool_id)
			//        .Where(a => a.storage_domain_type == StorageDomainType.Data || a.storage_domain_type == StorageDomainType.Master)
			//    );

			List<storage_domains> storageDomains;
			if (!vm.vmt_guid.Equals(Guid.Empty))
			{
				storageDomains = DataProvider.GetStorageDomainListByTemplate(vm.vmt_guid);
			}
			else
			{
				storageDomains = new List<storage_domains>();
				foreach (storage_domains a in DataProvider.GetStorageDomainList(vm.storage_pool_id))
				{
					if (a.storage_domain_type == StorageDomainType.Data || a.storage_domain_type == StorageDomainType.Master)
					{
						storageDomains.Add(a);
					}
				}
			}


			// filter only the Active storage domains (Active regarding the relevant storage pool).
			//storageDomains = storageDomains.Where(a => a.status.HasValue && a.status.Value == StorageDomainStatus.Active);
			List<storage_domains> list = new List<storage_domains>();
			foreach (storage_domains a in storageDomains)
			{
				if (a.status.HasValue && a.status.Value == StorageDomainStatus.Active)
				{
					list.Add(a);
				}
			}
			storageDomains = list;


			List<DiskImage> disks = DataProvider.GetVmDiskList(vm.vm_guid);
			if (disks.Count > 0)
			{
				//storageDomains = storageDomains.Where(a => a.id != disks[0].storage_id);
				list = new List<storage_domains>();
				foreach (storage_domains a in storageDomains)
				{
					if (!a.id.Equals(disks[0].storage_id.Value))
					{
						list.Add(a);
					}
				}
				storageDomains = list;
			}

			storageDomains.Sort(new Linq.StorageDomainByNameComparer());

			//var items = storageDomains.OrderBy(a => a.storage_name)
			//    .Select(a => new EntityModel() { Entity = a })
			//    .ToList();
			List<EntityModel> items = new List<EntityModel>();
			foreach (storage_domains a in storageDomains)
			{
				EntityModel m = new EntityModel();
				m.Entity = a;
				items.Add(m);
			}

			model.Items = items;
			if (items.Count == 1)
			{
				items[0].IsSelected = true;
			}


			if (items.Count == 0)
			{
				model.Message = "The system could not find available target Storage Domain.\nPossible reasons:\n  - No active Storage Domain available\n  - The Template that the VM is based on does not exist on active Storage Domain";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnMove", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnMove()
		{
			VM vm = (VM)SelectedItem;

			if (vm == null)
			{
				Cancel();
				return;
			}

			ListModel model = (ListModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<storage_domains> items = new List<storage_domains>();
			foreach (object item in model.Items)
			{
				EntityModel a = (EntityModel)item;
				if (a.IsSelected)
				{
					items.Add((storage_domains)a.Entity);
				}
			}

			// should be only one:
			if (items.Count == 0)
			{
				return;
			}


			List<VdcActionParametersBase> parameters = new List<VdcActionParametersBase>();
			foreach (storage_domains a in items)
			{
				parameters.Add(new MoveVmParameters(vm.vm_guid, a.id));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.MoveVm, parameters,
				result =>
				{
					ListModel localModel = (ListModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void Export()
		{
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				return;
			}

			if (Window != null)
			{
				return;
			}

			ExportVmModel model = new ExportVmModel();
			Window = model;
			model.Title = "Export Virtual Machine";
			model.HashName = "export_virtual_machine";

			//var storages = DataProvider.GetStorageDomainList(vm.storage_pool_id)
			//    .Where(a => a.storage_domain_type == StorageDomainType.ImportExport)
			//    .ToList();
			//model.Storage.Options = storages;
			//model.Storage.Value = storages.FirstOrDefault();
			List<storage_domains> storages = new List<storage_domains>();
			foreach (storage_domains a in DataProvider.GetStorageDomainList(vm.storage_pool_id))
			{
				if (a.storage_domain_type == StorageDomainType.ImportExport)
				{
					storages.Add(a);
				}
			}
			model.Storage.Items = storages;
			model.Storage.SelectedItem = Linq.FirstOrDefault(storages);


			bool noActiveStorage = true;
			foreach (storage_domains a in storages)
			{
				if (a.status == StorageDomainStatus.Active)
				{
					noActiveStorage = false;
					break;
				}
			}


			if (SelectedVmsOnDifferentDataCenters())
			{
				model.CollapseSnapshots.IsChangable = false;
				model.ForceOverride.IsChangable = false;

				model.Message = "Virtual Machines reside on several Data Centers. Make sure the exported Virtual Machines reside on the same Data Center.";


				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else if (storages.Count == 0)
			{
				model.CollapseSnapshots.IsChangable = false;
				model.ForceOverride.IsChangable = false;

				model.Message = "There is no Export Domain to Backup the Virtual Machine into. Attach an Export Domain to the Virtual Machine(s) Data Center.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});

			}
			//else if (storages.All(a => a.status != StorageDomainStatus.Active))
			else if (noActiveStorage)
			{
				model.CollapseSnapshots.IsChangable = false;
				model.ForceOverride.IsChangable = false;

				model.Message = "The relevant Export Domain in not active. Please activate it.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				showWarningOnExistingVms(model);

				model.Commands.Add(
					new UICommand("OnExport", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void showWarningOnExistingVms(ExportVmModel model)
		{
			Guid storageDomainId = ((storage_domains)model.Storage.SelectedItem).id;
			storage_pool storagePool = DataProvider.GetFirstStoragePoolByStorageDomain(storageDomainId);
			string existingVMs = string.Empty;
			if (storagePool != null)
			{
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmsFromExportDomain,
									new GetAllFromExportDomainQueryParamenters(storagePool.Id, storageDomainId) { GetAll = true });
				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					foreach (object selectedItem in SelectedItems)
					{
						VM vm = (VM)selectedItem;

						//if (((List<VM>)returnValue.ReturnValue).SingleOrDefault(a => a.vm_guid == vm.vm_guid) != null)
						VM foundVm = null;
						foreach (VM a in (List<VM>)returnValue.ReturnValue)
						{
							if (a.vm_guid.Equals(vm.vm_guid))
							{
								foundVm = a;
								break;
							}
						}

						if (foundVm != null)
						{
							existingVMs += "\u2022  " + vm.vm_name + "\n";
						}
					}
				}
				if (!String.IsNullOrEmpty(existingVMs))
				{
					model.Message = String.Format("VM(s):\n{0} already exist on the target Export Domain. If you want to override them, please check the 'Force Override' check-box.", existingVMs);
				}
			}
		}

		private bool SelectedVmsOnDifferentDataCenters()
		{
			//List<VM> vms = SelectedItems.Cast<VM>().ToList();
			//return vms.GroupBy(a => a.storage_pool_id).Count() > 1 ? true : false;

			List<VM> vms = new List<VM>();
			foreach (object selectedItem in SelectedItems)
			{
				VM a = (VM)selectedItem;
				vms.Add(a);
			}

			IDictionary<nguid, List<VM>> t = new Dictionary<nguid, List<VM>>();
			foreach (VM a in vms)
			{
				if (!t.ContainsKey(a.storage_pool_id))
				{
					t.Add(a.storage_pool_id, new List<VM>());
				}

				List<VM> list = t[a.storage_pool_id];
				list.Add(a);
			}

			return t.Count > 1;
		}

		private List<string> getTemplatesNotPresentOnExportDomain()
		{
			ExportVmModel model = (ExportVmModel)Window;
			Guid storageDomainId = ((storage_domains)model.Storage.SelectedItem).id;
			storage_pool storagePool = DataProvider.GetFirstStoragePoolByStorageDomain(storageDomainId);
			List<string> missingTemplates = new List<string>();

			if (storagePool != null)
			{
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetTemplatesFromExportDomain,
					  new GetAllFromExportDomainQueryParamenters(storagePool.Id, storageDomainId) { GetAll = true });
				Dictionary<string, List<string>> templateDic = new Dictionary<string, List<string>>();
				//check if relevant templates are already there
				if (returnValue != null && returnValue.Succeeded)
				{
					foreach (object selectedItem in SelectedItems)
					{
						VM vm = (VM)selectedItem;

						//if (vm.vmt_guid != Guid.Empty && ((Dictionary<VmTemplate, List<DiskImage>>)returnValue.ReturnValue).Keys.SingleOrDefault(a => vm.vmt_guid == a.vmt_guid) == null)

						bool hasMatch = false;
						foreach (VmTemplate a in ((Dictionary<VmTemplate, List<DiskImage>>)returnValue.ReturnValue).Keys)
						{
							if (vm.vmt_guid.Equals(a.Id))
							{
								hasMatch = true;
								break;
							}
						}

						if (!vm.vmt_guid.Equals(Guid.Empty) && !hasMatch)
						{
							if (!templateDic.ContainsKey(vm.vmt_name))
							{
								templateDic.Add(vm.vmt_name, new List<string>());
							}
							templateDic[vm.vmt_name].Add(vm.vm_name);
							//missingTemplates.Add(string.Format("Template '{0}' for VM '{1}'", vm.vmt_name, vm.vm_name));
						}
					}
					string tempStr;
					List<string> tempList;
					foreach (KeyValuePair<string, List<string>> keyValuePair in templateDic)
					{
						tempList = keyValuePair.Value;
						tempStr = "Template " + keyValuePair.Key + " (for ";
						int i;
						for (i = 0; i < tempList.Count - 1; i++)
						{
							tempStr += tempList[i] + ", ";
						}
						tempStr += tempList[i] + ")";
						missingTemplates.Add(tempStr);
					}
				}
				else
				{
					return null;
				}
			}
			return missingTemplates;
		}

		public void OnExport()
		{
			ExportVmModel model = (ExportVmModel)Window;
			Guid storageDomainId = ((storage_domains)model.Storage.SelectedItem).id;
			if (!model.Validate())
			{
				return;
			}

			List<string> missingTemplatesFromVms = getTemplatesNotPresentOnExportDomain();

			List<VdcActionParametersBase> parameters = new List<VdcActionParametersBase>();
			foreach (object a in SelectedItems)
			{
				VM vm = (VM)a;
				MoveVmParameters parameter = new MoveVmParameters(vm.vm_guid, storageDomainId);
				parameter.ForceOverride = (bool)model.ForceOverride.Entity;
				parameter.CopyCollapse = (bool)model.CollapseSnapshots.Entity;
				parameter.TemplateMustExists = true;

				parameters.Add(parameter);
			}


			if (!(bool)model.CollapseSnapshots.Entity)
			{
				if ((missingTemplatesFromVms == null || missingTemplatesFromVms.Count > 0))
				{
					ConfirmationModel confirmModel = new ConfirmationModel();
					ConfirmWindow = confirmModel;
					confirmModel.Title = "Template(s) not Found on Export Domain";
					confirmModel.HashName = "template_not_found_on_export_domain";

					confirmModel.Message = missingTemplatesFromVms == null ?
						"Could not read templates from Export Domain" :
						"The following templates are missing on the target Export Domain:";
					confirmModel.Items = missingTemplatesFromVms;

					confirmModel.Commands.Add(
						new UICommand("OnExportNoTemplates", this)
						{
							Title = "OK",
							IsDefault = true
						});
					confirmModel.Commands.Add(
						new UICommand("CancelConfirmation", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
				}
				else
				{
					if (model.Progress != null)
					{
						return;
					}

					model.StartProgress(null);

					Frontend.RunMultipleAction(VdcActionType.ExportVm, parameters,
						result =>
						{
							ExportVmModel localModel = (ExportVmModel)result.State;

							localModel.StopProgress();
							Cancel();
						},
						model
					);
				}
			}
			else
			{
				if (model.Progress != null)
				{
					return;
				}

				foreach (VdcActionParametersBase item in parameters)
				{
					MoveVmParameters parameter = (MoveVmParameters)item;
					parameter.TemplateMustExists = false;
				}


				model.StartProgress(null);

				Frontend.RunMultipleAction(VdcActionType.ExportVm, parameters,
					result =>
					{
						ExportVmModel localModel = (ExportVmModel)result.State;

						localModel.StopProgress();
						Cancel();
					},
					model
				);
			}
		}

		private void OnExportNoTemplates()
		{
			ExportVmModel model = (ExportVmModel)Window;
			Guid storageDomainId = ((storage_domains)model.Storage.SelectedItem).id;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				MoveVmParameters parameters = new MoveVmParameters(a.vm_guid, storageDomainId);
				parameters.ForceOverride = (bool)model.ForceOverride.Entity;
				parameters.CopyCollapse = (bool)model.CollapseSnapshots.Entity;
				parameters.TemplateMustExists = false;

				list.Add(parameters);
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.ExportVm, list,
				result =>
				{
					ExportVmModel localModel = (ExportVmModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void RunOnce()
		{
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				return;
			}

			RunOnceModel model = new RunOnceModel();
			Window = model;
			model.Title = "Run Virtual Machine(s)";
			model.HashName = "run_virtual_machine";
			model.AttachIso.Entity = false;

			List<string> images = DataProvider.GetIrsImageList(vm.storage_pool_id, false);
			model.IsoImage.Items = images;
			//model.IsoImage.Value = images.FirstOrDefault();
			model.IsoImage.SelectedItem = Linq.FirstOrDefault(images);
			model.AttachFloppy.Entity = false;
			images = DataProvider.GetFloppyImageList(vm.storage_pool_id, false);

			if (DataProvider.IsWindowsOsType(vm.vm_os))
			{
				// Add a pseudo floppy disk image used for Windows' sysprep.
				if (!vm.is_initialized)
				{
					images.Insert(0, @"[sysprep]");
					model.AttachFloppy.Entity = true;
				}
				else
				{
					images.Add(@"[sysprep]");
				}
			}
			model.FloppyImage.Items = images;
			//model.FloppyImage.Value = images.FirstOrDefault();
			model.FloppyImage.SelectedItem = Linq.FirstOrDefault(images);
			model.RunAsStateless.Entity = vm.is_stateless;
			model.HwAcceleration = true;

			//Boot sequence.
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmInterfacesByVmId,
				new GetVmByVmIdParameters(vm.vm_guid));

			bool hasNics = returnValue != null
				&& returnValue.Succeeded
				&& ((List<VmNetworkInterface>)returnValue.ReturnValue).Count > 0;

			if (!hasNics)
			{
				BootSequenceModel bootSequenceModel = model.BootSequence;
				bootSequenceModel.NetworkOption.IsChangable = false;
				bootSequenceModel.NetworkOption.ChangeProhibitionReasons.Add("Virtual Machine must have at least one network interface defined to boot from network.");
			}
			//passing Kernel parameters
			model.Kernel_parameters.Entity = vm.kernel_params;
			model.Kernel_path.Entity = vm.kernel_url;
			model.Initrd_path.Entity = vm.initrd_url;
			model.CustomProperties.Entity = vm.CustomProperties;
			model.CustomProperties.IsChangable = DataProvider.IsSupportCustomProperties(vm.vds_group_compatibility_version.ToString());
			model.IsLinux_Unassign_UnknownOS = DataProvider.IsLinuxOsType(vm.vm_os)
											|| vm.vm_os == VmOsType.Unassigned
											|| vm.vm_os == VmOsType.Other;
			model.IsWindowsOS = DataProvider.IsWindowsOsType(vm.vm_os);
			model.IsVmFirstRun.Entity = !vm.is_initialized;
			model.SysPrepDomainName.SelectedItem = vm.vm_domain;

			// Update Domain list
			AsyncDataProvider.GetDomainList(new AsyncQuery(model,
				(target, returnValue1) =>
				{
					RunOnceModel runOnceModel = (RunOnceModel)target;
					IList<string> domains = (IList<string>)returnValue1;

					// Get last selected domain
					string oldDomain = (string)runOnceModel.SysPrepDomainName.SelectedItem;

					if (oldDomain != null && !oldDomain.Equals(String.Empty) && !domains.Contains(oldDomain))
					{
						domains.Insert(0, oldDomain);	
					}

					runOnceModel.SysPrepDomainName.Items = domains;
					runOnceModel.SysPrepDomainName.SelectedItem = oldDomain ?? Linq.FirstOrDefault(domains);
				}), true
			);

			//Display protocols.
			EntityModel vncProtocol =
				new EntityModel
				{
					Title = "VNC",
					Entity = DisplayType.vnc
				};

			EntityModel qxlProtocol =
				new EntityModel()
				{
					Title = "Spice",
					Entity = DisplayType.qxl
				};

			List<EntityModel> items = new List<EntityModel>();
			items.Add(vncProtocol);
			items.Add(qxlProtocol);
			model.DisplayProtocol.Items = items;
			model.DisplayProtocol.SelectedItem = vm.default_display_type == DisplayType.vnc ? vncProtocol : qxlProtocol;

			model.CustomPropertiesKeysList = this.CustomPropertiesKeysList;

			model.Commands.Add(
				new UICommand("OnRunOnce", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRunOnce()
		{
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				Cancel();
				return;
			}

			RunOnceModel model = (RunOnceModel)Window;

			if (!model.Validate())
			{
				return;
			}

			BootSequenceModel bootSequenceModel = model.BootSequence;

			RunVmOnceParams param =
				new RunVmOnceParams()
				{
					VmId = vm.vm_guid,
					BootSequence = bootSequenceModel.Sequence,
					DiskPath = (bool)model.AttachIso.Entity ? (string)model.IsoImage.SelectedItem : String.Empty,
					FloppyPath = model.FloppyImagePath,
					KvmEnable = model.HwAcceleration,
					RunAndPause = (bool)model.RunAndPause.Entity,
					AcpiEnable = true,
					RunAsStateless = (bool)model.RunAsStateless.Entity,
					Reinitialize = model.Reinitialize,
					CustomProperties = (string)model.CustomProperties.Entity
				};

			//kernel params
			if (model.Kernel_path.Entity != null)
			{
				param.kernel_url = (string)model.Kernel_path.Entity;
			}
			if (model.Kernel_parameters.Entity != null)
			{
				param.kernel_params = (string)model.Kernel_parameters.Entity;
			}
			if (model.Initrd_path.Entity != null)
			{
				param.initrd_url = (string)model.Initrd_path.Entity;
			}

			//Sysprep params
			if (model.SysPrepDomainName.SelectedItem != null)
			{
				param.SysPrepDomainName = (string)model.SysPrepDomainName.SelectedItem;
			}
			if (model.SysPrepUserName.Entity != null)
			{
				param.SysPrepUserName = (string)model.SysPrepUserName.Entity;
			}
			if (model.SysPrepPassword.Entity != null)
			{
				param.SysPrepPassword = (string)model.SysPrepPassword.Entity;
			}

			EntityModel displayProtocolSelectedItem = (EntityModel)model.DisplayProtocol.SelectedItem;
			param.UseVnc = (DisplayType)displayProtocolSelectedItem.Entity == DisplayType.vnc;

			Frontend.RunActionAsyncroniousely(VdcActionType.RunVmOnce, param);

			Cancel();
		}

		private void NewTemplate()
		{
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				return;
			}

			if (Window != null)
			{
				return;
			}

			UnitVmModel model = new UnitVmModel(new NewTemplateVmModelBehavior(vm));
			Window = model;
			model.Title = "New Template";
			model.HashName = "new_template";
			model.IsNew = true;
			model.VmType = vm.vm_type;

			model.Initialize(SystemTreeSelectedItem);


			model.Commands.Add(
				new UICommand("OnNewTemplate", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void DisableNewTemplateModel(VmModel model, String errMessage)
		{
			model.Message = errMessage;

			model.Name.IsChangable = false;
			model.Description.IsChangable = false;
			model.Cluster.IsChangable = false;
			model.StorageDomain.IsChangable = false;
			model.IsTemplatePublic.IsChangable = false;

			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Close",
					IsCancel = true
				});
		}

		private void OnNewTemplate()
		{
			UnitVmModel model = (UnitVmModel)Window;
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				Cancel();
				return;
			}

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			string name = (string)model.Name.Entity;

			//Check name unicitate.
			if (!DataProvider.IsTemplateNameUnique(name))
			{
				model.Name.IsValid = false;
				model.Name.InvalidityReasons.Add("Name must be unique.");
				model.IsGeneralTabValid = false;
				return;
			}

			VM newvm =
				new VM
				{
					vm_guid = vm.vm_guid,
					vm_type = model.VmType,
					vm_os = (VmOsType)model.OSType.SelectedItem,
					num_of_monitors = (int)model.NumOfMonitors.SelectedItem,
					vm_domain = model.Domain.IsAvailable ? (string)model.Domain.SelectedItem : String.Empty,
					vm_mem_size_mb = (int)model.MemSize.Entity,
					MinAllocatedMem = (int)model.MinAllocatedMemory.Entity,
					vds_group_id = ((VDSGroup)model.Cluster.SelectedItem).ID,
					time_zone = model.TimeZone.IsAvailable && model.TimeZone.SelectedItem != null ? ((KeyValuePair<string, string>)model.TimeZone.SelectedItem).Key : String.Empty,
					num_of_sockets = (int)model.NumOfSockets.Entity,
					cpu_per_socket = (int)model.TotalCPUCores.Entity / (int)model.NumOfSockets.Entity,
					usb_policy = (UsbPolicy)model.UsbPolicy.SelectedItem,
					is_auto_suspend = false,
					is_stateless = (bool)model.IsStateless.Entity,
					default_boot_sequence = model.BootSequence,
					auto_startup = (bool)model.IsHighlyAvailable.Entity,
					iso_path = model.CdImage.IsChangable ? (string)model.CdImage.SelectedItem : String.Empty,
					initrd_url = vm.initrd_url,
					kernel_url = vm.kernel_url,
					kernel_params = vm.kernel_params
				};

			EntityModel displayProtocolSelectedItem = (EntityModel)model.DisplayProtocol.SelectedItem;
			newvm.default_display_type = (DisplayType)displayProtocolSelectedItem.Entity;

			EntityModel prioritySelectedItem = (EntityModel)model.Priority.SelectedItem;
			newvm.priority = (int)prioritySelectedItem.Entity;

			AddVmTemplateParameters addVmTemplateParameters = new AddVmTemplateParameters(newvm,
					(string)model.Name.Entity,
					(string)model.Description.Entity);
			addVmTemplateParameters.DestinationStorageDomainId = ((storage_domains)model.StorageDomain.SelectedItem).id;
			addVmTemplateParameters.PublicUse = (bool)model.IsTemplatePublic.Entity;
			
			
			model.StartProgress(null);

			Frontend.RunAction(VdcActionType.AddVmTemplate,
				addVmTemplateParameters,
				result =>
				{
					VmListModel vmListModel = (VmListModel)result.State;
					vmListModel.Window.StopProgress();
					VdcReturnValueBase returnValueBase = result.ReturnValue;
					if (returnValueBase != null && returnValueBase.Succeeded)
					{
						vmListModel.Cancel();
					}
				},
				this
			);
		}

		private void Migrate()
		{
			VM vm = (VM)SelectedItem;
			if (vm == null)
			{
				return;
			}

			if (Window != null)
			{
				return;
			}

			MigrateModel model = new MigrateModel();
			Window = model;
			model.Title = "Migrate Virtual Machine(s)";
			model.HashName = "migrate_virtual_machine";
			model.IsAutoSelect = true;
			model.VmList = Linq.Cast<VM>(SelectedItems);
			List<VDS> hosts = DataProvider.GetUpHostListByCluster(vm.vds_group_name);
			model.VmsOnSameCluster = true;
			Guid? run_on_vds = null;
			bool allRunOnSameVds = true;
			
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				if (!a.vds_group_id.Equals(((VM)SelectedItems[0]).vds_group_id))
				{
					model.VmsOnSameCluster = false;
				}
				if (run_on_vds == null)
				{
					run_on_vds = a.run_on_vds.Value;
				}
				else if(allRunOnSameVds && !run_on_vds.Equals(a.run_on_vds.Value))
				{
					allRunOnSameVds = false;
				}
			}

			model.IsHostSelAvailable = model.VmsOnSameCluster && hosts.Count > 0;

			if (model.VmsOnSameCluster && allRunOnSameVds)
			{
				VDS runOnSameVDS = null;
				foreach (VDS host in hosts)
				{
					if(host.vds_id.Equals(run_on_vds))
					{
						runOnSameVDS = host;
					}
				}
				hosts.Remove(runOnSameVDS);
			}
			if (hosts.Count == 0)
			{
				model.IsHostSelAvailable = false;

				if (allRunOnSameVds)
				{
					model.NoSelAvailable = true;

					model.Commands.Add(
						new UICommand("Cancel", this)
							{
								Title = "Close",
								IsDefault = true,
								IsCancel = true
							});
				}
			}
			else
			{
				model.Hosts.Items = hosts;
				model.Hosts.SelectedItem = Linq.FirstOrDefault(hosts);

				model.Commands.Add(
					new UICommand("OnMigrate", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnMigrate()
		{
			MigrateModel model = (MigrateModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			model.StartProgress(null);

			if (model.IsAutoSelect)
			{
				List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
				foreach (object item in SelectedItems)
				{
					VM a = (VM)item;
					list.Add(new MigrateVmParameters(true, a.vm_guid));
				}

				Frontend.RunMultipleAction(VdcActionType.MigrateVm, list,
					result =>
					{
						MigrateModel localModel = (MigrateModel)result.State;

						localModel.StopProgress();
						Cancel();
					},
					model
				);
			}
			else
			{
				List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
				foreach (object item in SelectedItems)
				{
					VM a = (VM)item;

					if (a.run_on_vds.Value.Equals(((VDS)model.Hosts.SelectedItem).vds_id))
					{
						continue;
					}
					
					list.Add(new MigrateVmToServerParameters(true, a.vm_guid, ((VDS)model.Hosts.SelectedItem).vds_id));
				}

				Frontend.RunMultipleAction(VdcActionType.MigrateVmToServer, list,
					result =>
					{
						MigrateModel localModel = (MigrateModel)result.State;

						localModel.StopProgress();
						Cancel();
					},
					model
				);
			}
		}

		private void Shutdown()
		{
			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Shut down Virtual Machine(s)";
			model.HashName = "shut_down_virtual_machine";
			model.Message = "Are you sure you want to Shut down the following Virtual Machines?";
			//model.Items = SelectedItems.Cast<VM>().Select(a => a.vm_name);
			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				items.Add(a.vm_name);
			}
			model.Items = items;


			model.Commands.Add(
				new UICommand("OnShutdown", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnShutdown()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				list.Add(new ShutdownVmParameters(a.vm_guid, true));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.ShutdownVm, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void Stop()
		{
			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Stop Virtual Machine(s)";
			model.HashName = "stop_virtual_machine";
			model.Message = "Are you sure you want to Stop the following Virtual Machines?";
			//model.Items = SelectedItems.Cast<VM>().Select(a => a.vm_name);
			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				items.Add(a.vm_name);
			}
			model.Items = items;


			model.Commands.Add(
				new UICommand("OnStop", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnStop()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				list.Add(new StopVmParameters(a.vm_guid, StopVmTypeEnum.NORMAL));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.StopVm, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void Pause()
		{
			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				list.Add(new HibernateVmParameters(a.vm_guid));
			}

			Frontend.RunMultipleAction(VdcActionType.HibernateVm, list,
				result =>
				{
				},
				null
			);
		}

		private void Run()
		{
			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				//use sysprep iff the vm is not initialized and vm has Win OS 
				bool reinitialize = !a.is_initialized && DataProvider.IsWindowsOsType(a.vm_os);
				list.Add(new RunVmParams(a.vm_guid) { Reinitialize = reinitialize });
			}

			Frontend.RunMultipleAction(VdcActionType.RunVm, list,
				result =>
				{
				},
				null
			);
		}

		private void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VM a = (VM)item;
				list.Add(new RemoveVmParameters(a.vm_guid, false));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVm, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void OnSave()
		{
			UnitVmModel model = (UnitVmModel)Window;
			VM selectedItem = (VM)SelectedItem;
			if (model.IsNew == false && selectedItem == null)
			{
				Cancel();
				return;
			}

			currentVm = model.IsNew ? new VM() : (VM)Cloner.Clone(selectedItem);

			if (!model.Validate())
			{
				return;
			}

			string name = (string)model.Name.Entity;

			//Check name unicitate.
			if (!DataProvider.IsVmNameUnique(name) && String.Compare(name, currentVm.vm_name, true) != 0)
			{
				model.Name.IsValid = false;
				model.Name.InvalidityReasons.Add("Name must be unique.");
				model.IsGeneralTabValid = false;
				return;
			}

			//Save changes.
			VmTemplate template = (VmTemplate)model.Template.SelectedItem;

			currentVm.vm_type = model.VmType;
			currentVm.vmt_guid = template.Id;
			currentVm.vm_name = name;
			currentVm.vm_os = (VmOsType)model.OSType.SelectedItem;
			currentVm.num_of_monitors = (int)model.NumOfMonitors.SelectedItem;
			currentVm.vm_description = (string)model.Description.Entity;
			currentVm.vm_domain = model.Domain.IsAvailable ? (string)model.Domain.SelectedItem : String.Empty;
			currentVm.vm_mem_size_mb = (int)model.MemSize.Entity;
			currentVm.MinAllocatedMem = (int)model.MinAllocatedMemory.Entity;
			Guid newClusterID = ((VDSGroup)model.Cluster.SelectedItem).ID;
			currentVm.vds_group_id = newClusterID;
			currentVm.time_zone = (model.TimeZone.IsAvailable && model.TimeZone.SelectedItem != null) ? ((KeyValuePair<string, string>)model.TimeZone.SelectedItem).Key : String.Empty;
			currentVm.num_of_sockets = (int)model.NumOfSockets.Entity;
			currentVm.cpu_per_socket = (int)model.TotalCPUCores.Entity / (int)model.NumOfSockets.Entity;
			currentVm.usb_policy = (UsbPolicy)model.UsbPolicy.SelectedItem;
			currentVm.is_auto_suspend = false;
			currentVm.is_stateless = (bool)model.IsStateless.Entity;
			currentVm.default_boot_sequence = model.BootSequence;
			currentVm.iso_path = model.CdImage.IsChangable ? (string)model.CdImage.SelectedItem : String.Empty;
			currentVm.auto_startup = (bool)model.IsHighlyAvailable.Entity;

			currentVm.initrd_url = (string)model.Initrd_path.Entity;
			currentVm.kernel_url = (string)model.Kernel_path.Entity;
			currentVm.kernel_params = (string)model.Kernel_parameters.Entity;

			currentVm.CustomProperties = (string)model.CustomProperties.Entity;

			EntityModel displayProtocolSelectedItem = (EntityModel)model.DisplayProtocol.SelectedItem;
			currentVm.default_display_type = (DisplayType)displayProtocolSelectedItem.Entity;

			EntityModel prioritySelectedItem = (EntityModel)model.Priority.SelectedItem;
			currentVm.priority = (int)prioritySelectedItem.Entity;


			VDS defaultHost = (VDS)model.DefaultHost.SelectedItem;
			if ((bool)model.IsAutoAssign.Entity)
			{
				currentVm.dedicated_vm_for_vds = null;
			}
			else
			{
				currentVm.dedicated_vm_for_vds = defaultHost.vds_id;
			}

			currentVm.MigrationSupport = MigrationSupport.MIGRATABLE;
			if ((bool)model.RunVMOnSpecificHost.Entity)
			{
				currentVm.MigrationSupport = MigrationSupport.PINNED_TO_HOST;
			}
			else if ((bool)model.DontMigrateVM.Entity)
			{
				currentVm.MigrationSupport = MigrationSupport.IMPLICITLY_NON_MIGRATABLE;
			}

			if (model.IsNew)
			{
				if (currentVm.vmt_guid.Equals(Guid.Empty))
				{
					if (model.Progress != null)
					{
						return;
					}

					model.StartProgress(null);

					Frontend.RunAction(VdcActionType.AddVmFromScratch, 
						new AddVmFromScratchParameters(currentVm, new List<DiskImageBase>(), Guid.Empty),
							result =>
							{
								VmListModel vmListModel = (VmListModel)result.State;
								vmListModel.Window.StopProgress();
								VdcReturnValueBase returnValueBase = result.ReturnValue;
								if (returnValueBase != null && returnValueBase.Succeeded)
								{
									vmListModel.Cancel();
									vmListModel.GuideContext = returnValueBase.ActionReturnValue;
									vmListModel.UpdateActionAvailability();
									vmListModel.GuideCommand.Execute();
								}
							}, this);
				}
				else
				{
					if (model.Progress != null)
					{
						return;
					}

					storage_domains storageDomain = (storage_domains)model.StorageDomain.SelectedItem;

					if ((bool)((EntityModel)model.Provisioning.SelectedItem).Entity)
					{
						List<DiskImage> templateDisks = DataProvider.GetTemplateDiskList(template.Id);
						foreach (DiskImage templateDisk in templateDisks)
						{
							//DiskModel disk = model.Disks.First(a => a.Name == templateDisk.internal_drive_mapping);
							DiskModel disk = null;
							foreach (DiskModel a in model.Disks)
							{
								if (a.Name == templateDisk.internal_drive_mapping)
								{
									disk = a;
									break;
								}
							}

							if (disk != null)
							{
								templateDisk.volume_type = (VolumeType)disk.VolumeType.SelectedItem;
								templateDisk.volume_format = DataProvider.GetDiskVolumeFormat((VolumeType)disk.VolumeType.SelectedItem, storageDomain.storage_type);
							}
						}

						Dictionary<string, DiskImageBase> dict = new Dictionary<string, DiskImageBase>();
						foreach (DiskImage a in templateDisks)
						{
							dict.Add(a.internal_drive_mapping, a);
						}

						model.StartProgress(null);

						Frontend.RunAction(VdcActionType.AddVmFromTemplate,
						new AddVmFromTemplateParameters(currentVm, dict, storageDomain.id),
							result =>
							{
								VmListModel vmListModel = (VmListModel)result.State;
								vmListModel.Window.StopProgress();
								VdcReturnValueBase returnValueBase = result.ReturnValue;
								if (returnValueBase != null && returnValueBase.Succeeded)
								{
									vmListModel.Cancel();
								}
							}, this);
					}
					else
					{
						if (model.Progress != null)
						{
							return;
						}

						model.StartProgress(null);

						Frontend.RunAction(VdcActionType.AddVm,
						new VmManagementParametersBase(currentVm)
							{
								StorageDomainId = storageDomain.id
							},
							result =>
							{
								VmListModel vmListModel = (VmListModel)result.State;
								vmListModel.Window.StopProgress();
								VdcReturnValueBase returnValueBase = result.ReturnValue;
								if (returnValueBase != null && returnValueBase.Succeeded)
								{
									vmListModel.Cancel();
								}
							}, this);
					}
				}
			}
			else // Update existing VM -> consists of editing VM cluster, and if succeeds - editing VM:
			{
				if (model.Progress != null)
				{
					return;
				}

				// runEditVM: should be true if Cluster hasn't changed or if 
				// Cluster has changed and Editing it in the Backend has succeeded:
				Guid oldClusterID = selectedItem.vds_group_id;
				if (oldClusterID.Equals(newClusterID) == false)
				{
					ChangeVMClusterParameters parameters =
						new ChangeVMClusterParameters(newClusterID, currentVm.vm_guid);

					model.StartProgress(null);

					Frontend.RunAction(VdcActionType.ChangeVMCluster, parameters,
						result =>
						{
							VmListModel vmListModel = (VmListModel)result.State;
							VdcReturnValueBase returnValueBase = result.ReturnValue;
							if (returnValueBase != null && returnValueBase.Succeeded)
							{
								Frontend.RunAction(VdcActionType.UpdateVm,
									new VmManagementParametersBase(vmListModel.currentVm),
										result1 =>
										{
											VmListModel vmListModel1 = (VmListModel)result1.State;
											vmListModel1.Window.StopProgress();
											VdcReturnValueBase retVal = result1.ReturnValue;
											bool isSucceeded = retVal.Succeeded;
											if (retVal != null && isSucceeded)
											{
												vmListModel1.Cancel();
											}
										}, vmListModel);
							}
							else
							{
								vmListModel.Window.StopProgress();
							}
						}, this);
				}
				else
				{
					if (model.Progress != null)
					{
						return;
					}

					model.StartProgress(null);

					Frontend.RunAction(VdcActionType.UpdateVm,
						new VmManagementParametersBase(currentVm),
							result =>
							{
								VmListModel vmListModel = (VmListModel)result.State;
								vmListModel.Window.StopProgress();
								VdcReturnValueBase returnValueBase = result.ReturnValue;
								if (returnValueBase != null && returnValueBase.Succeeded)
								{
									vmListModel.Cancel();
								}
							}, this);
				}
			}
		}

		private void RetrieveIsoImages()
		{
			VM vm = SelectedItem as VM;
			if (vm == null)
			{
				return;
			}

			Guid storagePoolId = vm.storage_pool_id;

			IsoImages.Clear();
			
			ChangeCDModel ejectModel = new ChangeCDModel { Title = ConsoleModel.EjectLabel };
			ejectModel.ExecutedEvent.addListener(this);
			IsoImages.Add(ejectModel);
			
			List<string> list = DataProvider.GetIrsImageList(storagePoolId, false);
			if (list.Count > 0)
			{
				foreach (string iso in list)
				{
					ChangeCDModel model = new ChangeCDModel { Title = iso };
					//model.Executed += changeCD;
					model.ExecutedEvent.addListener(this);
					IsoImages.Add(model);
				}

			}
			else
			{
				IsoImages.Add(new ChangeCDModel { Title = "No CDs" });
			}
			
			
		}

		private void changeCD(object sender, EventArgs e)
		{
			ChangeCDModel model = (ChangeCDModel)sender;

			//TODO: Patch!
			string isoName = model.Title;
			if (isoName == "No CDs")
			{
				return;
			}

			VM vm = SelectedItem as VM;
			if (vm == null)
			{
				return;
			}


			Frontend.RunMultipleAction(VdcActionType.ChangeDisk,
				new List<VdcActionParametersBase>
				{
					new ChangeDiskCommandParameters(vm.vm_guid, isoName == ConsoleModel.EjectLabel ? String.Empty : isoName)
				},
				result =>
				{
				},
				null
			);
		}

		public void Cancel()
		{
			Frontend.Unsubscribe();

			CancelConfirmation();

			GuideContext = null;
			Window = null;

			UpdateActionAvailability();
		}

		private void CancelConfirmation()
		{
			ConfirmWindow = null;
		}

		public void CancelError()
		{
			ErrorWindow = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();

			UpdateActionAvailability();
			UpdateConsoleModels();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();

			UpdateActionAvailability();
			UpdateConsoleModels();
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);

			switch (e.PropertyName)
			{
				case "status":
					UpdateActionAvailability();
					break;

				case "display_type":
					UpdateConsoleModels();
					break;
			}
		}

		private void UpdateActionAvailability()
		{
			IList items = SelectedItems != null && SelectedItem != null ? SelectedItems : new ArrayList();

			EditCommand.IsExecutionAllowed = items.Count == 1;
			RemoveCommand.IsExecutionAllowed = items.Count > 0 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.RemoveVm);
			RunCommand.IsExecutionAllowed = items.Count > 0 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.RunVm);
			PauseCommand.IsExecutionAllowed = items.Count > 0 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.HibernateVm);
			ShutdownCommand.IsExecutionAllowed = items.Count > 0 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.ShutdownVm);
			StopCommand.IsExecutionAllowed = items.Count > 0 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.StopVm);
			MigrateCommand.IsExecutionAllowed = items.Count > 0 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.MigrateVm);
			NewTemplateCommand.IsExecutionAllowed = items.Count == 1 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.AddVmTemplate);
			RunOnceCommand.IsExecutionAllowed = items.Count == 1 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.RunVmOnce);
			ExportCommand.IsExecutionAllowed = items.Count > 0 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.ExportVm);
			MoveCommand.IsExecutionAllowed = items.Count == 1 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.MoveVm);
			RetrieveIsoImagesCommand.IsExecutionAllowed = items.Count == 1 && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.ChangeDisk);
			AssignTagsCommand.IsExecutionAllowed = items.Count > 0;

			GuideCommand.IsExecutionAllowed = GuideContext != null
				|| (SelectedItem != null
				&& SelectedItems != null
				&& SelectedItems.Count == 1);
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ChangeCDModel.ExecutedEventDefinition))
			{
				changeCD(sender, args);
			}
			else if (ev.Equals(ConsoleModel.ErrorEventDefinition) && sender is SpiceConsoleModel)
			{
				SpiceConsoleModel_Error(sender, (ErrorCodeEventArgs)args);
			}
		}

		private void SpiceConsoleModel_Error(object sender, ErrorCodeEventArgs e)
		{
			ResourceManager rm = new ResourceManager("UICommon.Resources.RdpErrors.RdpErrors", Assembly.GetExecutingAssembly());

			ConfirmationModel model = new ConfirmationModel();
			if (ErrorWindow == null)
			{
				ErrorWindow = model;
			}
			model.Title = "Console Disconnected";
			model.HashName = "console_disconnected";
			model.Message = String.Format("Error connecting to Virtual Machine using Spice:\n{0}", rm.GetString("E" + e.ErrorCode));

			rm.ReleaseAllResources();


			model.Commands.Add(
				new UICommand("CancelError", this)
				{
					Title = "Close",
					IsDefault = true,
					IsCancel = true
				});
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewServerCommand)
			{
				NewServer();
			}
			else if (command == NewDesktopCommand)
			{
				NewDesktop();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == RunCommand)
			{
				Run();
			}
			else if (command == PauseCommand)
			{
				Pause();
			}
			else if (command == StopCommand)
			{
				Stop();
			}
			else if (command == ShutdownCommand)
			{
				Shutdown();
			}
			else if (command == MigrateCommand)
			{
				Migrate();
			}
			else if (command == NewTemplateCommand)
			{
				NewTemplate();
			}
			else if (command == RunOnceCommand)
			{
				RunOnce();
			}
			else if (command == ExportCommand)
			{
				Export();
			}
			else if (command == MoveCommand)
			{
				Move();
			}
			else if (command == GuideCommand)
			{
				Guide();
			}
			else if (command == RetrieveIsoImagesCommand)
			{
				RetrieveIsoImages();
			}
			else if (command == AssignTagsCommand)
			{
				AssignTags();
			}
			else if (command.Name == "OnAssignTags")
			{
				OnAssignTags();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnMove")
			{
				OnMove();
			}
			else if (command.Name == "OnExport")
			{
				OnExport();
			}
			else if (command.Name == "OnExportNoTemplates")
			{
				OnExportNoTemplates();
			}
			else if (command.Name == "CancelConfirmation")
			{
				CancelConfirmation();
			}
			else if (command.Name == "CancelError")
			{
				CancelError();
			}
			else if (command.Name == "OnRunOnce")
			{
				OnRunOnce();
			}
			else if (command.Name == "OnNewTemplate")
			{
				OnNewTemplate();
			}
			else if (command.Name == "OnMigrate")
			{
				OnMigrate();
			}
			else if (command.Name == "OnShutdown")
			{
				OnShutdown();
			}
			else if (command.Name == "OnStop")
			{
				OnStop();
			}
		}

		#region ISupportSystemTreeContext Members

		private SystemTreeItemModel systemTreeSelectedItem;
		public SystemTreeItemModel SystemTreeSelectedItem
		{
			get { return systemTreeSelectedItem; }
			set
			{
				systemTreeSelectedItem = value;
				OnPropertyChanged(new PropertyChangedEventArgs("SystemTreeSelectedItem"));
			}
		}

		#endregion
	}
}
