using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using org.ovirt.engine.ui.uicompat;
using VdcCommon;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public class VmInterfaceListModel : SearchableListModel
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		public VmInterfaceListModel()
		{
			Title = "Network Interfaces";

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			if (Entity != null)
			{
				SearchCommand.Execute();
			}

			UpdateActionAvailability();
		}

		protected override void SyncSearch()
		{
			VM vm = (VM)Entity;
			base.SyncSearch(VdcQueryType.GetVmInterfacesByVmId, new GetVmByVmIdParameters(vm.vm_guid));
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			VM vm = (VM)Entity;

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetVmInterfacesByVmId, new GetVmByVmIdParameters(vm.vm_guid));
			Items = AsyncResult.Data;
		}

		private void New()
		{
			VM vm = (VM)Entity;

			if (Window != null)
			{
				return;
			}

			List<VmNetworkInterface> interfaces = Linq.Cast<VmNetworkInterface>(Items);
			string newNicName = DataProvider.GetNewNicName(interfaces);

			VmInterfaceModel model = new VmInterfaceModel();
			Window = model;
			model.Title = "New Network Interface";
			model.HashName = "new_network_interface_vms";
			model.IsNew = true;
			model.NicType.Items = DataProvider.GetNicTypeList(vm.vm_os, false);
			model.NicType.SelectedItem = DataProvider.GetDefaultNicType(vm.vm_os);
			model.Name.Entity = newNicName;
			model.MAC.IsChangable = false;

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model1, Object result1)
				{
					VmInterfaceListModel vmInterfaceListModel = (VmInterfaceListModel)model1;
					VmInterfaceModel vmInterfaceModel = (VmInterfaceModel)vmInterfaceListModel.Window;

					List<network> networks = new List<network>();
					foreach (network a in (List<network>)result1)
					{
						if (a.Status == NetworkStatus.Operational)
						{
							networks.Add(a);
						}
					}

					if (vmInterfaceModel.IsNew)
					{
						vmInterfaceModel.Network.Items = networks;
						vmInterfaceModel.Network.SelectedItem = networks.Count > 0 ? networks[0] : null;
					}
					else
					{
						VmNetworkInterface nic = (VmNetworkInterface)vmInterfaceListModel.SelectedItem;

						vmInterfaceModel.Network.Items = networks;
						vmInterfaceModel.Network.SelectedItem = null;
						foreach (network a in networks)
						{
							if (a.name == nic.NetworkName)
							{
								vmInterfaceModel.Network.SelectedItem = a;
								break;
							}
						}
					}

				};//END_DELEGATE
			AsyncDataProvider.GetClusterNetworkList(_asyncQuery, vm.vds_group_id);


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Edit()
		{
			VM vm = (VM)Entity;
			VmNetworkInterface nic = (VmNetworkInterface)SelectedItem;

			if (Window != null)
			{
				return;
			}
		
			VmInterfaceModel model = new VmInterfaceModel();
			Window = model;
			model.Title = "Edit Network Interface";
			model.HashName = "edit_network_interface_vms";

			int? selectedNicType = nic.Type;
			List<VmInterfaceType> nicTypes = DataProvider.GetNicTypeList(vm.vm_os, (VmInterfaceType)selectedNicType == VmInterfaceType.rtl8139_pv);
			model.NicType.Items = nicTypes;

			if (!selectedNicType.HasValue || !nicTypes.Contains((VmInterfaceType)selectedNicType))
			{
				selectedNicType = (int)DataProvider.GetDefaultNicType(vm.vm_os);
			}

			model.NicType.SelectedItem = (VmInterfaceType)selectedNicType;

			model.Name.Entity = nic.Name;
			model.MAC.IsChangable = false;
			model.MAC.Entity = nic.MacAddress;

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model1, Object result1)
				{
					VmInterfaceListModel vmInterfaceListModel = (VmInterfaceListModel)model1;
					VmInterfaceModel vmInterfaceModel = (VmInterfaceModel)vmInterfaceListModel.Window;

					List<network> networks = new List<network>();
					foreach (network a in (List<network>)result1)
					{
						if (a.Status == NetworkStatus.Operational)
						{
							networks.Add(a);
						}
					}

					if (vmInterfaceModel.IsNew)
					{
						vmInterfaceModel.Network.Items = networks;
						vmInterfaceModel.Network.SelectedItem = networks.Count > 0 ? networks[0] : null;
					}
					else
					{
						VmNetworkInterface nic1 = (VmNetworkInterface)vmInterfaceListModel.SelectedItem;

						vmInterfaceModel.Network.Items = networks;
						vmInterfaceModel.Network.SelectedItem = null;
						foreach (network a in networks)
						{
							if (a.name == nic1.NetworkName)
							{
								vmInterfaceModel.Network.SelectedItem = a;
								break;
							}
						}
					}

				};//END_DELEGATE
			AsyncDataProvider.GetClusterNetworkList(_asyncQuery, vm.vds_group_id);


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnSave()
		{
			VM vm = (VM)Entity;
			VmInterfaceModel model = (VmInterfaceModel)Window;

			VmNetworkInterface nic = model.IsNew ? new VmNetworkInterface() : (VmNetworkInterface)Cloner.Clone((VmNetworkInterface)SelectedItem);

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			//Save changes.
			nic.Name = (string)model.Name.Entity;
			nic.NetworkName = ((network)model.Network.SelectedItem).name;
			if (model.NicType.SelectedItem == null)
			{
				nic.Type = null;
			}
			else
			{
				nic.Type = (int?)(VmInterfaceType)model.NicType.SelectedItem;
			}
			nic.MacAddress = model.MAC.IsChangable
				? (model.MAC.Entity == null ? null : ((string)(model.MAC.Entity)).ToLower())
				: model.IsNew
					? String.Empty
					: nic.MacAddress;


			model.StartProgress(null);

			Frontend.RunAction(model.IsNew ? VdcActionType.AddVmInterface : VdcActionType.UpdateVmInterface,
				new AddVmInterfaceParameters(vm.vm_guid, nic),
				result =>
				{
					VmInterfaceListModel localModel = (VmInterfaceListModel)result.State;

					localModel.PostOnSave(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnSave(VdcReturnValueBase returnValue)
		{
			VmInterfaceModel model = (VmInterfaceModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
			}
		}

		private void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Network Interface(s)";
			model.HashName = "remove_network_interface_vms";
			model.Message = "Network Interface(s)";
			
			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				VmNetworkInterface a = (VmNetworkInterface)item;
				items.Add(a.Name);
			}
			model.Items = items;

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			VM vm = (VM)Entity;
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VmNetworkInterface a = (VmNetworkInterface)item;
				RemoveVmInterfaceParameters parameters = new RemoveVmInterfaceParameters(vm.vm_guid, a.Id);
				list.Add(parameters);

			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVmInterface, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			VM vm = (VM)Entity;

			List<VM> items = new List<VM>();
			if (vm != null)
			{
				items.Add(vm);
			}

			NewCommand.IsExecutionAllowed = vm != null && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.AddVmInterface);
			EditCommand.IsExecutionAllowed = vm != null && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.UpdateVmInterface) && (SelectedItems != null && SelectedItems.Count == 1);
			RemoveCommand.IsExecutionAllowed = vm != null && VdcActionUtils.CanExecute(items, typeof(VM), VdcActionType.RemoveVmInterface) && (SelectedItems != null && SelectedItems.Count > 0);
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
		}
	}
}
