﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicommon.models.pools;
using org.ovirt.engine.ui.uicommon.models.vms;
using org.ovirt.engine.ui.uicompat;
using VdcCommon;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.Interfaces;

namespace org.ovirt.engine.ui.uicommon.models.userportal
{
	public interface IVmPoolResolutionService	
	{
		vm_pools ResolveVmPoolById(Guid id);
	}



	public class UserPortalListModel : IUserPortalListModel, IVmPoolResolutionService
	{
		#region Events

		public static EventDefinition SearchCompletedEventDefinition;
		public Event SearchCompletedEvent { get; private set; }

		#endregion

		#region Commands

		public UICommand NewDesktopCommand { get; private set; }
		public UICommand NewServerCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand SaveCommand { get; private set; }
		public UICommand RunOnceCommand { get; private set; }
		public UICommand ChangeCdCommand { get; private set; }
		public UICommand NewTemplateCommand { get; private set; }

		#endregion

		#region Properties

		private UnitVmModel vmModel;
		public UnitVmModel VmModel
		{
			get { return vmModel; }
			set
			{
				if (vmModel != value)
				{
					vmModel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("VmModel"));
				}
			}
		}

		private ConfirmationModel confirmationModel;
		public ConfirmationModel ConfirmationModel
		{
			get { return confirmationModel; }
			set
			{
				if (confirmationModel != value)
				{
					confirmationModel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmationModel"));
				}
			}
		}

		private RunOnceModel runOnceModel;
		public RunOnceModel RunOnceModel
		{
			get { return runOnceModel; }
			set
			{
				if (runOnceModel != value)
				{
					runOnceModel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("RunOnceModel"));
				}
			}
		}

		private AttachCdModel attachCdModel;
		public AttachCdModel AttachCdModel
		{
			get { return attachCdModel; }
			set
			{
				if (attachCdModel != value)
				{
					attachCdModel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("AttachCdModel"));
				}
			}
		}


		#endregion

		private EntityModel vmGeneralModel;
		private ListModel vmSnapshotListModel;
		private EntityModel vmMonitorModel;
		private ListModel vmDiskListModel;
		private ListModel vmInterfaceListModel;
		private ListModel permissionListModel;
		private ListModel vmEventListModel;
		private ListModel vmAppListModel;
		private EntityModel poolGeneralModel;
		private ListModel poolDiskListModel;
		private ListModel poolInterfaceListModel;
		public List<VM> vms { get; set; }
		public List<vm_pools> pools { get; set; }
		public VM tempVm { get; set; }
		public storage_domains storageDomain { get; set; }
		private List<string> CustomPropertiesKeysList;
		private readonly Dictionary<Guid, List<ConsoleModel>> cachedConsoleModels;

		static UserPortalListModel()
		{
			SearchCompletedEventDefinition = new EventDefinition("SearchCompleted", typeof(UserPortalListModel));
		}

		public UserPortalListModel()
		{
			SearchCompletedEvent = new Event(SearchCompletedEventDefinition);

			cachedConsoleModels = new Dictionary<Guid, List<ConsoleModel>>();

			NewDesktopCommand = new UICommand("NewDesktop", this);
			NewServerCommand = new UICommand("NewServer", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			SaveCommand = new UICommand("Save", this);
			RunOnceCommand = new UICommand("RunOnce", this);
			ChangeCdCommand = new UICommand("ChangeCD", this);
			NewTemplateCommand = new UICommand("NewTemplate", this);

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object result)
										  {
											  UserPortalListModel userPortalListModel = (UserPortalListModel)model;
											  if (result != null)
											  {
												  userPortalListModel.CustomPropertiesKeysList = new List<string>();
												  foreach (string s in ((string)result).Split(';'))
												  {
													  userPortalListModel.CustomPropertiesKeysList.Add(s);
												  }
											  }

										  };//END_DELEGATE
			AsyncDataProvider.GetCustomPropertiesList(_asyncQuery);
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				UserPortalListModel userPortalListModel = (UserPortalListModel)model;
				userPortalListModel.vms = (List<VM>)((VdcQueryReturnValue)ReturnValue).ReturnValue;
				userPortalListModel.OnVmAndPoolLoad();
			};//END_DELEGATE

			Frontend.RunQuery(VdcQueryType.GetUserVmsByUserIdAndGroups, new GetUserVmsByUserIdAndGroupsParameters(Frontend.LoggedInUser.UserId), _asyncQuery);

			AsyncQuery _asyncQuery1 = new AsyncQuery();
			_asyncQuery1.Model = this;
			/*START_DELEGATE*/_asyncQuery1.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				UserPortalListModel userPortalListModel = (UserPortalListModel)model;
				userPortalListModel.pools = (List<vm_pools>)((VdcQueryReturnValue)ReturnValue).ReturnValue;
				userPortalListModel.OnVmAndPoolLoad();
			};//END_DELEGATE

			  Frontend.RunQuery(VdcQueryType.GetAllVmPoolsAttachedToUser, new GetAllVmPoolsAttachedToUserParameters(Frontend.LoggedInUser.UserId), _asyncQuery1);

		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();
			SyncSearch();
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			vmGeneralModel = new VmGeneralModel();
			vmGeneralModel.IsAvailable = false;

			vmSnapshotListModel = new VmSnapshotListModel();
			vmSnapshotListModel.IsAvailable = false;

			vmMonitorModel = new VmMonitorModel();
			vmMonitorModel.IsAvailable = false;

			vmDiskListModel = new VmDiskListModel();
			vmDiskListModel.IsAvailable = false;

			vmInterfaceListModel = new VmInterfaceListModel();
			vmInterfaceListModel.IsAvailable = false;

			permissionListModel = new PermissionListModel();
			permissionListModel.IsAvailable = false;

			vmEventListModel = new VmEventListModel();
			vmEventListModel.IsAvailable = false;

			vmAppListModel = new VmAppListModel();
			vmAppListModel.IsAvailable = false;

			poolGeneralModel = new PoolGeneralModel();
			poolGeneralModel.IsAvailable = false;

			poolDiskListModel = new PoolDiskListModel();
			poolDiskListModel.IsAvailable = false;

			poolInterfaceListModel = new PoolInterfaceListModel();
			poolInterfaceListModel.IsAvailable = false;

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(vmGeneralModel);
			list.Add(poolGeneralModel);
			list.Add(vmInterfaceListModel);
			list.Add(poolInterfaceListModel);
			list.Add(vmDiskListModel);
			list.Add(poolDiskListModel);
			list.Add(vmSnapshotListModel);
			list.Add(permissionListModel);
			list.Add(vmEventListModel);
			list.Add(vmAppListModel);
			list.Add(vmMonitorModel);

			DetailModels = list;

			permissionListModel.IsAvailable = true;
			vmEventListModel.IsAvailable = true;
			vmAppListModel.IsAvailable = true;
		}

		protected override object ProvideDetailModelEntity(object selectedItem)
		{
			//Each item in this list model is not a business entity,
			//therefore select an Entity property to provide it to
			//the detail models.

			EntityModel model = (EntityModel)selectedItem;
			if (model == null)
			{
				return null;
			}

			return model.Entity;
		}

		protected override void UpdateDetailsAvailability()
		{
			base.UpdateDetailsAvailability();


			UserPortalItemModel item = (UserPortalItemModel)SelectedItem;

			vmGeneralModel.IsAvailable = item != null && !item.IsPool;
			vmSnapshotListModel.IsAvailable = item != null && !item.IsPool;
			vmMonitorModel.IsAvailable = item != null && !item.IsPool;
			vmDiskListModel.IsAvailable = item != null && !item.IsPool;
			vmInterfaceListModel.IsAvailable = item != null && !item.IsPool;
			vmEventListModel.IsAvailable = item != null && !item.IsPool;

			poolGeneralModel.IsAvailable = item != null && item.IsPool;
			poolDiskListModel.IsAvailable = item != null && item.IsPool;
			poolInterfaceListModel.IsAvailable = item != null && item.IsPool;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewDesktopCommand)
			{
				NewDesktop();
			}
			if (command == NewServerCommand)
			{
				NewServer();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == SaveCommand)
			{
				OnSave();
			}
			else if (command == RunOnceCommand)
			{
				RunOnce();
			}
			else if (command == ChangeCdCommand)
			{
				ChangeCD();
			}
			else if (command == NewTemplateCommand)
			{
				NewTemplate();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnRunOnce")
			{
				OnRunOnce();
			}
			else if (command.Name == "OnChangeCD")
			{
				OnChangeCD();
			}
			else if (command.Name == "OnNewTemplate")
			{
				OnNewTemplate();
			}
		}

		private void NewTemplate()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (selectedItem == null)
			{
				return;
			}

			VM vm = (VM)selectedItem.Entity;
			VmModel = new UnitVmModel(new NewTemplateVmModelBehavior(vm));
			VmModel.Title = "New Template";
			VmModel.HashName = "new_template";
			VmModel.IsNew = true;
			VmModel.VmType = vm.vm_type;
			VmModel.Initialize(null);

			VmModel.Commands.Add(
				new UICommand("OnNewTemplate", this)
				{
					Title = "OK",
					IsDefault = true
				});
			VmModel.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnNewTemplate()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (selectedItem == null)
			{
				Cancel();
				return;
			}
			
			UnitVmModel model = vmModel;

			if (!model.Validate())
			{
				model.IsValid = false;
			}
			else
			{
				string name = (string)model.Name.Entity;

				//Check name unicitate.
				AsyncDataProvider.IsTemplateNameUnique(new AsyncQuery(this,
					(target, returnValue) =>
					{
						UserPortalListModel userPortalListModel = (UserPortalListModel)target;
						bool isNameUnique = (bool)returnValue;
						if (!isNameUnique)
						{
							userPortalListModel.VmModel.Name.InvalidityReasons.Clear();
							userPortalListModel.VmModel.Name.InvalidityReasons.Add("Name must be unique.");
							userPortalListModel.VmModel.Name.IsValid = false;
							userPortalListModel.VmModel.IsValid = false;
						}
						else
						{
							userPortalListModel.PostNameUniqueCheck(userPortalListModel);
							Cancel();
						}
					})
					, name
				);
			}
		}

		public void PostNameUniqueCheck(UserPortalListModel userPortalListModel)
		{
			UnitVmModel model = userPortalListModel.VmModel;
			UserPortalItemModel selectedItem = (UserPortalItemModel)userPortalListModel.SelectedItem;
			VM vm = (VM)selectedItem.Entity;

			VM newvm =
				new VM
				{
					vm_guid = vm.vm_guid,
					vm_type = model.VmType,
					vm_os = (VmOsType)model.OSType.SelectedItem,
					num_of_monitors = (int)model.NumOfMonitors.SelectedItem,
					vm_domain = model.Domain.IsAvailable ? (string)model.Domain.SelectedItem : String.Empty,
					vm_mem_size_mb = (int)model.MemSize.Entity,
					MinAllocatedMem = (int)model.MinAllocatedMemory.Entity,
					vds_group_id = ((VDSGroup)model.Cluster.SelectedItem).ID,
					time_zone = model.TimeZone.IsAvailable && model.TimeZone.SelectedItem != null ? ((KeyValuePair<string, string>)model.TimeZone.SelectedItem).Key : String.Empty,
					num_of_sockets = (int)model.NumOfSockets.Entity,
					cpu_per_socket = (int)model.TotalCPUCores.Entity / (int)model.NumOfSockets.Entity,
					usb_policy = (UsbPolicy)model.UsbPolicy.SelectedItem,
					is_auto_suspend = false,
					is_stateless = (bool)model.IsStateless.Entity,
					default_boot_sequence = model.BootSequence,
					auto_startup = (bool)model.IsHighlyAvailable.Entity,
					iso_path = model.CdImage.IsChangable ? (string)model.CdImage.SelectedItem : String.Empty,
					initrd_url = vm.initrd_url,
					kernel_url = vm.kernel_url,
					kernel_params = vm.kernel_params
				};

			EntityModel displayProtocolSelectedItem = (EntityModel)model.DisplayProtocol.SelectedItem;
			newvm.default_display_type = (DisplayType)displayProtocolSelectedItem.Entity;

			EntityModel prioritySelectedItem = (EntityModel)model.Priority.SelectedItem;
			newvm.priority = (int)prioritySelectedItem.Entity;

			AddVmTemplateParameters addVmTemplateParameters = new AddVmTemplateParameters(newvm,
					(string)model.Name.Entity,
					(string)model.Description.Entity);
			addVmTemplateParameters.DestinationStorageDomainId = ((storage_domains)model.StorageDomain.SelectedItem).id;
			addVmTemplateParameters.PublicUse = (bool)model.IsTemplatePublic.Entity;

			Frontend.RunAction(VdcActionType.AddVmTemplate,
				addVmTemplateParameters,
				result =>
				{
				},
				this
			);
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		private void RunOnce()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (selectedItem == null || selectedItem.Entity == null)
			{
				return;
			}

			VM vm = (VM)selectedItem.Entity;

			RunOnceModel model = new RunOnceModel();
			RunOnceModel = model;
			model.Title = "Run Virtual Machine(s)";
			model.HashName  = "run_virtual_machine";
			model.AttachIso.Entity = false;
			model.AttachFloppy.Entity = false;
			model.RunAsStateless.Entity = vm.is_stateless;
			model.RunAndPause.Entity = false;
			model.HwAcceleration = true;

			AsyncQuery _asyncQuery0 = new AsyncQuery();
			_asyncQuery0.Model = this;

			/*START_DELEGATE*/_asyncQuery0.asyncCallback = delegate(Object model0, Object result0)
			{
				if (result0 != null)
				{
					storage_domains isoDomain = (storage_domains)result0;
					UserPortalListModel thisUserPortalListModel = (UserPortalListModel)model0;

					AsyncQuery _asyncQuery01 = new AsyncQuery();
					_asyncQuery01.Model = thisUserPortalListModel;

					/*START_DELEGATE*/_asyncQuery01.asyncCallback = delegate(Object model1, Object result)
					{
						UserPortalListModel userPortalListModel = (UserPortalListModel)model1;
						RunOnceModel runOnceModel = userPortalListModel.RunOnceModel;
						List<string> images = (List<string>)result;
						runOnceModel.IsoImage.Items = images;

						if (runOnceModel.IsoImage.IsChangable && runOnceModel.IsoImage.SelectedItem == null)
						{
							runOnceModel.IsoImage.SelectedItem = Linq.FirstOrDefault(images);
						}
					};//END_DELEGATE
					AsyncDataProvider.GetIrsImageList(_asyncQuery01, isoDomain.id, false);

					AsyncQuery _asyncQuery02 = new AsyncQuery();
					_asyncQuery02.Model = thisUserPortalListModel;

					/*START_DELEGATE*/_asyncQuery02.asyncCallback = delegate(Object model2, Object result)
					{
						UserPortalListModel userPortalListModel = (UserPortalListModel)model2;
						UserPortalItemModel userPortalItemModel = (UserPortalItemModel)userPortalListModel.SelectedItem;
						RunOnceModel runOnceModel = userPortalListModel.RunOnceModel;
						VM selectedVM = (VM)userPortalItemModel.Entity;
						List<string> images = (List<string>)result;

						if (DataProvider.IsWindowsOsType(selectedVM.vm_os))
						{
							// Add a pseudo floppy disk image used for Windows' sysprep.
							if (!selectedVM.is_initialized)
							{
								images.Insert(0, @"[sysprep]");
								runOnceModel.AttachFloppy.Entity = true;
							}
							else
							{
								images.Add(@"[sysprep]");
							}
						}
						runOnceModel.FloppyImage.Items = images;

						if (runOnceModel.FloppyImage.IsChangable && runOnceModel.FloppyImage.SelectedItem == null)
						{
							runOnceModel.FloppyImage.SelectedItem = Linq.FirstOrDefault(images);
						}
					};//END_DELEGATE
					AsyncDataProvider.GetFloppyImageList(_asyncQuery02, isoDomain.id, false);
				}

			};//END_DELEGATE
			AsyncDataProvider.GetIsoDomainByDataCenterId(_asyncQuery0, vm.storage_pool_id);

			//passing Kernel parameters
			model.Kernel_parameters.Entity = vm.kernel_params;
			model.Kernel_path.Entity = vm.kernel_url;
			model.Initrd_path.Entity = vm.initrd_url;

			model.CustomProperties.Entity = vm.CustomProperties;

			model.IsLinux_Unassign_UnknownOS = DataProvider.IsLinuxOsType(vm.vm_os)
											|| vm.vm_os == VmOsType.Unassigned
											|| vm.vm_os == VmOsType.Other;

			model.IsWindowsOS = DataProvider.IsWindowsOsType(vm.vm_os);
			model.IsVmFirstRun.Entity = !vm.is_initialized;
			model.SysPrepDomainName.SelectedItem = vm.vm_domain;

			// Update Domain list
			AsyncDataProvider.GetDomainList(new AsyncQuery(model,
				(target, returnValue1) =>
				{
					RunOnceModel runOnceModel = (RunOnceModel)target;
					IList<string> domains = (IList<string>)returnValue1;

					// Get last selected domain
					string oldDomain = (string)runOnceModel.SysPrepDomainName.SelectedItem;

					if (oldDomain != null && !oldDomain.Equals(String.Empty) && !domains.Contains(oldDomain))
					{
						domains.Insert(0, oldDomain);
					}

					runOnceModel.SysPrepDomainName.Items = domains;
					runOnceModel.SysPrepDomainName.SelectedItem = oldDomain ?? Linq.FirstOrDefault(domains);
				}), true
			);

			//Display protocols.
			EntityModel vncProtocol =
				new EntityModel
				{
					Title = "VNC",
					Entity = DisplayType.vnc
				};

			EntityModel qxlProtocol =
				new EntityModel()
				{
					Title = "Spice",
					Entity = DisplayType.qxl
				};

			List<EntityModel> items = new List<EntityModel>();
			items.Add(vncProtocol);
			items.Add(qxlProtocol);
			model.DisplayProtocol.Items = items;
			model.DisplayProtocol.SelectedItem = vm.default_display_type == DisplayType.vnc ? vncProtocol : qxlProtocol;

			model.CustomPropertiesKeysList = this.CustomPropertiesKeysList;


			//Boot sequence.           
			AsyncQuery _asyncQuery2 = new AsyncQuery();
			_asyncQuery2.Model = this;

			/*START_DELEGATE*/_asyncQuery2.asyncCallback = delegate(Object model3, Object ReturnValue)
			{
				UserPortalListModel userPortalListModel = (UserPortalListModel)model3;
				bool hasNics = ((List<VmNetworkInterface>)((VdcQueryReturnValue)ReturnValue).ReturnValue).Count > 0;

				if (!hasNics)
				{
					BootSequenceModel bootSequenceModel = userPortalListModel.RunOnceModel.BootSequence;
					bootSequenceModel.NetworkOption.IsChangable = false;
					bootSequenceModel.NetworkOption.ChangeProhibitionReasons.Add("Virtual Machine must have at least one network interface defined to boot from network.");
				}
			};//END_DELEGATE
			
			Frontend.RunQuery(VdcQueryType.GetVmInterfacesByVmId, new GetVmByVmIdParameters(vm.vm_guid), _asyncQuery2);
			
			model.Commands.Add(
				new UICommand("OnRunOnce", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRunOnce()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (selectedItem == null || selectedItem.Entity == null)
			{
				Cancel();
				return;
			}

			VM vm = (VM)selectedItem.Entity;

			RunOnceModel model = runOnceModel;

			if (!model.Validate())
			{
				return;
			}

			BootSequenceModel bootSequenceModel = model.BootSequence;

			RunVmOnceParams param =
				new RunVmOnceParams()
				{
					VmId = vm.vm_guid,
					BootSequence = bootSequenceModel.Sequence,
					DiskPath = (bool)model.AttachIso.Entity ? (string)model.IsoImage.SelectedItem : String.Empty,
					FloppyPath = model.FloppyImagePath,
					KvmEnable = model.HwAcceleration,
					RunAndPause = (bool)model.RunAndPause.Entity,
					AcpiEnable = true,
					RunAsStateless = (bool)model.RunAsStateless.Entity,
					Reinitialize = model.Reinitialize,
					CustomProperties = (string)model.CustomProperties.Entity
				};

			//kernel params
			if (model.Kernel_path.Entity != null)
			{
				param.kernel_url = (string)model.Kernel_path.Entity;
			}
			if (model.Kernel_parameters.Entity != null)
			{
				param.kernel_params = (string)model.Kernel_parameters.Entity;
			}
			if (model.Initrd_path.Entity != null)
			{
				param.initrd_url = (string)model.Initrd_path.Entity;
			}

			//Sysprep params
			if (model.SysPrepDomainName.SelectedItem != null)
			{
				param.SysPrepDomainName = (string)model.SysPrepDomainName.SelectedItem;
			}
			if (model.SysPrepUserName.Entity != null)
			{
				param.SysPrepUserName = (string)model.SysPrepUserName.Entity;
			}
			if (model.SysPrepPassword.Entity != null)
			{
				param.SysPrepPassword = (string)model.SysPrepPassword.Entity;
			}

			EntityModel displayProtocolSelectedItem = (EntityModel)model.DisplayProtocol.SelectedItem;
			param.UseVnc = (DisplayType)displayProtocolSelectedItem.Entity == DisplayType.vnc;

			Frontend.RunAction(VdcActionType.RunVmOnce, param,
				result =>
				{
				},
				this
			);

			Cancel();
		}


		private void UpdateActionAvailability()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;

			EditCommand.IsExecutionAllowed = selectedItem != null
				&& !selectedItem.IsPool;

			RemoveCommand.IsExecutionAllowed = selectedItem != null
				&& !selectedItem.IsPool
				&& VdcActionUtils.CanExecute(new List<VM> { (VM)selectedItem.Entity }, typeof(VM), VdcActionType.RemoveVm);

			RunOnceCommand.IsExecutionAllowed = selectedItem != null
				&& !selectedItem.IsPool
				&& VdcActionUtils.CanExecute(new List<VM> { (VM)selectedItem.Entity }, typeof(VM), VdcActionType.RunVmOnce);

			ChangeCdCommand.IsExecutionAllowed = selectedItem != null
				&& !selectedItem.IsPool
				&& VdcActionUtils.CanExecute(new List<VM> { (VM)selectedItem.Entity }, typeof(VM), VdcActionType.ChangeDisk);

			NewTemplateCommand.IsExecutionAllowed = selectedItem != null
				&& !selectedItem.IsPool
				&& VdcActionUtils.CanExecute(new List<VM> { (VM)selectedItem.Entity }, typeof(VM), VdcActionType.AddVmTemplate);
		}

		private void NewDesktop()
		{
			NewInternal(VmType.Desktop);
		}

		private void NewServer()
		{
			NewInternal(VmType.Server);
		}

		private void NewInternal(VmType vmType)
		{
			VmModel = new UnitVmModel(new UserPortalNewVmModelBehavior());
			VmModel.Title = "New " + (vmType == VmType.Server ? "Server" : "Desktop") + " Virtual Machine";
			vmModel.HashName = VmModel.Title.ToLower().Replace(' ', '_');
			VmModel.IsNew = true;
			VmModel.VmType = vmType;
			VmModel.CustomPropertiesKeysList = CustomPropertiesKeysList;

			VmModel.Initialize(null);

			// Ensures that the default provisioning is "Clone" for a new server and "Thin" for a new desktop.
			EntityModel selectedItem = null;
			bool selectValue = VmModel.VmType == VmType.Server;

			foreach (object item in VmModel.Provisioning.Items)
			{
				EntityModel a = (EntityModel)item;
				if ((bool)a.Entity == selectValue)
				{
					selectedItem = a;
					break;
				}
			}
			VmModel.Provisioning.SelectedItem = selectedItem;
		}

		private void Edit()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (selectedItem == null)
			{
				return;
			}

			VM vm = (VM)selectedItem.Entity;

			VmModel = new UnitVmModel(new UserPortalExistingVmModelBehavior(vm));
			VmModel.Title = "Edit " + (vm.vm_type == VmType.Server ? "Server" : "Desktop") + " Virtual Machine";
			vmModel.HashName = VmModel.Title.ToLower().Replace(' ', '_');
			VmModel.VmType = vm.vm_type;
			VmModel.CustomPropertiesKeysList = CustomPropertiesKeysList;

			VmModel.Initialize(null);
		}

		private void Remove()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			VM vm = (VM)selectedItem.Entity;

			ConfirmationModel = new ConfirmationModel();
			ConfirmationModel.Title = "Remove Virtual Machine";
			ConfirmationModel.HashName = "remove_virtual_machine";
			ConfirmationModel.Message = "Virtual Machine";

			List<string> list = new List<string>();
			list.Add(vm.vm_name);
			ConfirmationModel.Items = list;


			ConfirmationModel.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			ConfirmationModel.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			VM vm = (VM)selectedItem.Entity;

			Frontend.RunAction(VdcActionType.RemoveVm,
				new RemoveVmParameters(vm.vm_guid, false),
				result =>
				{
				},
				this
			);

			Cancel();
		}

		private void ChangeCD()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (selectedItem == null || selectedItem.Entity == null)
			{
				return;
			}

			VM vm = (VM)selectedItem.Entity;

			AttachCdModel model = new AttachCdModel();
			AttachCdModel = model;
			model.Title = "Change CD";
			model.HashName = "change_cd";
			
			AsyncQuery _asyncQuery0 = new AsyncQuery();
			_asyncQuery0.Model = this;

			/*START_DELEGATE*/_asyncQuery0.asyncCallback = delegate(Object model0, Object result0)
				{
					UserPortalListModel userPortalListModel0 = (UserPortalListModel)model0;
					List<string> images0 = new List<string> { "No CDs" };
					userPortalListModel0.AttachCdModel.IsoImage.Items = images0;
					userPortalListModel0.AttachCdModel.IsoImage.SelectedItem = Linq.FirstOrDefault(images0);
					
					if (result0 != null)
					{
						storage_domains isoDomain = (storage_domains)result0;

						AsyncQuery _asyncQuery = new AsyncQuery();
						_asyncQuery.Model = userPortalListModel0;

						/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model1, Object result)
							{
								UserPortalListModel userPortalListModel = (UserPortalListModel)model1;
								AttachCdModel _attachCdModel = userPortalListModel.AttachCdModel;
								List<string> images = (List<string>)result;
								if (images.Count > 0)
								{
									images.Insert(0, ConsoleModel.EjectLabel);
									_attachCdModel.IsoImage.Items = images;
								}
								if (_attachCdModel.IsoImage.IsChangable)
								{
									_attachCdModel.IsoImage.SelectedItem = Linq.FirstOrDefault(images);
								}
							};//END_DELEGATE
						AsyncDataProvider.GetIrsImageList(_asyncQuery, isoDomain.id, false);
					}
				};//END_DELEGATE

			AsyncDataProvider.GetIsoDomainByDataCenterId(_asyncQuery0, vm.storage_pool_id);

			model.Commands.Add(
				new UICommand("OnChangeCD", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnChangeCD()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (selectedItem == null || selectedItem.Entity == null)
			{
				Cancel();
				return;
			}

			VM vm = (VM)selectedItem.Entity;
			AttachCdModel model = AttachCdModel;
			String isoName = (model.IsoImage.SelectedItem.ToString() == ConsoleModel.EjectLabel) ? 
				String.Empty : model.IsoImage.SelectedItem.ToString();

			Frontend.RunAction(VdcActionType.ChangeDisk,
				new ChangeDiskCommandParameters(vm.vm_guid, isoName),
				result =>
				{
				},
				this
			);
		}

		private void OnSave()
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
			if (!VmModel.IsNew && selectedItem.Entity == null)
			{
				Cancel();
				return;
			}

			tempVm = VmModel.IsNew ? new VM() : (VM)Cloner.Clone(selectedItem.Entity);

			if (!VmModel.Validate())
			{
				return;
			}

			// Check name uniqueness.
			AsyncDataProvider.IsVmNameUnique(new AsyncQuery(this,
					(target, returnValue) =>
					{
						UserPortalListModel userPortalListModel = (UserPortalListModel)target;
						bool isNameUnique = (bool)returnValue;
						string newName = (string)VmModel.Name.Entity;
						string currentName = userPortalListModel.tempVm.vm_name;

						if (!isNameUnique && String.Compare(newName, currentName, true) != 0)
						{
							userPortalListModel.VmModel.Name.InvalidityReasons.Clear();
							userPortalListModel.VmModel.Name.InvalidityReasons.Add("Name must be unique.");
							userPortalListModel.VmModel.Name.IsValid = false;
							userPortalListModel.VmModel.IsValid = false;
							userPortalListModel.VmModel.IsGeneralTabValid = false;
						}
						else
						{
							userPortalListModel.PostVmNameUniqueCheck(userPortalListModel);
						}
					})
					, (string)VmModel.Name.Entity
				);
		}

		public void PostVmNameUniqueCheck(UserPortalListModel userPortalListModel)
		{
			UserPortalItemModel selectedItem = (UserPortalItemModel)userPortalListModel.SelectedItem;
			string name = (string)VmModel.Name.Entity;

			//Save changes.
			VmTemplate template = (VmTemplate)VmModel.Template.SelectedItem;

			tempVm.vm_type = VmModel.VmType;
			tempVm.vmt_guid = template.Id;
			tempVm.vm_name = name;
			tempVm.vm_os = (VmOsType)VmModel.OSType.SelectedItem;
			tempVm.num_of_monitors = (int)VmModel.NumOfMonitors.SelectedItem;
			tempVm.vm_description = (string)VmModel.Description.Entity;
			tempVm.vm_domain = VmModel.Domain.IsAvailable ? (string)VmModel.Domain.SelectedItem : String.Empty;
			tempVm.vm_mem_size_mb = (int)VmModel.MemSize.Entity;
			tempVm.MinAllocatedMem = (int)VmModel.MinAllocatedMemory.Entity;
			Guid newClusterID = ((VDSGroup)VmModel.Cluster.SelectedItem).ID;
			tempVm.vds_group_id = newClusterID;
			tempVm.time_zone = (VmModel.TimeZone.IsAvailable && VmModel.TimeZone.SelectedItem != null) ? ((KeyValuePair<string, string>)VmModel.TimeZone.SelectedItem).Key : String.Empty;
			tempVm.num_of_sockets = (int)VmModel.NumOfSockets.Entity;
			tempVm.cpu_per_socket = (int)VmModel.TotalCPUCores.Entity / (int)VmModel.NumOfSockets.Entity;
			tempVm.usb_policy = (UsbPolicy)VmModel.UsbPolicy.SelectedItem;
			tempVm.is_auto_suspend = false;
			tempVm.is_stateless = (bool)VmModel.IsStateless.Entity;
			tempVm.default_boot_sequence = VmModel.BootSequence;
			tempVm.iso_path = VmModel.CdImage.IsChangable ? (string)VmModel.CdImage.SelectedItem : String.Empty;
			tempVm.auto_startup = (bool)VmModel.IsHighlyAvailable.Entity;

			tempVm.initrd_url = (string)VmModel.Initrd_path.Entity;
			tempVm.kernel_url = (string)VmModel.Kernel_path.Entity;
			tempVm.kernel_params = (string)VmModel.Kernel_parameters.Entity;

			tempVm.CustomProperties = (string)VmModel.CustomProperties.Entity;

			EntityModel displayProtocolSelectedItem = (EntityModel)VmModel.DisplayProtocol.SelectedItem;
			tempVm.default_display_type = (DisplayType)displayProtocolSelectedItem.Entity;

			EntityModel prioritySelectedItem = (EntityModel)VmModel.Priority.SelectedItem;
			tempVm.priority = (int)prioritySelectedItem.Entity;


			VDS defaultHost = (VDS)VmModel.DefaultHost.SelectedItem;
			if ((bool)VmModel.IsAutoAssign.Entity)
			{
				tempVm.dedicated_vm_for_vds = null;
			}
			else
			{
				tempVm.dedicated_vm_for_vds = defaultHost.vds_id;
			}

			tempVm.MigrationSupport = MigrationSupport.MIGRATABLE;
			if ((bool)VmModel.RunVMOnSpecificHost.Entity)
			{
				tempVm.MigrationSupport = MigrationSupport.PINNED_TO_HOST;
			}
			else if ((bool)VmModel.DontMigrateVM.Entity)
			{
				tempVm.MigrationSupport = MigrationSupport.IMPLICITLY_NON_MIGRATABLE;
			}

			bool cancel = true;
			if (VmModel.IsNew)
			{
				if (tempVm.vmt_guid.Equals(Guid.Empty))
				{
					AddVmFromScratchParameters parameters = new AddVmFromScratchParameters(tempVm, new List<DiskImageBase>(), Guid.Empty);
					parameters.MakeCreatorExplicitOwner = true;

					Frontend.RunAction(VdcActionType.AddVmFromScratch, parameters,
						result =>
						{
						},
						this
					);
				}
				else
				{
					storageDomain = (storage_domains)VmModel.StorageDomain.SelectedItem;

					if ((bool)((EntityModel)VmModel.Provisioning.SelectedItem).Entity)
					{
						AsyncQuery _asyncQuery = new AsyncQuery();
						_asyncQuery.Model = this;
						/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object result)
						{
							UserPortalListModel userPortalListModel1 = (UserPortalListModel)model;
							List<DiskImage> templateDisks = (List<DiskImage>)result;
							foreach (DiskImage templateDisk in templateDisks)
							{
								DiskModel disk = null;
								foreach (DiskModel a in userPortalListModel1.VmModel.Disks)
								{
									if (a.Name == templateDisk.internal_drive_mapping)
									{
										disk = a;
										break;
									}
								}

								templateDisk.volume_type = (VolumeType)disk.VolumeType.SelectedItem;
								templateDisk.volume_format = DataProvider.GetDiskVolumeFormat((VolumeType)disk.VolumeType.SelectedItem, storageDomain.storage_type);
							}

							Dictionary<string, DiskImageBase> dict = new Dictionary<string, DiskImageBase>();
							foreach (DiskImage a in templateDisks)
							{
								dict.Add(a.internal_drive_mapping, a);
							}

							AddVmFromTemplateParameters parameters = new AddVmFromTemplateParameters(tempVm, dict, storageDomain.id);
							parameters.MakeCreatorExplicitOwner = true;

							Frontend.RunAction(VdcActionType.AddVmFromTemplate, parameters,
								a =>
								{
								},
								this
							);

							userPortalListModel1.Cancel();

						};//END_DELEGATE
						AsyncDataProvider.GetTemplateDiskList(_asyncQuery, template.Id);
						cancel = false;
					}
					else
					{
						VmManagementParametersBase parameters = new VmManagementParametersBase(tempVm);
						parameters.StorageDomainId = storageDomain.id;
						parameters.MakeCreatorExplicitOwner = true;

						Frontend.RunAction(VdcActionType.AddVm, parameters,
							result =>
							{
							},
							this
						);
					}
				}
			}
			else
			{
				Guid oldClusterID = ((VM)selectedItem.Entity).vds_group_id;
				if (oldClusterID.Equals(newClusterID) == false)
				{
					Frontend.RunAction(VdcActionType.ChangeVMCluster,
						new ChangeVMClusterParameters(newClusterID, tempVm.vm_guid),
						result =>
						{
							Frontend.RunAction(VdcActionType.UpdateVm,
								new VmManagementParametersBase(tempVm),
								a =>
								{
								},
								this
							);
						},
						this
					);
				}
				else
				{
					Frontend.RunAction(VdcActionType.UpdateVm,
						new VmManagementParametersBase(tempVm),
						a =>
						{
						},
						this
					);
				}
			}

			if (cancel)
			{
				Cancel();
			}
		}


		private void Cancel()
		{
			Frontend.Unsubscribe();

			ConfirmationModel = null;
			VmModel = null;
		}

		private void VmModel_DataCenter_ItemsChanged()
		{
			storage_pool dataCenter = null;
			foreach (object item in VmModel.DataCenter.Items)
			{
				storage_pool a = (storage_pool)item;

				if (VmModel.IsNew)
				{
					dataCenter = a;
					break;
				}
				else
				{
					UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
					VM vm = (VM)selectedItem.Entity;

					if (a.Id.Equals(vm.storage_pool_id))
					{
						dataCenter = a;
						break;
					}
				}
			}
			
			if (!VmModel.IsNew && dataCenter == null)
			{
				UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
				VM vm = (VM)selectedItem.Entity;
				AsyncQuery _asyncQuery = new AsyncQuery();
				_asyncQuery.Model = this;
				/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object result)
												{
													UserPortalListModel userPortalListModel = (UserPortalListModel)model;
													List<storage_pool> list = new List<storage_pool> { (storage_pool)result };
													userPortalListModel.VmModel.DataCenter.Items = list;
													userPortalListModel.VmModel.DataCenter.SelectedItem = Linq.FirstOrDefault(list);

												};//END_DELEGATE
				AsyncDataProvider.GetDataCenterById(_asyncQuery, vm.storage_pool_id);
			}
			else
			{
				VmModel.DataCenter.SelectedItem = dataCenter;
			}
		}

		private void VmModel_Cluster_ItemsChanged()
		{
			if (!VmModel.IsNew)
			{
				UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
				VM vm = (VM)selectedItem.Entity;
				VDSGroup cluster = null;

				foreach (object item in VmModel.Cluster.Items)
				{
					VDSGroup a = (VDSGroup)item;
					if (a.ID.Equals(vm.vds_group_id))
					{
						cluster = a;
						break;
					}
				}
				VmModel.Cluster.SelectedItem = cluster;

				VmModel.Cluster.IsChangable = vm.status == VMStatus.Down;
			}
		}

		private void VmModel_DefaultHost_ItemsChanged()
		{
			if (!VmModel.IsNew)
			{
				UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
				VM vm = (VM)selectedItem.Entity;
				VDS host = null;

				foreach (object item in VmModel.DefaultHost.Items)
				{
					VDS a = (VDS)item;
					if (a.vds_id.Equals(((vm.dedicated_vm_for_vds != null) ? vm.dedicated_vm_for_vds : Guid.Empty)))
					{
						host = a;
						break;
					}
				}
				if (host == null)
				{
					VmModel.IsAutoAssign.Entity = true;
				}
				else
				{
					VmModel.DefaultHost.SelectedItem = host;
					VmModel.IsAutoAssign.Entity = false;
				}
			}
		}

		private void VmModel_DisplayProtocol_ItemsChanged()
		{
			if (!VmModel.IsNew)
			{
				UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
				VM vm = (VM)selectedItem.Entity;
				EntityModel displayType = null;

				foreach (object item in VmModel.DisplayProtocol.Items)
				{
					EntityModel a = (EntityModel)item;
					DisplayType dt = (DisplayType)a.Entity;
					if (dt == vm.default_display_type)
					{
						displayType = a;
						break;
					}
				}
				VmModel.DisplayProtocol.SelectedItem = displayType;
			}
		}

		private void VmModel_Priority_ItemsChanged()
		{
			if (!VmModel.IsNew)
			{
				UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
				VM vm = (VM)selectedItem.Entity;
				AsyncQuery _asyncQuery = new AsyncQuery();
				_asyncQuery.Model = this;
				/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object result)
												{
													UserPortalListModel userPortalListModel = (UserPortalListModel)model;
													int roundPriority = (int)result;
													EntityModel priority = null;

													foreach (object item in userPortalListModel.VmModel.Priority.Items)
													{
														EntityModel a = (EntityModel)item;
														int p = (int)a.Entity;
														if (p == roundPriority)
														{
															priority = a;
															break;
														}
													}
													userPortalListModel.VmModel.Priority.SelectedItem = priority;

												};//END_DELEGATE
				AsyncDataProvider.GetRoundedPriority(_asyncQuery, vm.priority);
			}
		}

		private void VmModel_TimeZone_ItemsChanged()
		{
			if (!VmModel.IsNew)
			{
				UserPortalItemModel selectedItem = (UserPortalItemModel)SelectedItem;
				VM vm = (VM)selectedItem.Entity;

				if (!String.IsNullOrEmpty(vm.time_zone))
				{
					VmModel.TimeZone.SelectedItem = Linq.FirstOrDefault((IEnumerable<KeyValuePair<string, string>>)VmModel.TimeZone.Items, new Linq.TimeZonePredicate(vm.time_zone));
				}
			}
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ItemsChangedEventDefinition) && sender == VmModel.DataCenter)
			{
				VmModel_DataCenter_ItemsChanged();
			}
			else if (ev.Equals(ItemsChangedEventDefinition) && sender == VmModel.Cluster)
			{
				VmModel_Cluster_ItemsChanged();
			}
			else if (ev.Equals(ItemsChangedEventDefinition) && sender == VmModel.DefaultHost)
			{
				VmModel_DefaultHost_ItemsChanged();
			}
			else if (ev.Equals(ItemsChangedEventDefinition) && sender == VmModel.DisplayProtocol)
			{
				VmModel_DisplayProtocol_ItemsChanged();
			}
			else if (ev.Equals(ItemsChangedEventDefinition) && sender == VmModel.Priority)
			{
				VmModel_Priority_ItemsChanged();
			}
			else if (ev.Equals(ItemsChangedEventDefinition) && sender == VmModel.TimeZone)
			{
				VmModel_TimeZone_ItemsChanged();
			}
		}

		public override void OnVmAndPoolLoad()
		{
			if (vms != null && pools != null)
			{
				//Complete search.

				//Remove pools that has provided VMs.
				List<vm_pools> filteredPools = new List<vm_pools>();
				poolMap = new Dictionary<Guid, vm_pools>();

				foreach (vm_pools pool in pools)
				{
					//Add pool to map.
					poolMap.Add(pool.vm_pool_id, pool);

					bool found = false;
					foreach (VM vm in vms)
					{
						if (vm.VmPoolId != null && vm.VmPoolId.Equals(pool.vm_pool_id))
						{
							found = true;
							break;
						}
					}

					if (!found)
					{
						filteredPools.Add(pool);
					}
				}

				//Merge VMs and Pools, and create item models.
				IList all = Linq.Concat(vms, filteredPools);
				Linq.Sort(all, new Linq.VmAndPoolByNameComparer());

				List<Model> items = new List<Model>();
				foreach (object item in all)
				{
					UserPortalItemModel model = new UserPortalItemModel(this);
					model.Entity = item;
					items.Add(model);

					UpdateConsoleModel(model);
				}

				// In userportal 'Extended View': Set 'CanConnectAutomatically' to true if there's one and only one up VM.
				CanConnectAutomatically = GetUpVms(items).Count == 1 && 
					((UICommand)GetUpVms(items)[0].DefaultConsole.ConnectCommand).IsExecutionAllowed;

				Items = items;

				vms = null;
				pools = null;
				
				SearchCompletedEvent.raise(this, EventArgs.Empty);
			}
		}

		private void UpdateConsoleModel(UserPortalItemModel item)
		{
			if (item.Entity != null)
			{
				VM vm = item.Entity as VM;
				if (vm == null)
				{
					return;
				}
				
				// Caching console model if needed
				if (!cachedConsoleModels.ContainsKey(vm.vm_guid))
				{
					SpiceConsoleModel spiceConsoleModel = new SpiceConsoleModel();
					spiceConsoleModel.ErrorEvent.addListener(this);
					VncConsoleModel vncConsoleModel = new VncConsoleModel();
					RdpConsoleModel rdpConsoleModel = new RdpConsoleModel();

					cachedConsoleModels.Add(vm.vm_guid, new List<ConsoleModel> { spiceConsoleModel, vncConsoleModel, rdpConsoleModel });
				}

				// Getting cached console model
				List<ConsoleModel> cachedModels = cachedConsoleModels[vm.vm_guid];
				foreach (ConsoleModel cachedModel in cachedModels)
				{
					cachedModel.Entity = null;
					cachedModel.Entity = vm;
				}

				// Set default console by vm's display type
				item.DefaultConsole = vm.display_type == DisplayType.vnc ? cachedModels[1] : cachedModels[0];

				// Adjust item's default console for userportal 'Extended View'
				item.DefaultConsole.ForceVmStatusUp = false;

				// Update additional console
				if (DataProvider.IsWindowsOsType(vm.vm_os))
				{
					item.AdditionalConsole = cachedModels[2];
					item.HasAdditionalConsole = true;
				}
				else
				{
					item.AdditionalConsole = null;
					item.HasAdditionalConsole = false;
				}
			}
		}
	}
}
