using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.vms;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.dataprovider;

namespace org.ovirt.engine.ui.uicommon.models.templates
{
	public class TemplateInterfaceListModel : SearchableListModel
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		//TODO: Check if we really need the following property.
		private VmTemplate EntityStronglyTyped
		{
			get { return Entity as VmTemplate; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		public TemplateInterfaceListModel()
		{
			Title = "Network Interfaces";

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			SearchCommand.Execute();
			UpdateActionAvailability();
		}

		public override void Search()
		{
			if (EntityStronglyTyped != null)
			{
				base.Search();
			}
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.GetTemplateInterfacesByTemplateId, new GetVmTemplateParameters(EntityStronglyTyped.Id));
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetTemplateInterfacesByTemplateId, new GetVmTemplateParameters(EntityStronglyTyped.Id));
			Items = AsyncResult.Data;
		}

		private void New()
		{
			if (Window != null)
			{
				return;
			}

			VmInterfaceModel model = new VmInterfaceModel();
			Window = model;
			model.Title = "New Network Interface";
			model.IsNew = true;

			AsyncDataProvider.GetClusterNetworkList(new AsyncQuery(this,
				(target, returnValue) =>
				{
					TemplateInterfaceListModel vmInterfaceListModel = (TemplateInterfaceListModel)target;
					List<network> network_list = returnValue != null ? (List<network>)returnValue : new List<network>();
					vmInterfaceListModel.PostGetClusterNetworkList_New(network_list);
				})
				,EntityStronglyTyped.vds_group_id
			);
		}

		public void PostGetClusterNetworkList_New(List<network> network_list)
		{
			List<network> networks = new List<network>();
			foreach (network a in network_list)
			{
				if (a.Status == NetworkStatus.Operational)
				{
					networks.Add(a);
				}
			}

			List<VmNetworkInterface> nics = Linq.Cast<VmNetworkInterface>(Items);
			int nicCount = nics.Count;
			string newNicName = DataProvider.GetNewNicName(nics);

			VmInterfaceModel model = (VmInterfaceModel)Window;
			model.Network.Items = networks;
			model.Network.SelectedItem = networks.Count > 0 ? networks[0] : null;
			model.NicType.Items = DataProvider.GetNicTypeList(EntityStronglyTyped.os, false);
			model.NicType.SelectedItem = DataProvider.GetDefaultNicType(EntityStronglyTyped.os);
			model.Name.Entity = newNicName;
			model.MAC.IsAvailable = false;

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Edit()
		{
			if (Window != null)
			{
				return;
			}

			VmInterfaceModel model = new VmInterfaceModel();
			Window = model;
			model.Title = "Edit Network Interface";

			AsyncDataProvider.GetClusterNetworkList(new AsyncQuery(this,
				(target, returnValue) =>
				{
					TemplateInterfaceListModel vmInterfaceListModel = (TemplateInterfaceListModel)target;
					List<network> network_list = returnValue != null ? (List<network>)returnValue : new List<network>();
					vmInterfaceListModel.PostGetClusterNetworkList_Edit(network_list);
				})
				,EntityStronglyTyped.vds_group_id
			);
		}

		public void PostGetClusterNetworkList_Edit(List<network> network_list)
		{
			VmNetworkInterface nic = (VmNetworkInterface)SelectedItem;
			int nicCount = Linq.Cast<VmNetworkInterface>(Items).Count;
			List<network> networks = new List<network>();
			foreach (network a in network_list)
			{
				if (a.Status == NetworkStatus.Operational)
				{
					networks.Add(a);
				}
			}

			VmInterfaceModel model = (VmInterfaceModel)Window;
			model.Network.Items = networks;
			network network = null;
			foreach (network a in networks)
			{
				if (a.name == nic.NetworkName)
				{
					network = a;
					break;
				}
			}
			model.Network.SelectedItem = network;

			int? selectedNicType = nic.Type;
			List<VmInterfaceType> nicTypes = DataProvider.GetNicTypeList(EntityStronglyTyped.os, (VmInterfaceType)selectedNicType == VmInterfaceType.rtl8139_pv);
			model.NicType.Items = nicTypes;

			if (!selectedNicType.HasValue || !nicTypes.Contains((VmInterfaceType)selectedNicType))
			{
				selectedNicType = (int)DataProvider.GetDefaultNicType(EntityStronglyTyped.os);
			}

			model.NicType.SelectedItem = (VmInterfaceType)(selectedNicType);

			model.Name.Entity = nic.Name;
			model.MAC.IsAvailable = false;


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnSave()
		{
			VmInterfaceModel model = (VmInterfaceModel)Window;
			VmNetworkInterface nic = model.IsNew ? new VmNetworkInterface() : (VmNetworkInterface)Cloner.Clone((VmNetworkInterface)SelectedItem);

			if (!model.Validate())
			{
				return;
			}

			//Save changes.
			nic.Name = (string)model.Name.Entity;
			nic.NetworkName = ((network)model.Network.SelectedItem).name;
			if (model.NicType.SelectedItem == null)
			{
				nic.Type = null;
			}
			else
			{
				nic.Type = (int?)(VmInterfaceType)model.NicType.SelectedItem;
			}
			nic.MacAddress = model.MAC.IsChangable
							? (model.MAC.Entity == null ? null : ((string)(model.MAC.Entity)).ToLower())
							: model.IsNew
								? String.Empty
								: nic.MacAddress;


			if (model.IsNew)
			{
				Frontend.RunMultipleAction(VdcActionType.AddVmTemplateInterface,
					new List<VdcActionParametersBase>
                    {
        				new AddVmTemplateInterfaceParameters(EntityStronglyTyped.Id, nic)
                    },
					result =>
					{
						Cancel();
					},
					null
				);
			}
			else
			{
				Frontend.RunMultipleAction(VdcActionType.UpdateVmTemplateInterface,
						new List<VdcActionParametersBase>
                        {
					        new AddVmTemplateInterfaceParameters(EntityStronglyTyped.Id, nic)
                        },
						result =>
						{
							Cancel();
						},
					null
				);
			}
		}

		private void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Network Interface(s)";
			model.Message = "Network Interface(s)";
			
			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				VmNetworkInterface a = (VmNetworkInterface)item;
				items.Add(a.Name);
			}
			model.Items = items;

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VmNetworkInterface a = (VmNetworkInterface)item;
				list.Add(new RemoveVmTemplateInterfaceParameters(EntityStronglyTyped.Id, a.Id));
			}

			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVmTemplateInterface, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			EditCommand.IsExecutionAllowed = SelectedItems != null && SelectedItems.Count == 1 && SelectedItem != null;
			RemoveCommand.IsExecutionAllowed = SelectedItems != null && SelectedItems.Count > 0;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
		}
	}
}
