using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Text;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using VdcCommon;

namespace org.ovirt.engine.ui.uicommon.models.datacenters
{
	public class DataCenterStorageListModel : SearchableListModel
	{
		#region Commands

		public UICommand AttachStorageCommand { get; private set; }
		public UICommand AttachISOCommand { get; private set; }
		public UICommand AttachBackupCommand { get; private set; }
		public UICommand DetachCommand { get; private set; }
		public UICommand ActivateCommand { get; private set; }
		public UICommand MaintenanceCommand { get; private set; }

		#endregion

		#region Properties

		public new storage_pool Entity
		{
			get { return (storage_pool)base.Entity; }
			set { base.Entity = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		public DataCenterStorageListModel()
		{
			Title = "Storage";

			AttachStorageCommand = new UICommand("AttachStorage", this);
			AttachISOCommand = new UICommand("AttachISO", this);
			AttachBackupCommand = new UICommand("AttachBackup", this);
			DetachCommand = new UICommand("Detach", this);
			ActivateCommand = new UICommand("Activate", this);
			MaintenanceCommand = new UICommand("Maintenance", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();
			SearchCommand.Execute();
		}

		public override void Search()
		{
			if (Entity != null)
			{
				// omer - overriding AsyncSearch - using query instead of search
				//SearchString = String.Format("storage: datacenter={0}", Entity.name);
				base.Search();
			}
		}

		protected override void SyncSearch()
		{
			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				SearchableListModel searchableListModel = (SearchableListModel)model;
				searchableListModel.Items = (List<storage_domains>)((VdcQueryReturnValue)ReturnValue).ReturnValue;
			};//END_DELEGATE

			Frontend.RunQuery(VdcQueryType.GetStorageDomainsByStoragePoolId,
				new StoragePoolQueryParametersBase(Entity.Id),
				_asyncQuery);
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetStorageDomainsByStoragePoolId, new StoragePoolQueryParametersBase(Entity.Id));
			Items = AsyncResult.Data;
		}

		public void Maintenance()
		{
			//Frontend.RunMultipleActions(VdcActionType.DeactivateStorageDomain,
			//    SelectedItems.Cast<storage_domains>()
			//    .Select(a => (VdcActionParametersBase)new StorageDomainPoolParametersBase(a.id, Entity.id))
			//    .ToList()
			//);
			List<VdcActionParametersBase> pb = new List<VdcActionParametersBase>();
			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				pb.Add(new StorageDomainPoolParametersBase(a.id, Entity.Id));
			}

			Frontend.RunMultipleAction(VdcActionType.DeactivateStorageDomain, pb);
		}

		public void Activate()
		{
			//Frontend.RunMultipleActions(VdcActionType.ActivateStorageDomain,
			//    SelectedItems.Cast<storage_domains>()
			//    .Select(a => (VdcActionParametersBase)new StorageDomainPoolParametersBase(a.id, Entity.id))
			//    .ToList()
			//);
			List<VdcActionParametersBase> pb = new List<VdcActionParametersBase>();
			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				pb.Add(new StorageDomainPoolParametersBase(a.id, Entity.Id));
			}


			Frontend.RunMultipleAction(VdcActionType.ActivateStorageDomain, pb);
		}

		public void AttachBackup()
		{
			AttachInternal(StorageDomainType.ImportExport, "Attach Export Domain", "attach_export_domain");
		}

		public void AttachISO()
		{
			AttachInternal(StorageDomainType.ISO, "Attach ISO Library", "attach_iso_library");
		}

		public void AttachStorage()
		{
			AttachInternal(StorageDomainType.Data, "Attach Storage", "attach_storage");
		}

		private void AttachInternal(StorageDomainType storageType, string title, string hashName)
		{
			if (Window != null)
			{
				return;
			}

			//var items = Items != null ? Items.Cast<storage_domains>().ToList() : new List<storage_domains>();
			List<storage_domains> items = Items != null ? new List<storage_domains>(Linq.Cast<storage_domains>(Items)) : new List<storage_domains>();

			ListModel model = new ListModel();
			Window = model;
			model.Title = title;
			model.HashName = hashName;
			List<EntityModel> models;
			if (storageType == StorageDomainType.ISO)
			{
				List<storage_domains> list = DataProvider.GetISOStorageDomainList();
				//models = list.Where(a=>items.All(b=>b.id != a.id)).Select(a=>new EntityModel(){Entity = a}).ToList();
				models = new List<EntityModel>();
				foreach (storage_domains a in list)
				{
					//if (Linq.All<storage_domains>(items, delegate(storage_domains b) { return b.id != a.id; }))
					if (!Linq.IsSDItemExistInList(items, a.id))
					{
						models.Add(new EntityModel() { Entity = a });
					}
				}

			}
			else
			{
				List<storage_domains> list = DataProvider.GetStorageDomainList();

				models = new List<EntityModel>();
				bool addToList;
				Version version3_0 = new Version(3,0);
				foreach (storage_domains a in list)
				{
					addToList = false;

					if(!Linq.IsSDItemExistInList(items, a.id) && a.storage_domain_type == storageType)
					{
						if(storageType == StorageDomainType.Data && a.storage_type == Entity.storage_pool_type &&
						      a.storage_domain_shared_status == StorageDomainSharedStatus.Unattached)
						{
							if(Entity.StoragePoolFormatType == null)
							{
								//compat logic: in case its not v1 and the version is less than 3.0 - continue.
								if (a.StorageStaticData.StorageFormat != StorageFormatType.V1 && Entity.compatibility_version.compareTo(version3_0) < 0)
								{
									continue;
								}
								addToList = true;
							}
							else if ((StorageFormatType)Entity.StoragePoolFormatType == a.StorageStaticData.StorageFormat)
							{
								addToList = true;
							}
						}
						else if(storageType == StorageDomainType.ImportExport &&
						      a.storage_domain_shared_status == StorageDomainSharedStatus.Unattached)
						{
							addToList = true;
						}
						
						if(addToList)
						{
							models.Add(new EntityModel() { Entity = a });
						}
					}
				}
			}
			model.Items = models;

			if (models.Count == 0)
			{
				model.Message = "There are no compatible Storage Domains to attach to this Data Center. Please add new Storage from the Storage tab.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnAttach", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		public void OnAttach()
		{
			ListModel model = (ListModel)Window;

			if (Entity == null)
			{
				Cancel();
				return;
			}

			//var items = model.Items
			//    .Cast<EntityModel>()
			//    .Where(Selector.GetIsSelected)
			//    .Select(a => (storage_domains)a.Entity)
			//    .ToList();
			List<storage_domains> items = new List<storage_domains>();
			foreach (EntityModel a in Linq.Cast<EntityModel>(model.Items))
			{
				if (a.IsSelected)
				{
					items.Add((storage_domains)a.Entity);
				}
			}


			if (items.Count > 0)
			{
				//Frontend.RunMultipleActions(VdcActionType.AttachStorageDomainToPool,
				//    items
				//    .Select(a => (VdcActionParametersBase)new StorageDomainPoolParametersBase(a.id, Entity.id))
				//    .ToList()
				//);
				List<VdcActionParametersBase> pb = new List<VdcActionParametersBase>();
				foreach (storage_domains a in items)
				{
					pb.Add(new StorageDomainPoolParametersBase(a.id, Entity.Id));
				}


				Frontend.RunMultipleAction(VdcActionType.AttachStorageDomainToPool, pb);
			}

			Cancel();
		}

		public void Detach()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Detach Storage";
			model.HashName = "detach_storage";
			model.Message = "Are you sure you want to Detach the following storage(s)?";

			//model.Items = SelectedItems.Cast<storage_domains>().Select(a => a.storage_name);
			model.Items = new List<string>();

			if (ContainsLocalStorage(model))
			{
				model.Latch.IsAvailable = true;
				model.Latch.IsChangable = true;

				model.Note = "Note: " + GetLocalStoragesFormattedString() + " will be removed!";
			}

			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				((List<string>)model.Items).Add(a.storage_name);
			}

			model.Commands.Add(
				new UICommand("OnDetach", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private string GetLocalStoragesFormattedString()
		{
			string localStorages = "";
			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				if (a.storage_type == StorageType.LOCALFS)
				{
					localStorages += a.storage_name + ", ";
				}
			}
			return localStorages.Remove(localStorages.Length - 2);
		}

		private bool ContainsLocalStorage(ConfirmationModel model)
		{
			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				if (a.storage_type == StorageType.LOCALFS)
				{
					return true;
				}
			}
			return false;
		}

		public void OnDetach()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			// A list of 'detach' action parameters
			List<VdcActionParametersBase> pb_detach = new List<VdcActionParametersBase>();
			// A list of 'remove' action parameters
			List<VdcActionParametersBase> pb_remove = new List<VdcActionParametersBase>();

			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				// For local storage - remove; otherwise - detach
				if (a.storage_type == StorageType.LOCALFS)
				{
					VDS locaVds = DataProvider.GetLocalStorageHost(a.storage_pool_name);
					pb_remove.Add(new RemoveStorageDomainParameters(a.id)
					{
						VdsId = (locaVds != null ? locaVds.vds_id : null),
						DoFormat = true
					});
				}
				else
				{
					pb_detach.Add(new DetachStorageDomainFromPoolParameters(a.id, Entity.Id));
				}
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveStorageDomain,
				pb_remove,
				result1 =>
				{
					object[] array = (object[])result1.State;
					ConfirmationModel localModel1 = (ConfirmationModel)array[0];
					List<VdcActionParametersBase> parameters = (List<VdcActionParametersBase>)array[1];

					Frontend.RunMultipleAction(VdcActionType.DetachStorageDomainFromPool,
						parameters,
						result2 =>
						{
							ConfirmationModel localModel2 = (ConfirmationModel)result2.State;

							localModel2.StopProgress();
							Cancel();
						},
						localModel1
					);
				},
				new object[] { model, pb_detach }
			);
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void ItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
		{
			base.ItemsCollectionChanged(sender, e);
			UpdateActionAvailability();
		}

		protected override void ItemsChanged()
		{
			base.ItemsChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		protected override void ItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.ItemPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		protected override bool NotifyPropertyChangeForAnyItem
		{
			get { return true; }
		}

		private void UpdateActionAvailability()
		{
			List<storage_domains> items = Items != null ? Linq.Cast<storage_domains>(Items) : new List<storage_domains>();
			List<storage_domains> selectedItems = SelectedItems != null ? Linq.Cast<storage_domains>(SelectedItems) : new List<storage_domains>();

			if (Entity != null)
			{
				AttachStorageCommand.IsExecutionAllowed = Entity.storage_pool_type != StorageType.LOCALFS;
			}

			bool isMasterPresents = false;
			foreach (storage_domains a in items)
			{
				if (a.storage_domain_type == StorageDomainType.Master && a.status.HasValue && a.status.Value == StorageDomainStatus.Active)
				{
					isMasterPresents = true;
					break;
				}
			}

			bool isISOPresents = false;
			foreach (storage_domains a in items)
			{
				if (a.storage_domain_type == StorageDomainType.ISO)
				{
					isISOPresents = true;
					break;
				}
			}
			AttachISOCommand.IsExecutionAllowed = false;
			AttachISOCommand.IsExecutionAllowed = items.Count > 0 && isMasterPresents && !isISOPresents;


			bool isBackupPresents = false;
			foreach (storage_domains a in items)
			{
				if (a.storage_domain_type == StorageDomainType.ImportExport)
				{
					isBackupPresents = true;
					break;
				}
			}
			AttachBackupCommand.IsExecutionAllowed = items.Count > 0 && isMasterPresents && !isBackupPresents;


			DetachCommand.IsExecutionAllowed = selectedItems.Count > 0
				&& VdcActionUtils.CanExecute(selectedItems, typeof(storage_domains), VdcActionType.DetachStorageDomainFromPool);

			ActivateCommand.IsExecutionAllowed = selectedItems.Count == 1
				&& VdcActionUtils.CanExecute(selectedItems, typeof(storage_domains), VdcActionType.ActivateStorageDomain);

			MaintenanceCommand.IsExecutionAllowed = selectedItems.Count == 1
				&& VdcActionUtils.CanExecute(selectedItems, typeof(storage_domains), VdcActionType.DeactivateStorageDomain);
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == AttachStorageCommand)
			{
				AttachStorage();
			}
			else if (command == AttachISOCommand)
			{
				AttachISO();
			}
			else if (command == AttachBackupCommand)
			{
				AttachBackup();
			}
			else if (command == DetachCommand)
			{
				Detach();
			}
			else if (command == ActivateCommand)
			{
				Activate();
			}
			else if (command == MaintenanceCommand)
			{
				Maintenance();
			}
			else if (command.Name == "OnAttach")
			{
				OnAttach();
			}
			else if (command.Name == "OnDetach")
			{
				OnDetach();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
