
IF NOT EXISTS (SELECT NAME FROM sys.objects WHERE TYPE = 'P' AND NAME = 'RethrowError')
BEGIN
	EXEC('CREATE PROCEDURE [dbo].RethrowError AS RETURN')
END
GO

ALTER PROCEDURE RethrowError AS
    /* Return if there is no error information to retrieve. */
    IF ERROR_NUMBER() IS NULL
        RETURN;

    DECLARE
        @ErrorMessage    NVARCHAR(4000),
        @ErrorNumber     INT,
        @ErrorSeverity   INT,
        @ErrorState      INT,
        @ErrorLine       INT,
        @ErrorProcedure  NVARCHAR(200); 

    /* Assign variables to error-handling functions that
       capture information for RAISERROR. */

    SELECT
        @ErrorNumber = ERROR_NUMBER(),
        @ErrorSeverity = ERROR_SEVERITY(),
        @ErrorState = ERROR_STATE(),
        @ErrorLine = ERROR_LINE(),
        @ErrorProcedure = ISNULL(ERROR_PROCEDURE(), '-'); 

    /* Building the message string that will contain original
       error information. */

    SELECT @ErrorMessage = 
        N'Error %d, Level %d, State %d, Procedure %s, Line %d, ' + 
         'Message: '+ ERROR_MESSAGE(); 

    /* Raise an error: msg_str parameter of RAISERROR will contain
	   the original error information. */

    RAISERROR(@ErrorMessage, @ErrorSeverity, 1,
        @ErrorNumber,    /* parameter: original error number. */
        @ErrorSeverity,  /* parameter: original error severity. */
        @ErrorState,     /* parameter: original error state. */
        @ErrorProcedure, /* parameter: original error procedure name. */
        @ErrorLine       /* parameter: original error line number. */
        );

GO

----------------------------------------------------------------
-- [dbo].[async_tasks] Table
--
IF NOT EXISTS (SELECT NAME FROM sys.objects WHERE TYPE = 'P' AND NAME = 'Insertasync_tasks')
BEGIN
	EXEC('CREATE PROCEDURE [dbo].[Insertasync_tasks] AS RETURN')
END

GO

ALTER PROCEDURE [dbo].[Insertasync_tasks]
    @action_type int,
	@result int,
	@status int,
	@task_id uniqueidentifier,
	@action_parameters image = null
AS
BEGIN
	SET NOCOUNT ON
	
	BEGIN TRY
    INSERT INTO [dbo].[async_tasks] ([action_type], [result], [status], [task_id], [action_parameters])
	VALUES (@action_type, @result, @status, @task_id, @action_parameters)
    
    END TRY

    BEGIN CATCH
		EXEC RethrowError;
	END CATCH
    
    SET NOCOUNT OFF
END    

GO

IF NOT EXISTS (SELECT NAME FROM sys.objects WHERE TYPE = 'P' AND NAME = 'Updateasync_tasks')
BEGIN
	EXEC('CREATE PROCEDURE [dbo].[Updateasync_tasks] AS RETURN')
END

GO

ALTER PROCEDURE [dbo].[Updateasync_tasks]
    @action_type int,
	@result int,
	@status int,
	@task_id uniqueidentifier,
	@action_parameters image = null
AS
BEGIN

	--The [dbo].[async_tasks] table doesn't have a timestamp column. Optimistic concurrency logic cannot be generated
	SET NOCOUNT ON

	BEGIN TRY
	UPDATE [dbo].[async_tasks] 
	SET [action_type] = @action_type, [result] = @result, [status] = @status, [action_parameters] = @action_parameters
	WHERE [task_id]=@task_id

	IF @@ROWCOUNT = 0
	BEGIN
		RAISERROR('Concurrent update error. Updated aborted.', 16, 2)
	END
    END TRY

    BEGIN CATCH
		EXEC RethrowError;
	END CATCH	

	SET NOCOUNT OFF
END

GO

IF NOT EXISTS (SELECT NAME FROM sys.objects WHERE TYPE = 'P' AND NAME = 'Deleteasync_tasks')
BEGIN
	EXEC('CREATE PROCEDURE [dbo].[Deleteasync_tasks] AS RETURN')
END

GO

ALTER PROCEDURE [dbo].[Deleteasync_tasks]
	 @task_id uniqueidentifier
AS
BEGIN
	SET NOCOUNT ON
	
    DELETE FROM [dbo].[async_tasks]
	WHERE [task_id]=@task_id
    
    SET NOCOUNT OFF
END

GO

IF NOT EXISTS (SELECT NAME FROM sys.objects WHERE TYPE = 'P' AND NAME = 'GetAllFromasync_tasks')
BEGIN
	EXEC('CREATE PROCEDURE [dbo].[GetAllFromasync_tasks] AS RETURN')
END

GO

ALTER PROCEDURE [dbo].[GetAllFromasync_tasks]    
AS
BEGIN
	SET TRANSACTION ISOLATION LEVEL READ UNCOMMITTED
	SET NOCOUNT ON
	
	SELECT
	[async_tasks].[action_type] AS 'action_type',
	[async_tasks].[result] AS 'result',
	[async_tasks].[status] AS 'status',
	[async_tasks].[task_id] AS 'task_id',
	[async_tasks].[action_parameters] AS 'action_parameters'
FROM [dbo].[async_tasks] [async_tasks]

	SET NOCOUNT OFF
END

GO

IF NOT EXISTS (SELECT NAME FROM sys.objects WHERE TYPE = 'P' AND NAME = 'Getasync_tasksBytask_id')
BEGIN
	EXEC('CREATE PROCEDURE [dbo].[Getasync_tasksBytask_id] AS RETURN')
END

GO

ALTER PROCEDURE [dbo].[Getasync_tasksBytask_id] 
	@task_id uniqueidentifier
AS
BEGIN
	SET TRANSACTION ISOLATION LEVEL READ UNCOMMITTED
	SET NOCOUNT ON
	
	SELECT
	[async_tasks].[action_type] AS 'action_type',
	[async_tasks].[result] AS 'result',
	[async_tasks].[status] AS 'status',
	[async_tasks].[task_id] AS 'task_id',
	[async_tasks].[action_parameters] AS 'action_parameters'
	FROM [dbo].[async_tasks] [async_tasks]
	WHERE [task_id]=@task_id

	SET NOCOUNT OFF
END

GO


