#!/usr/bin/python

# Imports
import sys
import os
import logging
import traceback
import datetime
import subprocess
import pwd
from optparse import OptionParser
import yum
from StringIO import StringIO
import time
import common_utils as utils

# Consts
RPM_NAME = "rhevm"
#TODO: Work with a real list here
RPM_LIST = "rhevm-doc rhevm-spice-client-x64-msi rhevm-notification-service \
rhevm-genericapi rhevm rhevm-tools-common rhevm-spice-client-x86-cab \
rhevm-spice-client-x86-msi rhevm-backend rhevm-admin-portal-wpf \
quartz-rhevm rhevm-iso-uploader rhevm-jboss-deps rhevm-log-collector \
rhevm-userportal rhevm-restapi rhevm-config rhevm-spice-client-x64-cab \
rhevm-setup rhevm-dbscripts vdsm-bootstrap rhev-guest-tools-iso"

RPM_LOCK_LIST = "rhevm-genericapi rhevm rhevm-backend rhevm-admin-portal-wpf rhevm-jboss-deps \
rhevm-userportal rhevm-restapi rhevm-config rhevm-tools-common rhevm-notification-service"

RPM_BACKEND = "rhevm-backend"
RPM_DBSCRIPTS = "rhevm-dbscripts"
RPM_SETUP = "rhevm-setup"

DB_NAME = "rhevm"
DB_USER = "postgres"
SERVER_NAME = "127.0.0.1"
BACKUP_DIR = "/usr/share/rhevm/db-backups"
BACKUP_FILE = "rhevm_db_backup"
LOG_PATH = "/var/log/rhevm"
LOG_FILE = "rhevm-upgrade.log"
PGDUMP_EXEC = "/usr/bin/pg_dump"
DROPDB_EXEC = "/usr/bin/dropdb"
PSQL_EXEC = "/usr/bin/psql"
PGRESTORE_EXEC = "/usr/bin/pg_restore"
DB_UPDATE_EXEC = "/usr/share/rhevm/dbscripts/upgrade.sh"
YUM_EXEC = "/usr/bin/yum"
YUM_VERSION_LOCK_FILE = "/etc/yum/pluginconf.d/versionlock.list"
RPM_EXEC = "/bin/rpm"
SERVICE_EXEC="/sbin/service"
ETL_SERVICE="/etc/init.d/rhevm-etl"

FILE_JBOSS_LOG4J_XML_SRC="/usr/share/rhevm/conf/jboss-log4j.xml"
FILE_JBOSS_LOG4J_XML_DEST="/var/lib/jbossas/server/rhevm-slimmed/conf/jboss-log4j.xml"

#MSGS
MSG_ERROR_USER_NOT_ROOT = "Error: insufficient permissions for user %s, you must run with user root."
MSG_NO_ROLLBACK = "Error: Current installation "
MSG_RC_ERROR = "Return Code is not zero"
MSG_INFO_NO_UPGRADE_AVAIL = "No updates available"
MSG_INFO_UPGRADE_AVAIL = "%d Updates available:"
MSG_ERROR_NO_ROLLBACK_AVAIL = "Error: Installed packages are missing from the yum repositories\n\
Please check your yum repositories or use --no-yum-rollback"
MSG_ERROR_NEW_SETUP_AVAIL="\nError: New %s rpm available in yum. Please execute `yum update %s` or use --force-current-setup-rpm" % (RPM_SETUP, RPM_SETUP)
MSG_ERROR_BACKUP_DB = "Error: Database backup failed"
MSG_ERROR_RESTORE_DB = "Error: Database restore failed"
MSG_ERROR_DROP_DB = "Error: Database drop failed"
MSG_ERROR_UPDATE_DB = "Error: Database update failed"
MSG_ERROR_YUM_HISTORY_LIST = "Error: Can't get history from yum"
MSG_ERROR_YUM_HISTORY_GETLAST = "Error: Can't find last install transaction in yum"
MSG_ERROR_YUM_HISTORY_UNDO = "Error: Can't rollback yum"
MSG_ERROR_YUM_LOCK = "Error: Can't edit yum lock file"
MSG_ERROR_YUM_UPDATE = "Error: Can't update rhevm rpm"
MSG_ERROR_CHECK_LOG = "Error: RHEV Manager Upgrade failed.\nplease check log at %s"
MSG_ERR_FAILED_START_JBOSS_SERVICE = "Error: Can't start JBoss"
MSG_ERR_FAILED_JBOSS_SERVICE_STILL_RUN = "Error: Can't stop jboss service. Please shut it down manually."
MSG_ERR_FAILED_STP_JBOSS_SERVICE = "Error: Can't stop JBoss"
MSG_ERR_FAILED_STATUS_JBOSS_SERVICE = "Error: Can't get JBoss service status"
MSG_ERR_FAILED_START_SERVICE = "Error: Can't start the %s service"
MSG_ERR_FAILED_STOP_SERVICE = "Error: Can't stop the %s service"
MSG_ERR_SQL_CODE="Failed running sql query"

MSG_INFO_DONE = "DONE"
MSG_INFO_ERROR = "ERROR"
MSG_INFO_REASON = " **Reason: %s**\n"
MSG_INFO_STOP_JBOSS = "Stopping JBoss Service"
MSG_INFO_BACKUP_DB = "Backing Up RHEVM DB"
MSG_INFO_YUM_UPDATE = "Updating rpms"
MSG_INFO_DB_UPDATE = "Updating RHEVM DB"
MSG_INFO_RUN_POST = "Running post install configuration"
MSG_ERROR_RHEVM_UPGRADE = "\n **Error: Failed upgrading rhevm, rolling back**"
MSG_INFO_DB_RESTORE = "Restoring RHEVM DB"
MSG_INFO_YUM_ROLLBACK = "Rolling back rpms"
MSG_INFO_NO_YUM_ROLLBACK = "Skipping yum rollback"
MSG_INFO_START_JBOSS = "Starting JBoss"
MSG_INFO_DB_BACKUP_FILE = "DB Backup available at "
MSG_INFO_LOG_FILE = "Upgrade log available at"
MSG_INFO_CHECK_UPDATE = "Checking for updates... (This may take several minutes)"
MSG_INFO_UPGRADE_OK = "RHEV Manager upgrade completed successfully!"
MSG_INFO_STOP_INSTALL_EXIT="Upgrade stopped, Goodbye."
MSG_ERR_EXP_FAILED_UPD_LOG4J="Failed to configure application logging (could not edit %s)"
MSG_INFO_UPDATE_JBOSS_PROFILE="Updating JBoss Profile"

MSG_ALERT_STOP_JBOSS="\nDuring the upgrade process, RHEV Manager will not be accessible.\n\
All existing running virtual machines will continue but you will not be able to\n\
start or stop any new virtual machines during the process.\n"
INFO_Q_STOP_JBOSS="Would you like to proceed"
MSG_INFO_REPORTS="Perform the following steps to upgrade the history service \
or the reporting package:\n\
1. Execute: yum update rhevm-reports*\n\
2. Execute: rhevm-dwh-setup\n\
3. Execute: rhevm-reports-setup"

# COLORS
RED="\033[0;31m"
GREEN="\033[92m"
BLUE="\033[94m"
YELLOW="\033[93m"
NO_COLOR="\033[0m"

SPACE_LENGTH = 70

messages = []

# Code
def getOptions():
    parser = OptionParser()
    parser.add_option("-r", "--no-yum-rollback",
                      action="store_false", dest="yum_rollback", default=True,
                      help="don't rollback yum transaction")

    parser.add_option("-u", "--unattended",
                      action="store_true", dest="unattended_upgrade", default=False,
                      help="unattended upgrade (this option will stop jboss service before upgrading rhevm)")

    parser.add_option("-s", "--force-current-setup-rpm",
                      action="store_true", dest="force_current_setup_rpm", default=False,
                      help="Ignore new %s rpm"%(RPM_SETUP))

    parser.add_option("-c", "--check-update",
                      action="store_true", dest="check_update", default=False,
                      help="Check for available package updates")

    (options, args) = parser.parse_args()
    return (options, args)

def askYesNo(question=None):
    """
    service func to ask yes/no
    input from user
    """
    message = StringIO()
    userQuestion = "%s? (yes|no): "%(question)
    logging.debug("asking user: %s"%userQuestion)
    message.write(userQuestion)
    message.seek(0)
    answer = raw_input(message.read())
    logging.debug("user answered: %s"%(answer))
    answer = answer.lower()
    if answer == "yes" or answer == "y":
        return True
    elif answer == "no" or answer == "n":
        return False
    else:
        return askYesNo(question)

def _getColoredText (text, color):
    """ 
        gets text string and color
        and returns a colored text.
        the color values are RED/BLUE/GREEN/YELLOW
        everytime we color a text, we need to disable
        the color at the end of it, for that
        we use the NO_COLOR chars.
    """
    return color + text + NO_COLOR

def getCurrentDateTime(isUtc=None):
    now = None
    if (isUtc is not None):
        now = datetime.datetime.utcnow()
    else:
        now = datetime.datetime.now()
    return now.strftime("%Y_%m_%d_%H_%M_%S")

def checkJbossService():
    """
    Ask user to stop jboss service before
    upgrading rhevm

    returns: true is user choose to stop jboss
    false otherwise
    """
    logging.debug("checking the status of jbossas service")
    cmd = "%s jbossas status" % (SERVICE_EXEC)
    output, rc = execExternalCmd(cmd, True, MSG_ERR_FAILED_STATUS_JBOSS_SERVICE)
    if " is running" in output:
        logging.debug("jbossas service is up and running")

        print MSG_ALERT_STOP_JBOSS
        answer = askYesNo(INFO_Q_STOP_JBOSS)

        # If user choose yes -> return true (stop jbossas)
        if answer:
            return True
        else:
            logging.debug("User chose not to stop jboss")
            return False
    else:
        # If jboss is not running, we don't need to stop it
        return True

def execSqlCommand(userName, dbName, sqlQuery, failOnError=False, errMsg=MSG_ERR_SQL_CODE):
    logging.debug("running sql query %s on db: \'%s\'."%(dbName, sqlQuery))
    cmd = "%s -U %s -d %s -c \"%s\""%(PSQL_EXEC, userName, dbName, sqlQuery)
    return execExternalCmd(cmd, failOnError, errMsg)

def execExternalCmd(command, failOnError=False, msg=MSG_RC_ERROR):
    logging.debug("cmd = %s"%(command))
    p = subprocess.Popen(command, shell=True,
        stdin=subprocess.PIPE, stdout=subprocess.PIPE,
        stderr=subprocess.PIPE)
    out, err = p.communicate()
    logging.debug("output = %s"%(out))
    logging.debug("stderr = %s"%(err))
    logging.debug("retcode = %s"%(p.returncode))
    output = out + err
    if failOnError and p.returncode != 0:
        raise OSError(msg)
    return ("".join(output.splitlines(True)), p.returncode)

def initLogging():
    global LOG_FILE
    try:
        if not os.path.isdir(LOG_PATH):
            os.makedirs(LOG_PATH)
        LOG_FILE = "%s/rhevm-upgrade_%s.log"%(LOG_PATH, getCurrentDateTime())
        level = logging.DEBUG
        # TODO: Move to mode="a"?
        hdlr = logging.FileHandler(filename = LOG_FILE, mode='w')
        fmts='%(asctime)s::%(levelname)s::%(module)s::%(lineno)d::%(name)s:: %(message)s'
        dfmt='%Y-%m-%d %H:%M:%S'
        fmt = logging.Formatter(fmts, dfmt)
        hdlr.setFormatter(fmt)
        logging.root.addHandler(hdlr)
        logging.root.setLevel(level)
    except:
        logging.error(traceback.format_exc())
        raise Exception("Failed to initiate logger")

def _verifyUserPermissions():
    username = pwd.getpwuid(os.getuid())[0]
    if (username != 'root'):
        print MSG_ERROR_USER_NOT_ROOT%(username)
        sys.exit(1)

def retry(func, expectedException=Exception, tries=None, timeout=None, sleep=1):
    """
    Retry a function. Wraps the retry logic so you don't have to
    implement it each time you need it.

    :param func: The callable to run.
    :param expectedException: The exception you expect to receive when the function fails.
    :param tries: The number of time to try. None\0,-1 means infinite.
    :param timeout: The time you want to spend waiting. This **WILL NOT** stop the method.
                    It will just not run it if it ended after the timeout.
    :param sleep: Time to sleep between calls in seconds.
    """
    if tries in [0, None]:
        tries = -1

    if timeout in [0, None]:
        timeout = -1

    startTime = time.time()

    while True:
        tries -= 1
        try: 
            logging.debug("retrying function attempt number %s"%tries)
            return func()
        except expectedException:
            if tries == 0:
                raise

            if (timeout > 0) and ((time.time() - startTime) > timeout):
                raise

            time.sleep(sleep)

def isDBUp():
    """
    check if rhevm db is up
    """
    logging.debug("checking if %s db is already installed and running.."%DB_NAME)
    (out, rc) = execSqlCommand(DB_USER, DB_NAME, "select 1", True)

class Service():
    def __init__(self, serviceName):
        self.wasStopped = False
        self.wasStarted = False
        self.serviceName = serviceName

    def isServiceAvailable(self):
        if os.path.exists("/etc/init.d/%s" % self.serviceName):
            return True
        return False

    def start(self, raiseFailure = False):
        logging.debug("starting %s", self.serviceName)
        (output, rc) = self._serviceFacility(self.serviceName, "start")
        if rc == 0:
            self.wasStarted = True
        elif raiseFailure:
            raise Exception(MSG_ERR_FAILED_START_SERVICE % self.serviceName)

        return (output, rc)

    def stop(self, raiseFailure = False):
        logging.debug("stopping %s", self.serviceName)
        (output, rc) = self._serviceFacility(self.serviceName, "stop")
        if rc == 0:
            self.wasStopped = True
        elif raiseFailure:
                raise Exception(MSG_ERR_FAILED_STOP_SERVICE % self.serviceName)

        return (output, rc)

    def conditionalStart(self, raiseFailure = False):
        """
        Will only start if wasStopped is set to True
        """
        if self.wasStopped:
            logging.debug("Service %s was stopped. starting it again"%self.serviceName)
            return self.start(raiseFailure)
        else:
            logging.debug("Service was not stopped. there for we're not starting it")
            return (False, False)

    def status(self):
        logging.debug("getting status for %s", self.serviceName)
        (output, rc) = self._serviceFacility(self.serviceName, "status")
        return (output, rc)

    def _serviceFacility(self, serviceName, action): 
        """
        Execute the command "service serviceName action"
        returns: output, rc
        """
        logging.debug("executing action %s on service %s", serviceName, action)
        cmd = [SERVICE_EXEC, serviceName, action]
        return utils.execCmd(cmd, None, failOnError=False)

class RhevYum():
    def __init__(self):
        self.updated = False
        self.yumbase = None
        self.upackages = []
        self.ipackages = []
        self.__initbase()

    def __initbase(self):
        self.yumbase = yum.YumBase()
        self.yumbase.conf.cache = False # Do not relay on existing cache
        self.yumbase.cleanMetadata()
        self.yumbase.cleanSqlite()

    def _lock(self):
        logging.debug("Yum lock started")
        #TODO: seperate exces below
        cmd = "%s -q %s >> %s"%(RPM_EXEC, RPM_LOCK_LIST, YUM_VERSION_LOCK_FILE)
        output, rc = execExternalCmd(cmd, True, MSG_ERROR_YUM_LOCK)
        logging.debug("Yum lock completed successfully")

    def _unlock(self):
        logging.debug("Yum unlock started")
        # Read file content
        fd = file(YUM_VERSION_LOCK_FILE)
        fileText = fd.readlines()
        fd.close()

        # Change content:
        fd = file(YUM_VERSION_LOCK_FILE, 'w')
        for line in fileText:
            if not RPM_NAME in line:
                fd.write(line)
        fd.close()
        logging.debug("Yum unlock completed successfully")

    def update(self):
        self._unlock()
        try:
            # yum update rhevm
            # TODO: Run test transaction
            logging.debug("Yum update started")
            cmd = "%s update -q -y %s"%(YUM_EXEC, RPM_LIST)
            output, rc = execExternalCmd(cmd, True, MSG_ERROR_YUM_UPDATE)
            logging.debug("Yum update completed successfully")
            self.updated = True
        finally:
            self._lock()

    def updateAvailable(self):
        logging.debug("Yum list updates started")
        
        # Get packages info from yum
        rpms = RPM_LIST.split()
        self._unlock()
        try:
            logging.debug("Getting list of packages to upgrade")
            pkgs = self.yumbase.doPackageLists(patterns=rpms)
        finally:
            self._lock()
        
        # Save update candidates
        if pkgs.available:
            self.upackages = [str(i) for i in sorted(pkgs.available)] # list of rpm names to update
            logging.debug("%s Packages marked for update:"%(len(self.upackages)))
            logging.debug(self.upackages)
        else:
            logging.debug("No packages marked for update")

        # Save installed packages
        self.ipackages = [str(i) for i in sorted(pkgs.installed)] # list of rpm names already installed
        logging.debug("Installed packages:")
        logging.debug(self.ipackages)

        logging.debug("Yum list updated completed successfully")
        
        # Return 
        if pkgs.available:
            return True
        else:
            return False
    
    def rollbackAvailable(self):
        logging.debug("Yum rollback-avail started")
        
        # Get All available packages in yum
        rpms = RPM_LIST.split()
        pkgs = self.yumbase.pkgSack.returnPackages(patterns=rpms)
        available = [str(i) for i in sorted(pkgs)] # list of available rpm names
        logging.debug("%s Packages available in yum:"%(len(available)))
        logging.debug(available)

        # Verify all installed packages available in yum 
        # self.ipackages is populated in updateAvailable
        for installed in self.ipackages:
            if installed not in available:
                logging.debug("%s not available in yum"%(installed))
                return False

        logging.debug("Yum rollback-avail completed successfully")
        return True

    def rollback(self):
        self._unlock()
        try:
            # yum history undo 17
            # Do rollback only if update went well
            if self.updated:
                tid = self.getLastTid()
                logging.debug("Yum rollback started")
                cmd = "%s history -y undo %s"%(YUM_EXEC, tid)
                output, rc = execExternalCmd(cmd, True, MSG_ERROR_YUM_HISTORY_UNDO)
                logging.debug("Yum rollback completed successfully")
            else:
                logging.debug("No rollback needed")
        finally:
            self._lock()

    def getLastTid(self): 
        # TODO: Make sure it's our transaction 
        logging.debug("Yum getLastTid started")
        cmd = "%s history list %s"%(YUM_EXEC, self.upackages[0])
        output, rc = execExternalCmd(cmd, True, MSG_ERROR_YUM_HISTORY_LIST)
        logging.debug("Yum getLastTid completed successfully")

        # Parse last tid
        for line in output.splitlines():
            lsplit = line.split("|")
            if len(lsplit) > 3:
                if 'Update' in lsplit[3]:
                    return lsplit[0].strip()

        raise ValueError(MSG_ERROR_YUM_HISTORY_GETLAST)

    def isCandidateForUpdate(self, rpm):
        candidate = False
        for package in self.upackages:
            if rpm in package:
                candidate = True
        return candidate

    def getUpdateCandidates(self):
        return self.upackages

class DB():
    def __init__(self):
        date = getCurrentDateTime()
        self.sqlfile = "%s/%s_%s.sql" % (BACKUP_DIR, BACKUP_FILE, date)
        self.updated = False

    def __del__(self):
        if self.updated:
            logging.debug(MSG_INFO_DB_BACKUP_FILE + self.sqlfile)
            print "* %s %s" % (MSG_INFO_DB_BACKUP_FILE, self.sqlfile)

    def backup(self):
        # pg_dump -C -E UTF8  --column-inserts --disable-dollar-quoting  --disable-triggers -U postgres --format=p -f $dir/$file  rhevm
        logging.debug("DB Backup started")
        cmd = "%s -C -E UTF8 --column-inserts --disable-dollar-quoting  --disable-triggers -U %s --format=p -f %s %s"\
            %(PGDUMP_EXEC, DB_USER, self.sqlfile, DB_NAME)
        output, rc = execExternalCmd(cmd, True, MSG_ERROR_BACKUP_DB)
        logging.debug("DB Backup completed successfully")

    def restore(self):
        #psql -U postgres -f <backup directory>/<backup_file> 
        if self.updated:
            logging.debug("DB Restore started")
            # Drop
            cmd = "%s -U %s %s"%(DROPDB_EXEC, DB_USER, DB_NAME)
            output, rc = execExternalCmd(cmd, True, MSG_ERROR_DROP_DB)
            
            # Restore
            cmd = "%s -U %s -f %s"%(PSQL_EXEC, DB_USER, self.sqlfile)
            output, rc = execExternalCmd(cmd, True, MSG_ERROR_RESTORE_DB)
            logging.debug("DB Restore completed successfully")
        else:
            logging.debug("No DB Restore needed")

    def update(self):
        cwd = os.getcwd()
        os.chdir(os.path.dirname(DB_UPDATE_EXEC))
        
        # Make sure we always returning to cwd 
        try:
            self.updated = True
            logging.debug("DB Update started")
            # ./upgrade.sh -s ${SERVERNAME} -d ${DATABASE} -u ${USERNAME};
            cmd = "%s -s %s -d %s -u %s"%(DB_UPDATE_EXEC, SERVER_NAME, DB_NAME, DB_USER) 
            output, rc = execExternalCmd(cmd, True, MSG_ERROR_UPDATE_DB)
            logging.debug("DB Update completed successfully")
        finally:
            os.chdir(cwd)

def restartPostgresql():
    """
    restart the postgresql service
    """

    logging.debug("Restarting the postgresql service")
    postgresql = Service("postgresql")
    postgresql.stop(True)
    postgresql.start(True)

    # Now we want to make sure the postgres service is up
    # before we continue to the upgrade
    retry(isDBUp, tries=10, timeout=30)

def stopJboss():
    logging.debug("stopping jboss service.")
    output, rc = execExternalCmd("/sbin/service jbossas stop", True, MSG_ERR_FAILED_STP_JBOSS_SERVICE)
    
    # JBoss service sometimes return zero rc even if service is still up
    if "[FAILED]" in output and "Timeout: Shutdown command was sent, but process is still running" in output:
        raise OSError(MSG_ERR_FAILED_JBOSS_SERVICE_STILL_RUN)

def startJboss():
    logging.debug("starting jboss service.")
    output, rc = execExternalCmd("/sbin/service jbossas start", True, MSG_ERR_FAILED_START_JBOSS_SERVICE)

def runPost():
    logging.debug("Running post script")
    import post_upgrade as post
    post.run()
    logging.debug("Post script completed successfully")

def runFunc(funcList, dispString):
    print "%s..."%(dispString),
    sys.stdout.flush()
    spaceLen = SPACE_LENGTH - len(dispString)
    try:
        for func in funcList:
            func()
        print ("[ " + _getColoredText(MSG_INFO_DONE, GREEN) + " ]").rjust(spaceLen)
    except:
        print ("[ " + _getColoredText(MSG_INFO_ERROR, RED) + " ]").rjust(spaceLen)
        raise

def isUpdateRelatedToDb(yumo):
    """
    Verifies current update needs DB manipulation (backup/update/rollback)
    """

    logging.debug("Verifing update is related to db")

    related = False
    for rpm in RPM_BACKEND, RPM_DBSCRIPTS:
        if yumo.isCandidateForUpdate(rpm):
            related = True

    logging.debug("isUpdateRelatedToDb value is %s"%(str(related)))
    return related

def editLog4jXml():
    try:
        utils.copyFile(FILE_JBOSS_LOG4J_XML_SRC, FILE_JBOSS_LOG4J_XML_DEST)
        logging.debug("editing log file %s"%(FILE_JBOSS_LOG4J_XML_DEST))
        logHandler = utils.XMLConfigFileHandler(FILE_JBOSS_LOG4J_XML_DEST)
        logHandler.open()
        nodes = logHandler.xpathEval("//appender[@name='RHEVM_LOG' or @name='PUBLICAPI_LOG']/param[@name='File']")
        logging.debug(nodes)
        for node in nodes:
            string = node.prop("value")
            logging.debug(node)
            logging.debug("Current value is %s"%(string))
            newString = string.replace("${jboss.server.log.dir}","/var/log")
            node.setProp("value",newString)
        logHandler.close()
    except:
        logging.error(traceback.format_exc())
        raise Exception(MSG_ERR_EXP_FAILED_UPD_LOG4J % FILE_JBOSS_LOG4J_XML_DEST)

def printMessages():
    for msg in messages:
        logging.info(msg)
        print "* %s" % msg.strip()

def addAdditionalMessages(addReports=False):
    global messages
    messages.append(MSG_INFO_LOG_FILE + " " + LOG_FILE)

    if addReports:
        messages.append(MSG_INFO_REPORTS)


def stopRhevmDbRelatedServices(etlService, notificationService):
    """
    shut down etl and notifier services
    in order to disconnect any open sessions to the db
    """
    # If the rhevm-etl service is installed, then try and stop it.
    if etlService.isServiceAvailable():
        try:
            etlService.stop(True)
        except:
            logging.warn("Failed to stop rhevm-etl")
            logging.warn(traceback.format_exc())
            messages.append(MSG_ERR_FAILED_STOP_SERVICE % "rhevm-etl")

    # If the rhevm-notifierd service is up, then try and stop it.
    if notificationService.isServiceAvailable():
        try:
            (status, rc) = notificationService.status()
            if utils.verifyStringFormat(status, ".*running.*"):
                logging.debug("stopping rhevm-notifierd service..")
                notificationService.stop()
        except:
            logging.warn("Failed to stop rhevm-notifierd service")
            logging.warn(traceback.format_exc())
            messages.append(MSG_ERR_FAILED_STOP_SERVICE % "rhevm-notifierd")

def startRhevmDbRelatedServices(etlService, notificationService):
    """
    bring back any service we stopped
    we won't start services that are down
    but weren't stopped by us
    """
    if etlService.isServiceAvailable():
        (output, rc) = etlService.conditionalStart()
        if rc != 0:
            logging.warn("Failed to start rhevm-etl")
            messages.append(MSG_ERR_FAILED_START_SERVICE % "rhevm-etl")

    if notificationService.isServiceAvailable():
        (output, rc) = notificationService.conditionalStart()
        if rc != 0:
            logging.warn("Failed to start rhevm-notifierd: exit code %d" % rc)
            messages.append(MSG_ERR_FAILED_START_SERVICE % "rhevm-notifierd")

def main(options):
    rhyum = RhevYum()
    db = DB()
    
    # Check for upgrade, else exit
    print MSG_INFO_CHECK_UPDATE
    if not rhyum.updateAvailable():
        logging.debug(MSG_INFO_NO_UPGRADE_AVAIL)
        print MSG_INFO_NO_UPGRADE_AVAIL
        sys.exit(0)
    else:
        updates = rhyum.getUpdateCandidates()
        print MSG_INFO_UPGRADE_AVAIL % (len(updates))
        for package in updates:
            print " * %s" % package
        if options.check_update:
            sys.exit(100)

    # Check for setup package
    if rhyum.isCandidateForUpdate(RPM_SETUP) and not options.force_current_setup_rpm:
        logging.debug(MSG_ERROR_NEW_SETUP_AVAIL)
        print MSG_ERROR_NEW_SETUP_AVAIL
        sys.exit(3)

    # Make sure we will be able to rollback
    if not rhyum.rollbackAvailable() and options.yum_rollback:
        logging.debug(MSG_ERROR_NO_ROLLBACK_AVAIL)
        print MSG_ERROR_NO_ROLLBACK_AVAIL
        print MSG_ERROR_CHECK_LOG%(LOG_FILE)
        sys.exit(2)

    # No rollback in this case
    try:
        # We ask the user before stoping jboss or take command line option
        if options.unattended_upgrade or checkJbossService():
            # Stopping jboss
            runFunc([stopJboss], MSG_INFO_STOP_JBOSS )
        else:
            # This means that user chose not to stop jboss
            logging.debug("exiting gracefully")
            print MSG_INFO_STOP_INSTALL_EXIT
            sys.exit(0)

        # Backup DB
        if isUpdateRelatedToDb(rhyum):
            runFunc([db.backup], MSG_INFO_BACKUP_DB)

    except Exception as e:
        print e
        raise

    # In case of failure, do rollback
    try:
        # yum update
        runFunc([rhyum.update], MSG_INFO_YUM_UPDATE)

        # define db connections services
        etlService = Service("rhevm-etl")
        notificationService = Service("rhevm-notifierd")

        # check if update is relevant to db update
        if isUpdateRelatedToDb(rhyum):
            stopRhevmDbRelatedServices(etlService, notificationService)

            # Update the db
            runFunc([restartPostgresql, db.update], MSG_INFO_DB_UPDATE)

            # Bring up any services we shut down before db upgrade
            startRhevmDbRelatedServices(etlService, notificationService)

        # Edit log4j.xml file and copy it into the jbossas profile
        runFunc([editLog4jXml], MSG_INFO_UPDATE_JBOSS_PROFILE)

        # post install conf
        runFunc([runPost], MSG_INFO_RUN_POST)

    except:
        logging.error(traceback.format_exc())
        logging.error("Rolling back update")

        print MSG_ERROR_RHEVM_UPGRADE
        print MSG_INFO_REASON%(sys.exc_info()[1])

        # db restore
        if isUpdateRelatedToDb(rhyum):
            runFunc([db.restore], MSG_INFO_DB_RESTORE)

        # yum rollback
        if options.yum_rollback:
            runFunc([rhyum.rollback], MSG_INFO_YUM_ROLLBACK)
        else:
            print MSG_INFO_NO_YUM_ROLLBACK
            logging.debug("Skipping yum rollback")

        raise

    finally:
        # start jboss
        runFunc([startJboss], MSG_INFO_START_JBOSS)

    # Print log location on success
    addAdditionalMessages(etlService.isServiceAvailable())
    print "\n%s\n" % MSG_INFO_UPGRADE_OK
    printMessages()

if __name__ == '__main__':
    try:
        # Must run as root
        _verifyUserPermissions()

        # Init logging facility
        initLogging()

        # get iso and domain from user arguments
        (options, args) = getOptions()

        main(options)

    except SystemExit:
        raise

    except:
        print MSG_ERROR_CHECK_LOG%(LOG_FILE)
        logging.error(traceback.format_exc())
        sys.exit(1)
