using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Text;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.events
{
	public class EventListModel : SearchableListModel
	{
		private readonly ITimer timer;

		#region Commands

		public UICommand RefreshCommand { get; private set; }

		#endregion

		#region Properties

		private AuditLog lastEvent;
		public AuditLog LastEvent
		{
			get { return lastEvent; }
			private set
			{
				if (lastEvent != value)
				{
					lastEvent = value;
					OnPropertyChanged(new PropertyChangedEventArgs("LastEvent"));
				}
			}
		}

		private bool isAdvancedView;
		public bool IsAdvancedView
		{
			get { return isAdvancedView; }
			set
			{
				if (isAdvancedView != value)
				{
					isAdvancedView = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsAdvancedView"));
				}
			}
		}

		#endregion

		public EventListModel()
		{
			Title = "Events";

			RefreshCommand = new UICommand("Refresh", this);

			DefaultSearchString = "Events:";
			SearchString = DefaultSearchString;

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;

			timer = (ITimer)TypeResolver.Instance.Resolve(typeof(ITimer));
			timer.Interval = Configurator.PollingTimerInterval;
			timer.TickEvent.addListener(this);
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("event");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			Items = new ObservableCollection<AuditLog>();
			LastEvent = null;
			timer.Start();
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();
			SyncSearch();
		}

		private void Refresh()
		{
			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				EventListModel eventListModel = (EventListModel)model;
				List<AuditLog> list = (List<AuditLog>)((VdcQueryReturnValue)ReturnValue).ReturnValue;
				eventListModel.UpdateItems(list);
			};//END_DELEGATE

			SearchParameters searchParameters = new SearchParameters(SearchString, SearchType.AuditLog)
			                                    	{
			                                    		MaxCount = SearchPageSize,
			                                    		SearchFrom = LastEvent != null ? LastEvent.audit_log_id : 0,
			                                    		Refresh = false
			                                    	};

			  Frontend.RunQuery(VdcQueryType.Search, searchParameters, _asyncQuery);
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ProvideTickEvent.Definition))
			{
				RefreshCommand.Execute();
			}
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == RefreshCommand)
			{
				Refresh();
				UpdatePagingAvailability();
			}
		}

		internal override void EnsureAsyncSearchStopped()
		{
			base.EnsureAsyncSearchStopped();

			timer.Stop();
		}

		private void UpdateItems(List<AuditLog> source)
		{
			if (Items == null)
			{
				return;
			}

			IList<AuditLog> list = (IList<AuditLog>)Items;

			source.Sort(new Linq.AuditLogComparer());

			foreach (AuditLog item in source)
			{
				if (list.Count == SearchPageSize)
				{
					list.RemoveAt(list.Count - 1);
				}

				list.Insert(0, item);
			}

			LastEvent = Linq.FirstOrDefault(list);
		}

	}
}
