using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.configure;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using org.ovirt.engine.ui.uicompat;

namespace org.ovirt.engine.ui.uicommon.models.datacenters
{
	public class DataCenterListModel : ListWithDetailsModel, ISupportSystemTreeContext
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand ForceRemoveCommand { get; private set; }
		public UICommand ActivateCommand { get; private set; }
		public UICommand GuideCommand { get; private set; }
		public UICommand RecoveryStorageCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		protected object[] SelectedKeys
		{
			//get { return SelectedItems == null ? new object[0] : SelectedItems.Cast<storage_pool>().Select(a => a.id).Cast<object>().ToArray(); }
			get
			{
				if (SelectedItems == null)
					return new object[0];
				else
				{
					List<object> objL = new List<object>();
					foreach (storage_pool a in Linq.Cast<storage_pool>(SelectedItems))
						objL.Add(a.Id);
					return objL.ToArray();
				}
			}
		}

		public object GuideContext { get; set; }

		#endregion


		public DataCenterListModel()
		{
			Title = "Data Centers";

			DefaultSearchString = "DataCenter:";
			SearchString = DefaultSearchString;

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			ForceRemoveCommand = new UICommand("ForceRemove", this) {IsExecutionAllowed = true};
			RecoveryStorageCommand = new UICommand("RecoveryStorage", this);
			ActivateCommand = new UICommand("Activate", this);
			GuideCommand = new UICommand("Guide", this);

			UpdateActionAvailability();

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;
		}

		public void Guide()
		{
			DataCenterGuideModel model = new DataCenterGuideModel();
			Window = model;
			model.Title = "New Data Center - Guide Me";
			model.HashName = "new_data_center_-_guide_me";

			model.Entity = GuideContext != null
				? DataProvider.GetDataCenterById(GuideContext is guid ? (Guid)(guid)GuideContext : (Guid)GuideContext)
				: null;


			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Configure Later",
					IsDefault = true,
					IsCancel = true
				});
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new DataCenterStorageListModel());
			list.Add(new DataCenterNetworkListModel());
			list.Add(new DataCenterClusterListModel());
			list.Add(new PermissionListModel());
			list.Add(new DataCenterEventListModel());
			DetailModels = list;
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("datacenter");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.Search, new SearchParameters(SearchString, SearchType.StoragePool)
			                                     	{
			                                     		MaxCount= SearchPageSize
			                                     	});

		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterSearch(SearchString, SearchType.StoragePool, SearchPageSize);
			Items = AsyncResult.Data;
		}

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			DataCenterModel model = new DataCenterModel();
			Window = model;
			model.Title = "New Data Center";
			model.HashName = "new_data_center";
			model.IsNew = true;
			model.StorageTypeList.SelectedItem = StorageType.NFS;


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void Edit()
		{
			storage_pool dataCenter = (storage_pool)SelectedItem;

			if (Window != null)
			{
				return;
			}

			DataCenterModel model = new DataCenterModel();
			Window = model;
			model.DataCenterId = dataCenter.Id;
			model.Title = "Edit Data Center";
			model.HashName = "edit_data_center";
			model.Name.Entity = dataCenter.name;

			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.DataCenter)
			{
				model.Name.IsChangable = false;
				model.Name.Info = "Cannot edit Data Center's Name in tree context";
			}

			model.Description.Entity = dataCenter.description;
			model.OriginalName = dataCenter.name;
			if (DataProvider.GetStorageDomainList(dataCenter.Id).Count != 0)
			{
				model.StorageTypeList.IsChangable = false;
				model.StorageTypeList.ChangeProhibitionReasons.Add("Cannot change Repository type with Storage Domains attached to it");
			}

			model.StorageTypeList.SelectedItem = dataCenter.storage_pool_type;

			//Version
			foreach (object a in model.Version.Items)
			{
				Version item = (Version)a;
				if (item == dataCenter.compatibility_version)
				{
					model.Version.SelectedItem = item;
					break;
				}
			}


			model.Commands.Add(
					new UICommand("OnSave", this)
					{
						Title = "OK",
						IsDefault = true
					});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Data Center(s)";
			model.HashName = "remove_data_center";
			model.Message = "Data Center(s)";

			List<string> list = new List<string>();
			foreach (storage_pool a in Linq.Cast<storage_pool>(SelectedItems))
			{
				list.Add(a.name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void ForceRemove()
		{
			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Force Remove Data Center";
			model.HashName = "force_remove_data_center";
			model.Message = "Data Center(s)";
			model.Latch.IsAvailable = true;
			model.Latch.IsChangable = true;

			List<string> list = new List<string>();
			foreach (storage_pool a in Linq.Cast<storage_pool>(SelectedItems))
			{
				list.Add(a.name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnForceRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void RecoveryStorage()
		{
			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Data Center Re-Initialize";
			model.HashName = "data_center_re-initialize";
			model.Latch.IsAvailable = true;
			model.Latch.IsChangable = true;

			//IEnumerable<storage_domains> list = DataProvider.GetStorageDomainList();
			//List<EntityModel> models = list
			//    .Where(a => (a.storage_domain_type == StorageDomainType.Data && a.storage_type == ((storage_pool)SelectedItem).storage_pool_type) &&
			//         (a.storage_domain_shared_status == StorageDomainSharedStatus.Unattached)
			//        )
			//    .Select(a => new EntityModel() { Entity = a })
			//    .ToList();
			List<EntityModel> models = new List<EntityModel>();
			foreach (storage_domains a in DataProvider.GetStorageDomainList())
			{
				if (a.storage_domain_type == StorageDomainType.Data && a.storage_type == ((storage_pool)SelectedItem).storage_pool_type &&
					 (a.storage_domain_shared_status == StorageDomainSharedStatus.Unattached))
					models.Add(new EntityModel() { Entity = a });
			}

			model.Items = models;

			if (models.Count > 0)
			{
				EntityModel entityModel = models.Count != 0 ? models[0] : null;
				if (entityModel != null)
				{
					entityModel.IsSelected = true;
				}
			}


			if (models.Count == 0)
			{
				model.Message = "There are no compatible Storage Domains to attach to this Data Center. Please add new Storage from the Storage tab.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnRecover", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		public void OnRecover()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (!model.Validate())
			{
				return;
			}

			//storage_domains master = DataProvider.GetStorageDomainList(((storage_pool)SelectedItem).id).FirstOrDefault(a=>a.storage_domain_type == StorageDomainType.Master);
			storage_domains master = null;
			foreach (storage_domains a in DataProvider.GetStorageDomainList(((storage_pool)SelectedItem).Id))
				if (a.storage_domain_type == StorageDomainType.Master)
				{
					master = a;
					break;
				}


			List<storage_domains> items = new List<storage_domains>();
			foreach (EntityModel a in Linq.Cast<EntityModel>(model.Items))
			{
				if (a.IsSelected)
				{
					items.Add((storage_domains)a.Entity);
				}
			}

			if (items.Count > 0)
			{
				if (model.Progress != null)
				{
					return;
				}

				List<VdcActionParametersBase> parameters = new List<VdcActionParametersBase>();
				foreach (storage_domains a in items)
				{
					parameters.Add(new RecoveryStoragePoolParameters(((storage_pool)SelectedItem).Id, a.id));
				}


				model.StartProgress(null);

				Frontend.RunMultipleAction(VdcActionType.RecoveryStoragePool, parameters,
					result =>
					{
						ConfirmationModel localModel = (ConfirmationModel)result.State;

						localModel.StopProgress();
						Cancel();
					},
					model
				);
			}
			else
			{
				Cancel();
			}
		}

		public void Activate()
		{
			//Frontend.RunMultipleActions(VdcActionType.ActivateStoragePool,
			//    SelectedItems.Cast<storage_pool>()
			//    .Select(a => (VdcActionParametersBase)new StoragePoolParametersBase(a.id))
			//    .ToList()
			//);
		}

		public void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> parameters = new List<VdcActionParametersBase>();
			foreach (storage_pool a in Linq.Cast<storage_pool>(SelectedItems))
			{
				parameters.Add(new StoragePoolParametersBase(a.Id));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveStoragePool, parameters,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		public void OnForceRemove()
		{
			ConfirmationModel model = (ConfirmationModel) Window;
			if(!model.Validate())
			{
				return;
			}
			VdcActionParametersBase parametersBase = new VdcActionParametersBase();
			parametersBase = new StoragePoolParametersBase(((storage_pool)SelectedItem).Id) { ForceDelete = true };
			Frontend.RunAction(VdcActionType.RemoveStoragePool, parametersBase);
			Cancel();
		}

		public void Cancel()
		{
			CancelConfirmation();

			GuideContext = null;
			Window = null;

			UpdateActionAvailability();
		}

		public void CancelConfirmation()
		{
			ConfirmWindow = null;
		}

		public void OnSave()
		{
			DataCenterModel model = (DataCenterModel)Window;

			if (!model.Validate())
			{
				return;
			}

			if (!model.IsNew && (Version)model.Version.SelectedItem != ((storage_pool)SelectedItem).compatibility_version)
			{
				ConfirmationModel confirmModel = new ConfirmationModel();
				ConfirmWindow = confirmModel;
				confirmModel.Title = "Change Data Center Compatibility Version";
				confirmModel.HashName = "change_data_center_compatibility_version";
				confirmModel.Message = "You are about to change the Data Center Compatibility Version. Are you sure you want to continue?";

				confirmModel.Commands.Add(
					new UICommand("OnSaveInternal", this)
					{
						Title = "OK",
						IsDefault = true
					});
				confirmModel.Commands.Add(
					new UICommand("CancelConfirmation", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
			else
			{
				OnSaveInternal();
			}
		}

		public void OnSaveInternal()
		{
			DataCenterModel model = (DataCenterModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			storage_pool dataCenter = model.IsNew ? new storage_pool() : (storage_pool)Cloner.Clone(SelectedItem);

			//cancel confirm window if there is
			CancelConfirmation();

			//Save changes.
			dataCenter.name = (string)model.Name.Entity;
			dataCenter.description = (string)model.Description.Entity;
			dataCenter.storage_pool_type = (StorageType)model.StorageTypeList.SelectedItem;
			dataCenter.compatibility_version = (Version)model.Version.SelectedItem;


			model.StartProgress(null);

			Frontend.RunAction(model.IsNew ? VdcActionType.AddEmptyStoragePool : VdcActionType.UpdateStoragePool,
				new StoragePoolManagementParameter(dataCenter),
				result =>
				{
					DataCenterListModel localModel = (DataCenterListModel)result.State;

					localModel.PostOnSaveInternal(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnSaveInternal(VdcReturnValueBase returnValue)
		{
			DataCenterModel model = (DataCenterModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();

				if (model.IsNew)
				{
					GuideContext = returnValue.ActionReturnValue;
					UpdateActionAvailability();
					GuideCommand.Execute();
				}
			}
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void ItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
		{
			base.ItemsCollectionChanged(sender, e);

			//Try to select an item corresponding to the system tree selection.
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.DataCenter)
			{
				storage_pool dataCenter = (storage_pool)SystemTreeSelectedItem.Entity;

				SelectedItem = Linq.FirstOrDefault(Linq.Cast<storage_pool>(Items), new Linq.DataCenterPredicate(dataCenter.Id));
			}
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			List<storage_pool> items = SelectedItems != null
				? new List<storage_pool>(Linq.Cast<storage_pool>(SelectedItems))
				: new List<storage_pool>();

			bool isAllDown = true;
			foreach (storage_pool item in items)
			{
				if (item.status == StoragePoolStatus.Up || item.status == StoragePoolStatus.Contend)
				{
					isAllDown = false;
					break;
				}
			}

			EditCommand.IsExecutionAllowed = SelectedItem != null && items.Count == 1;
			RemoveCommand.IsExecutionAllowed = items.Count > 0 && isAllDown;

			storage_pool storagePoolItem = (SelectedItem != null ? (storage_pool)SelectedItem : null);

			ForceRemoveCommand.IsExecutionAllowed = storagePoolItem != null && items.Count == 1
											&& storagePoolItem.status != StoragePoolStatus.Up
											&& storagePoolItem.status != StoragePoolStatus.Contend
											&& storagePoolItem.status != StoragePoolStatus.Uninitialized;
			
			GuideCommand.IsExecutionAllowed = GuideContext != null
				|| (SelectedItem != null
				&& SelectedItems != null
				&& SelectedItems.Count == 1);

			ActivateCommand.IsExecutionAllowed = items.Count > 0;
			if (ActivateCommand.IsExecutionAllowed)
				foreach (storage_pool a in items)
				{
					if (a.status == StoragePoolStatus.Up || a.status == StoragePoolStatus.Uninitialized)
					{
						ActivateCommand.IsExecutionAllowed = false;
						break;
					}
				}


			RecoveryStorageCommand.IsExecutionAllowed = items != null && items.Count == 1;


			//System tree dependent actions.
			bool isAvailable = !(SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.DataCenter);

			NewCommand.IsAvailable = isAvailable;
			RemoveCommand.IsAvailable = isAvailable;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == ForceRemoveCommand)
			{
				ForceRemove();
			}
			else if (command == ActivateCommand)
			{
				Activate();
			}
			else if (command == GuideCommand)
			{
				Guide();
			}
			else if (command == RecoveryStorageCommand)
			{
				RecoveryStorage();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnForceRemove")
			{
				OnForceRemove();
			}
			else if (command.Name == "OnSaveInternal")
			{
				OnSaveInternal();
			}
			else if (command.Name == "CancelConfirmation")
			{
				CancelConfirmation();
			}
			else if (command.Name == "OnRecover")
			{
				OnRecover();
			}
		}


		private SystemTreeItemModel systemTreeSelectedItem;
		public SystemTreeItemModel SystemTreeSelectedItem
		{
			get { return systemTreeSelectedItem; }
			set
			{
				if (systemTreeSelectedItem != value)
				{
					systemTreeSelectedItem = value;
					OnSystemTreeSelectedItemChanged();
				}
			}
		}

		private void OnSystemTreeSelectedItemChanged()
		{
			UpdateActionAvailability();
		}
	}
}
