using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using System.ComponentModel;
using System.Threading;
using org.ovirt.engine.ui.uicommon.models.common;
using org.ovirt.engine.ui.uicompat;
using System.IO;

namespace org.ovirt.engine.ui.uicommon.models
{
	public class Model : PropertyChangeNotifier, IEventListener, ICommandTarget, IProvidePropertyChangedEvent
	{
		#region Events

		public Event PropertyChangedEvent { get; private set; }

		#endregion

		#region Properties

		protected Configurator Configurator { get; private set; }
		protected ILogger Logger { get; private set; }

		public UICommand LastExecutedCommand { get; private set; }


		private ProgressModel progress;
		/// <summary>
		/// Represents a progress operation on the model.
		/// </summary>
		public ProgressModel Progress
		{
			get { return progress; }
			private set
			{
				if (progress != value)
				{
					progress = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Progress"));
				}
			}
		}

		#region Multi-purpose
		
		private string hashName { get; set; }
		public string HashName 
		{ 
			get
			{
				return hashName;
			} 
			set 
			{ 
				hashName = value;
				OpenDocumentationCommand = new UICommand("OpenDocumentation", this);
				OnPropertyChanged(new PropertyChangedEventArgs("HashName"));
			}
		}

		/// <summary>
		/// Gets or sets the title of this model. There is no specific
		/// purpose for this property, use it on your convinience.
		/// </summary>
		private string title;
		public string Title
		{
			get { return title; }
			set
			{
				if (title != value)
				{
					title = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Title"));
				}
			}
		}

		private bool isValid;
		public bool IsValid
		{
			get { return isValid; }
			set
			{
				if (isValid != value)
				{
					isValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsValid"));

					if (isValid)
					{
						InvalidityReasons.Clear();
					}
				}
			}
		}

		public IList<string> InvalidityReasons { get; set; }

		private bool isAvailable;
		public bool IsAvailable
		{
			get { return isAvailable; }
			set
			{
				if (isAvailable != value)
				{
					isAvailable = value;
					OnIsAvailableChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("IsAvailable"));
				}
			}
		}

		private bool isChangable;
		public bool IsChangable
		{
			get { return isChangable; }
			set
			{
				if (isChangable != value)
				{
					isChangable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsChangable"));

					if (isChangable)
					{
						ChangeProhibitionReasons.Clear();
					}
				}
			}
		}

		public IList<string> ChangeProhibitionReasons { get; private set; }

		private bool isSelected;
		public bool IsSelected
		{
			get { return isSelected; }
			set
			{
				if (isSelected != value)
				{
					isSelected = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsSelected"));
				}
			}
		}

		private bool isSelectable;
		public bool IsSelectable
		{
			get { return isSelectable; }
			set
			{
				if (isSelectable != value)
				{
					isSelectable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsSelectable"));
				}
			}
		}

		public IList<UICommand> Commands { get; set; }

		public UICommand OpenDocumentationCommand { get; set; }

		private string message;
		public string Message
		{
			get { return message; }
			set
			{
				if (message != value)
				{
					message = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Message"));
				}
			}
		}

		private string info { get; set; }
		public string Info
		{
			get
			{
				return info;
			}
			set
			{
				info = value;
				OnPropertyChanged(new PropertyChangedEventArgs("Info"));
			}
		}

		#endregion

		#endregion

		public Model()
		{
			PropertyChangedEvent = new Event(ProvidePropertyChangedEvent.Definition);

			//Resolve commonly used types.
			Configurator = (Configurator)TypeResolver.Instance.Resolve(typeof(Configurator));
			Logger = (ILogger)TypeResolver.Instance.Resolve(typeof(ILogger));

			InvalidityReasons = new ObservableCollection<string>();
			IsValid = true;

			ChangeProhibitionReasons = new ObservableCollection<string>();
			IsChangable = true;
			IsAvailable = true;

			IsSelectable = true;

			Commands = new ObservableCollection<UICommand>();
		}

		/// <summary>
		/// Override this method to initialize model, for example populate
		/// some properties with data here rather than in constructor.
		/// But instantiation still should be done in constructor.
		/// </summary>
		public virtual void Initialize()
		{
		}

		protected virtual void OnIsAvailableChanged()
		{
		}

		protected override void OnPropertyChanged(PropertyChangedEventArgs e)
		{
			base.OnPropertyChanged(e);
			PropertyChangedEvent.raise(this, e);
		}

		public virtual void eventRaised(Event ev, object sender, EventArgs args)
		{
		}

		public virtual void ExecuteCommand(UICommand command)
		{
			LastExecutedCommand = command;
			if(command == OpenDocumentationCommand)
			{
				OnPropertyChanged(new PropertyChangedEventArgs("OpenDocumentation"));
			}
		}

		/// <summary>
		/// Assigns a new instance to the Progress property, indicating
		/// start of some operation on the model.
		/// </summary>
		/// <param name="operation"></param>
		public void StartProgress(string operation)
		{
			Progress =
				new ProgressModel
				{
					CurrentOperation = operation
				};
		}

		/// <summary>
		/// Assigns null to the Progress property, indicating
		/// end of some operation on the model.
		/// </summary>
		public void StopProgress()
		{
			Progress = null;
		}
	}
}
