﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using org.ovirt.engine.ui.uicommon.validation;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using VdcCommon.Interfaces;
using VdcFrontend;

namespace org.ovirt.engine.ui.uicommon.models.storage
{
	public class StorageModel : ListModel, ISupportSystemTreeContext
	{
		public static readonly Guid UnassignedDataCenterId = Guid.Empty;
		private readonly StorageModelBehavior behavior;

		#region Properties

		public new IStorageModel SelectedItem
		{
			get { return (IStorageModel)base.SelectedItem; }
			set { base.SelectedItem = value; }
		}

		/// <summary>
		/// Gets or sets the storage being edited. Null if it's a new one.
		/// </summary>
		public storage_domains Storage { get; set; }

		public string OriginalName { get; set; }

		public EntityModel Name { get; private set; }
		public ListModel DataCenter { get; private set; }
		public ListModel Host { get; private set; }
		public ListModel Format { get; private set; }

		#endregion


		public StorageModel(StorageModelBehavior behavior)
		{
			this.behavior = behavior;
			this.behavior.Model = this;

			Name = new EntityModel();
			DataCenter = new ListModel();
			DataCenter.SelectedItemChangedEvent.addListener(this);
			Host = new ListModel();
			Host.SelectedItemChangedEvent.addListener(this);
			Format = new ListModel();
		}

		public override void Initialize()
		{
			base.Initialize();

			InitDataCenter();
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(SelectedItemChangedEventDefinition))
			{
				if (sender == DataCenter)
				{
					DataCenter_SelectedItemChanged();
				}
				else if (sender == Host)
				{
					Host_SelectedItemChanged();
				}
			}
			else if (ev.Equals(NfsStorageModel.PathChangedEventDefinition))
			{
				NfsStorageModel_PathChanged(sender, args);
			}
		}

		private void NfsStorageModel_PathChanged(object sender, EventArgs args)
		{
			NfsStorageModel senderModel = (NfsStorageModel)sender;

			foreach (object item in Items)
			{
				if (item is NfsStorageModel && item != sender)
				{
					NfsStorageModel model = (NfsStorageModel)item;
					model.Path.Entity = senderModel.Path.Entity;
				}
			}
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();

			if (SelectedItem != null)
			{
				UpdateFormat();
				UpdateHost();
				SelectedItem.UpdateCommand.Execute();
			}
		}

		protected override void ItemsChanged()
		{
			base.ItemsChanged();

			if (Items != null)
			{
				foreach (object item in Items)
				{
					IStorageModel model = (IStorageModel)item;
					model.Container = this;

					if (item is NfsStorageModel)
					{
						NfsStorageModel nfsModel = (NfsStorageModel)item;
						nfsModel.PathChangedEvent.addListener(this);
					}
				}
			}
		}

		private void DataCenter_SelectedItemChanged()
		{
			UpdateItemsAvailability();
			UpdateFormat();
			UpdateHost();
		}

		private void Host_SelectedItemChanged()
		{
			VDS host = (VDS)Host.SelectedItem;
			if (SelectedItem != null)
			{
				//When changing host clear items for san storage model.
				if (SelectedItem is SanStorageModelBase && Storage == null)
				{
					SanStorageModelBase sanStorageModel = (SanStorageModelBase)SelectedItem;
					sanStorageModel.Items = null;
				}

				if (host != null)
				{
					SelectedItem.UpdateCommand.Execute();

					//Update path prefix for local storage.
					string prefix = host.vds_type == VDSType.oVirtNode
					                	? DataProvider.GetLocalFSPath()
					                	: String.Empty;

					foreach (object item in Items)
					{
						if (item is LocalStorageModel)
						{
							LocalStorageModel model = (LocalStorageModel)item;
							model.Path.Entity = prefix;
							model.Path.IsChangable = String.IsNullOrEmpty(prefix);
						}
					}
				}
			}
		}

		private void InitDataCenter()
		{
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type != SystemTreeItemType.System)
			{
				switch (SystemTreeSelectedItem.Type)
				{
					case SystemTreeItemType.DataCenter:
					case SystemTreeItemType.Cluster:
					case SystemTreeItemType.Storages:
					case SystemTreeItemType.Storage:
						{
							SystemTreeItemModel dataCenterItem = SystemTreeItemModel.FindAncestor(SystemTreeItemType.DataCenter, SystemTreeSelectedItem);
							storage_pool dc = (storage_pool)dataCenterItem.Entity;

							DataCenter.Items = new List<storage_pool> { dc };
							DataCenter.SelectedItem = dc;
							DataCenter.IsChangable = false;
							DataCenter.Info = "Cannot choose Storage's Data Center in tree context";
						}
						break;

					case SystemTreeItemType.Host:
						{
							VDS host = (VDS)SystemTreeSelectedItem.Entity;

							Host.IsChangable = false;
							Host.Info = "Cannot choose Storage's Host in tree context";
							Host.SelectedItem = host;

							SystemTreeItemModel dataCenterItem = SystemTreeItemModel.FindAncestor(SystemTreeItemType.DataCenter, SystemTreeSelectedItem);
							storage_pool dc = (storage_pool)dataCenterItem.Entity;

							DataCenter.Items = new List<storage_pool> { dc };
							DataCenter.SelectedItem = dc;
							DataCenter.IsChangable = false;
							DataCenter.Info = "Cannot choose Storage's Data Center in tree context";
						}
						break;
				}
			}
			else
			{
				IList<storage_pool> dataCenters = new List<storage_pool>();

				if (Storage == null || Storage.storage_domain_shared_status == StorageDomainSharedStatus.Unattached)
				// We are either adding a new storage or editing an unattached storage
				// -> fill DataCenters drop-down with all possible Data-Centers, choose the empty one:
				// [TODO: In case of an Unattached SD, choose only DCs of the same type]
				{
					dataCenters = behavior.FilterDataCenter(DataProvider.GetDataCenterList());
					AddEmptyDataCenterToList(dataCenters);

					storage_pool oldSelectedItem = (storage_pool)DataCenter.SelectedItem;
					DataCenter.Items = dataCenters;

					if (oldSelectedItem != null)
					{
						DataCenter.SelectedItem = Linq.FirstOrDefault(dataCenters, new Linq.DataCenterPredicate(oldSelectedItem.Id));
					}
					else
					{
						DataCenter.SelectedItem = Storage == null
							? Linq.FirstOrDefault(dataCenters)	// New -> select first DC (should be non-empty)
							: Linq.FirstOrDefault(dataCenters, new Linq.DataCenterPredicate(UnassignedDataCenterId));	// Edit -> select empty DC
					}
				}

				else // "Edit Storage" mode:
				{
					IList<storage_pool> dataCentersWithStorage = behavior.FilterDataCenter(DataProvider.GetDataCentersByStorageDomain(Storage.id));
					if (dataCentersWithStorage.Count < 1 || dataCentersWithStorage[0] == null)
					{
						// [*** shouldn't happen - this is just to prevent unexpected crashes/behavior ***]
						// SD isn't attached to any DC -> just add Unassigned item:
						AddEmptyDataCenterToList(dataCenters);
					}
					else
					// storage is in (at least) one (real) DC 
					//-> set this DC as the only item in the list:
					{
						dataCenters = new List<storage_pool> { dataCentersWithStorage[0] };
					}

					DataCenter.Items = dataCenters;
					DataCenter.SelectedItem = Linq.FirstOrDefault(dataCenters);
				}
			}
		}

		private static void AddEmptyDataCenterToList(IList<storage_pool> dataCenters)
		{
			dataCenters.Add(
				new storage_pool
				{
					Id = UnassignedDataCenterId,
					name = "(none)"
				});
		}

		private void UpdateHost()
		{
			if (DataCenter.Items == null)
			{
				return;
			}


			storage_pool dataCenter = (storage_pool)DataCenter.SelectedItem;
			IEnumerable<VDS> hosts = new List<VDS>();

			if (SelectedItem is LocalStorageModel && (dataCenter == null || dataCenter.Id.Equals(UnassignedDataCenterId)))
			{
				List<storage_pool> dataCenterList = (List<storage_pool>)DataCenter.Items;
				List<storage_pool> localDCList = new List<storage_pool>();
				string dataCenterQueryLine = "";

				foreach (storage_pool storagePool in dataCenterList)
				{
					if (storagePool.storage_pool_type == StorageType.LOCALFS)
					{
						localDCList.Add(storagePool);
					}
				}

				if (localDCList.Count > 0)
				{
					int i = 0;
					for (; i < localDCList.Count - 1; i++)
					{
						dataCenterQueryLine += "datacenter=" + localDCList[i].name + " or ";
					}
					dataCenterQueryLine += "datacenter=" + localDCList[i].name;

					VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
																		new SearchParameters(
																			"Hosts: status=Up " + dataCenterQueryLine, SearchType.VDS));
					if (returnValue != null && returnValue.Succeeded)
					{
						hosts = (List<VDS>)returnValue.ReturnValue;
					}
				}
			}
			else
			{
				hosts = dataCenter == null || dataCenter.Id.Equals(UnassignedDataCenterId)
					? DataProvider.GetHostList()
					: DataProvider.GetHostListByDataCenter(dataCenter.name);

				hosts = Linq.Where(hosts, new Linq.HostStatusPredicate(VDSStatus.Up));
			}

			//Allow only hosts with version above 2.2 for export storage.
			List<VDS> list = new List<VDS>();
			if (SelectedItem != null && SelectedItem.Role == StorageDomainType.ImportExport)
			{
				foreach (VDS host in hosts)
				{
					if (host.vds_group_compatibility_version.compareTo(new Version("2.2")) >= 0)
					{
						list.Add(host);
					}
				}
				hosts = list;
			}


			VDS oldSelectedItem = (VDS)Host.SelectedItem;
			Host.Items = hosts;

			//Try to select previously selected host.
			if (oldSelectedItem != null)
			{
				Host.SelectedItem = Linq.FirstOrDefault(hosts, new Linq.HostPredicate(oldSelectedItem.vds_id));
			}

			//Try to select an SPM host when edit storage.
			if (Host.SelectedItem == null && Storage != null)
			{
				foreach (VDS host in hosts)
				{
					if (host.spm_status == VdsSpmStatus.SPM)
					{
						Host.SelectedItem = host;
						break;
					}
				}
			}

			//Select a default - first host in the list.
			if (Host.SelectedItem == null)
			{
				Host.SelectedItem = Linq.FirstOrDefault(hosts);
			}
		}

		private void UpdateFormat()
		{
			storage_pool dataCenter = (storage_pool)DataCenter.SelectedItem;

			StorageFormatType selectItem = StorageFormatType.V1;

			List<StorageFormatType> formats = new List<StorageFormatType>();

			if (dataCenter != null && SelectedItem != null)
			{
				if (!dataCenter.Id.Equals(UnassignedDataCenterId))
				{
					Format.IsChangable = false;

					// If data center has format defined and the selected-item role is Data, choose it.
					if (dataCenter.StoragePoolFormatType != null && SelectedItem.Role == StorageDomainType.Data)
					{
						formats.Add(dataCenter.StoragePoolFormatType.Value);
						selectItem = dataCenter.StoragePoolFormatType.Value;
					}
					// If selected-item role is ISO or Export, add only the 'V1' option.
					// (*** Note that currently both ISO and Export can be only NFS, so theoretically they are covered by 
					// the next "else if..." condition; however, just in case we will support non-NFS ISO/Export in the future
					// and in order to make the code more explicit, it is here. ***)
					else if (SelectedItem.Role == StorageDomainType.ISO || SelectedItem.Role == StorageDomainType.ImportExport)
					{
						formats.Add(StorageFormatType.V1);
					}
					else if (SelectedItem.Type == StorageType.NFS || SelectedItem.Type == StorageType.LOCALFS)
					{
						formats.Add(StorageFormatType.V1);
					}
					else if ((SelectedItem.Type == StorageType.ISCSI || SelectedItem.Type == StorageType.FCP)
							 && dataCenter.compatibility_version.compareTo(new Version("3.0")) < 0)
					{
						formats.Add(StorageFormatType.V1);
					}
					else if ((SelectedItem.Type == StorageType.ISCSI || SelectedItem.Type == StorageType.FCP)
							 && dataCenter.compatibility_version.compareTo(new Version("3.0")) >= 0)
					{
						formats.Add(StorageFormatType.V2);
						selectItem = StorageFormatType.V2;
					}
				}
				else // Unassigned DC:
				{
					Format.IsChangable = true;

					formats.Add(StorageFormatType.V1);

					if ((SelectedItem.Type == StorageType.FCP || SelectedItem.Type == StorageType.ISCSI)
						&& SelectedItem.Role == StorageDomainType.Data)
					{
						formats.Add(StorageFormatType.V2);
						selectItem = StorageFormatType.V2;
					}
				}
			}

			Format.Items = formats;
			Format.SelectedItem = selectItem;
		}

		private void UpdateItemsAvailability()
		{
			if (Items == null)
			{
				return;
			}

			behavior.UpdateItemsAvailability();

			bool chooseFirst = false;
			if (SelectedItem != null)
			{
				Model selectedModel = (Model)SelectedItem;
				if (!selectedModel.IsSelectable)
				{
					chooseFirst = true;
				}
			}
			else
			{
				chooseFirst = true;
			}

			if (chooseFirst)
			{
				// Choose first allowed type (it will be data role in case of
				// New Domain and ISO role in case of Import Domain).
				foreach (IStorageModel item in Linq.Cast<IStorageModel>(Items))
				{
					Model model = (Model)item;
					if (model.IsSelectable)
					{
						SelectedItem = item;
						break;
					}
				}
			}
		}

		public bool Validate()
		{
			int nameMaxLength = DataProvider.GetStorageDomainMaxNameLength();

			Name.ValidateEntity(
				new IValidation[]
				{
					new NotEmptyValidation(),
					new RegexValidation
					{
						Expression = @"^[A-Za-z0-9_-]{1," + nameMaxLength + "}$",
						Message = "Name can contain only 'A-Z', 'a-z', '0-9', '_' or '-' characters, max length: " + nameMaxLength
					}
				});

			string name = (string)Name.Entity;

			if (String.Compare(name, OriginalName, true) != 0 && !DataProvider.IsStorageDomainNameUnique(name))
			{
				Name.IsValid = false;
				Name.InvalidityReasons.Add("Name must be unique.");
			}


			Host.ValidateSelectedItem(new[] { new NotEmptyValidation() });
			ValidateSelectedItem(new[] { new NotEmptyValidation() });


			return Name.IsValid
				   && Host.IsValid
				   && IsValid
				   && SelectedItem.Validate();
		}


		public SystemTreeItemModel SystemTreeSelectedItem { get; set; }
	}





	public abstract class StorageModelBehavior
	{
		public StorageModel Model { get; set; }

		public virtual IList<storage_pool> FilterDataCenter(IList<storage_pool> source)
		{
			return source;
		}

		public virtual void UpdateItemsAvailability()
		{
		}
	}


	public class NewEditStorageModelBehavior : StorageModelBehavior
	{
		public override void UpdateItemsAvailability()
		{
			base.UpdateItemsAvailability();

			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			//Allow Data storage type corresponding to the selected data-center type + ISO and Export that are NFS only:
			foreach (IStorageModel item in Linq.Cast<IStorageModel>(Model.Items))
			{
				Model model = (Model)item;

				model.IsSelectable = dataCenter != null &&
					((dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId) && item.Role == StorageDomainType.Data) ||
					(!dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId) &&
						((item.Role == StorageDomainType.Data && item.Type == dataCenter.storage_pool_type) ||
						(item.Role == StorageDomainType.ImportExport
							&& item.Type == StorageType.NFS
							&& dataCenter.status != StoragePoolStatus.Uninitialized
							&& DataProvider.GetExportDomainByDataCenterId(dataCenter.Id) == null) ||
						item.Role == StorageDomainType.ISO
							&& item.Type == StorageType.NFS
							&& dataCenter.status != StoragePoolStatus.Uninitialized
							&& DataProvider.GetIsoDomainByDataCenterId(dataCenter.Id) == null)) ||
					(Model.Storage != null && item.Type == Model.Storage.storage_type));
			}
		}
	}


	public class ImportStorageModelBehavior : StorageModelBehavior
	{
		public override IList<storage_pool> FilterDataCenter(IList<storage_pool> source)
		{
			return Linq.ToList(Linq.Where(source, new Linq.DataCenterStatusPredicate(StoragePoolStatus.Up)));
		}

		public override void UpdateItemsAvailability()
		{
			base.UpdateItemsAvailability();

			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			foreach (IStorageModel item in Linq.Cast<IStorageModel>(Model.Items))
			{
				Model model = (Model)item;

				storage_domains isoStorage = DataProvider.GetIsoDomainByDataCenterId(dataCenter.Id);
				storage_domains exportStorage = DataProvider.GetExportDomainByDataCenterId(dataCenter.Id);

				// available type/function items are:
				// all in case of Unassigned DC.
				// ISO in case the specified DC doesn't have an attached ISO domain.
				// Export in case the specified DC doesn't have an attached export domain.
				model.IsSelectable =
					(dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId) ||
					(item.Role == StorageDomainType.ISO && isoStorage == null) ||
					(item.Role == StorageDomainType.ImportExport && exportStorage == null));
			}
		}
	}
}
