using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VdcCommon.BusinessEntities;
using System.ComponentModel;
using VdcFrontend;
using System.Collections.ObjectModel;
using org.ovirt.engine.ui.uicommon.models.users;
using VdcCommon.Users;
using VdcCommon.Interfaces;
using System.Collections;

namespace org.ovirt.engine.ui.uicommon.models.configure
{
	public class PermissionListModel : SearchableListModel
	{
		#region Commands

		public UICommand AddCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		public new object Entity
		{
			get { return base.Entity; }
			set { base.Entity = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion


		public PermissionListModel()
		{
			Title = "Permissions";

			AddCommand = new UICommand("New", this);
			RemoveCommand = new UICommand("Remove", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			SearchCommand.Execute();
			UpdateActionAvailability();
		}

		public override void Search()
		{
			if (Entity != null)
			{
				base.Search();
			}
		}

		protected override void SyncSearch()
		{
			VdcObjectType objType = getObjectType();
			bool directOnly = (objType == VdcObjectType.VM ? true : false);
			base.SyncSearch(VdcQueryType.GetPermissionsForObject,
				new GetPermissionsForObjectParameters
				{
					ObjectId = getEntityGuid(),
					VdcObjectType = objType,
					DirectOnly = directOnly
				});
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();


			VdcObjectType objType = getObjectType();
			bool directOnly = (objType == VdcObjectType.VM ? true : false);

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetPermissionsForObject,
				new GetPermissionsForObjectParameters
				{
					ObjectId = getEntityGuid(),
					VdcObjectType = objType,
					DirectOnly = directOnly
				});

			Items = AsyncResult.Data;
		}

		private void Add()
		{
			if (Window != null)
			{
				return;
			}

			AdElementListModel model = new AdElementListModel();
			Window = model;
			model.Title = "Add Permission to User";
			model.HashName = "add_permission_to_user";

			model.Commands.Add(
				new UICommand("OnAdd", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Permission";
			model.HashName = "remove_permission";
			model.Message = "Permission";
			List<string> items = new List<string>();
			foreach (object a in SelectedItems)
			{
				items.Add("Role " + ((permissions)a).RoleName + " on User " + ((permissions)a).OwnerName);
			}
			model.Items = items;

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			if (SelectedItems != null && SelectedItems.Count > 0)
			{
				ConfirmationModel model = (ConfirmationModel)Window;

				if (model.Progress != null)
				{
					return;
				}

				List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
				foreach (object perm in SelectedItems)
				{
					list.Add(new PermissionsOperationsParametes { Permission = (permissions)perm });
				}


				model.StartProgress(null);

				Frontend.RunMultipleAction(VdcActionType.RemovePermission, list,
					result =>
					{
						ConfirmationModel localModel = (ConfirmationModel)result.State;

						localModel.StopProgress();
						Cancel();
					},
					model
				);
			}

			Cancel();
		}

		private void OnAdd()
		{
			AdElementListModel model = (AdElementListModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.IsEveryoneSelected && model.SelectedItems == null)
			{
				Cancel();
				return;
			}


			List<DbUser> items = new List<DbUser>();
			if (model.IsEveryoneSelected)
			{
				items.Add(new DbUser { user_id = UserListModel.EveryoneUserId });
			}
			else
			{
				foreach (object item in model.Items)
				{
					EntityModel entityModel = (EntityModel) item;
					if (entityModel.IsSelected)
					{
						items.Add((DbUser) entityModel.Entity);
					}
				}
			}

			roles role = (roles)model.Role.SelectedItem;
			//adGroup/user


			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (DbUser user in items)
			{
				permissions perm = new permissions { ad_element_id = user.user_id, role_id = role.Id };
				perm.ObjectId = getEntityGuid();
				perm.ObjectType = this.getObjectType();

				if (user.IsGroup)
				{
					list.Add(new PermissionsOperationsParametes { Permission = perm, AdGroup = new ad_groups(user.user_id, user.name, user.domain) });
				}
				else
				{
					list.Add(new PermissionsOperationsParametes { Permission = perm, VdcUser = new VdcUser(user.user_id, user.username, user.domain) });
				}
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.AddPermission, list,
				result =>
				{
					AdElementListModel localModel = (AdElementListModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			RemoveCommand.IsExecutionAllowed = (SelectedItems != null && SelectedItems.Count > 0);
			if (RemoveCommand.IsExecutionAllowed == false)
			{
				return;
			}
			Guid entityGuid = getEntityGuid();
			foreach (object p in SelectedItems)
			{
				if (((permissions)p).ObjectId != entityGuid)
				{
					RemoveCommand.IsExecutionAllowed = false;
					return;
				}
			}
		}

		private Guid getEntityGuid()
		{
			return DataProvider.GetEntityGuid(Entity);
		}

		private VdcObjectType getObjectType()
		{
			if (Entity is VM)
			{
				return VdcObjectType.VM;
			}
			if (Entity is storage_pool)
			{
				return VdcObjectType.StoragePool;
			}
			if (Entity is VDSGroup)
			{
				return VdcObjectType.VdsGroups;
			}
			if (Entity is VDS)
			{
				return VdcObjectType.VDS;
			}
			if (Entity is storage_domains)
			{
				return VdcObjectType.Storage;
			}
			if (Entity is VmTemplate)
			{
				return VdcObjectType.VmTemplate;
			}
			if (Entity is vm_pools)
			{
				return VdcObjectType.VmPool;
			}
			return VdcObjectType.Unknown;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == AddCommand)
			{
				Add();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnAdd")
			{
				OnAdd();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
