using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
namespace SearchBackend
{
    


    /// <summary>
    /// A base class for all condition field auto completers
    /// </summary>
    public class BaseConditionFieldAutoCompleter : BaseAutoCompleter, IConditionFieldAutoCompleter
    {
        protected Dictionary<string, List<valueValidationFunction>> mValidationDict = null;
        private Dictionary<string, Type>                          mTypeDict       = null;
        protected Dictionary<string, string>                        mColumnNameDict = null;
        protected List<string> mNotFreeTextSearchableFieldsList = new List<string>();


        public BaseConditionFieldAutoCompleter()
        {
            mValidationDict = new Dictionary<string,List<valueValidationFunction>>();
            mTypeDict = new Dictionary<string, Type>();
            mColumnNameDict = new Dictionary<string, string>();
        }

        //public static ITagsHandler TagsHandler;

        public Dictionary<string, Type> TypeDictionary
        {
            get { return mTypeDict; }
        }

        protected void buildBasicValidationTable()
        {
            valueValidationFunction charValidation = validCahracters;
            valueValidationFunction intValidationFunc = validInteger;
            valueValidationFunction decimalValidationFunction = validDecimal;
            valueValidationFunction dateValidationFunc = validDateTime;
            valueValidationFunction timeSpanValidationFunc = validTimeSpan;
            valueValidationFunction valueValidationFunc = validateFieldValueByValueAC;
            valueValidationFunction dateEnumValidationFunc = validateDateEnumValueByValueAC;

            foreach (string key in mVerbs.Keys)
            {
                List<valueValidationFunction> curList = new List<valueValidationFunction>();
                Type curType  = mTypeDict[key];
                if (curType == typeof(decimal))
                {
                    curList.Add(decimalValidationFunction);
                }
                else if (curType == typeof(int))
                {
                    curList.Add(intValidationFunc);
                }
                else if (curType == typeof(DateTime))
                {
                    curList.Add(dateValidationFunc);
                }
                else if (curType == typeof(TimeSpan))
                {
                    curList.Add(timeSpanValidationFunc);
                }
                else
                {
                    curList.Add(charValidation);
                }
                IConditionValueAutoCompleter tmp = getFieldValueAutoCompleter(key);
                if (tmp != null)
                {
                    if (tmp.GetType() == typeof(DateEnumValueAutoCompleter))
                    {
                        curList.Add(dateEnumValidationFunc);
                    }
                    else
                    {
                        curList.Add(valueValidationFunc);
                    }
                }
                mValidationDict.Add(key, curList);
            }
        }


        public bool validateFieldValue(string fieldName, string fieldValue)
        {
            if (mValidationDict.ContainsKey(fieldName))
            {
                List<valueValidationFunction> validationList = mValidationDict[fieldName];
                foreach (valueValidationFunction curValidationFunc in validationList)
                {
                    if (!curValidationFunc(fieldName, fieldValue))
                    {
                        return false;
                    }
                }
            }
            return true;
        }


        public string getDbFieldName(string fieldName)
        {
            string retval = null;
            if (mColumnNameDict.ContainsKey(fieldName))
            {
                retval = mColumnNameDict[fieldName];
            }
            return retval;
        }

        public Type getDbFieldType(string fieldName)
        {
            Type retval = null;
            if (mTypeDict.ContainsKey(fieldName))
            {
                retval = mTypeDict[fieldName];
            }
            return retval;

        }

        public string buildFreeTextConditionSql(string tableName, string relations, string value)
        {
            StringBuilder sb = new StringBuilder(" ( ");
            bool firstTime = true;
            if (!String.IsNullOrEmpty(value) && value != "''")
            {
                value = String.Format("N'%{0}%'", value.Trim('\''));
            }
            if (relations == "=")
            {
                relations = "LIKE";
            }
            else if (relations == "!=")
            {
                relations = "NOT LIKE";
            }
            foreach (string field in mColumnNameDict.Keys)
            {
                if (mTypeDict[field] == typeof(string) && 
                    !mNotFreeTextSearchableFieldsList.Contains(field))
                {
                    if (firstTime)
                    {
                        firstTime = false;
                    }
                    else
                    {
                        sb.Append(" OR ");
                    }
                    sb.Append(String.Format(" {0}.{1} {2} {3}", tableName, mColumnNameDict[field], relations, value));
                }
            }
            sb.Append(" ) ");
            return sb.ToString();
        }


        public bool validCahracters(string field, string value)
        {
            Regex validChar = new Regex(@"^[^\<\>&^#!']*$");
            //Regex validChar = new Regex("^[a-zA-Z0-9%-_@:\\*\\. ]*$");
            return validChar.IsMatch(value);
        }

        public bool validDateTime(string field, string value)
        {
            DateTime test;
            DateTime dbOldestPossibleDate = DateTime.Parse("01/01/1753 00:00:00"); //For some reason this is the oldest possible time in SQL server
            bool retval = false;
            if (DateTime.TryParse(value, out test))
            {
                if (test >= dbOldestPossibleDate)
                {
                    retval = true;
                }
            }
            else//check for enum
            {
                foreach (int val in Enum.GetValues(typeof(DateEnumForSearch)))
                {
                    if (value.ToUpper() == Enum.GetName(typeof(DateEnumForSearch), val).ToUpper())
                    {
                        return true;
                    }
                }
                //check for week before
                foreach (DayOfWeek day in Enum.GetValues(typeof(DayOfWeek)))
                {
                    if (value.ToUpper() == day.ToString().ToUpper())
                    {
                        return true;
                    }
                }
            }
            return retval;
        }

        public bool validTimeSpan(string field, string value)
        {
            TimeSpan test;
            bool retval = false;
            if (TimeSpan.TryParse(value, out test))
            {
                retval = true;
            }
            return retval;
        }

        public bool validInteger(string field, string value)
        {
            int test;
            bool retval = false;
            if (Int32.TryParse(value, out test))
            {
                retval = true;
            }
            return retval;
        }

        public bool validDecimal(string field, string value)
        {
            decimal test;
            bool retval = false;
            if (Decimal.TryParse(value, out test))
            {
                retval = true;
            }
            return retval;
        }

        public bool validateFieldValueByValueAC(string field, string value)
        {
            bool retval = true;
            IConditionValueAutoCompleter vlaueAc = getFieldValueAutoCompleter(field);
            if (vlaueAc != null)
            {
                retval = vlaueAc.validate(value);
            }
            return retval;
        }

        public bool validateDateEnumValueByValueAC(string field, string value)
        {
            bool retval = true;
            IConditionValueAutoCompleter vlaueAc = getFieldValueAutoCompleter(field);
            if (vlaueAc != null)//check if this enum first
            {
                retval = vlaueAc.validate(value);
            }
            if (!retval)//check for week before
            {
                foreach (DayOfWeek day in Enum.GetValues(typeof(DayOfWeek)))
                {
                    if (value.ToUpper() == day.ToString().ToUpper())//Enum.GetName(typeof(DayOfWeek), day).ToUpper()
                    {
                        return true;
                    }
                }
            }
            if (!retval)//check for free date
            {
                DateTime result;
                if (DateTime.TryParse(value.Trim('\''), out result))
                {
                    retval = true;
                }
            }
            
            return retval;
        }

        public virtual IAutoCompleter getFieldRelationshipAutoCompleter(string fieldName)
        {
            return null;
        }

        public virtual IConditionValueAutoCompleter getFieldValueAutoCompleter(string fieldName)
        {
            return null;
        }

        public virtual void formatValue(string fieldName, ref string relations, ref string value)
        {
            switch (fieldName)
            {
                case "TIME":
                case "CREATIONDATE":
                    DateTime result;
                    //check if it is a real date
                    if (!DateTime.TryParse(value.Trim('\''), out result))
                    {
                        result = DealWithDateEnum(value);
                    }
                    if (relations == "=")
                    {
                        relations = "between";
                        value = string.Format("'{0}' and '{1}'", result, result.AddDays(1));
                    }
                    else
                    {   // ">" or "<"
                        value = string.Format("'{0}'", result);
                    }
                    break;
                //ITAMAR-OK - not supposed to be used in client side
                    /*
                case "TAG":
                    value = value.StartsWith("N'") ? value.Substring(2) : value;
                    if (relations == "=")
                    {
                        relations = "IN";
                        value = value.Trim('\'');
                        tags tag = TagsHandler.GetTagByTagName(value);
                        if (tag != null)
                        {
                            value = String.Format("({0})",
                                TagsHandler.GetTagIdAndChildrenIds(tag.tag_id));
                        }
                        else
                        {
                            value = "(-1)";
                        }
                    }
                    else if (relations == "LIKE")
                    {
                        relations = "IN";
                        value = value.Trim('\'').Replace("%", "*");
                    }
                    break;
                    */
            }

        }

        private const string DATE_FORMAT = "MMM dd,yyyy";
        private static DateTime DealWithDateEnum(string value)
        {
            DateTime formatedValue = DateTime.Now;
            int result;
            if (int.TryParse(value.Trim('\''), out result))
            {
                DateEnumForSearch dateEnumVal = (DateEnumForSearch)Enum.ToObject(typeof(DateEnumForSearch), result);
                switch (dateEnumVal)
                {
                    case DateEnumForSearch.Today:
                        formatedValue = DateTime.Now;
                        break;
                    case DateEnumForSearch.Yesterday:
                        formatedValue = DateTime.Now.AddDays(-1);
                        break;

                    default:
                        break;
                }
            }
            else
            {
                DateTime date;
                for (int i = -2; i > -8; i--)
                {
                    date = DateTime.Now.AddDays(i);
                    if (date.DayOfWeek.ToString().ToUpper() == value.Trim('\'').ToUpper())
                    {
                        formatedValue = date;
                        return formatedValue;
                    }
                }
            }
            return formatedValue;
        }

        public string buildConditionSql( string fieldName, string customizedValue, string customizedRelation, string tableName)
        {
            string condition;
            formatValue(fieldName, ref customizedRelation, ref customizedValue);
            if ((customizedValue == "''" || customizedValue.ToLower() == "n'null'") && ((customizedRelation == "=") || (customizedRelation == "!="))) //handling the NULL value in case the search is for empty value;
            {
                string nullRelation = (customizedRelation == "=") ? "IS" : "IS NOT";
                string SqlCond = (customizedRelation == "=") ? "OR" : "AND";
                condition = String.Format("( {0}.{1} {2} {3} {4}  {0}.{1} {5}  NULL)",
                                          tableName, getDbFieldName(fieldName), customizedRelation, customizedValue, SqlCond, nullRelation);
            }
            else
            {
                condition = String.Format(" {0}.{1} {2} {3} ", tableName, getDbFieldName(fieldName), customizedRelation, customizedValue);
            }
            return condition;
        }
    }
}
