﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading;
using org.ovirt.engine.ui.uicommon;
using org.ovirt.engine.ui.uicommon.models;
using org.ovirt.engine.ui.uicompat;
using Microsoft.Practices.Unity;
using VdcCommon.BusinessEntities;
using VdcFrontend;

namespace UI.WPFClient
{
	public sealed class ModelEventArgs<T> : EventArgs where T : Model
	{
		public T Model { get; private set; }

		public ModelEventArgs(T model)
		{
			Model = model;
		}
	}

	public class ShellModel : EntityModel, IFrontendEventsHandler
	{
		#region Events

		public event EventHandler<ModelEventArgs<MessageModel>> NewMessage = delegate { };

		#endregion

		public ShellModel(SynchronizationContext synchronizationContext)
		{
			GuiSynchronizationContext = synchronizationContext;

			if (!Frontend.IsUserLoggedIn)
			{
				GoToLogin();
			}
			else
			{
				GoToCommon();
			}
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			if (ev.Equals(LoginModel.LoggedInEventDefinition))
			{
				GoToCommon();
			}
			else if (ev.Equals(CommonModel.SignedOutEventDefinition))
			{
				GoToLogin();
			}
		}

		private void GoToCommon()
		{
			var model = new CommonModel();
			model.SignedOutEvent.addListener(this);
			Entity = model;
		}

		private void GoToLogin()
		{
			var model = new LoginModel();
			model.LoggedInEvent.addListener(this);
			Entity = model;
		}

		public void ConnectionClosed(Exception ex)
		{
			GuiSynchronizationContext.Send(
				a =>
				{
					if (Entity is CommonModel)
					{
						var commonModel = (CommonModel)Entity;
						commonModel.SignOutCommand.Execute();
					}

					var model =
						new MessageModel
						{
							Title = "Connection Lost",
							DocumentBuilder = new ConnectionClosedDocumentBuilder(ex),
							IsExclusive = true
						};
					model.Closed +=
						(s, args) =>
						{
							//Go to login after a connection closed message.
							GoToLogin();
						};

					//Change title of the default button.
					model.Commands.First().Title = "Login";

					NewMessage(this, new ModelEventArgs<MessageModel>(model));
				}
				, null);
		}

		public void RunActionFailed(List<VdcReturnValueBase> returnValues)
		{
			if (returnValues == null) return;

			GuiSynchronizationContext.Send(
				a =>
				{
					var messages = new List<List<string>>();
					foreach (VdcReturnValueBase returnValue in returnValues)
					{
						if (returnValue == null)
						{
							continue;
						}

						var currentItemDescriptionAndMessages = new List<string>();

						// add the description as the first item in the sub-list:
						currentItemDescriptionAndMessages.Add(returnValue.Description ?? string.Empty);

						// add the CanDoActionMessages as the rest of the sub-list:
						currentItemDescriptionAndMessages.AddRange(
							returnValue.CanDoActionMessages == null ? new List<string>() : returnValue.CanDoActionMessages.ToList());

						// add the sub-list as an item to the main list:
						messages.Add(currentItemDescriptionAndMessages);
					}

					var model =
						new MessageModel
						{
							Title = "Operation Canceled",
							DocumentBuilder = new CanDoActionDocumentBuilder(messages)
						};
					NewMessage(this, new ModelEventArgs<MessageModel>(model));
				},
				null);
		}

		public void RunActionExecutionFailed(VdcActionType action, VdcFault fault)
		{
			if (fault == null) return;

			string faultMessage = fault.Message ?? string.Empty;

			GuiSynchronizationContext.Send(
				a =>
				{
					var model =
						new MessageModel
						{
							Title = "Operation Canceled",
							DocumentBuilder =
								new GeneralFailureDocumentBuilder(
								new[]
								{
									faultMessage,
									String.Format("(Error code: {0})", fault.ErrorCode)
								})
						};
					NewMessage(this, new ModelEventArgs<MessageModel>(model));
				},
				null);
		}

		public void RunQueryFailed(List<VdcQueryReturnValue> returnValue)
		{
			//Do nothing.
		}

		public void PublicConnectionClosed(Exception ex)
		{
			//Do nothing.
		}

		public SynchronizationContext GuiSynchronizationContext { get; private set; }
	}
}
