using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Resources;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using org.ovirt.engine.ui.uicommon.dataprovider;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public class SpiceConsoleModel : ConsoleModel, IFrontendMultipleQueryAsyncCallback
	{
		#region Events

		public static EventDefinition SpiceDisconnectedEventDefinition;
		public static EventDefinition SpiceMenuItemSelectedEventDefinition;
		public static EventDefinition UsbAutoShareChangedEventDefinition;

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		private SpiceMenu menu;
		public ISpice spice { get; set; }

		static SpiceConsoleModel()
		{
			SpiceDisconnectedEventDefinition = new EventDefinition("SpiceDisconnected", typeof(SpiceConsoleModel));
			SpiceMenuItemSelectedEventDefinition = new EventDefinition("SpiceMenuItemSelected", typeof(SpiceConsoleModel));
			UsbAutoShareChangedEventDefinition = new EventDefinition("UsbAutoShareChanged", typeof(SpiceConsoleModel));
		}

		public SpiceConsoleModel()
		{
			Title = "Spice";

			spice = (ISpice)TypeResolver.Instance.Resolve(typeof(ISpice));
			Configurator.Configure(spice);
		}

		protected override void Connect()
		{
			if (Entity != null)
			{
				Logger.Debug("Connecting to Spice console...");
				if (!spice.IsInstalled)
				{
					Logger.Info("Spice client is not installed.");
					spice.Install();
					return;
				}

				//Check a spice version.
				if (Configurator.IsAdmin && spice.CurrentVersion.CompareTo(spice.DesiredVersion) < 0)
				{
					Logger.Info("Spice client version is not as desired (" + spice.DesiredVersion + ")");
					spice.Install();
					return;
				}

				//Don't connect if there VM is not running on any host.
				if (Entity.run_on_vds == null)
				{
					return;
				}


				SendVmTicket();
			}
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(spice.DisconnectedEvent))
			{
				Spice_Disconnected(sender, (ErrorCodeEventArgs)args);
			}
			else if (ev.Equals(spice.MenuItemSelectedEvent))
			{
				Spice_MenuItemSelected(sender, (SpiceMenuItemEventArgs)args);
			}
		}

		private void Spice_MenuItemSelected(object sender, SpiceMenuItemEventArgs e)
		{
			if (Entity != null)
			{
				//SpiceMenuCommandItem item = menu.Descendants()
				//	.OfType<SpiceMenuCommandItem>()
				//	.FirstOrDefault(a => a.Id == e.MenuItemId);
				SpiceMenuCommandItem item = null;
				foreach (SpiceMenuItem a in menu.Descendants())
				{
					if (a.GetType() == typeof(SpiceMenuCommandItem) && a.Id == e.MenuItemId)
					{
						item = (SpiceMenuCommandItem)a;
						break;
					}
				}
				if (item != null)
				{
					switch (item.CommandName)
					{
						case CommandPlay:
							//use sysprep iff the vm is not initialized and vm has Win OS 
							bool reinitialize = !Entity.is_initialized && DataProvider.IsWindowsOsType(Entity.vm_os);
							Frontend.RunMultipleAction(VdcActionType.RunVm,
								new List<VdcActionParametersBase>
								{
									
						    		new RunVmParams(Entity.vm_guid) {RunAsStateless = Entity.is_stateless, Reinitialize = reinitialize}
								});
							break;

						case CommandSuspend:
							Frontend.RunMultipleAction(VdcActionType.HibernateVm,
								new List<VdcActionParametersBase>
								{
						    		new HibernateVmParameters(Entity.vm_guid)
								});
							break;

						case CommandStop:
							Frontend.RunMultipleAction(VdcActionType.ShutdownVm,
								new List<VdcActionParametersBase>
								{
						    		new ShutdownVmParameters(Entity.vm_guid,true)
								});
							break;

						case CommandChangeCD:
							Frontend.RunMultipleAction(VdcActionType.ChangeDisk,
								new List<VdcActionParametersBase>
								{
						    		new ChangeDiskCommandParameters(Entity.vm_guid, item.Text == EjectLabel ? String.Empty : item.Text)
								});
							break;
					}
				}
			}
		}

		private void Spice_Disconnected(object sender, ErrorCodeEventArgs e)
		{
			spice.DisconnectedEvent.removeListener(this);
			spice.MenuItemSelectedEvent.removeListener(this);

			IsConnected = false;
			UpdateActionAvailability();

			if (e.ErrorCode > 100)
			{
				ErrorEvent.raise(this, e);
			}
		}

		private void Cancel()
		{
			Window = null;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command.Name == "Cancel")
			{
				Cancel();
			}
		}

		protected override void UpdateActionAvailability()
		{
			base.UpdateActionAvailability();

			ConnectCommand.IsExecutionAllowed = !IsConnected
				&& Entity != null
				&& Entity.display_type != DisplayType.vnc
				&& IsVmConnectReady();
		}

		private void ExecuteQuery(VM vm)
		{
			AsyncQuery _asyncQuery0 = new AsyncQuery();
			_asyncQuery0.Model = this;

			/*START_DELEGATE*/_asyncQuery0.asyncCallback = delegate(Object model0, Object result0)
				{
					SpiceConsoleModel thisSpiceConsoleModel = (SpiceConsoleModel)model0;
					VM thisVm = thisSpiceConsoleModel.Entity;

					storage_domains isoDomain = null;
					if (result0 != null)
					{
						isoDomain = (storage_domains)result0;
					}

					List<VdcQueryType> queryTypeList = new List<VdcQueryType>();
					queryTypeList.Add(VdcQueryType.GetVdsByVdsId);
					queryTypeList.Add(VdcQueryType.GetConfigurationValue);
					queryTypeList.Add(VdcQueryType.GetConfigurationValue);
					queryTypeList.Add(VdcQueryType.GetConfigurationValue);
					queryTypeList.Add(VdcQueryType.GetConfigurationValue);
					queryTypeList.Add(VdcQueryType.GetVdsCertificateSubjectByVdsId);
					queryTypeList.Add(VdcQueryType.GetCACertificate);
					queryTypeList.Add(VdcQueryType.GetConfigurationValue);
					queryTypeList.Add(VdcQueryType.GetConfigurationValue);

					List<VdcQueryParametersBase> parametersList = new List<VdcQueryParametersBase>();
					parametersList.Add(new GetVdsByVdsIdParameters(thisVm.run_on_vds.Value));
					parametersList.Add(new GetConfigurationValueParameters(ConfigurationValues.SSLEnabled));
					parametersList.Add(new GetConfigurationValueParameters(ConfigurationValues.CipherSuite));
					parametersList.Add(new GetConfigurationValueParameters(ConfigurationValues.SpiceSecureChannels));
					parametersList.Add(new GetConfigurationValueParameters(ConfigurationValues.EnableSpiceRootCertificateValidation));
					parametersList.Add(new GetVdsByVdsIdParameters(thisVm.run_on_vds.Value));
					parametersList.Add(new VdcQueryParametersBase());
					parametersList.Add(new GetConfigurationValueParameters(ConfigurationValues.SpiceToggleFullScreenKeys));
					parametersList.Add(new GetConfigurationValueParameters(ConfigurationValues.SpiceReleaseCursorKeys));

					if (isoDomain != null)
					{
						queryTypeList.Add(VdcQueryType.GetAllIsoImagesList);

						GetAllIsoImagesListParameters getIsoPamams = new GetAllIsoImagesListParameters();
						getIsoPamams.StorageDomainId = isoDomain.id;
						getIsoPamams.ForceRefresh = false;
						parametersList.Add(getIsoPamams);
					}

					Frontend.RunMultipleQueries(queryTypeList, parametersList, thisSpiceConsoleModel);
				};//END_DELEGATE

			AsyncDataProvider.GetIsoDomainByDataCenterId(_asyncQuery0, vm.storage_pool_id);
		}

		private string ticket;

		public void Executed(FrontendMultipleQueryAsyncResult result)
		{
			IList<VdcQueryReturnValue> returnValues = result.ReturnValues;

			bool success = true;
			foreach (VdcQueryReturnValue returnValue in returnValues)
			{
				if (!returnValue.Succeeded)
				{
					success = false;
					break;
				}
			}


			if (!success)
			{
				bool enableSpiceRootCertificateValidation = (bool) result.ReturnValues[4].ReturnValue;
				VdcQueryReturnValue caCertificateReturnValue = result.ReturnValues[6];

				// If only the caCertificate query failed - ignore failure (goto OnSuccess)
				if (!caCertificateReturnValue.Succeeded && !enableSpiceRootCertificateValidation)
				{
					// Verify that all queries (except caCertificate) succeeded
					// If succeeded goto 'OnSuccess'; Otherwise, 'OnFailure'.
					foreach (VdcQueryReturnValue returnValue in returnValues)
					{
						if (!returnValue.Succeeded && returnValue != caCertificateReturnValue)
						{
							return;
						}
					}
				}
			}


			string cipherSuite = null;
			string spiceSecureChannels = null;

			bool isSSLEnabled = (bool)returnValues[1].ReturnValue;
			if (isSSLEnabled)
			{
				cipherSuite = (string)returnValues[2].ReturnValue;
				spiceSecureChannels = (string)returnValues[3].ReturnValue;
			}

			string certificateSubject = String.Empty;
			string caCertificate = String.Empty;

			if ((bool)returnValues[4].ReturnValue)
			{
				certificateSubject = (string)returnValues[5].ReturnValue;
				caCertificate = (string)returnValues[6].ReturnValue;
			}

			spice.Host = Entity.display_ip;
			spice.Port = Entity.display.GetValueOrDefault();
			spice.Password = ticket;
			spice.NumberOfMonitors = Entity.num_of_monitors;
			spice.GuestHostName = Entity.vm_host.Split(' ')[0];
			if (Entity.display_secure_port.HasValue)
			{
				spice.SecurePort = Entity.display_secure_port.Value;
			}
			if (!String.IsNullOrEmpty(spiceSecureChannels))
			{
				spice.SslChanels = spiceSecureChannels;
			}
			if (!String.IsNullOrEmpty(cipherSuite))
			{
				spice.CipherSuite = cipherSuite;
			}

			spice.HostSubject = certificateSubject;
			spice.TrustStore = caCertificate;

			string toggleFullScreenKeys = (string) returnValues[7].ReturnValue;
			string releaseCursorKeys = (string) returnValues[8].ReturnValue;
			string ctrlAltDel = "ctrl+alt+del";
			string ctrlAltEnd = "ctrl+alt+end";
			
			string toggleFullScreenKeysTranslated = DataProvider.GetComplexValueFromSpiceRedKeysResource(toggleFullScreenKeys ?? "shift+f11");
			string releaseCursorKeysTranslated = DataProvider.GetComplexValueFromSpiceRedKeysResource(releaseCursorKeys ?? "shift+f12");
			string ctrlAltDelTranslated = DataProvider.GetComplexValueFromSpiceRedKeysResource(ctrlAltDel);
			string ctrlAltEndTranslated = DataProvider.GetComplexValueFromSpiceRedKeysResource(ctrlAltEnd);

			spice.Title = Entity.vm_name + ":%d" +
				(string.IsNullOrEmpty(releaseCursorKeysTranslated) ? string.Empty : (" - Press " + releaseCursorKeysTranslated + " to Release Cursor"));

			// If 'AdminConsole' is true, send true; otherwise, false should be sent only for VMs with SPICE driver installed.
			spice.AdminConsole = Configurator.SpiceAdminConsole ? true : Entity.SpiceDriverVersion != null ? false : true;

			// Update 'UsbListenPort' value
			spice.UsbListenPort = Configurator.IsUsbEnabled && Entity.usb_policy == UsbPolicy.Enabled ? 
				Configurator.SpiceDefaultUsbPort : Configurator.SpiceDisableUsbListenPort;

			//At lease one of the hot-keys is not empty -> send it to SPICE:
			if (!String.IsNullOrEmpty(releaseCursorKeys) || !String.IsNullOrEmpty(toggleFullScreenKeys))
			{
				string comma =
					(!string.IsNullOrEmpty(releaseCursorKeys) &&
					 !string.IsNullOrEmpty(toggleFullScreenKeys)) ?
					"," : "";

				string releaseCursorKeysParameter = string.IsNullOrEmpty(releaseCursorKeys) ?
					"" :
					"release-cursor=" + releaseCursorKeys;

				string toggleFullScreenKeysParameter = string.IsNullOrEmpty(toggleFullScreenKeys) ?
					"" :
					"toggle-fullscreen=" + toggleFullScreenKeys;

				spice.HotKey = releaseCursorKeysParameter + comma + toggleFullScreenKeysParameter;
			}

			spice.LocalizedStrings = new[] { "USB", "USB Devices,No USB devices,Client's SPICE USB Redirector is not installed" };


			//Create menu.
			int id = 1;
			menu = new SpiceMenu();

			menu.Items.Add(new SpiceMenuCommandItem(ID_SYS_MENU_SEND_CTRL_ALT_DEL, "S&end " + ctrlAltDelTranslated + "\t" + ctrlAltEndTranslated, ""));
			menu.Items.Add(new SpiceMenuCommandItem(ID_SYS_MENU_TOGGLE_FULL_SCREEN, "&Toggle Full Screen\t" + toggleFullScreenKeysTranslated, ""));

			SpiceMenuContainerItem specialKes = new SpiceMenuContainerItem(id, "&Special Keys");
			menu.Items.Add(specialKes);

			specialKes.Items.Add(new SpiceMenuCommandItem(ID_SYS_MENU_SEND_SHIFT_F11, "&" + toggleFullScreenKeysTranslated, ""));
			specialKes.Items.Add(new SpiceMenuCommandItem(ID_SYS_MENU_SEND_SHIFT_F12, "&" + releaseCursorKeysTranslated, ""));
			specialKes.Items.Add(new SpiceMenuCommandItem(ID_SYS_MENU_SEND_CTRL_ALT_END, "&" + ctrlAltEndTranslated, ""));
			menu.Items.Add(new SpiceMenuSeparatorItem(id));
			id++;

			SpiceMenuContainerItem changeCDItem = new SpiceMenuContainerItem(id, "Change CD");
			id++;

			List<string> isos = new List<string>();

			if (returnValues.Count > 9)
			{
				List<RepoFileMetaData> repoList = (List<RepoFileMetaData>)returnValues[9].ReturnValue;
				foreach (RepoFileMetaData RepoFileMetaData in repoList)
				{
					isos.Add(RepoFileMetaData.RepoFileName);
				}
			}

			isos = isos.Count > 0
					? isos
					: new List<string> { "No CDs" };

			isos.Sort();

			foreach (string fileName in isos)
			{
				changeCDItem.Items.Add(new SpiceMenuCommandItem(id, fileName, CommandChangeCD));
				id++;
			}
			changeCDItem.Items.Add(new SpiceMenuCommandItem(id, EjectLabel, CommandChangeCD));
			id++;

			menu.Items.Add(changeCDItem);
			menu.Items.Add(new SpiceMenuSeparatorItem(id));
			id++;
			menu.Items.Add(new SpiceMenuCommandItem(id, "Play", CommandPlay));
			id++;
			menu.Items.Add(new SpiceMenuCommandItem(id, "Suspend", CommandSuspend));
			id++;
			menu.Items.Add(new SpiceMenuCommandItem(id, "Stop", CommandStop));

			spice.Menu = menu.ToString();

			spice.GuestID = Entity.vm_guid.ToString();

			//Subscribe to events.
			spice.DisconnectedEvent.addListener(this);
			spice.MenuItemSelectedEvent.addListener(this);

			if (String.IsNullOrEmpty(Entity.display_ip) || Entity.display_ip == "0")
			{
				VDS host = (VDS)returnValues[0].ReturnValue;
				if (host == null)
				{
					return;
				}

				AsyncQuery _asyncQuery = new AsyncQuery();
				_asyncQuery.Model = this;
				/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
				{
					SpiceConsoleModel spiceConsoleModel = (SpiceConsoleModel)model;
					IEnumerable networkInterfaces = (IEnumerable)((VdcQueryReturnValue)ReturnValue).ReturnValue;
					IEnumerator networkInterfacesIterator = networkInterfaces.GetEnumerator();
					while (networkInterfacesIterator.MoveNext())
					{
						VdsNetworkInterface currentNetworkInterface = (VdsNetworkInterface)networkInterfacesIterator.Current;
						if (currentNetworkInterface == null)
						{
							continue;
						}
						if (currentNetworkInterface.IsManagement)
						{
							spiceConsoleModel.spice.Host = currentNetworkInterface.Address;
							spiceConsoleModel.SpiceConnect();
							return;
						}
					}
				};//END_DELEGATE

				Frontend.RunQuery(VdcQueryType.GetVdsInterfacesByVdsId,
					new GetVdsByVdsIdParameters(host.vds_id),
					_asyncQuery);
			}
			else
			{
				//Try to connect.
				SpiceConnect();
			}
		}

		private void SendVmTicket()
		{
			//Create ticket for single sign on.
			Frontend.RunAction(VdcActionType.SetVmTicket,
				new SetVmTicketParameters(Entity.vm_guid, null, 120),
				result =>
				{
					SpiceConsoleModel model = (SpiceConsoleModel)result.State;

					model.PostSendVmTicket(result.ReturnValue);
				},
				this
			);
		}

		public void PostSendVmTicket(VdcReturnValueBase returnValue)
		{
			if (returnValue == null || !returnValue.Succeeded)
			{
				return;
			}

			ticket = (string)returnValue.ActionReturnValue;

			// Only if the VM has agent and we connect through user-portal
			// we attempt to perform SSO (otherwise an error will be thrown)
			if (!Configurator.IsAdmin && Entity.GuestAgentVersion != null && Entity.status == VMStatus.Up)
			{
				Logger.Info("SpiceConsoleManager::Connect: Attempting to perform SSO on Desktop " + Entity.vm_name);

				Frontend.RunAction(VdcActionType.VmLogon,
					new VmOperationParameterBase(Entity.vm_guid),
					result =>
					{
						SpiceConsoleModel model = (SpiceConsoleModel)result.State;

						VdcReturnValueBase returnValue1 = result.ReturnValue;
						bool success1 = returnValue1 != null && returnValue1.Succeeded;

						if (success1)
						{
							model.ExecuteQuery(Entity);
						}
						else
						{
							string vmName = returnValue1 != null ? returnValue1.Description : "";
							model.Logger.Info("SpiceConsoleManager::Connect: Failed to perform SSO on Destkop " + vmName + " continuing without SSO.");
						}
					},
					this
				);
			}
			else
			{
				ExecuteQuery(Entity);
			}
		}

		public void SpiceConnect()
		{
			try
			{
				spice.Connect();
				IsConnected = true;
				UpdateActionAvailability();
			}
			catch (Exception ex)
			{
				Logger.Error("Exception on Spice connect", ex);
			}
		}


		private const string CommandStop = "Stop";
		private const string CommandPlay = "Play";
		private const string CommandSuspend = "Suspend";
		private const string CommandChangeCD = "ChangeCD";

		private const int ID_SYS_MENU_DISCONNECT = 0x1200;
		private const int ID_SYS_MENU_SEND_CTRL_ALT_DEL = 0x1300;
		private const int ID_SYS_MENU_TOGGLE_FULL_SCREEN = 0x1400;
		private const int ID_SYS_MENU_SEND_SHIFT_F11 = 0x1500;
		private const int ID_SYS_MENU_SEND_SHIFT_F12 = 0x1600;
		private const int ID_SYS_MENU_SEND_CTRL_ALT_END = 0x1700;
		private const int ID_SYS_MENU_USB_DEVICES = 0x1800;
	}
}
