﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.Users;
using org.ovirt.engine.ui.uicommon.validation;

namespace org.ovirt.engine.ui.uicommon.models.userportal
{
	public class UserPortalLoginModel : LoginModel
	{
		#region Commands

		public UICommand ChangePasswordCommand { get; set; }

		#endregion

		#region Properties

		public EntityModel NewPassword { get; private set; }
		public EntityModel VerifyPassword { get; private set; }
		public EntityModel IsAutoConnect { get; private set; }

		private bool isChangingPassword;
		public bool IsChangingPassword
		{
			get { return isChangingPassword; }
			set
			{
				if (isChangingPassword != value)
				{
					isChangingPassword = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsChangingPassword"));
				}
			}
		}

		public const string RHEVMUserRoleId = "00000000-0000-0000-0001-000000000001";
		public Guid Everyone = new Guid("eee00000-0000-0000-0000-123456789eee");
		public Guid Blank = new Guid("00000000-0000-0000-0000-000000000000");
		public Guid UserTemplateBasedVM = new Guid("def00009-0000-0000-0000-def000000009");

		public VdcUser LoggedUser { get; protected set; }
		public EntityModel IsRHEVMUser { get; set; }
		public List<ActionGroup> LoggedUserActionGroupList { get; set; }
		public List<ActionGroup> RHEVMUserActionGroupList { get; set; }
		public int RolesCounter { get; set; }
 
		#endregion

		public UserPortalLoginModel()
		{
			ChangePasswordCommand = new UICommand("ChangePassword", this);

			NewPassword = new EntityModel();
			VerifyPassword = new EntityModel();

			IsRHEVMUser = new EntityModel { Entity = true };
			IsAutoConnect = new EntityModel { Entity = true };
		}

		public override void Login()
		{
			//Completely override the base class functionality.

			if (!Validate())
			{
				return;
			}

			//Clear config cache on login (to make sure we don't use old config in a new session)
			DataProvider.ClearConfigCache();


			Frontend.RunAction(VdcActionType.LoginUser,
				new LoginUserParameters((string)UserName.Entity, (string)Password.Entity, (string)Domain.SelectedItem, String.Empty, String.Empty, String.Empty),
				result =>
				{
					UserPortalLoginModel model = (UserPortalLoginModel)result.State;

					VdcReturnValueBase returnValue = result.ReturnValue;
					bool success = returnValue != null && returnValue.Succeeded;

					if (success)
					{
						model.LoggedUser = (VdcUser)returnValue.ActionReturnValue;
						model.LoggedInEvent.raise(this, EventArgs.Empty);
					}
					else
					{
						//TODO: Based on a returned result set IsPasswordChanging true.

						model.Password.Entity = String.Empty;
						if (returnValue != null)
						{
							model.Message = Linq.FirstOrDefault(returnValue.CanDoActionMessages);
						}

						model.LoginFailedEvent.raise(this, EventArgs.Empty);
					}
				},
				this
			);
		}

		private void ChangePassword()
		{
			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.ChangeUserPassword,
				new ChangeUserPasswordParameters((string)UserName.Entity, (string)Password.Entity, (string)NewPassword.Entity, (string)Domain.SelectedItem));

			if (returnValue != null && returnValue.Succeeded)
			{
				//TODO:
			}
		}

		protected override bool Validate()
		{
			bool baseValidation = base.Validate();

			if (IsChangingPassword)
			{
				NewPassword.ValidateEntity(new IValidation[] { new NotEmptyValidation() });
				VerifyPassword.ValidateEntity(new IValidation[] { new NotEmptyValidation() });

				//Check that the verify password field matches new password.
				if (!((string)NewPassword.Entity).Equals((string)VerifyPassword.Entity))
				{
					VerifyPassword.IsValid = false;
					VerifyPassword.InvalidityReasons.Add("TODO: Verify password field doesn't match a new password.");
				}
			}

			return baseValidation
				&& NewPassword.IsValid
				&& VerifyPassword.IsValid;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == ChangePasswordCommand)
			{
				ChangePassword();
			}
		}

		
		// Update IsRHEVMUser flag.
		// Get 'RHEVMUser' role's ActionGroups (and proceed to Step2).
		public void UpdateIsRHEVMUser(VdcUser LoggedUser)
		{
			RHEVMUserActionGroupList = new List<ActionGroup>();
			this.LoggedUser = LoggedUser;

			AsyncDataProvider.GetRoleActionGroupsByRoleId(new AsyncQuery(this,
				(target, returnValue) =>
				{
					UserPortalLoginModel loginModel = (UserPortalLoginModel)target;
					loginModel.RHEVMUserActionGroupList = (List<ActionGroup>)returnValue;
					loginModel.GetUserRoles(loginModel);

				}),
				new Guid(RHEVMUserRoleId)
			);
		}

		// Get logged user's permissions and create a list of roles associated with the user (and proceed to Step3).
		// Use only as 'Step2' of 'UpdateIsRHEVMUser'
		public void GetUserRoles(object targetObject)
		{
			UserPortalLoginModel loginModel = (UserPortalLoginModel)targetObject;
			AsyncDataProvider.GetPermissionsByAdElementId(new AsyncQuery(targetObject,
				(target, returnValue) =>
				{
					List<permissions> permissions = (List<permissions>)returnValue;
					List<Guid> roleIdList = new List<Guid>();

					// Create user's role IDs list
					foreach (permissions permission in permissions)
					{
						if (IsEveryoneUserPortalBasedVmPermission(permission))
						{
							continue;
						}
						
						if (!roleIdList.Contains(permission.role_id))
						{
							roleIdList.Add(permission.role_id);
						}
					}

					// If roles have been found - get user's ActionGroups; Otherwise, proceed to last step.
					UserPortalLoginModel loginModel1 = (UserPortalLoginModel)target;
					loginModel1.LoggedUserActionGroupList = new List<ActionGroup>();
					if (roleIdList.Count > 0)
					{
						loginModel1.RolesCounter = roleIdList.Count;
						loginModel1.UpdateUserActionGroups(loginModel1, roleIdList);
					}
					else
					{
						CheckIsRHEVMUser(loginModel1);
					}
				}),
				loginModel.LoggedUser.UserId
			);
		}

		/// <summary>
		/// Determines whether a given permission is a permission on Everyone associated with the "UserTemplateBasedVM" Role.
		/// </summary>
		/// <param name="permission">given permission</param>
		/// <returns>True if permission is on Everyone and associated with the "UserTemplateBasedVM" Role, false otherwise.</returns>
		public bool IsEveryoneUserPortalBasedVmPermission(permissions permission)
		{
			return permission.ad_element_id.Value.Equals(Everyone) && permission.role_id.Value.Equals(UserTemplateBasedVM);
		}

		// Create a list of ActionGroups associated with the user by retrieving each role's ActionGroups (and proceed to Step4).
		// Use only as 'Step3' of 'UpdateIsRHEVMUser'
		public void UpdateUserActionGroups(object targetObject, List<Guid> roleIdList)
		{
			foreach (Guid roleID in roleIdList)
			{
				AsyncDataProvider.GetRoleActionGroupsByRoleId(new AsyncQuery(targetObject,
					(target, returnValue) =>
					{
						UserPortalLoginModel loginModel = (UserPortalLoginModel)target;
						List<ActionGroup> roleActionGroupList = (List<ActionGroup>)returnValue;
						foreach (ActionGroup actionGroup in roleActionGroupList)
						{
							if (!loginModel.LoggedUserActionGroupList.Contains(actionGroup))
							{
								loginModel.LoggedUserActionGroupList.Add(actionGroup);
							}
						}

						// Proceed to last step after retrieving the ActionGroups of every role.
						if (--loginModel.RolesCounter == 0)
						{
							CheckIsRHEVMUser(loginModel);
						}
					}),
					roleID
				);
			}
		}

		// If 'LoggedUserActionGroupList' contains an ActionGroup that doesn't exist in RHEVMUserActionGroupList
		// the logged user is not 'RHEVMUser' - Update IsRHEVMUser to false; Otherwise, true.
		// Raise 'LoggedIn' event after updating the flag.
		// Use only as 'Step4' of 'UpdateIsRHEVMUser'
		public void CheckIsRHEVMUser(object targetObject)
		{
			UserPortalLoginModel loginModel = (UserPortalLoginModel)targetObject;
			loginModel.IsRHEVMUser.Entity = null;
			bool isRHEVMUser = true;

			foreach (ActionGroup actionGroup in loginModel.LoggedUserActionGroupList)
			{
				if (!loginModel.RHEVMUserActionGroupList.Contains(actionGroup))
				{
					isRHEVMUser = false;
					break;
				}
			}

			loginModel.IsRHEVMUser.Entity = isRHEVMUser;
		}
	}
}
