using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.validation;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;

namespace org.ovirt.engine.ui.uicommon.models.hosts
{
	public class HostInterfaceModel : EntityModel
	{
		#region Properties

		public EntityModel Address { get; private set; }
		public EntityModel Subnet { get; private set; }
		public ListModel Network { get; private set; }
		public EntityModel CheckConnectivity { get; private set; }
		public ListModel BondingOptions { get; private set; }
		public List<VdsNetworkInterface> Networks { get; set; }

		private string name;
		public string Name
		{
			get { return name; }
			set
			{
				if (name != value)
				{
					name = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Name"));
				}
			}
		}

		private NetworkBootProtocol bootProtocol;
		public NetworkBootProtocol BootProtocol
		{
			get
			{
				return bootProtocol;
			}

			set
			{
				if (bootProtocol != value)
				{
					bootProtocol = value;
					BootProtocolChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("BootProtocol"));
				}
			}
		}

		private bool noneBootProtocolAvailable = true;
		public bool NoneBootProtocolAvailable
		{
			get
			{
				return noneBootProtocolAvailable;
			}

			set
			{
				if (noneBootProtocolAvailable != value)
				{
					noneBootProtocolAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("NoneBootProtocolAvailable"));
				}
			}
		}


		private bool bootProtocolsAvailable;
		public bool BootProtocolsAvailable
		{
			get
			{
				return bootProtocolsAvailable;
			}

			set
			{
				if (bootProtocolsAvailable != value)
				{
					bootProtocolsAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("BootProtocolsAvailable"));
				}
			}
		}

		public bool IsStaticAddress
		{
			get
			{
				return BootProtocol == NetworkBootProtocol.StaticIp;
			}
		}

		private bool commitChanges;
		public bool CommitChanges
		{
			get { return commitChanges; }
			set
			{
				if (commitChanges != value)
				{
					commitChanges = value;
					OnPropertyChanged(new PropertyChangedEventArgs("CommitChanges"));
				}
			}
		}

		private bool bondingOptionsOverrideNotification { get; set; }
		public bool BondingOptionsOverrideNotification
		{
			get
			{
				return bondingOptionsOverrideNotification;
			}
			set
			{
				bondingOptionsOverrideNotification = value;
				OnPropertyChanged(new PropertyChangedEventArgs("BondingOptionsOverrideNotification"));
			}
		}

		#endregion

		public HostInterfaceModel()
		{
			Address = new EntityModel();
			Subnet = new EntityModel();
			Network = new ListModel();
			Network.SelectedItemChangedEvent.addListener(this);

			CheckConnectivity = new EntityModel();
			CheckConnectivity.Entity = false;
			BondingOptions = new ListModel();
			// call the Network_ValueChanged method to set all
			// properties according to default value of Network:
			Network_SelectedItemChanged(null);
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ListModel.SelectedItemChangedEventDefinition) && sender == Network)
			{
				Network_SelectedItemChanged(null);
			}
		}

		void Network_SelectedItemChanged(EventArgs e)
		{
			UpdateCanSpecify();

			network network = (network)Network.SelectedItem;
			BootProtocolsAvailable = (network != null && network.name == "None") ? false : true;

			if (Networks != null)
			{
				foreach (VdsNetworkInterface item in Networks)
				{
					if (item.NetworkName == network.name)
					{
						Address.Entity = String.IsNullOrEmpty(item.Address) ? null : item.Address;
						Subnet.Entity = String.IsNullOrEmpty(item.Subnet) ? null : item.Subnet;
						BootProtocol = !NoneBootProtocolAvailable && item.BootProtocol == NetworkBootProtocol.None ? NetworkBootProtocol.Dhcp : item.BootProtocol;
						break;
					}
				}
			}
			
		}

		private void BootProtocolChanged()
		{
			UpdateCanSpecify();

			Address.IsValid = true;
			Subnet.IsValid = true;
		}

		private void UpdateCanSpecify()
		{
			network network = (network)Network.SelectedItem;
			bool isChangable = IsStaticAddress && network != null && !network.Id.Equals(Guid.Empty);
			Address.IsChangable = isChangable;
			Subnet.IsChangable = isChangable;
		}

		internal virtual bool Validate()
		{
			Network.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });

			Address.IsValid = true;
			Subnet.IsValid = true;

			network net = (network)Network.SelectedItem;
			if (IsStaticAddress &&
				Network.SelectedItem != null &&
				!net.Id.Equals(Guid.Empty))
			{
				Address.ValidateEntity(new IValidation[] { new NotEmptyValidation(), new IpAddressValidation() });
				Subnet.ValidateEntity(new IValidation[] { new NotEmptyValidation(), new IpAddressValidation() });
			}

			return Network.IsValid
				   && Address.IsValid
				   && Subnet.IsValid;
		}
	}
}
