using System;
using System.Collections.Generic;
using System.Text;
using org.ovirt.engine.ui.uicommon.models;
using org.ovirt.engine.ui.uicommon.models.tags;
using org.ovirt.engine.ui.uicommon.models.users;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.genericapi.parameters;
using org.ovirt.engine.ui.genericapi.returnvalues;
using org.ovirt.engine.ui.genericapi.uiqueries;
using VdcCommon.Users;
using VdcCommon.VdcQueries;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Collections;
using VdcCommon.Interfaces;
using VdcCommon;
using System.Resources;
using System.Reflection;
using System.Globalization;

namespace org.ovirt.engine.ui.uicommon
{
	/// <summary>
	/// Contains method for retrieving common data (mostly via frontend).
	/// </summary>
	/// <remarks>
	/// All method returning list of objects must avoid
	/// returning a null value, but an empty list.
	/// </remarks>
	public static class DataProvider
	{
		public const int SearchLimit = 9999;


		public static string GetLocalFSPath()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.RhevhLocalFSPath));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return String.Empty;
		}

		public static string GetLinuxMountPointRegex()
		{
			//32-bit IPv4 Internet Protocol (IP): RFC 1918
			//^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$

			//FQDN: RFC's 952/1123
			//^([a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?\.)+[a-zA-Z]{2,}$

			//Linux path
			//(.*?/|.*?\\)?([^\./|^\.\\]+)(?:\.([^\\]*)|)$

			//[IP:/path or FQDN:/path]
			return @"^((?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)|(([a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?\.)+[a-zA-Z]{2,}))\:/(.*?/|.*?\\)?([^\./|^\.\\]+)(?:\.([^\\]*)|)$";
		}
		public static List<UserPermissionModel> GetUserPermissionMatrix(Guid userId)
		{
			//            var roles = GetRoleList().ToDictionary(a => a.id);
			Dictionary<Guid, roles> roles = new Dictionary<Guid, roles>();
			foreach (roles role in GetRoleList())
			{
				roles.Add(role.Id, role);
			}

			VdcQueryReturnValue returnValue = Frontend.RunQuery(
					VdcQueryType.GetPermissionsByAdElementId,
					new MultilevelAdministrationByAdElementIdParameters(userId)
				);

			if (returnValue == null || !returnValue.Succeeded)
			{
				return null;
			}
			List<permissions> permissions = (List<permissions>)returnValue.ReturnValue;

			List<UserPermissionModel> userPermissions = new List<UserPermissionModel>();

			foreach (permissions permission in permissions)
			{
				UserPermissionModel userPermission = new UserPermissionModel();
				userPermission.Id = permission.Id;
				userPermission.Role = new ListModel() { SelectedItem = roles[permission.role_id].name };
				List<TagModel> tags = new List<TagModel>();
				foreach (tags tag in permission.Tags)
				{
					TagModel tagModel = new TagModel();
					EntityModel entityModel = new EntityModel();
					entityModel.Entity = tag.tag_name;
					tagModel.Name = entityModel;
					tags.Add(tagModel);
				}
				userPermission.Tags = tags;
				userPermissions.Add(userPermission);
			}
			//return permissions
			//    .Select(a =>
			//        new UserPermissionModel
			//        {
			//            Id = a.id,
			//            Role = public ListModel(container) { Value = roles[a.role_id].name },
			//            Tags = a.Tags.Select(b => new TagModel { Name = new EntityModel(Container) { Value = b.tag_name } }).ToList()
			//        }
			//    )
			//    .ToList();
			return userPermissions;
		}

		public static List<event_subscriber> GetEventNotificationList(Guid userId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(
					VdcQueryType.GetEventSubscribersBySubscriberId,
					new GetEventSubscribersBySubscriberIdParameters(userId)
				);
			if (returnValue == null || !returnValue.Succeeded)
			{
				return new List<event_subscriber>();
			}

			return (List<event_subscriber>)returnValue.ReturnValue;
		}

		public static List<network> GetNetworkList(Guid storagePoolId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllNetworks,
				new GetAllNetworkQueryParamenters(storagePoolId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<network>)returnValue.ReturnValue;
			}

			return new List<network>();
		}

		public static List<network> GetClusterNetworkList(Guid clusterId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllNetworksByClusterId,
				new VdsGroupQueryParamenters(clusterId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<network>)returnValue.ReturnValue;
			}

			return new List<network>();
		}

		public static List<roles> GetRoleList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllRoles,
				new MultilevelAdministrationsQueriesParameters());

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<roles>)returnValue.ReturnValue;
			}

			return new List<roles>();
		}

		private static string cachedDefaultTimeZone;
		public static string GetDefaultTimeZone()
		{
			if (cachedDefaultTimeZone == null)
			{
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetDefualtTimeZone,
					new VdcQueryParametersBase());

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					cachedDefaultTimeZone = ((KeyValuePair<string, string>)returnValue.ReturnValue).Key;
				}
				else
				{
					cachedDefaultTimeZone = String.Empty;
				}
			}

			return cachedDefaultTimeZone;
		}

		private static Dictionary<string, string> cachedTimeZones;
		public static Dictionary<string, string> GetTimeZoneList()
		{
			if (cachedTimeZones == null)
			{
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetTimeZones,
					new VdcQueryParametersBase());

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					cachedTimeZones = (Dictionary<string, string>)returnValue.ReturnValue;
				}
				else
				{
					cachedTimeZones = new Dictionary<string, string>();
				}
			}

			return cachedTimeZones;
		}

		public static List<VDSGroup> GetClusterList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllVdsGroups, new VdcQueryParametersBase());
			if (returnValue != null && returnValue.Succeeded)
			{
				List<VDSGroup> list = (List<VDSGroup>)returnValue.ReturnValue;
				if (list != null)
				{
					//return Linq.OrderBy<VDSGroup>( groups.OrderBy(a => a.name).ToList();
					list.Sort(new Linq.VdsGroupByNameComparer());
					return list;
				}
			}

			return new List<VDSGroup>();
		}

		public static List<VDSGroup> GetClusterList(Guid storagePoolID)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVdsGroupsByStoragePoolId,
				 new StoragePoolQueryParametersBase(storagePoolID));

			if (returnValue != null && returnValue.Succeeded)
			{
				List<VDSGroup> list = (List<VDSGroup>)returnValue.ReturnValue;
				if (list != null)
				{
					//return groups.OrderBy(a => a.name).ToList();
					list.Sort(new Linq.VdsGroupByNameComparer());
					return list;
				}
			}

			return new List<VDSGroup>();
		}

		public static List<string> GetDomainList(bool filterInternalDomain)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetDomainList,
				new GetDomainListParameters() { FilterInternalDomain = filterInternalDomain });

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<string>)returnValue.ReturnValue;
			}

			return new List<string>();
		}

		//NOTE: Moved to AsyncDataProvider.
		//public static List<string> GetDomainListViaPublic()
		//{
		//    VdcQueryReturnValue returnValue = Frontend.RunPublicQuery(VdcQueryType.GetDomainList,
		//        new VdcQueryParametersBase());

		//    if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
		//    {
		//        return (List<string>)returnValue.ReturnValue;
		//    }

		//    return new List<string>();
		//}

		//NOTE: Moved to AsyncDataProvider.
		//public static bool IsBackendAvailable()
		//{
		//    return Frontend.RunPublicQuery(VdcQueryType.GetDomainList, new VdcQueryParametersBase()) != null ? true : false;
		//}

		public static int GetMinimalVmMemSize()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VMMinMemorySizeInMB));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 1;
		}

		public static int GetMaximalVmMemSize32OS()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VM32BitMaxMemorySizeInMB));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 262144;
		}

		public static int GetMaximalVmMemSize64OS()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VM64BitMaxMemorySizeInMB));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 262144;
		}

		public static List<VmTemplate> GetTemplateList(Guid storagePoolId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmTemplatesByStoragePoolId, new GetVmTemplatesByStoragePoolIdParameters(storagePoolId));
			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//              var list = ((List<VmTemplate>)returnValue.ReturnValue)
				//                  .Where(a => a.status == VmTemplateStatus.OK)
				//                  .OrderBy(a => a.name)
				//                  .ToList();
				VmTemplate blankTemplate = new VmTemplate();
				List<VmTemplate> list = new List<VmTemplate>();
				foreach (VmTemplate template in (List<VmTemplate>)returnValue.ReturnValue)
				{
					if (template.Id.Equals(Guid.Empty))
					{
						blankTemplate = template;
					}
					else if (template.status == VmTemplateStatus.OK)
					{
						list.Add(template);
					}
				}

				list.Sort(new Linq.VmTemplateByNameComparer());
				list.Insert(0, blankTemplate);
				//              VmTemplate blankTemplate = list.First(a => (Guid)a.vmt_guid == Guid.Empty);
				//				list.Remove(blankTemplate);
				//				list.Insert(0, blankTemplate);

				return list;
			}

			return new List<VmTemplate>();
		}

		public static VmTemplate GetTemplateByID(Guid templateGUID)
		{
			VdcQueryReturnValue returnValue =
				Frontend.RunQuery(
					VdcQueryType.GetVmTemplate,
					new GetVmTemplateParameters(templateGUID));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (VmTemplate)returnValue.ReturnValue;
			}

			return null;
		}

		public static List<storage_domains> GetStorageDomainListByTemplate(Guid templateId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStorageDomainsByVmTemplateId,
				new GetStorageDomainsByVmTemplateIdQueryParameters(templateId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<storage_domains>)returnValue.ReturnValue;
			}

			return new List<storage_domains>();
		}

		public static storage_domains GetIsoDomainByDataCenterId(Guid dataCenterId)
		{
			StoragePoolQueryParametersBase getIsoParams = new StoragePoolQueryParametersBase(dataCenterId);
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStorageDomainsByStoragePoolId, getIsoParams);

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				List<storage_domains> storageDomains = (List<storage_domains>)returnValue.ReturnValue;

				foreach (storage_domains domain in storageDomains)
				{
					if (domain.storage_domain_type == StorageDomainType.ISO)
					{
						return domain;
					}
				}
			}

			return null;
		}

		public static storage_domains GetExportDomainByDataCenterId(Guid dataCenterId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStorageDomainsByStoragePoolId,
				new StoragePoolQueryParametersBase(dataCenterId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				List<storage_domains> storageDomains = (List<storage_domains>)returnValue.ReturnValue;
				foreach (storage_domains domain in storageDomains)
				{
					if (domain.storage_domain_type == StorageDomainType.ImportExport)
					{
						return domain;
					}
				}
			}

			return null;
		}

		public static List<string> GetIrsImageList(Guid dataCenterId, bool forceRefresh)
		{
			storage_domains isoDomain = GetIsoDomainByDataCenterId(dataCenterId);
			if (isoDomain != null)
			{
				return GetIsoListByIsoDomainId(isoDomain.id, forceRefresh);
			}

			return new List<string>();
		}

		public static List<string> GetIsoListByIsoDomainId(Guid isoDomainId, bool forceRefresh)
		{
			GetAllIsoImagesListParameters param = new GetAllIsoImagesListParameters();
			param.StorageDomainId = isoDomainId;
			param.ForceRefresh = forceRefresh;

			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllIsoImagesList, param);
			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				List<RepoFileMetaData> listRepoFileList = (List<RepoFileMetaData>)returnValue.ReturnValue;
				List<string> fileNameList = new List<string>();
				foreach (RepoFileMetaData RepoFileMetaData in listRepoFileList)
				{
					fileNameList.Add(RepoFileMetaData.RepoFileName);
				}

				fileNameList.Sort();
				return fileNameList;
			}

			return new List<string>();
		}

		public static string GetDefaultExportPath()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.ExportDefaultPath));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return String.Empty;
		}

		public static string GetDefaultImportPath()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.ImportDefaultPath));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return String.Empty;
		}

		public static List<tags> GetAllTagsList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllTags,
				new VdcQueryParametersBase());

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return ((List<tags>)returnValue.ReturnValue);
			}

			return new List<tags>();
		}

		public static tags GetRootTag()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetRootTag, new VdcQueryParametersBase());
			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				tags tag = (tags)returnValue.ReturnValue;

				tags root = new tags(tag.description, tag.parent_id, tag.IsReadonly, tag.tag_id, tag.tag_name);
				if (tag.Children != null)
				{
					fillTagsRecursive(root, tag.Children);
				}

				return root;
			}

			return new tags();
		}

		private static void fillTagsRecursive(tags tagToFill, IList<tags> children)
		{
			List<tags> list = new List<tags>();

			foreach (tags tag in children)
			{
				//tags child = new tags(tag.description, tag.parent_id, tag.IsReadonly, tag.tag_id, tag.tag_name);
				if (tag.type == TagsType.GeneralTag)
				{
					list.Add(tag);
					if (tag.Children != null)
					{
						fillTagsRecursive(tag, tag.Children);
					}
				}

			}

			tagToFill.Children = list;
		}

		public static List<tags> GetAttachedTagsToVm(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetTagsByVmId,
				new GetTagsByVmIdParameters(id.ToString()));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<tags>)returnValue.ReturnValue)
				//					.Where(a => a.type == TagsType.GeneralTag)
				//					.ToList();
				List<tags> ret = new List<tags>();
				foreach (tags tags in (List<tags>)returnValue.ReturnValue)
				{
					if (tags.type == TagsType.GeneralTag)
					{
						ret.Add(tags);
					}
				}
				return ret;
			}

			return new List<tags>();
		}

		public static List<tags> GetAttachedTagsToHost(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetTagsByVdsId,
				new GetTagsByVdsIdParameters(Convert.ToString(id)));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//return ((List<tags>)returnValue.ReturnValue)
				//    .Where(a => a.type == TagsType.GeneralTag)
				//    .ToList();
				List<tags> ret = new List<tags>();
				foreach (tags tags in (List<tags>)returnValue.ReturnValue)
				{
					if (tags.type == TagsType.GeneralTag)
					{
						ret.Add(tags);
					}
				}
				return ret;
			}

			return new List<tags>();
		}

		public static List<tags> GetAttachedTagsToUser(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetTagsByUserId,
				new GetTagsByUserIdParameters(id.ToString()));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<tags>)returnValue.ReturnValue)
				//					.Where(a => a.type == TagsType.GeneralTag)
				//					.ToList();
				List<tags> ret = new List<tags>();
				foreach (tags tags in (List<tags>)returnValue.ReturnValue)
				{
					if (tags.type == TagsType.GeneralTag)
					{
						ret.Add(tags);
					}
				}
				return ret;
			}

			return new List<tags>();
		}

		public static List<tags> GetAttachedTagsToUserGroup(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetTagsByUserGroupId,
				new GetTagsByUserGroupIdParameters(id.ToString()));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<tags>)returnValue.ReturnValue)
				//					.Where(a => a.type == TagsType.GeneralTag)
				//					.ToList();
				List<tags> ret = new List<tags>();
				foreach (tags tags in (List<tags>)returnValue.ReturnValue)
				{
					if (tags.type == TagsType.GeneralTag)
					{
						ret.Add(tags);
					}
				}
				return ret;
			}

			return new List<tags>();
		}

		public static List<tags> GetAttachedTagsBySubscriberId(Guid id, string event_name)
		{
			//			return ((List<event_subscriber>)GetEventNotificationList(id)).Where(a => a.event_up_name == event_name && !string.IsNullOrEmpty(a.tag_name)).Select(b => new tags(b.event_up_name, null, false, i++, b.tag_name)).ToList();
			List<tags> tags = new List<tags>();
			foreach (event_subscriber event_subscriber in (List<event_subscriber>)GetEventNotificationList(id))
			{
				if (event_subscriber.event_up_name == event_name && !string.IsNullOrEmpty(event_subscriber.tag_name))
				{
					tags.Add(new tags(event_subscriber.event_up_name, null, false, new Guid(), event_subscriber.tag_name));
				}
			}
			return tags;
		}

		public static List<string> GetHostTypeList(bool showPowerclient)
		{
			//TODO: We can translate it here too
			List<string> ret = new List<string>();
			foreach (string s in Enum.GetNames(typeof(VDSType)))
			{
				if (s == VDSType.PowerClient.ToString())
				{
					if (showPowerclient)
					{
						ret.Add(s);
					}
				}
				else
				{
					ret.Add(s);
				}
			}
			return ret;
		}

		public static List<EventNotificationEntity> GetEventNotificationTypeList()
		{
			List<EventNotificationEntity> ret = new List<EventNotificationEntity>();
			//TODO: We can translate it here too
			foreach (EventNotificationEntity entity in Enum.GetValues(typeof(EventNotificationEntity)))
			{
				if (entity != EventNotificationEntity.UNKNOWN)
				{
					ret.Add(entity);
				}
			}
			return ret;
		}

		public static IDictionary<EventNotificationEntity, HashSet<AuditLogType>> GetAvailableNotificationEvents()
		{
			return VdcEventNotificationUtils.GetNotificationEvents();
		}

		public static List<bookmarks> GetBookmarkList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllBookmarks,
				new VdcQueryParametersBase());

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<bookmarks>)returnValue.ReturnValue;
			}

			return new List<bookmarks>();
		}

		public static List<VmInterfaceType> GetNicTypeList(VmOsType osType, bool hasDualmode)
		{
			List<VmInterfaceType> list = new List<VmInterfaceType>();
			foreach (VmInterfaceType item in Enum.GetValues(typeof(VmInterfaceType)))
			{
				list.Add(item);
			}

			list.Remove(VmInterfaceType.rtl8139_pv); //Dual mode NIC should be available only for existing NICs that have that type already
			if (IsWindowsOsType(osType))
			{
				list.Remove(VmInterfaceType.e1000);
				if (osType == VmOsType.WindowsXP && hasDualmode)
				{
					list.Add(VmInterfaceType.rtl8139_pv);
				}
			}

			return list;
		}

		public static VmInterfaceType GetDefaultNicType(VmOsType osType)
		{
			return VmInterfaceType.pv;
		}

		private static List<int> cachedNumOfMonitors;
		public static List<int> GetNumOfMonitorList()
		{
			if (cachedNumOfMonitors == null)
			{
				VdcQueryReturnValue returnValue = GetConfigFromCache(
					new GetConfigurationValueParameters(ConfigurationValues.ValidNumOfMonitors));

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					//					cachedNumOfMonitors = ((List<string>)returnValue.ReturnValue).Select(a => Convert.ToInt32(a)).ToList();
					List<int> nums = new List<int>();
					foreach (string num in (List<string>)returnValue.ReturnValue)
					{
						nums.Add(Convert.ToInt32(num));
					}
					return nums;
				}
				else
				{
					cachedNumOfMonitors = new List<int>();
				}
			}

			return cachedNumOfMonitors;
		}

		public static bool IsSupportCustomProperties(string version)
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.SupportCustomProperties) { Version = version });

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (bool)returnValue.ReturnValue;
			}

			return true;
		}

		public static Int32 GetMaxNumOfVmCpus(string version)
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.MaxNumOfVmCpus) { Version = version });

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (Int32)returnValue.ReturnValue;
			}

			return 1;
		}

		public static string GetAuthenticationMethod()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.AuthenticationMethod));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return String.Empty;
		}

		public static int GetMaxNumOfVmSockets(string version)
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
			new GetConfigurationValueParameters(ConfigurationValues.MaxNumOfVmSockets) { Version = version });

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 1;
		}

		public static int GetMaxNumOfCPUsPerSocket(string version)
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
			new GetConfigurationValueParameters(ConfigurationValues.MaxNumOfCpuPerSocket) { Version = version });

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 1;
		}

		private static Dictionary<Version, List<ServerCpu>> cachedCPUs;
		public static List<ServerCpu> GetCPUList(Version Version)
		{
			if (Version != null && (cachedCPUs == null ? InitCachedCPUsDict() : true)
				&& ((!cachedCPUs.ContainsKey(Version)) ||
				((cachedCPUs.ContainsKey(Version)) && cachedCPUs[Version] == null)))
			{
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllServerCpuList,
					new GetAllServerCpuListParameters(Version));

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					cachedCPUs[Version] = (List<ServerCpu>)returnValue.ReturnValue;
				}
			}

			return Version != null && cachedCPUs.ContainsKey(Version) ?
				cachedCPUs[Version] ?? new List<ServerCpu>()
				:
				new List<ServerCpu>();
		}

		private static bool InitCachedCPUsDict()
		{
			cachedCPUs = new Dictionary<Version, List<ServerCpu>>();
			return cachedCPUs != null;
		}

		//#537775, #622235
		public static void GetWipeAfterDeleteDefaultsByStorageType(StorageType storageType, EntityModel WipeAfterDeleteEntityModel, bool IsThisNewCommand)
		{
			if (storageType == StorageType.NFS || storageType == StorageType.LOCALFS)
			{
				WipeAfterDeleteEntityModel.IsChangable = false;
			}
			else
			{
				WipeAfterDeleteEntityModel.IsChangable = true;
				if (IsThisNewCommand)
				{
					WipeAfterDeleteEntityModel.Entity = GetSANWipeAfterDelete();
				}
			}
		}

		public static string[] GetADPathList()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.ComputerADPaths));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return ((string)returnValue.ReturnValue).Split(';');
			}

			return new string[] { };
		}

		public static List<StorageType> GetStoragePoolTypeList()
		{
			return
				new List<StorageType>
				{
					StorageType.NFS,
					StorageType.ISCSI,
					StorageType.FCP,
					StorageType.LOCALFS
				};
		}

		public static bool IsVersionMatchStorageType(Version version, StorageType type)
		{
			return !(type == StorageType.LOCALFS && version.CompareTo(new Version(2, 2)) <= 0);
		}


		public static string GetRpmVersionViaPublic()
		{
			VdcQueryReturnValue returnValue = Frontend.RunPublicQuery(VdcQueryType.GetConfigurationValue,
				new GetConfigurationValueParameters(ConfigurationValues.ProductRPMVersion));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return string.Empty;
		}

		public static string GetRpmVersion()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.ProductRPMVersion));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return string.Empty;
		}

		/// <summary>
		/// Used to retrieve versions for editing cluster
		/// </summary>
		/// <param name="vdsGroupId"></param>
		/// <returns></returns>
		public static List<Version> GetClusterVersions(guid vdsGroupId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAvailableClusterVersions,
				new GetAvailableClusterVersionsParameters { VdsGroupId = vdsGroupId });

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<Version>)returnValue.ReturnValue;
			}
			else
			{
				return new List<Version>();
			}
		}

		/// <summary>
		/// Used to retrieve versions for creating cluster
		/// </summary>
		/// <param name="storagePoolId"></param>
		/// <returns></returns>
		public static List<Version> GetDataCenterClusterVersions(Guid? storagePoolId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAvailableClusterVersionsByStoragePool,
			   new GetAvailableClusterVersionsByStoragePoolParameters
			   {
				   StoragePoolId = storagePoolId
			   });

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				List<Version> list = (List<Version>)returnValue.ReturnValue;
				list.Sort();
				return list;
			}

			return new List<Version>();
		}

		public static List<Version> GetDataCenterVersions(Guid? storagePoolId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAvailableStoragePoolVersions,
				new GetAvailableStoragePoolVersionsParameters
				{
					StoragePoolId = storagePoolId
				});

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return Linq.OrderByDescending((List<Version>)returnValue.ReturnValue);
			}

			return new List<Version>();
		}

		public static int GetClusterServerMemoryOverCommit()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.MaxVdsMemOverCommitForServers));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 0;
		}

		public static int GetClusterDesktopMemoryOverCommit()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.MaxVdsMemOverCommit));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 0;
		}

		public static int GetClusterDefaultMemoryOverCommit()
		{
			return 100;
		}

		public static List<storage_pool> GetDataCenterList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("DataCenter: sortby name", SearchType.StoragePool)
				{
					MaxCount = SearchLimit
				});

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return Linq.Cast<storage_pool>((List<IVdcQueryable>)returnValue.ReturnValue);
				//return ((List<IVdcQueryable>)returnValue.ReturnValue)
				//	.Cast<storage_pool>()
				//	.ToList();
			}

			return new List<storage_pool>();
		}

		public static List<VmOsType> GetOSList()
		{
			return new List<VmOsType>
			       {
					   VmOsType.Unassigned,
			       		VmOsType.RHEL6,
			       		VmOsType.RHEL6x64,
			       		VmOsType.RHEL5,
			       		VmOsType.RHEL5x64,
			       		VmOsType.RHEL4,
			       		VmOsType.RHEL4x64,
			       		VmOsType.RHEL3,
			       		VmOsType.RHEL3x64,
			       		VmOsType.OtherLinux,
			       		VmOsType.WindowsXP,
			       		VmOsType.Windows2003,
			       		VmOsType.Windows2003x64,
			       		VmOsType.Windows7,
			       		VmOsType.Windows7x64,
			       		VmOsType.Windows2008,
			       		VmOsType.Windows2008x64,
			       		VmOsType.Windows2008R2x64,
			       		VmOsType.Other
			       };
		}

		public static IEnumerable GetUsbPolicyList()
		{
			return Enum.GetValues(typeof(UsbPolicy));
		}

		public static List<VDS> GetUpHostList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Host: status=up", SearchType.VDS));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<IVdcQueryable>)returnValue.ReturnValue).Cast<VDS>().ToList();
				return Linq.Cast<VDS>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<VDS>();
		}

		public static List<VDS> GetUpSpmsByStorage(Guid storageDomainId)
		{
			List<storage_domains> storageDomainWithPools = null;

			// get the storage domain instance - each one with a different storage pool:
			VdcQueryReturnValue returnValue = Frontend.RunQuery(
				VdcQueryType.GetStorageDomainListById,
				new StorageDomainQueryParametersBase(storageDomainId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				storageDomainWithPools = (List<storage_domains>)returnValue.ReturnValue;
			}

			// filter only the storage domain instances that are active:
			//			storageDomainWithPools =
			//				storageDomainWithPools == null ?
			//				null :
			//				storageDomainWithPools.Where(a => a.status.HasValue && a.status.Value == StorageDomainStatus.Active).ToList();
			if (storageDomainWithPools != null)
			{
				List<storage_domains> list = new List<storage_domains>();

				foreach (storage_domains domain in storageDomainWithPools)
				{
					if (domain.status.HasValue && (domain.status.Value == StorageDomainStatus.Active))
					{
						list.Add(domain);
					}
				}
				storageDomainWithPools = list;
			}

			if (storageDomainWithPools != null && storageDomainWithPools.Count > 0)
			{
				// build search query according to storage pool names:
				StringBuilder sbSearch = new StringBuilder();
				sbSearch.Append(
					string.Format(
						"Host: status=up and datacenter={0}",
						storageDomainWithPools[0].storage_pool_name));

				for (int i = 1; i < storageDomainWithPools.Count; i++)
				{
					sbSearch.Append(string.Format(" or datacenter={0}", storageDomainWithPools[i].storage_pool_name));
				}

				returnValue = Frontend.RunQuery(
					VdcQueryType.Search,
					new SearchParameters(sbSearch.ToString(), SearchType.VDS));

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{

					//					return ((List<IVdcQueryable>)returnValue.ReturnValue).Cast<VDS>().Where(a => a.spm_status == VdsSpmStatus.SPM).ToList();
					List<VDS> list = new List<VDS>();
					foreach (IVdcQueryable a in (List<IVdcQueryable>)returnValue.ReturnValue)
					{
						VDS vds = (VDS)a;
						if (vds.spm_status == VdsSpmStatus.SPM)
						{
							list.Add(vds);
						}
					}
					return list;
				}
			}

			return new List<VDS>();
		}

		public static List<VDS> GetUpHostListByCluster(string cluster)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Host: cluster = " + cluster + " and status = up", SearchType.VDS));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return Linq.Cast<VDS>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<VDS>();
		}

		public static List<VDS> GetHostListByDataCenter(string dataCenterName)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Host: datacenter = " + dataCenterName + " sortby name", SearchType.VDS));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return Linq.Cast<VDS>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<VDS>();
		}

		public static List<VDS> GetHostListByCluster(string cluster)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Host: cluster = " + cluster + " sortby name", SearchType.VDS));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//var list = ((List<IVdcQueryable>)returnValue.ReturnValue)
				//    .Cast<VDS>()
				//    .ToList();
				List<VDS> list = Linq.Cast<VDS>((List<IVdcQueryable>)returnValue.ReturnValue);
				return list;
			}

			return new List<VDS>();
		}

		public static VDS GetHostById(Guid hostId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVdsByVdsId,
				new GetVdsByVdsIdParameters(hostId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (VDS)returnValue.ReturnValue;
			}

			return null;
		}

		public static List<VDS> GetHostList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Host:", SearchType.VDS)
				{
					MaxCount = SearchLimit
				});

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<IVdcQueryable>)returnValue.ReturnValue).Cast<VDS>().ToList();
				return Linq.Cast<VDS>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<VDS>();
		}

		public static List<VM> GetServerList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Vms: status=up sortby cpu_usage desc", SearchType.VM));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<IVdcQueryable>)returnValue.ReturnValue).Cast<VM>().ToList();
				return Linq.Cast<VM>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<VM>();
		}

		//Assumption that we have only 1 data/export storage domain per pool otherwise review the code
		public static List<storage_domains> GetStorageDomainList(Guid storagePoolId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStorageDomainsByStoragePoolId,
				new StoragePoolQueryParametersBase(storagePoolId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<storage_domains>)returnValue.ReturnValue;
			}

			return new List<storage_domains>();
		}

		public static List<storage_pool> GetDataCentersByStorageDomain(Guid storageDomainId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStoragePoolsByStorageDomainId,
				new StorageDomainQueryParametersBase(storageDomainId));


			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null && ((List<storage_pool>)returnValue.ReturnValue).Count > 0)
			{
				return (List<storage_pool>)(returnValue.ReturnValue);
			}

			return new List<storage_pool>();
		}

		public static storage_pool GetFirstStoragePoolByStorageDomain(Guid storageDomainId)
		{
			List<storage_pool> storagePools = GetDataCentersByStorageDomain(storageDomainId);
			return storagePools.Count > 0 ? storagePools[0] : null;
		}

		public static List<storage_domains> GetDataDomainsListByDomain(Guid storageDomainId)
		{
			storage_pool pool = GetFirstStoragePoolByStorageDomain(storageDomainId);
			if (pool != null)
			{
				//				return GetStorageDomainList(pool.id).Where(a => a.id != storageDomainId).ToList();
				List<storage_domains> list = new List<storage_domains>();
				foreach (storage_domains domain in GetStorageDomainList(pool.Id))
				{
					if (!domain.id.Equals(storageDomainId))
					{
						list.Add(domain);
					}
				}
				return list;
			}

			return new List<storage_domains>();
		}

		public static List<VDSGroup> GetClusterListByStorageDomain(Guid storage_domain_id)
		{

			storage_pool pool = GetFirstStoragePoolByStorageDomain(storage_domain_id);
			if (pool != null)
			{
				return GetClusterList(pool.Id);
			}

			return new List<VDSGroup>();
		}

		public static storage_domains GetStorageDomainById(Guid storageDomainId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStorageDomainById,
				new StorageDomainQueryParametersBase(storageDomainId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (storage_domains)(returnValue.ReturnValue);
			}

			return null;
		}

		public static storage_domains GetStorageDomainByDiskList(IEnumerable disksList)
		{
			List<DiskImage> diskImageList =
				disksList == null ? new List<DiskImage>() : Linq.Cast<DiskImage>(disksList);

			if (diskImageList.Count > 0)
			{
				DiskImage firstDisk = diskImageList[0];
				return GetStorageDomainByDisk(firstDisk);
			}

			return null;
		}

		public static storage_domains GetStorageDomainByDisk(DiskImage disk)
		{
			if (disk != null)
			{
				Guid selectedStorageDomainId = (disk.storage_id != null) ? disk.storage_id.Value : Guid.Empty;
				if (!selectedStorageDomainId.Equals(Guid.Empty))
				{
					storage_domains selectedStorageDomain =
						GetStorageDomainById(selectedStorageDomainId);

					if (selectedStorageDomain != null)
					{
						return selectedStorageDomain;
					}
				}
			}

			return null;
		}

		public static List<storage_domains> GetStorageDomainList()
		{
			SearchParameters searchParameters = new SearchParameters("Storage:", SearchType.StorageDomain);
			searchParameters.MaxCount = SearchLimit;
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				searchParameters);

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<IVdcQueryable>)returnValue.ReturnValue).Cast<storage_domains>().ToList();
				return Linq.Cast<storage_domains>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<storage_domains>();
		}

		public static List<storage_domains> GetISOStorageDomainList()
		{
			SearchParameters searchParams = new SearchParameters("Storage:", SearchType.StorageDomain);
			searchParams.MaxCount = 9999;

			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search, searchParams);

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				// filter only the ISO storage domains into the return value:
				List<storage_domains> allStorageDomains = (List<storage_domains>)returnValue.ReturnValue;
				List<storage_domains> isoStorageDomains = new List<storage_domains>();
				foreach (storage_domains storageDomain in allStorageDomains)
				{
					if (storageDomain.storage_domain_type == StorageDomainType.ISO)
					{
						isoStorageDomains.Add(storageDomain);
					}
				}

				return isoStorageDomains;
			}

			return new List<storage_domains>();
		}

		public static bool IsSSLEnabled()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.SSLEnabled));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (bool)returnValue.ReturnValue;
			}

			return false;
		}

		public static bool EnableSpiceRootCertificateValidation()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.EnableSpiceRootCertificateValidation));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (bool)returnValue.ReturnValue;
			}

			return false;
		}

		public static string GetHostCertSubjectByHostID(Guid hostID)
		{
			VdcQueryReturnValue returnValue =
				Frontend.RunQuery(
					VdcQueryType.GetVdsCertificateSubjectByVdsId,
					new GetVdsByVdsIdParameters(hostID));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return null;
		}

		public static string GetCACertificate()
		{
			VdcQueryReturnValue returnValue =
				Frontend.RunQuery(
					VdcQueryType.GetCACertificate,
					new VdcQueryParametersBase());

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return null;
		}

		public static string GetSpiceSecureChannelList()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.SpiceSecureChannels));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return String.Empty;
		}

		public static string GetCipherSuite()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.CipherSuite));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return String.Empty;
		}

		public static bool IsUSBEnabledByDefault()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.EnableUSBAsDefault));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (bool)returnValue.ReturnValue;
			}

			return false;
		}

		public static string GetSpiceToggleFullScreenKeys()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.SpiceToggleFullScreenKeys));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return "shift+f11";
		}

		public static string GetSpiceReleaseCursorKeys()
		{
			VdcQueryReturnValue returnValue = DataProvider.GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.SpiceReleaseCursorKeys));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (string)returnValue.ReturnValue;
			}

			return "shift+f12";
		}

		public static List<ActionGroup> GetRoleActionGroupsByRoleId(Guid roleId)
		{
			// TODO getRoleActionGroupsByRoleId instead
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetRoleActionGroupsByRoleId,
				new MultilevelAdministrationByRoleIdParameters(roleId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<ActionGroup>)returnValue.ReturnValue;
			}
			return new List<ActionGroup>();
		}

		public static bool IsLicenseValid(out IList<string> reasons)
		{
			//LicenseReturnValue returnValue = null;
			//try
			//{
			//    returnValue = (LicenseReturnValue)Frontend.RunPublicQuery(VdcQueryType.IsLicenseValid,
			//         new VdcQueryParametersBase());
			//}
			//catch
			//{
			//}

			//if (returnValue != null && returnValue.Succeeded && returnValue.Messages.Count == 0)
			//{
			//    reasons = new List<string>();
			//    return true;
			//}

			reasons = null; // returnValue.Messages;
			return true;
		}

		//public static IDictionary<string, string> GetLicenseProperties()
		//{
		//    var returnValue = Frontend.RunPublicQuery(VdcQueryType.GetLicenseProperties,
		//        new VdcQueryParametersBase());

		//    if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
		//    {
		//        return (IDictionary<string, string>)returnValue.ReturnValue;
		//    }

		//    return new Dictionary<string, string>();
		//}

		public static bool IsLicenseHasDesktops()
		{
			//var dict = GetLicenseProperties();
			//return dict.ContainsKey("ProductTypeProperty") && dict["ProductTypeProperty"].Contains("Desktop");
			return true;
		}

		public static int GetClusterDefaultCpuOverCommit()
		{
			VdcQueryReturnValue returnValue = DataProvider.GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.CpuOverCommitDurationMinutes));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 0;
		}

		public static VdsSelectionAlgorithm GetDefaultVdsSelectionAlgorithm()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VdsSelectionAlgorithm));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (VdsSelectionAlgorithm)Enum.Parse(typeof(VdsSelectionAlgorithm), (string)returnValue.ReturnValue);
			}

			return VdsSelectionAlgorithm.None;
		}

		public static int GetHighUtilizationForEvenDistribution()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.HighUtilizationForEvenlyDistribute));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 0;
		}

		public static int GetHighUtilizationForPowerSave()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.HighUtilizationForPowerSave));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 0;
		}

		public static int GetLowUtilizationForPowerSave()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.LowUtilizationForPowerSave));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 0;
		}

		public static int GetVcpuConsumptionPercentage()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VcpuConsumptionPercentage));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 10;
		}

		public static List<DiskImageBase> GetDiskPresetList(
			VmType vmType,
			StorageType storageType)
		{
			// Get basic preset list from backend:
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetDiskConfigurationList,
				new VdcQueryParametersBase());

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//List<DiskImageBase> presetList =
				//	((List<DiskImageBase>)returnValue.ReturnValue)
				//	.Where(a => a.disk_type == DiskType.System || a.disk_type == DiskType.Data)
				//	.ToList();
				List<DiskImageBase> list = new List<DiskImageBase>();
				DiskImageBase presetData = null;
				DiskImageBase presetSystem = null;
				foreach (DiskImageBase disk in (List<DiskImageBase>)returnValue.ReturnValue)
				{
					if (disk.disk_type == DiskType.System || disk.disk_type == DiskType.Data)
					{
						list.Add(disk);
					}
					if (disk.disk_type == DiskType.System && presetSystem == null)
					{
						presetSystem = disk;
					}
					else if (disk.disk_type == DiskType.Data && presetData == null)
					{
						presetData = disk;
					}
				}
				List<DiskImageBase> presetList = list;

				//DiskImageBase presetData = presetList.FirstOrDefault(a => a.disk_type == DiskType.Data);
				//DiskImageBase presetSystem = presetList.FirstOrDefault(a => a.disk_type == DiskType.System);

				// Set default volume type by vm type:
				// Set volume format by storage type and volume type:
				if (presetData != null)
				{
					presetData.volume_type = VolumeType.Preallocated;
					presetData.volume_format = GetDiskVolumeFormat(presetData.volume_type, storageType);
				}
				if (presetSystem != null)
				{
					presetSystem.volume_type = vmType == VmType.Server ? VolumeType.Preallocated : VolumeType.Sparse;
					presetSystem.volume_format = GetDiskVolumeFormat(presetSystem.volume_type, storageType);
				}

				return presetList;
			}

			return new List<DiskImageBase>();
		}

		public static VolumeFormat GetDiskVolumeFormat(VolumeType volumeType, StorageType storageType)
		{
			switch (storageType)
			{
				case StorageType.NFS:
					return VolumeFormat.RAW;

				case StorageType.ISCSI:
				case StorageType.FCP:
					switch (volumeType)
					{
						case VolumeType.Sparse:
							return VolumeFormat.COW;

						case VolumeType.Preallocated:
							return VolumeFormat.RAW;

						default:
							return VolumeFormat.Unassigned;
					}

				case StorageType.LOCALFS:
					return VolumeFormat.RAW;

				default:
					return VolumeFormat.Unassigned;
			}
		}

		public static List<VolumeType> GetVolumeTypeList()
		{
			return
				new List<VolumeType>
				{
					VolumeType.Preallocated,
					VolumeType.Sparse
				};
		}

		public static List<DiskInterface> GetDiskInterfaceList(VmOsType osType, Version Version)
		{
			return osType == VmOsType.WindowsXP && (Version == null || Version.CompareTo(new Version("2.2")) < 0)
				? new List<DiskInterface> { DiskInterface.IDE }
				: new List<DiskInterface> { DiskInterface.IDE, DiskInterface.VirtIO };
		}

		public static DiskInterface GetDefaultDiskInterface(VmOsType osType, IList<DiskImage> disks)
		{
			return

				osType == VmOsType.WindowsXP ?
				DiskInterface.IDE :
					disks != null && disks.Count > 0 ?
					disks[0].disk_interface
					:
				DiskInterface.VirtIO;
		}

		public static VolumeFormat[] GetVolumeFormatList()
		{
			return new VolumeFormat[] { VolumeFormat.COW, VolumeFormat.RAW };
		}

		public static List<VdsNetworkInterface> GetFreeBondList(Guid hostId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVdsFreeBondsByVdsId,
				new GetVdsByVdsIdParameters(hostId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<VdsNetworkInterface>)returnValue.ReturnValue;
			}

			return new List<VdsNetworkInterface>();
		}

		public static List<RpmVersion> GetoVirtISOsList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetoVirtISOs,
				new VdcQueryParametersBase());

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<RpmVersion>)returnValue.ReturnValue;
			}

			return new List<RpmVersion>();
		}

		public static bool IsDataCenterNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.IsStoragePoolWithSameNameExist,
				new IsStoragePoolWithSameNameExistParameters(name));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return !(bool)returnValue.ReturnValue;
			}

			return true;
		}

		public static bool IsStorageDomainNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters(String.Format("Storage: name={0}", name), SearchType.StorageDomain));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				List<storage_domains> list = (List<storage_domains>)returnValue.ReturnValue;
				return list.Count == 0;
			}
			return true;
		}

		public static bool IsClusterNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.IsVdsGroupWithSameNameExist,
				new IsVdsGroupWithSameNameExistParameters(name));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return !(bool)returnValue.ReturnValue;
			}

			return true;
		}

		public static bool IsHostNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.IsVdsWithSameNameExist,
				new IsVdsWithSameNameExistParameters(name));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return !(bool)returnValue.ReturnValue;
			}

			return true;
		}

		public static bool IsVmNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.IsVmWithSameNameExist,
				new IsVmWithSameNameExistParameters(name));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return !(bool)returnValue.ReturnValue;
			}

			return true;
		}

		public static bool IsTemplateNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.IsVmTemlateWithSameNameExist,
				new IsVmTemlateWithSameNameExistParameters(name));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return !(bool)returnValue.ReturnValue;
			}

			return false;
		}

		public static bool IsPoolNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.IsVmPoolWithSameNameExists,
				new IsVmPoolWithSameNameExistsParameters(name));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return !(bool)returnValue.ReturnValue;
			}

			return false;
		}

		public static bool IsRoleNameUnique(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllRoles,
				new MultilevelAdministrationsQueriesParameters());

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//return ((List<roles>)returnValue.ReturnValue).FirstOrDefault(a => String.Compare(a.name, name, true) == 0) == null;
				foreach (roles role in (List<roles>)returnValue.ReturnValue)
				{
					if (String.Compare(role.name, name, true) == 0)
					{
						return false;
					}
				}
				return true;
			}

			return false;
		}

		public static List<string> GetPmTypeList(Version ClusterVersion)
		{
			List<string> list = new List<string>();

			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VdsFenceType)
				{
					Version = ClusterVersion != null ? ClusterVersion.ToString() : null
				});

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				string[] array = ((string)returnValue.ReturnValue).Split(',');
				foreach (string item in array)
				{
					list.Add(item);
				}
			}

			return list;
		}

		private static Dictionary<string, List<string>> cachedPmMap;

		public static List<string> GetPmOptions(string pmType)
		{
			if (cachedPmMap == null)
			{
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAgentFenceOptions,
					new VdcQueryParametersBase());

				if (returnValue != null && returnValue.Succeeded)
				{
					cachedPmMap = new Dictionary<string, List<string>>();

					Dictionary<string, Dictionary<string, object>> dict = (Dictionary<string, Dictionary<string, object>>)returnValue.ReturnValue;
					foreach (KeyValuePair<string, Dictionary<string, object>> pair in dict)
					{
						List<string> list = new List<string>();
						foreach (KeyValuePair<string, object> p in pair.Value)
						{
							list.Add(p.Key);
						}

						cachedPmMap.Add(pair.Key, list);
					}
				}
			}

			return cachedPmMap[pmType];
		}

		public static List<DiskImage> GetVmDiskList(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllDisksByVmId,
				new GetAllDisksByVmIdParameters(id));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<DiskImage>)returnValue.ReturnValue;
			}

			return new List<DiskImage>();
		}

		public static List<DiskImage> GetTemplateDiskList(Guid templateId)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmTemplatesDisks,
				new GetVmTemplatesDisksParameters(templateId));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<DiskImage>)returnValue.ReturnValue;
			}

			return new List<DiskImage>();
		}

		public static List<VmNetworkInterface> GetVmNicList(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmInterfacesByVmId,
				new GetVmByVmIdParameters(id));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<VmNetworkInterface>)returnValue.ReturnValue;
			}

			return new List<VmNetworkInterface>();
		}

		public static List<DiskImage> GetSnapshotList(Guid id, string drive, out Guid previewingImageId)
		{
			GetAllVmSnapshotsByDriveQueryReturnValue returnValue = null;
			try
			{
				returnValue = (GetAllVmSnapshotsByDriveQueryReturnValue)Frontend.RunQuery(VdcQueryType.GetAllVmSnapshotsByDrive,
				 new GetAllVmSnapshotsByDriveParameters(id, drive));
			}
			catch
			{
			}

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				previewingImageId = returnValue.TryingImage;
				return (List<DiskImage>)returnValue.ReturnValue;
			}

			previewingImageId = Guid.Empty;
			return new List<DiskImage>();
		}

		public static List<string> GetFloppyImageList(Guid dataCenterId, bool forceRefresh)
		{
			storage_domains isoDomain = GetIsoDomainByDataCenterId(dataCenterId);

			if (isoDomain != null)
			{
				GetAllIsoImagesListParameters parameters = new GetAllIsoImagesListParameters();
				parameters.StorageDomainId = isoDomain.id;
				parameters.ForceRefresh = forceRefresh;

				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllFloppyImagesList, parameters);

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					List<RepoFileMetaData> listRepoFileList = (List<RepoFileMetaData>)returnValue.ReturnValue;
					List<string> fileNameList = new List<string>();
					foreach (RepoFileMetaData RepoFileMetaData in listRepoFileList)
					{
						fileNameList.Add(RepoFileMetaData.RepoFileName);
					}

					fileNameList.Sort();
					return fileNameList;
				}
			}

			return new List<string>();
		}


		public static storage_server_connections GetStorageConnectionById(string id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStorageServerConnectionById,
				new StorageServerConnectionQueryParametersBase(id));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (storage_server_connections)returnValue.ReturnValue;
			}

			return null;
		}

		public static int GetDefaultPoolLeasePeriod()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VmPoolLeaseDays));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 0;
		}

		public static DateTime GetDefaultPoolLeaseStartTime()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VmPoolLeaseStartTime));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return DateTime.Parse((string)returnValue.ReturnValue);
			}

			return DateTime.Now;
		}

		public static DateTime GetDefaultPoolLeaseEndTime()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VmPoolLeaseEndTime));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return DateTime.Parse((string)returnValue.ReturnValue);
			}

			return DateTime.Now;
		}

		public static List<DbUser> GetUserList()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("User:", SearchType.DBUser));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//				return ((List<IVdcQueryable>)returnValue.ReturnValue).Cast<DbUser>().ToList();
				return Linq.Cast<DbUser>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<DbUser>();
		}

		public static VM GetVmById(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmByVmId,
				new GetVmByVmIdParameters(id));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (VM)returnValue.ReturnValue;
			}

			return null;
		}

		//NOTE: Moved to AsyncDataProvider.
		//public static vm_pools GetPoolById(Guid poolId)
		//{
		//    VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmPoolById,
		//        new GetVmPoolByIdParameters(poolId));

		//    if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
		//    {
		//        return (vm_pools)returnValue.ReturnValue;
		//    }

		//    return null;
		//}


		public static VDSGroup GetClusterById(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVdsGroupById,
				new GetVdsGroupByIdParameters(id));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (VDSGroup)returnValue.ReturnValue;
			}

			return null;
		}

		public static VDSGroup GetClusterByName(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters(String.Format("Cluster: name={0}", name), SearchType.Cluster));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return Linq.FirstOrDefault(Linq.Cast<VDSGroup>((List<IVdcQueryable>)returnValue.ReturnValue));
			}

			return null;
		}

		public static storage_pool GetDataCenterById(Guid id)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStoragePoolById,
				new StoragePoolQueryParametersBase(id));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (storage_pool)returnValue.ReturnValue;
			}

			return null;
		}

		public static storage_pool GetDataCenterByName(string name)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters(String.Format("DataCenter: name={0}", name), SearchType.StoragePool));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return Linq.FirstOrDefault(Linq.Cast<storage_pool>((List<IVdcQueryable>)returnValue.ReturnValue));
			}

			return null;
		}

		public static List<VM> GetVmList(string poolName)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters("Vms: pool=" + poolName, SearchType.VM));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				//return ((List<IVdcQueryable>)returnValue.ReturnValue).Cast<VM>().ToList();
				return Linq.Cast<VM>((List<IVdcQueryable>)returnValue.ReturnValue);
			}

			return new List<VM>();
		}

		public static VM GetAnyVm(string poolName)
		{
			List<VM> vms = GetVmList(poolName);
			return vms.Count > 0 ? vms[0] : null;
		}

		private static int _cachedSearchResultsLimit = -1;
		public static int GetSearchResultsLimit()
		{
			if (_cachedSearchResultsLimit == -1)
			{
				VdcQueryReturnValue returnValue = GetConfigFromCache(
					new GetConfigurationValueParameters(ConfigurationValues.SearchResultsLimit));

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					_cachedSearchResultsLimit = (int)returnValue.ReturnValue;
				}
				else
				{
					_cachedSearchResultsLimit = 0;
				}
			}

			return _cachedSearchResultsLimit;
		}

		public static List<VdsNetworkInterface> GetInterfaceOptionsForEditNetwork(
			VdsNetworkInterface originalInterface,
			network networkToEdit,
			Guid vdsID,
			out string defaultInterfaceName)
		{
			List<VdsNetworkInterface> interfaceList = GetAllHostInterfaces(vdsID);
			List<VdsNetworkInterface> ifacesOptions = new List<VdsNetworkInterface>();
			foreach (VdsNetworkInterface i in interfaceList)
			{
				if (string.IsNullOrEmpty(i.NetworkName) && string.IsNullOrEmpty(i.BondName))
				{
					ifacesOptions.Add(i);
				}
			}

			//			List<Interface> ifacesOptions =

			//			interfaceList
			// Filter only Interfaces that are not connected to a network:
			//				.Where(a => string.IsNullOrEmpty(a.network_name))

			// Filter only Interfaces that are not slaves of a bond:
			//				.Where(a => string.IsNullOrEmpty(a.bond_name))

			//				.ToList();

			if (!originalInterface.VlanId.HasValue) // no vlan:
			{
				// Filter out the Interfaces that have child vlan Interfaces:
				//				ifacesOptions.RemoveAll(
				//					delegate(Interface i)
				//					{
				//						return InterfaceHasChildVlanInterfaces(vdsID, i);
				//					});
				List<VdsNetworkInterface> ifacesOptionsTemp = new List<VdsNetworkInterface>();
				foreach (VdsNetworkInterface i in ifacesOptions)
				{
					if (!InterfaceHasChildVlanInterfaces(vdsID, i))
					{
						ifacesOptionsTemp.Add(i);
					}
				}
				ifacesOptions = ifacesOptionsTemp;

				if (originalInterface.Bonded.HasValue && originalInterface.Bonded.Value)
				{
					// eth0 -- \
					//          |---> bond0 -> <networkToEdit>
					// eth1 -- /
					// ---------------------------------------
					// - originalInterface: 'bond0'
					// --> We want to add 'eth0' and and 'eth1' as optional Interfaces
					// (note that choosing one of them will break the bond):
					//					ifacesOptions.AddRange(interfaceList.Where(a => a.bond_name == originalInterface.name).ToList());
					foreach (VdsNetworkInterface i in interfaceList)
					{
						if (i.BondName == originalInterface.Name)
						{
							ifacesOptions.Add(i);
						}
					}
				}

				// add the original interface as an option and set it as the default option:
				ifacesOptions.Add(originalInterface);
				defaultInterfaceName = originalInterface.Name;
			}

			else // vlan:
			{
				VdsNetworkInterface vlanParent = GetVlanParentInterface(vdsID, originalInterface);
				if (vlanParent != null &&
					vlanParent.Bonded.HasValue &&
					vlanParent.Bonded.Value &&
					!InterfaceHasSiblingVlanInterfaces(vdsID, originalInterface))
				{
					// eth0 -- \
					//          |--- bond0 ---> bond0.3 -> <networkToEdit>
					// eth1 -- /
					// ---------------------------------------------------
					// - originalInterface: 'bond0.3'
					// - vlanParent: 'bond0'
					// - 'bond0.3' has no vlan siblings
					// --> We want to add 'eth0' and and 'eth1' as optional Interfaces.
					// (note that choosing one of them will break the bond):
					//					ifacesOptions.AddRange(interfaceList.Where(a => a.bond_name == vlanParent.name).ToList());
					foreach (VdsNetworkInterface i in interfaceList)
					{
						if (i.BondName == vlanParent.Name)
						{
							ifacesOptions.Add(i);
						}
					}
				}

				// the vlanParent should already be in ifacesOptions
				// (since it has no network_name or bond_name).

				defaultInterfaceName = vlanParent.Name;
			}

			return ifacesOptions;
		}

		private static List<VdsNetworkInterface> GetAllHostInterfaces(Guid vdsID)
		{
			List<VdsNetworkInterface> interfaceList = new List<VdsNetworkInterface>();

			VdcQueryReturnValue retValue =
				Frontend.RunQuery(
					VdcQueryType.GetVdsInterfacesByVdsId,
					new GetVdsByVdsIdParameters(vdsID));

			if (retValue != null && retValue.Succeeded)
			{
				interfaceList = (List<VdsNetworkInterface>)retValue.ReturnValue;
			}

			return interfaceList;
		}

		private static VdsNetworkInterface GetVlanParentInterface(Guid vdsID, VdsNetworkInterface iface)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVlanParanet,
				new GetAllChildVlanInterfacesQueryParameters(vdsID, iface));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (VdsNetworkInterface)returnValue.ReturnValue;
			}

			return null;
		}

		private static bool InterfaceHasChildVlanInterfaces(Guid vdsID, VdsNetworkInterface iface)
		{
			List<VdsNetworkInterface> childVlanInterfaces = new List<VdsNetworkInterface>();
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllChildVlanInterfaces,
				new GetAllChildVlanInterfacesQueryParameters(vdsID, iface));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				childVlanInterfaces = (List<VdsNetworkInterface>)(returnValue.ReturnValue);
			}

			if (childVlanInterfaces.Count > 0)
			{
				return true;
			}

			return false;
		}

		private static bool InterfaceHasSiblingVlanInterfaces(Guid vdsID, VdsNetworkInterface iface)
		{
			List<VdsNetworkInterface> siblingVlanInterfaces = new List<VdsNetworkInterface>();
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetAllSiblingVlanInterfaces,
				new GetAllChildVlanInterfacesQueryParameters(vdsID, iface));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				siblingVlanInterfaces = (List<VdsNetworkInterface>)returnValue.ReturnValue;
			}

			if (siblingVlanInterfaces.Count > 0)
			{
				return true;
			}

			return false;
		}

		public static Dictionary<string, int> GetSystemStatistics()
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetSystemStatistics,
				new GetSystemStatisticsQueryParameters(-1));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (Dictionary<string, int>)returnValue.ReturnValue;
			}

			return new Dictionary<string, int>();
		}

		public static int GetDiskMaxSize()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.MaxDiskSize));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 2047;
		}

		public static int GetMaxVmsInPool()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.MaxVmsInPool));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 1000;
		}

		public static List<VM> GetUserVmList(Guid userId, string groupNames)
		{
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetUserVmsByUserIdAndGroups,
				new GetUserVmsByUserIdAndGroupsParameters(/*user.UserId, user.GroupNames*/));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (List<VM>)returnValue.ReturnValue;
			}

			return new List<VM>();
		}

		internal static string GetNewNicName(List<VmNetworkInterface> existingInterfaces)
		{
			int maxIfaceNumber = 0;
			if (existingInterfaces != null)
			{
				foreach (VmNetworkInterface iface in existingInterfaces)
				{
					int ifaceNumber;
					// name of Interface is "eth<n>" (<n>: integer).
					if (iface.Name.Length > 3 && int.TryParse(iface.Name.Substring(3), out ifaceNumber))
					{
						if (ifaceNumber > maxIfaceNumber)
						{
							maxIfaceNumber = ifaceNumber;
						}
					}
				}
			}

			return "nic" + (maxIfaceNumber + 1);
		}

		private static Dictionary<string, ResourceManager> _resourcesCache = new Dictionary<string, ResourceManager>();

		public static string GetValueFromResource(string resourcePath, string key)
		{
			if (string.IsNullOrEmpty(resourcePath) || string.IsNullOrEmpty(key))
			{
				return string.Empty;
			}

			lock (_resourcesCache)
			{
				if (!_resourcesCache.ContainsKey(resourcePath))
				{
					_resourcesCache.Add(resourcePath, new ResourceManager(resourcePath, Assembly.GetExecutingAssembly()));
				}
			}

			if (_resourcesCache.ContainsKey(resourcePath))
			{
				try
				{
					return _resourcesCache[resourcePath].GetString(key, CultureInfo.CurrentCulture);
				}
				catch { }
			}

			return key;
		}

		public static string GetValueFromSpiceRedKeysResource(string key)
		{
			return GetValueFromResource("UICommon.Resources.SpiceRedKeys.SpiceRedKeys", key);
		}

		/// <summary>
		/// Gets a value composed of "[string1]+[string2]+..." and returns "[string1Translated]+[string2Translated]+..."
		/// </summary>
		/// <param name="complexValue">string in the form of "[string1]+[string2]+..."</param>
		/// <returns>string in the form of "[string1Translated]+[string2Translated]+..."</returns>
		public static string GetComplexValueFromSpiceRedKeysResource(string complexValue)
		{
			if (string.IsNullOrEmpty(complexValue))
			{
				return string.Empty;
			}
			List<string> values = new List<string>();

			foreach (string s in complexValue.Split(new char[] { '+' }))
			{
				values.Add(GetValueFromSpiceRedKeysResource(s));
			}
			return string.Join("+", values.ToArray());
			//			return string.Join("+", complexValue.Split(new char[] { '+' }).Select(a => GetValueFromSpiceRedKeysResource(a)).ToArray());
		}

		public static int GetMaxVmPriority()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.VmPriorityMaxValue));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				return (int)returnValue.ReturnValue;
			}

			return 100;
		}

		public static int RoundPriority(int priority)
		{
			int max = GetMaxVmPriority();
			int medium = max / 2;

			int[] levels = new[] { 1, medium, max };

			for (int i = 0; i < levels.Length; i++)
			{
				int lengthToLess = levels[i] - priority;
				int lengthToMore = levels[i + 1] - priority;

				if (lengthToMore < 0)
					continue;

				return Math.Abs(lengthToLess) < lengthToMore
					? levels[i]
					: levels[i + 1];
			}


			return 0;
		}

		private static int cacheDataCenterMaxNameLength = -1;

		internal static int GetDataCenterMaxNameLength()
		{
			if (cacheDataCenterMaxNameLength == -1) // cache not initialized -> call Backend:
			{
				VdcQueryReturnValue returnValue = GetConfigFromCache(
					new GetConfigurationValueParameters(ConfigurationValues.StoragePoolNameSizeLimit));

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					cacheDataCenterMaxNameLength = (int)returnValue.ReturnValue;
				}
				else
				{
					cacheDataCenterMaxNameLength = 1;
				}
			}

			return cacheDataCenterMaxNameLength;
		}

		private static int cacheStorageDomainMaxNameLength = -1;

		internal static int GetStorageDomainMaxNameLength()
		{
			if (cacheStorageDomainMaxNameLength == -1) // cache not initialized -> call Backend:
			{
				VdcQueryReturnValue returnValue = DataProvider.GetConfigFromCache(
					new GetConfigurationValueParameters(ConfigurationValues.StorageDomainNameSizeLimit));

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					cacheStorageDomainMaxNameLength = (int)returnValue.ReturnValue;
				}
				else
				{
					cacheStorageDomainMaxNameLength = 1;
				}
			}

			return cacheStorageDomainMaxNameLength;
		}

		// dictionary to hold cache of all config values (per version) queried by client, if the request for them succeeded.
		private static Dictionary<KeyValuePair<ConfigurationValues, string>, VdcQueryReturnValue> CachedConfigValues = new Dictionary<KeyValuePair<ConfigurationValues, string>, VdcQueryReturnValue>();

		// helper method to clear the config cache (currently used on each login)
		public static void ClearConfigCache()
		{
			if (CachedConfigValues != null)
			{
				CachedConfigValues.Clear();
			}

			userActionGroups = null;
			cacheCustomProperties = null;
			windowsOsTypes = null;
			linuxOsTypes = null;
			x64OsTypes = null;
			hasAdminSystemPermission = null;
		}

		// method to get an item from config while caching it (config is not supposed to change during a session)
		public static VdcQueryReturnValue GetConfigFromCache(GetConfigurationValueParameters parameters)
		{
			KeyValuePair<ConfigurationValues, string> config_key = new KeyValuePair<ConfigurationValues, string>(parameters.ConfigValue, parameters.Version);

			// populate cache if not in cache already
			if (!CachedConfigValues.ContainsKey(config_key))
			{

				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetConfigurationValue,
					parameters);

				// only put result in cache if query succeeded
				if (returnValue != null && returnValue.Succeeded)
				{
					CachedConfigValues[config_key] = returnValue;
				}
				// return actual return value on error
				else
				{
					return returnValue;
				}
			}
			// return value from cache (either it was in, or the query succeeded, and it is now in the cache
			return CachedConfigValues[config_key];
		}

		private static bool GetSANWipeAfterDelete()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.SANWipeAfterDelete));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				// use bool? to force conversion to Boolean in java (ReturnValue is Object in java code)b
				return ((bool?)returnValue.ReturnValue).Value;
			}
			return false;
		}

		public static Guid GetEntityGuid(object entity)
		{
			if (entity is VM)
			{
				return ((VM)entity).vm_guid;
			}
			else if (entity is storage_pool)
			{
				return ((storage_pool)entity).Id;
			}
			else if (entity is VDSGroup)
			{
				return ((VDSGroup)entity).ID;
			}
			else if (entity is VDS)
			{
				return ((VDS)entity).vds_id;
			}
			else if (entity is storage_domains)
			{
				return ((storage_domains)entity).id;
			}
			else if (entity is VmTemplate)
			{
				return ((VmTemplate)entity).Id;
			}
			else if (entity is vm_pools)
			{
				return ((vm_pools)entity).vm_pool_id;
			}
			else if (entity is DbUser)
			{
				return ((DbUser)entity).user_id;
			}
			return new Guid();
		}

		private static List<ActionGroup> userActionGroups = null;
		public static List<ActionGroup> GetUserActionGroups()
		{
			if (userActionGroups != null)
			{
				return userActionGroups;
			}
			UIQueryReturnValue ret = Frontend.RunUIQuery(UIQueryType.GetUserActionGroups, new UIQueryParametersBase());
			if (ret.Succeeded && ret.ReturnValue != null)
			{
				userActionGroups = (List<ActionGroup>)ret.ReturnValue;
				return userActionGroups;
			}

			return new List<ActionGroup>();
		}

		private static Dictionary<string, string> cacheCustomProperties;

		/// <summary>
		/// Gets a dictionary in which the keys are the valid custom property keys allowed
		/// and the values are the valid RegExp to validate the custom property values with.
		/// </summary>
		/// <returns>dictionary of valid keys and valid values' RegExps.</returns>
		public static Dictionary<string, string> GetCustomPropertiesList()
		{
			if (cacheCustomProperties != null)
			{
				return cacheCustomProperties;
			}
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetVmCustomProperties, new VdcQueryParametersBase());
			if (returnValue.Succeeded && returnValue.ReturnValue != null && ((string)returnValue.ReturnValue).Equals(string.Empty) == false)
			{
				string temp = (string)returnValue.ReturnValue;
				string[] tempArray = temp.Split(';');
				cacheCustomProperties = new Dictionary<string, string>();
				foreach (string keyRegexpPair in tempArray)
				{
					string[] keyAndRegexp = keyRegexpPair.Split('=');
					string key = keyAndRegexp[0];
					string regexp = null;
					// if there is no "=", it means that there is no RegExp to
					// validate with, which means that all strings are valid.
					if (keyAndRegexp.Length > 1)
					{
						regexp = keyAndRegexp[1];
					}

					if (!cacheCustomProperties.ContainsKey(key))
					{
						cacheCustomProperties.Add(key, regexp);
					}
				}
				return cacheCustomProperties;
			}

			return new Dictionary<string, string>();
		}

		private static List<VmOsType> windowsOsTypes;
		private static List<VmOsType> linuxOsTypes;
		private static List<VmOsType> x64OsTypes;

		public static List<VmOsType> GetWindowsOsTypes()
		{
			if (windowsOsTypes != null)
			{
				return windowsOsTypes;
			}

			// TODO: Uncomment once the gwt is using generic api instead of backend!

			//UIQueryReturnValue ret = Frontend.RunUIQuery(UIQueryType.GetWindowsOsTypes, new UIQueryParametersBase());
			//if (ret.Succeeded && ret.ReturnValue != null)
			//{
			//    windowsOsTypes = (List<VmOsType>)ret.ReturnValue;
			//    return windowsOsTypes;
			//}

			//return new List<VmOsType>();

			/***** TODO: remove once the gwt is using generic api instead of backend! *****/
			windowsOsTypes = new List<VmOsType>()
                {
                    VmOsType.Windows2003,
                    VmOsType.Windows2003x64,
                    VmOsType.Windows2008,
                    VmOsType.Windows2008R2x64,
                    VmOsType.Windows2008x64,
                    VmOsType.Windows7,
                    VmOsType.Windows7x64,
                    VmOsType.WindowsXP
                };

			return windowsOsTypes;
			/*******************************************************************************/
		}

		public static bool IsWindowsOsType(VmOsType osType)
		{
			if (GetWindowsOsTypes().Contains(osType))
			{
				return true;
			}

			return false;
		}

		public static List<VmOsType> GetLinuxOsTypes()
		{
			if (linuxOsTypes != null)
			{
				return linuxOsTypes;
			}

			// TODO: Uncomment once the gwt is using generic api instead of backend!

			//UIQueryReturnValue ret = Frontend.RunUIQuery(UIQueryType.GetLinuxOsTypes, new UIQueryParametersBase());
			//if (ret.Succeeded && ret.ReturnValue != null)
			//{
			//    linuxOsTypes = (List<VmOsType>)ret.ReturnValue;
			//    return linuxOsTypes;
			//}

			//return new List<VmOsType>();

			/***** TODO: remove once the gwt is using generic api instead of backend! *****/
			linuxOsTypes = new List<VmOsType>()
                {
                    VmOsType.OtherLinux,
                    VmOsType.RHEL3,
                    VmOsType.RHEL3x64,
                    VmOsType.RHEL4,
                    VmOsType.RHEL4x64,
                    VmOsType.RHEL5,
                    VmOsType.RHEL5x64,
                    VmOsType.RHEL6,
                    VmOsType.RHEL6x64
                };

			return linuxOsTypes;
			/*******************************************************************************/
		}

		public static bool IsLinuxOsType(VmOsType osType)
		{
			if (GetLinuxOsTypes().Contains(osType))
			{
				return true;
			}

			return false;
		}

		public static List<VmOsType> Get64bitOsTypes()
		{
			if (x64OsTypes != null)
			{
				return x64OsTypes;
			}

			// TODO: Uncomment once the gwt is using generic api instead of backend!

			//UIQueryReturnValue ret = Frontend.RunUIQuery(UIQueryType.Get64bitOsTypes, new UIQueryParametersBase());
			//if (ret.Succeeded && ret.ReturnValue != null)
			//{
			//    x64OsTypes = (List<VmOsType>)ret.ReturnValue;
			//    return x64OsTypes;
			//}

			//return new List<VmOsType>();

			/***** TODO: remove once the gwt is using generic api instead of backend! *****/
			x64OsTypes = new List<VmOsType>()
            {
                VmOsType.RHEL3x64,
                VmOsType.RHEL4x64,
                VmOsType.RHEL5x64,
                VmOsType.RHEL6x64,
                VmOsType.Windows2003x64,
                VmOsType.Windows2008R2x64,
                VmOsType.Windows2008x64,
                VmOsType.Windows7x64
            };

			return x64OsTypes;
			/*******************************************************************************/
		}

		public static bool Is64bitOsType(VmOsType osType)
		{
			if (Get64bitOsTypes().Contains(osType))
			{
				return true;
			}

			return false;
		}

		private static bool? hasAdminSystemPermission = null;
		public static bool HasAdminSystemPermission()
		{
			if (hasAdminSystemPermission == null)
			{
				VdcUser vdcUser = Frontend.LoggedInUser;
				if (vdcUser == null)
				{
					hasAdminSystemPermission = false;
					return false;
				}
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetPermissionsByAdElementId,
																	new MultilevelAdministrationByAdElementIdParameters(
																		vdcUser.UserId));
				if (returnValue == null || !returnValue.Succeeded)
				{
					hasAdminSystemPermission = false;
					return false;
				}
				List<permissions> permissions = (List<permissions>)returnValue.ReturnValue;

				foreach (permissions permission in permissions)
				{
					if (permission.ObjectType == VdcObjectType.System && permission.RoleType == RoleType.ADMIN)
					{
						hasAdminSystemPermission = true;
						return true;
					}
				}
				if (hasAdminSystemPermission == null)
				{
					hasAdminSystemPermission = false;
				}
			}
			return (bool)hasAdminSystemPermission;
		}

		public static VDS GetLocalStorageHost(string storagePoolName)
		{
			SearchParameters sp =
				new SearchParameters(String.Format("hosts: datacenter={0}", storagePoolName), SearchType.VDS);

			VdcQueryReturnValue result = Frontend.RunQuery(VdcQueryType.Search, sp);
			if ((result != null) && result.Succeeded && (result.ReturnValue != null))
			{
				foreach (IVdcQueryable res in (List<IVdcQueryable>)(result.ReturnValue))
				{
					return (VDS)res;
				}
			}
			return null;
		}

		public static List<KeyValuePair<string, EntityModel>> GetBondingOptionList(out KeyValuePair<string, EntityModel> defaultItem)
		{
			List<KeyValuePair<string, EntityModel>> list = new List<KeyValuePair<string, EntityModel>>();
			EntityModel entityModel = new EntityModel();
			entityModel.Entity = "(Mode 1) Active-Backup";
			list.Add(new KeyValuePair<string, EntityModel>("mode=1 miimon=100", entityModel));
			entityModel = new EntityModel();
			entityModel.Entity = "(Mode 2) Load balance (balance-xor)";
			list.Add(new KeyValuePair<string, EntityModel>("mode=2", entityModel));
			entityModel = new EntityModel();
			entityModel.Entity = "(Mode 4) Dynamic link aggregation (802.3ad)";
			defaultItem = new KeyValuePair<string, EntityModel>("mode=4", entityModel);
			list.Add(defaultItem);
			entityModel = new EntityModel();
			entityModel.Entity = "(Mode 5) Adaptive transmit load balancing (balance-tlb)";
			list.Add(new KeyValuePair<string, EntityModel>("mode=5", entityModel));
			entityModel = new EntityModel();
			entityModel.Entity = "";
			list.Add(new KeyValuePair<string, EntityModel>("custom", entityModel));
			return list;
		}

		public static string GetDefaultBondingOption()
		{
			return "mode=802.3ad miimon=150";
		}

        public static bool IsDomainAlreadyExist(Guid? storagePoolId, string path, out string storageName)
        {
            GetStorageDomainsByConnectionParameters param = new GetStorageDomainsByConnectionParameters();
            param.Connection = path;
            if (storagePoolId.HasValue)
            {
                param.StoragePoolId = storagePoolId.Value;
            }

            VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetStorageDomainsByConnection, param);

            if (returnValue != null && returnValue.Succeeded)
            {
                IList<storage_domains> storages = (List<storage_domains>)returnValue.ReturnValue;
                if (storages.Count > 0)
                {
                    storageName = storages[0].storage_name;
                    return true;
                }
            }

            storageName = null;
            return false;
        }

        public static bool IsDomainAlreadyExist(string path, out string storageName)
		{
            return IsDomainAlreadyExist(null, path, out storageName);
		}

		public static string GetMarketplaceUrl()
		{
			VdcQueryReturnValue returnValue = GetConfigFromCache(
				new GetConfigurationValueParameters(ConfigurationValues.CustomPublicConfig_AppsWebSite));

			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				string url = (string)returnValue.ReturnValue;

				string rpmVersion = GetRpmVersion();
				if (!String.IsNullOrEmpty(rpmVersion))
				{
					return url.Replace("$EngineVersion", rpmVersion);
				}
			}

			return String.Empty;
		}
	}
}
