using System;
using System.Collections.Generic;
using System.Text;
using System.Windows;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.validation;
using VdcFrontend;
using VdcCommon.BusinessEntities;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.hosts
{
	public class HostModel : Model, ITaskTarget
	{
		#region Const

		public const int HostNameMaxLength = 255;
		public const string PmSecureKey = "secure";
		public const string PmPortKey = "port";
		public const string PmSlotKey = "slot";
		public const string BeginTestStage = "BeginTest";
		public const string EndTestStage = "EndTest";

		#endregion Const

		#region Commands

		public UICommand TestCommand { get; private set; }

		#endregion

		#region Properties

		public bool IsNew
		{
			get { return !HostId.HasValue; }
		}

		public Guid? HostId { get; set; }
		public string OriginalName { get; set; }

		public EntityModel Name { get; private set; }
		public EntityModel Host { get; private set; }
		public EntityModel ManagementIp { get; private set; }
		public ListModel DataCenter { get; private set; }
		public ListModel Cluster { get; private set; }
		public EntityModel Port { get; private set; }
		public EntityModel RootPassword { get; private set; }
		public EntityModel OverrideIpTables { get; private set; }
		public EntityModel IsPm { get; private set; }
		public EntityModel PmUserName { get; private set; }
		public EntityModel PmPassword { get; private set; }
		public ListModel PmType { get; private set; }
		public EntityModel PmSecure { get; private set; }
		public EntityModel PmPort { get; private set; }
		public EntityModel PmSlot { get; private set; }
		public EntityModel PmOptions { get; private set; }

		bool isGeneralTabValid;
		public bool IsGeneralTabValid
		{
			get { return isGeneralTabValid; }
			set
			{
				if (isGeneralTabValid != value)
				{
					isGeneralTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsGeneralTabValid"));
				}
			}
		}

		bool isPowerManagementTabValid;
		public bool IsPowerManagementTabValid
		{
			get { return isPowerManagementTabValid; }
			set
			{
				if (isPowerManagementTabValid != value)
				{
					isPowerManagementTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsPowerManagementTabValid"));
				}
			}
		}

		private bool isPowerManagementSelected;
		public bool IsPowerManagementSelected
		{
			get
			{
				return isPowerManagementSelected;
			}
			set
			{
				if (isPowerManagementSelected != value)
				{
					isPowerManagementSelected = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsPowerManagementSelected"));
				}
			}
		}

		public Dictionary<string, string> PmOptionsMap
		{
			get
			{
				Dictionary<string, string> dict = new Dictionary<string, string>();

				//Add well known pm options.
				if (PmPort.IsAvailable)
				{
					dict[PmPortKey] = PmPort.Entity == null ? string.Empty : (string)PmPort.Entity;
				}
				if (PmSlot.IsAvailable)
				{
					dict[PmSlotKey] = PmSlot.Entity == null ? string.Empty : (string)PmSlot.Entity;
				}
				if (PmSecure.IsAvailable)
				{
					dict[PmSecureKey] = PmSecure.Entity.ToString();
				}

				//Add unknown pm options.
				//Assume Validate method was called before this getter.
				string pmOptions = (string)PmOptions.Entity;
				if (!String.IsNullOrEmpty(pmOptions))
				{
					foreach (string pair in pmOptions.Split(','))
					{
						string[] array = pair.Split('=');
						if (array.Length == 2)
						{
							dict.Add(array[0], array[1]);
						}
						else if (array.Length == 1)
						{
							dict.Add(array[0], String.Empty);
						}
					}
				}

				return dict;
			}
			set
			{
				string pmOptions = String.Empty;

				foreach (KeyValuePair<string, string> pair in value)
				{
					string k = pair.Key;
					string v = pair.Value;

					switch (k)
					{
						//Handle well known pm options.
						case PmPortKey:
							PmPort.Entity = string.IsNullOrEmpty(value[k]) ? string.Empty : value[k];
							break;

						case PmSlotKey:
							PmSlot.Entity = string.IsNullOrEmpty(value[k]) ? string.Empty : value[k];
							break;

						case PmSecureKey:
							PmSecure.Entity = Convert.ToBoolean(value[k]);
							break;

						default:
							//Compose custom string from unknown pm options.
							if (String.IsNullOrEmpty(v))
							{
								pmOptions += String.Format("{0},", k);
							}
							else
							{
								pmOptions += String.Format("{0}={1},", k, v);
							}
							break;
					}
				}

				if (!String.IsNullOrEmpty(pmOptions))
				{
					PmOptions.Entity = pmOptions.Remove(pmOptions.Length - 1);
				}
			}
		}

		#endregion

		public HostModel()
		{
			TestCommand = new UICommand("Test", this);

			Name = new EntityModel();
			Host = new EntityModel();
			ManagementIp = new EntityModel();
			DataCenter = new ListModel();
			DataCenter.SelectedItemChangedEvent.addListener(this);
			Cluster = new ListModel();
			Cluster.SelectedItemChangedEvent.addListener(this);
			Port = new EntityModel();
			RootPassword = new EntityModel();
			OverrideIpTables = new EntityModel { Entity = false };
			PmUserName = new EntityModel();
			PmPassword = new EntityModel();
			PmType = new ListModel();
			PmType.SelectedItemChangedEvent.addListener(this);
			PmSecure = new EntityModel();
			PmSecure.IsAvailable = false;
			PmSecure.Entity = false;
			PmPort = new EntityModel();
			PmPort.IsAvailable = false;
			PmSlot = new EntityModel();
			PmSlot.IsAvailable = false;
			PmOptions = new EntityModel();

			IsPm = new EntityModel();
			IsPm.EntityChangedEvent.addListener(this);
			IsPm.Entity = false;

			IsGeneralTabValid = IsPowerManagementTabValid = true;
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ListModel.SelectedItemChangedEventDefinition) && sender == DataCenter)
			{
				DataCenter_SelectedItemChanged();
			}
			else if (ev.Equals(ListModel.SelectedItemChangedEventDefinition) && sender == Cluster)
			{
				Cluster_SelectedItemChanged();
			}
			else if (ev.Equals(ListModel.SelectedItemChangedEventDefinition) && sender == PmType)
			{
				PmType_SelectedItemChanged();
			}
			else if (ev.Equals(EntityModel.EntityChangedEventDefinition) && sender == IsPm)
			{
				IsPm_EntityChanged();
			}
		}

		void IsPm_EntityChanged()
		{
			UpdatePmModels();
		}

		private void DataCenter_SelectedItemChanged()
		{
			storage_pool dataCenter = (storage_pool)DataCenter.SelectedItem;
			if (dataCenter != null)
			{
				VDSGroup oldCluster = (VDSGroup)Cluster.SelectedItem;

				List<VDSGroup> clusters = DataProvider.GetClusterList(dataCenter.Id);
				Cluster.Items = clusters;

				if (oldCluster != null)
				{
					Cluster.SelectedItem = Linq.FirstOrDefault(clusters, new Linq.ClusterPredicate(oldCluster.ID));
				}

				if (Cluster.SelectedItem == null)
				{
					Cluster.SelectedItem = Linq.FirstOrDefault(clusters);
				}
			}
		}

		void Cluster_SelectedItemChanged()
		{
			VDSGroup cluster = (VDSGroup)Cluster.SelectedItem;
			if (cluster != null)
			{
				string pmType = (string)PmType.SelectedItem;
				PmType.Items = DataProvider.GetPmTypeList(cluster.compatibility_version);
				if (((List<string>)PmType.Items).Contains(pmType))
				{
					PmType.SelectedItem = pmType;
				}
				else
				{
					PmType.SelectedItem = null;
				}
			}
		}

		void PmType_SelectedItemChanged()
		{
			UpdatePmModels();
		}

		private void UpdatePmModels()
		{
			string pmType = (string)PmType.SelectedItem;
			List<string> pmOptions = !String.IsNullOrEmpty(pmType) ? DataProvider.GetPmOptions(pmType) : new List<string>();

			PmPort.IsAvailable = pmOptions.Contains(PmPortKey);
			PmSlot.IsAvailable = pmOptions.Contains(PmSlotKey);
			PmSecure.IsAvailable = pmOptions.Contains(PmSecureKey);


			bool isPm = (bool)IsPm.Entity;

			TestCommand.IsExecutionAllowed = isPm;

			ManagementIp.IsChangable = (bool)IsPm.Entity;
			ManagementIp.IsValid = true;
			PmUserName.IsChangable = (bool)IsPm.Entity;
			PmUserName.IsValid = true;
			PmPassword.IsChangable = (bool)IsPm.Entity;
			PmPassword.IsValid = true;
			PmType.IsChangable = (bool)IsPm.Entity;
			PmType.IsValid = true;
			PmOptions.IsChangable = (bool)IsPm.Entity;
			PmSecure.IsChangable = (bool)IsPm.Entity;
			PmPort.IsChangable = (bool)IsPm.Entity;
			PmPort.IsValid = true;
			PmSlot.IsChangable = (bool)IsPm.Entity;
		}

		public void Test()
		{
			//Validate user input.
			if ((bool)IsPm.Entity)
			{
				Cluster.IsValid = true;
				Cluster.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
				ValidatePmModels();
			}

			if (!ManagementIp.IsValid
				|| !PmUserName.IsValid
				|| !PmPassword.IsValid
				|| !PmType.IsValid
				|| !PmPort.IsValid
				|| !PmOptions.IsValid)
			{
				return;
			}


			Message = "Testing in progress. It will take a few seconds. Please wait...";
			TestCommand.IsExecutionAllowed = false;

			VDSGroup cluster = (VDSGroup)Cluster.SelectedItem;

			Task.Create(this,
				new List<object>
				{
					BeginTestStage,
					ManagementIp.Entity,
					PmType.SelectedItem,
					PmUserName.Entity,
					PmPassword.Entity,
					cluster.storage_pool_id != null ? cluster.storage_pool_id.Value : Guid.Empty,
					PmOptionsMap
				})
				.Run();
		}

		private void ValidatePmModels()
		{
			ManagementIp.ValidateEntity(new IValidation[] { new NotEmptyValidation(), new HostAddressValidation() });
			PmUserName.ValidateEntity(new IValidation[] { new NotEmptyValidation() });
			PmPassword.ValidateEntity(new IValidation[] { new NotEmptyValidation() });
			PmType.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			PmPort.ValidateEntity(new IValidation[] { new IntegerValidation { Minimum = 1, Maximum = 65535 } });
			PmOptions.ValidateEntity(new IValidation[] { new KeyValuePairValidation(true) });
		}

		internal bool Validate()
		{
			string hostNameRegex = String.Format(@"^[0-9a-zA-Z-_\.]{{1,{0}}}$", HostNameMaxLength);
			string hostNameMessage = String.Format(
				@"This field can't contain blanks or special characters, must " +
				@"be at least one character long, legal values are 0-9, a-z, '_', '.' " +
				@"and a length of up to {0} characters.", HostNameMaxLength);

			Name.ValidateEntity(
				new IValidation[]
				{
					new NotEmptyValidation(),
					new RegexValidation {Expression = hostNameRegex, Message = hostNameMessage}
				});

			Host.ValidateEntity(
				new IValidation[]
				{
					new NotEmptyValidation(),
					new HostAddressValidation()
				});

			Port.ValidateEntity(
				new IValidation[]
				{
					new NotEmptyValidation(),
					new IntegerValidation {Minimum = 1, Maximum = 65535}
				});


			DataCenter.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			Cluster.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });


			string name = (string)Name.Entity;

			//Check name unicitate.
			if (String.Compare(name, OriginalName, true) != 0 && !DataProvider.IsHostNameUnique(name))
			{
				Name.IsValid = false;
				Name.InvalidityReasons.Add("Name must be unique.");
			}

			if ((bool)IsPm.Entity)
			{
				ValidatePmModels();
			}

			IsGeneralTabValid =
				Name.IsValid
				&& Host.IsValid
				&& Port.IsValid
				&& Cluster.IsValid;

			IsPowerManagementTabValid =
				ManagementIp.IsValid
				&& PmUserName.IsValid
				&& PmPassword.IsValid
				&& PmType.IsValid
				&& PmPort.IsValid
				&& PmOptions.IsValid;

			return Name.IsValid
				   && Host.IsValid
				   && Port.IsValid
				   && Cluster.IsValid
				   && ManagementIp.IsValid
				   && PmUserName.IsValid
				   && PmPassword.IsValid
				   && PmType.IsValid
				   && PmPort.IsValid
				   && PmOptions.IsValid;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == TestCommand)
			{
				Test();
			}
		}

		public void run(TaskContext context)
		{
			List<object> state = (List<object>)context.State;
			string stage = (string)state[0];

			switch (stage)
			{
				case BeginTestStage:
					{
						string message = null;

						GetNewVdsFenceStatusParameters param = new GetNewVdsFenceStatusParameters();
						if (HostId.HasValue)
						{
							param.VdsId = HostId.Value;
						}
						param.ManagementIp = (string)state[1];
						param.PmType = (string)state[2];
						param.User = (string)state[3];
						param.Password = (string)state[4];
						param.StoragePoolId = (Guid)state[5];
						param.FencingOptions = new valueObjectMap((Dictionary<string, string>)state[6], false);

						VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetNewVdsFenceStatus, param);
						if (returnValue != null && returnValue.ReturnValue != null)
						{
							FenceStatusReturnValue fenceStatusReturnValue = (FenceStatusReturnValue) returnValue.ReturnValue;
							message = fenceStatusReturnValue.ToString();
						}
						else
						{
							message = "Test Failed (unknown error).";
						}

						context.InvokeUIThread(this,
							new List<object>
							{
								EndTestStage,
								message
							});
					}
					break;

				case EndTestStage:
					{
						Message = (string)state[1];
						TestCommand.IsExecutionAllowed = true;
					}
					break;
			}
		}
	}
}
