#!/usr/bin/python

# Imports
import sys
import os
import logging
import traceback
import tempfile
import shutil
import pwd
from optparse import OptionParser
from StringIO import StringIO
import common_utils as utils
import types
import basedefs

# Consts
RPM_NAME = "rhevm"
LOG_FILE = "rhevm-cleanup.log"

# PATH
LOG_PATH = "/var/log/rhevm"
PKI_DIR = "/etc/pki/rhevm"
TRUSTORE = os.path.join (PKI_DIR, ".truststore")
KEYSTORE = os.path.join (PKI_DIR, ".keystore")
JBOSS_SERVER_DIR = "/var/lib/jbossas/server"
VAR_SLIMMED_DIR = os.path.join(JBOSS_SERVER_DIR, basedefs.JBOSS_PROFILE_NAME)
ETC_JBOSS_DIR = "/etc/jbossas"
ETC_SLIMMED_DIR = os.path.join(ETC_JBOSS_DIR, basedefs.JBOSS_PROFILE_NAME)
PKI_BACKUP_DIR = "/etc/pki/rhevm-backups"

#MSGS
MSG_ERROR_USER_NOT_ROOT = "Error: insufficient permissions for user %s, you must run with user root."
MSG_RC_ERROR = "Return Code is not zero"
MSG_ERROR_BACKUP_DB = "Error: Database backup failed"
MSG_ERROR_DROP_DB = "Error: Database drop failed"
MSG_ERROR_CHECK_LOG = "Error: RHEV Manager Cleanup failed.\nplease check log at %s"
MSG_ERR_FAILED_JBOSS_SERVICE_STILL_RUN = "Error: Can't stop jboss service. Please shut it down manually."
MSG_ERR_FAILED_STP_JBOSS_SERVICE = "Error: Can't stop JBoss"
MSG_ERR_FAILED_STATUS_JBOSS_SERVICE = "Error: Can't get JBoss service status"

MSG_INFO_DONE = "DONE"
MSG_INFO_ERROR = "ERROR"
MSG_INFO_STOP_JBOSS = "Stopping JBoss Service"
MSG_INFO_BACKUP_DB = "Backing Up Database"
MSG_INFO_REMOVE_DB = "Removing Database"
MSG_INFO_REMOVE_SLIMMED = "Removing RHEV Manager JBoss profile"
MSG_INFO_REMOVE_CA = "Removing CA"
MSG_INFO_DB_BACKUP_FILE = "DB Backup available at "
MSG_INFO_LOG_FILE = "Cleanup log available at "
MSG_INFO_CLEANUP_OK = "\nRHEV Manager cleanup finished successfully!"
MSG_INFO_FINISHED_WITH_ERRORS = "\nRHEV Manager cleanup finished with errors, please see log file"
MSG_INFO_STOP_INSTALL_EXIT = "Cleanup stopped, Goodbye."
MSG_INFO_KILL_DB_CONNS="Stopping all connections to DB"

MSG_ALERT_CLEANUP = "WARNING: Executing RHEV Manager cleanup utility.\n\
This utility will wipe all existing data including configuration settings, certificates and database.\n\
In addition, all existing DB connections will be closed."
MSG_PROCEED_QUESTION = "Would you like to proceed"

#global err msg list
err_messages = []

# Code
def getOptions():
    parser = OptionParser()

    parser.add_option("-u", "--unattended",
                      action="store_true", dest="unattended_clean", default=False,
                      help="unattended cleanup")

    parser.add_option("-d", "--dont-drop-db",
                      action="store_false", dest="drop_db", default=True,
                      help="Don't drop database")

    parser.add_option("-c", "--dont-remove-ca",
                      action="store_false", dest="remove_ca", default=True,
                      help="Don't remove CA")

    parser.add_option("-s", "--dont-remove-profile",
                      action="store_false", dest="remove_slimmed", default=True,
                      help="Don't remove rhevm-slimmed JBoss profile")

    (options, args) = parser.parse_args()
    return (options, args)

def askYesNo(question=None):
    """
    service func to ask yes/no
    input from user
    """
    message = StringIO()
    userQuestion = "%s? (yes|no): " % (question)
    logging.debug("asking user: %s", userQuestion)
    message.write(userQuestion)
    message.seek(0)
    answer = raw_input(message.read())
    logging.debug("user answered: %s", answer)
    answer = answer.lower()
    if answer == "yes" or answer == "y":
        return True
    elif answer == "no" or answer == "n":
        return False
    else:
        return askYesNo(question)

def _getColoredText (text, color):
    """
        gets text string and color
        and returns a colored text.
        the color values are RED/BLUE/GREEN/YELLOW
        everytime we color a text, we need to disable
        the color at the end of it, for that
        we use the NO_COLOR chars.
    """
    return color + text + basedefs.NO_COLOR

def askForUserApproval():
    """
    Ask user to proceed with cleanup
    """

    logging.debug("Asking user for approval")
    print MSG_ALERT_CLEANUP
    answer = askYesNo(MSG_PROCEED_QUESTION)

    if answer:
        logging.debug("User chose to proceed")
        return True
    else:
        logging.debug("User chose to exit")
        return False

def initLogging():
    global LOG_FILE
    try:
        if not os.path.isdir(LOG_PATH):
            os.makedirs(LOG_PATH)
        LOG_FILE = "%s/rhevm-cleanup_%s.log" % (LOG_PATH, utils.getCurrentDateTime())
        level = logging.DEBUG
        # TODO: Move to mode="a"?
        hdlr = logging.FileHandler(filename = LOG_FILE, mode='w')
        fmts = '%(asctime)s::%(levelname)s::%(module)s::%(lineno)d::%(name)s:: %(message)s'
        dfmt = '%Y-%m-%d %H:%M:%S'
        fmt = logging.Formatter(fmts, dfmt)
        hdlr.setFormatter(fmt)
        logging.root.addHandler(hdlr)
        logging.root.setLevel(level)
    except:
        logging.error(traceback.format_exc())
        raise Exception("Failed to initiate logger")

def _verifyUserPermissions():
    username = pwd.getpwuid(os.getuid())[0]
    if (username != 'root'):
        print MSG_ERROR_USER_NOT_ROOT % (username)
        sys.exit(1)

class CA():
    def backup(self):
        logging.debug("CA backup started")
        if not os.path.isdir(PKI_BACKUP_DIR):
            os.mkdir(PKI_BACKUP_DIR)

        # Do backup
        now = utils.getCurrentDateTime()
        backupDir = os.path.join(PKI_BACKUP_DIR, "rhevm-%s" % (now))
        logging.debug("Copy %s to %s", PKI_DIR, backupDir)
        shutil.copytree(PKI_DIR, backupDir, True)

        logging.debug("CA backup completed successfully")

    def remove(self):
        logging.debug("CA Remove started")

        # Remove trustore
        if os.path.exists(TRUSTORE):
            logging.debug("Removing %s", TRUSTORE)
            os.remove(TRUSTORE)
        else:
            logging.debug("%s doesn't exists", TRUSTORE)

        # Remove keystore
        if os.path.exists(KEYSTORE):
            logging.debug("Removing %s", KEYSTORE)
            os.remove(KEYSTORE)
        else:
            logging.debug("%s doesn't exists", KEYSTORE)

        logging.debug("CA Remove completed successfully")

class Slimmed():
    def __init__(self):
        pass

    def backup(self):
        """
        Backup rhevm-slimmed JBoss profile
        """
        logging.debug("RHEVM Slimmed profile backup started")

        now = utils.getCurrentDateTime()

        # Backup profile under /var
        if os.path.exists(VAR_SLIMMED_DIR):
            backupDir = os.path.join(JBOSS_SERVER_DIR, "%s-%s" % (basedefs.JBOSS_PROFILE_NAME, now))
            logging.debug("Copy %s to %s", VAR_SLIMMED_DIR, backupDir)
            shutil.copytree(VAR_SLIMMED_DIR, backupDir, True)
        else:
            logging.debug("%s doesn't exists", VAR_SLIMMED_DIR)

        # Backup profile under /etc
        if os.path.exists(ETC_SLIMMED_DIR):
            backupDir = os.path.join(ETC_JBOSS_DIR, "%s-%s" % (basedefs.JBOSS_PROFILE_NAME, now))
            logging.debug("Copy %s to %s", ETC_SLIMMED_DIR, backupDir)
            shutil.copytree(ETC_SLIMMED_DIR, backupDir, True)
        else:
            logging.debug("%s doesn't exists", ETC_SLIMMED_DIR)

        logging.debug("RHEVM Slimmed profile backup completed successfully")

    def remove(self):
        """
        Remove rhevm-slimmed JBoss profile
        """
        logging.debug("RHEVM Slimmed profile remove started")

        # Remove profile from /etc/jboss
        if os.path.exists(ETC_SLIMMED_DIR):
            logging.debug("Removing %s", ETC_SLIMMED_DIR)
            shutil.rmtree(ETC_SLIMMED_DIR)
        else:
            logging.debug("%s doesn't exists", ETC_SLIMMED_DIR)

        # Remvoe profile from /var/lib/jboss
        if os.path.exists(VAR_SLIMMED_DIR):
            logging.debug("Removing %s", VAR_SLIMMED_DIR)
            shutil.rmtree(VAR_SLIMMED_DIR)
        else:
            logging.debug("%s doesn't exists", VAR_SLIMMED_DIR)

        logging.debug("RHEVM Slimmed profile remove completed successfully")

class DB():
    def __init__(self):
        self.sqlfile = tempfile.mkstemp(suffix=".sql", dir=basedefs.DIR_DB_BACKUPS)[1]
        self.dropped = False

    def __del__(self):
        if self.dropped:
            logging.debug(MSG_INFO_DB_BACKUP_FILE + self.sqlfile)
            print MSG_INFO_DB_BACKUP_FILE,
            print self.sqlfile
        else:
            os.remove(self.sqlfile)

    def backup(self):
        """
        Backup rhevm db using pg_dump
        """
        # pg_dump -C -E UTF8  --column-inserts --disable-dollar-quoting  --disable-triggers -U postgres --format=p -f $dir/$file  rhevm
        logging.debug("DB Backup started")
        cmd = [basedefs.EXEC_PGDUMP, "-C", "-E", "UTF8", "--column-inserts", "--disable-dollar-quoting",  "--disable-triggers", "-U",
                basedefs.DB_ADMIN, "--format=p", "-f", self.sqlfile, basedefs.DB_NAME]
        output, rc = utils.execCmd(cmd, None, True, MSG_ERROR_BACKUP_DB, [])
        logging.debug("DB Backup completed successfully")

    def drop(self):
        """
        Drops rhevm db using dropdb
        """
        logging.debug("DB Drop started")
        cmd = [basedefs.EXEC_DROPDB, "-w", "-U", basedefs.DB_ADMIN, basedefs.DB_NAME]
        output, rc = utils.execCmd(cmd, None, True, MSG_ERROR_DROP_DB, [])
        self.dropped = True
        logging.debug("DB Drop completed successfully")

    def exists(self):
        """
        check if db exists
        """
        logging.debug("verifying is db exists")

        # Making sure postgresql service is up
        postgresql = utils.Service("postgresql")
        postgresql.conditionalStart()

        try:
            # We want to make check if postgresql service is up
            logging.debug("making sure postgresql service is up")
            utils.retry(utils.checkIfRhevmDbIsUp, tries=5, timeout=15, sleep=3)
        except:
            # If we're here, it means something is wrong, either there is a real db error
            # or rhevm db is not installed, let's check
            logging.debug("checking if rhevm db is already installed..")
            (out, rc) = utils.execSqlCommand(basedefs.DB_ADMIN, basedefs.DB_NAME, "select 1")
            if (rc != 0):
                if utils.verifyStringFormat(out,".*FATAL:\s*database\s*\"rhevm\"\s*does not exist"):
                    # This means that the db is not installed, so we return false
                    return False

        return True

def stopJboss():
    logging.debug("stoping jboss service.")

    cmd = [basedefs.EXEC_SERVICE, "jbossas", "stop"]
    output, rc = utils.execCmd(cmd, None, True, MSG_ERR_FAILED_STP_JBOSS_SERVICE, [])

    # JBoss service sometimes return zero rc even if service is still up
    if "[FAILED]" in output and "Timeout: Shutdown command was sent, but process is still running" in output:
        raise OSError(MSG_ERR_FAILED_JBOSS_SERVICE_STILL_RUN)

def restartPostgresql():
    """
    restart the postgresql service
    """
    logging.debug("Restarting the postgresql service")
    postgresql = utils.Service("postgresql")
    postgresql.stop(True)
    postgresql.start(True)

    # We want to make sure the postgres service is up
    utils.retry(utils.checkIfRhevmDbIsUp, tries=10, timeout=30, sleep=3)

def runFunc(funcs, dispString):
    print "%s..." % (dispString),
    sys.stdout.flush()
    spaceLen = basedefs.SPACE_LEN - len(dispString)
    try:
        if type(funcs) is types.ListType:
            for funcName in funcs:
                funcName()
        elif type(funcs) is types.FunctionType:
            funcs()
        print ("[ " + _getColoredText(MSG_INFO_DONE, basedefs.GREEN) + " ]").rjust(spaceLen)
    except:
        print ("[ " + _getColoredText(MSG_INFO_ERROR, basedefs.RED) + " ]").rjust(spaceLen + 2)
        logging.error(traceback.format_exc())
        err_messages.append(sys.exc_info()[1])

def _printErrlMessages():
    print MSG_INFO_FINISHED_WITH_ERRORS
    for msg in err_messages:
        logging.info('%s'%(msg))
        print ('%s'%(msg))

def main(options):
    db = DB()
    ca = CA()
    slimmed = Slimmed()

    if not options.unattended_clean:
        # Ask user to proceed
        if not askForUserApproval():
            print MSG_INFO_STOP_INSTALL_EXIT
            sys.exit(0)

    print

    # Stop JBoss
    runFunc(stopJboss, MSG_INFO_STOP_JBOSS)

    # Close all DB connections and drop DB (only if db exists)
    if db.exists() and options.drop_db:
        runFunc(restartPostgresql, MSG_INFO_KILL_DB_CONNS)
        runFunc([db.backup, db.drop], MSG_INFO_REMOVE_DB)

    # Remove CA
    if options.remove_ca:
        runFunc([ca.backup, ca.remove], MSG_INFO_REMOVE_CA)

    # Remove Slimmed profile
    if options.remove_slimmed:
        runFunc([slimmed.backup, slimmed.remove], MSG_INFO_REMOVE_SLIMMED)

    if len(err_messages) == 0:
        print MSG_INFO_CLEANUP_OK
    else:
        _printErrlMessages()

    print MSG_INFO_LOG_FILE,
    print LOG_FILE

if __name__ == '__main__':
    try:
        # Must run as root
        _verifyUserPermissions()

        # Init logging facility
        initLogging()

        # get iso and domain from user arguments
        (options, args) = getOptions()

        main(options)

    except SystemExit:
        raise

    except BaseException as e:
        logging.error(traceback.format_exc())
        print
        print e
        print MSG_ERROR_CHECK_LOG % (LOG_FILE)
        sys.exit(1)
