using System;
using System.Collections;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using CookComputing.XmlRpc;

namespace VdcUtils
{
    public interface IObjectDescriptorContainer
    {
        bool CanUpdateField(object obj,string fieldName, Enum status);
    }
    static public class ObjectDescriptor
    {
        public static string ToString(object obj)
        {
            Type type = obj.GetType();
            PropertyInfo[] properties = type.GetProperties();
            StringBuilder builder = new StringBuilder(String.Format("Class Name: {0}", type.Name));
            builder.AppendLine();
            foreach (PropertyInfo property in properties)
            {
                string propertyName = property.Name;
                object objValue = property.GetValue(obj, null);
                string strObjectValue;
                if (objValue != null)
                {
                    if (!(objValue is string) && !(objValue is IDictionary) &&  objValue is IEnumerable)
                    {
                        StringBuilder tempBuilder = new StringBuilder();
                        foreach (object o in (IEnumerable)objValue)
                        {
                            tempBuilder.Append(o.ToString());
                            tempBuilder.AppendLine();
                        }
                        strObjectValue = tempBuilder.ToString();
                    }
                    else
                    {
                        strObjectValue = objValue.ToString();
                    }
                }
                else
                {
                    strObjectValue = "Null";
                }
                builder.AppendFormat("{0,-30}{1}", propertyName, strObjectValue);
                builder.AppendLine();
            }
            FieldInfo[] fields = type.GetFields(BindingFlags.Public | BindingFlags.Instance);
            foreach (FieldInfo field in fields)
            {
                string propertyName = field.Name;
                object objValue = field.GetValue(obj);
                string strObjectValue;
                if (objValue != null)
                {
                    if (!(objValue is string) && !(objValue is IDictionary) && objValue is IEnumerable)
                    {
                        StringBuilder tempBuilder = new StringBuilder();
                        foreach (object o in (IEnumerable)objValue)
                        {
                            tempBuilder.Append(o.ToString());
                            tempBuilder.AppendLine();
                        }
                        strObjectValue = tempBuilder.ToString();
                    }
                    else
                    {
                        strObjectValue = objValue.ToString();
                    }
                }
                else
                {
                    strObjectValue = "Null";
                }
                builder.AppendFormat("{0,-30}{1}", propertyName, strObjectValue);
                builder.AppendLine();
            }
            return builder.ToString();
        }
        public static void ToStringBuilder(XmlRpcStruct xmlRpc, StringBuilder builder)
        {
            foreach (DictionaryEntry entry in xmlRpc)
            {
                if (entry.Value is XmlRpcStruct)
                {
                    builder.Append(string.Format("{0}:", entry.Key));
                    builder.AppendLine();
                    ToStringBuilder((XmlRpcStruct)entry.Value,builder);
                    builder.AppendLine();
                }
                else if (!(entry.Value is string) && entry.Value is IEnumerable)
                {
                    builder.Append(string.Format("{0}:", entry.Key));
                    builder.AppendLine();
                    ToStringBuilder((IEnumerable)(entry.Value),builder);
                    builder.AppendLine();
                }
                else
                {
                    builder.Append(string.Format("{0} = {1}", entry.Key, entry.Value.ToString()));
                    builder.AppendLine();
                }
            }
        }
        public static void ToStringBuilder(XmlRpcStruct[] xmlRpc, StringBuilder builder)
        {
            foreach (XmlRpcStruct entry in xmlRpc)
            {
                ToStringBuilder(entry, builder);
            }
        }
        private static void ToStringBuilder(IEnumerable xmlRpc,StringBuilder builder)
        {
            
            foreach (object value in xmlRpc)
            {
                if (value is XmlRpcStruct)
                {
                    ToStringBuilder((XmlRpcStruct)(value),builder);
                    builder.AppendLine();
                }
                else if (value is IEnumerable)
                {
                    ToStringBuilder((IEnumerable)(value),builder);
                }
                else
                {
                    builder.Append(value.ToString());
                    builder.AppendLine();
                }
            }
        }
    }
    public class ObjectIdentityChecker
    {
        private IObjectDescriptorContainer mContainer;
        private static Dictionary<string, Type> mAliases = new
            Dictionary<string, Type>();
        private static Dictionary<Type, ObjectIdentityChecker> mIdentities = new
            Dictionary<Type, ObjectIdentityChecker>();

        private static Dictionary<Type, Type> mStatusTypes = new Dictionary<Type, Type>();

        private Dictionary<Enum, List<string>> mDictionary = new Dictionary<Enum, List<string>>();
        private List<string> mPermitted = new List<string>();
        
        public ObjectIdentityChecker(Type type)
        {
            mIdentities[type] = this;
        }
        public ObjectIdentityChecker(Type type, IEnumerable<string>aliases):this(type)
        {
            foreach (string alias in aliases)
            {
                mAliases[alias] = type;
            }
        }
        public ObjectIdentityChecker(Type type, IEnumerable<string> aliases,Type enumType)
            : this(type,aliases)
        {
            mStatusTypes[type] = enumType;
        }

        public IObjectDescriptorContainer Container
        {
            set { mContainer = value; }
        }

        public static bool CanUpdateField(object fieldContainer, string fieldName, Enum status)
        {
            return CanUpdateField(fieldContainer.GetType().Name, fieldName, status, fieldContainer);
        }
        public static bool CanUpdateField(string objectType, string fieldName, Enum status,object fieldContainer)
        {
            Type type;
            if (mAliases.TryGetValue(objectType, out type))
            {
                return CanUpdateField(type, fieldName, status, fieldContainer);
            }
            else
            {
                throw new Exception(string.Format("status type {0} not exist", type));
            }
        }
        public static bool CanUpdateField(Type objectType, string fieldName, Enum status, 
                                            object fieldContainer)
        {
            ObjectIdentityChecker checker;
            if (mIdentities.TryGetValue(objectType, out checker))
            {
                return checker.IsFieldUpdatable(status, fieldName, fieldContainer);
            }
            return true;
        }
        public static bool CanUpdateField(string objectType,string fieldName, string status)
        {
            Type type;
            if (mAliases.TryGetValue(objectType, out type))
            {
                Type statusType;
                if (mStatusTypes.TryGetValue(type, out statusType))
                {
                    Enum currentStatus;
                    try
                    {
                        currentStatus = (Enum) Enum.Parse(statusType, status, true);
                    }
                    catch(ArgumentException)
                    {
                        throw new Exception
                        (string.Format("status type {0} not contain type {1}", statusType,status));
                    }
                    return CanUpdateField(type, fieldName, currentStatus);
                }
                else
                {
                    throw new Exception(string.Format("status type {0} not exist",type));
                }
            }
            else
            {
                throw new Exception(String.Format("object type {0} not exist",objectType));
            }
        }

        public void AddField(Enum status, string fieldName)
        {
            List<string> values;
            if (!mDictionary.TryGetValue(status,out values))
            {
                values = new List<string>();
                mDictionary[status] = values;
            }
            if (!values.Contains(fieldName))
            {
                values.Add(fieldName);
            }
        }
        public void AddField(IEnumerable<Enum> statuses, string fieldName)
        {
            foreach (Enum status in statuses)
            {
                AddField(status, fieldName);
            }
        }
        public void AddFields(IEnumerable<Enum> statuses, IEnumerable<string> fields)
        {
            foreach (string field in fields)
            {
                AddField(statuses, field);
            }
        }

        public void AddPermittedField(string fieldName)
        {
            if (!mPermitted.Contains(fieldName))
            {
                mPermitted.Add(fieldName);
            }
        }
        public void AddPermittedFields(String[] fieldNames)
        {
            foreach (string fieldName in fieldNames)
            {
                AddPermittedField(fieldName);
            }
        }

        public bool IsFieldUpdatable(string name)
        {
            return mPermitted.Contains(name);
        }

        bool IsFieldUpdatable(Enum status,string name, object fieldContainer)
        {
            bool returnValue = true;
            if (!IsFieldUpdatable(name))
            {
                if (fieldContainer != null && mContainer != null 
                    && !mContainer.CanUpdateField(fieldContainer,name,status))
                {
                    returnValue = false;
                }
                else
                {
                    List<string> values;
                    if (mDictionary.TryGetValue(status, out values) && values != null)
                    {
                        returnValue = values.Contains(name);                        
                    }
                    else
                    {
                        returnValue = false;
                    }
                }
                if (!returnValue)
                {
                    LogError(name, status);
                }                
            }
            return returnValue;
        }
        public bool IsUpdateValid(object source, object destination)
        {
            if (source.GetType() != destination.GetType())
            {
                return false;
            }
            foreach (string fieldName in GetChangedFields(source, destination))
            {
                if (!IsFieldUpdatable(fieldName))
                {
                    return false;
                }
            }
            return true;
        }

        public bool IsUpdateValid(object source, object destination,Enum status)
        {
            if (source.GetType() != destination.GetType())
            {
                return false;
            }
            foreach (string fieldName in GetChangedFields(source, destination))
            {
                if (!IsFieldUpdatable(status, fieldName,null))
                {
                    QLogger.Instance.Warn(String.Format("ObjectIdentityChecker.IsUpdateValid:: Not updatable field '{0}' was updated", fieldName));
                    return false;
                }
            }
            return true;
        }
        public bool IsFieldsUpdated (object source,object destination, IEnumerable<string>fields)
        {
            List<string> changedFields = GetChangedFields(source, destination);
            foreach (string field in fields)
            {
                if (changedFields.Contains(field))
                {
                    return true;
                }
            }
            return false;
        }
        public static List<string> GetChangedFields(object source, object destination)
        {
            List<string> returnValue = new List<string>();
            if (source.GetType() == destination.GetType())
            {
                Type objectType = source.GetType();
                PropertyInfo[] properties = objectType.GetProperties();
                foreach (PropertyInfo property in properties)
                {
                    object sourceValue = property.GetValue(source, null);
                    object destinationValue = property.GetValue(destination, null);
                    
                    if (property.CanWrite &&
                        sourceValue != null && !sourceValue.Equals(destinationValue) ||
                        ((sourceValue == null && destinationValue != null)))
                    {
                        returnValue.Add(property.Name);
                    }
                }
            }
            return returnValue;
        }

        /// <summary>
        /// Logs the error.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <param name="status">The status.</param>
        private static void LogError(string name, Enum status)
        {
            QLogger.Instance.ErrorFormat("Field {0} cannot be updated when status is {1}",
                             name, status);

        }
    }
    public class CommandParametersInitializer : ICloneable
    {
        public Dictionary<Type, Queue<string>> mParameters = new Dictionary<Type, Queue<string>>();

        public void AddParameter(Type type, string parameterName)
        {
            Queue<string> values;
            if (!mParameters.TryGetValue(type,out values))
            {
                values = new Queue<string>();
                mParameters.Add(type,values);
            }
            if (!values.Contains(parameterName))
            {
                values.Enqueue(parameterName);
            }
        }
        public void AddParameters(Type type, IEnumerable<string>parameterNames)
        {
            Queue<string> values;
            if (!mParameters.TryGetValue(type, out values))
            {
                values = new Queue<string>();
                mParameters.Add(type, values);
            }
            foreach (string param in parameterNames)
            {
                values.Enqueue(param);
            }
         
        }
        public void InitializeParameter(object obj,object value)
        {
            Type type = obj.GetType();
            Queue<string> values;
            if (mParameters.TryGetValue(value.GetType(),out values) && values.Count != 0)
            {
                string paramName = values.Dequeue();
                FieldInfo field = type.GetField(paramName, 
                                                BindingFlags.NonPublic | 
                                                BindingFlags.Instance);
                if (field != null)
                {
                    field.SetValue(obj, value);
                }
            }
        }

        #region ICloneable Members

        public object Clone()
        {
            CommandParametersInitializer newInstance = new CommandParametersInitializer();
            foreach (Type type in mParameters.Keys)
            {
                Queue<string> values = mParameters[type];
                newInstance.AddParameters(type,values);
            }
            return newInstance;
        }

        #endregion
    }
}
