using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

namespace SearchBackend
{

    public class ADSyntaxChecker : ISyntaxChecker
    {
        private readonly AdSearchObjecAutoCompleter mSearchObjectAC;
        private readonly BaseAutoCompleter mColonAC;
        private readonly BaseAutoCompleter mPluralAC;
        private readonly Dictionary<SyntaxObjectType, SyntaxObjectType[]> mStateMap;

        private readonly Regex mFirstDQRegexp;
        private readonly Regex mNonSpaceRegexp;                        
 
        internal ADSyntaxChecker()
        {
            mSearchObjectAC         = new AdSearchObjecAutoCompleter();
            mColonAC                = new BaseAutoCompleter(":");
            mPluralAC               = new BaseAutoCompleter("S");

            mFirstDQRegexp =  new Regex("^\\s*\"$");
            mNonSpaceRegexp = new Regex("^\\S+$");

            mStateMap = new Dictionary<SyntaxObjectType, SyntaxObjectType[]>();
            SyntaxObjectType[] beginArray = {SyntaxObjectType.SEARCH_OBJECT};
            mStateMap.Add(SyntaxObjectType.BEGIN, beginArray);
            SyntaxObjectType[] searchObjectArray = { SyntaxObjectType.COLON };
            mStateMap.Add(SyntaxObjectType.SEARCH_OBJECT, searchObjectArray);
            SyntaxObjectType[] colonArray = { SyntaxObjectType.CONDITION_FIELD, SyntaxObjectType.END };
            mStateMap.Add(SyntaxObjectType.COLON, colonArray);
            SyntaxObjectType[] conditionFieldArray = { SyntaxObjectType.CONDITION_RELATION };
            mStateMap.Add(SyntaxObjectType.CONDITION_FIELD, conditionFieldArray);
            SyntaxObjectType[] conditionRelationArray = { SyntaxObjectType.CONDITION_VALUE };
            mStateMap.Add(SyntaxObjectType.CONDITION_RELATION, conditionRelationArray);
            SyntaxObjectType[] conditionValueArray = { SyntaxObjectType.CONDITION_FIELD };
            mStateMap.Add(SyntaxObjectType.CONDITION_VALUE, conditionValueArray);
        }


        public SyntaxContainer analyzeSyntaxState(string searchText, bool final)
        {
            SyntaxContainer retval = new SyntaxContainer(searchText);
            IConditionFieldAutoCompleter AdConditionFieldAC;
            if (searchText.ToUpper().Contains("ADUSER"))
            {
                AdConditionFieldAC = new AdUserConditionFieldAutoCompleter();
            }
            else
            {
                AdConditionFieldAC = new AdGroupConditionFieldAutoCompleter();
            }
            IAutoCompleter conditionRelationAC;
            char[] searchCharArr = searchText.ToCharArray();
            bool betweenDoubleQuotes = false;
            int curStartPos = 0;
            string curConditionField = string.Empty;
            for (int idx = 0; idx < searchCharArr.Length; idx++)
            {
                SyntaxObjectType curState = retval.getState();
                char curChar = searchCharArr[idx];
                if ((curChar == ' ') && (curState != SyntaxObjectType.CONDITION_RELATION))
                {
                    curStartPos += 1;
                    continue;
                }
                string strRealObj = searchText.Substring(curStartPos, idx - curStartPos + 1);
                string nextObject = strRealObj.ToUpper();
                switch (curState)
                {
                    case SyntaxObjectType.BEGIN:
                        //we have found a search-object
                        if (!mSearchObjectAC.validate(nextObject))
                        {
                            if (!mSearchObjectAC.validateCompletion(nextObject))
                            {
                                //ERROR INVALID-SEARCH OBJECT
                                retval.setErr(SyntaxError.INVALID_SEARCH_OBJECT, curStartPos, idx - curStartPos + 1);
                                return retval;
                            }
                        }
                        else
                        {
                            if (searchCharArr.Length >= idx + 2) // Check that this maybe a plural
                            {
                                // Validate that the next character is an 's'
                                if (mPluralAC.validate(searchText.Substring(idx + 1, 1)))
                                {
                                    // Then just move things along.
                                    idx++;
                                    StringBuilder sb = new StringBuilder(nextObject);
                                    sb.Append('S');
                                    nextObject = sb.ToString();
                                }
                            }
                            retval.addSyntaxObject(SyntaxObjectType.SEARCH_OBJECT, nextObject, curStartPos, idx + 1);
                            retval.valid = true;
                            curStartPos = idx + 1;
                        }     
                        break;

                    case SyntaxObjectType.SEARCH_OBJECT:
                        
                        if (!mColonAC.validate(nextObject))
                        {
                            if (!mColonAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.COLON_NOT_NEXT_TO_SEARCH_OBJECT, curStartPos, idx + 1);
                                return retval;
                            }
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.COLON, nextObject, idx, idx + 1);
                            curStartPos = idx + 1;
                            retval.valid = true;
                        }
                        break;

                    case SyntaxObjectType.COLON:
                    case SyntaxObjectType.CONDITION_VALUE:
                        if (AdConditionFieldAC.validate(nextObject))
                        {
                            retval.addSyntaxObject(SyntaxObjectType.CONDITION_FIELD, nextObject, curStartPos, idx + 1);
                            curConditionField = nextObject;
                            curStartPos = idx + 1;

                        }
                        else if (!AdConditionFieldAC.validateCompletion(nextObject))// && (!mSortbyAC.validateCompletion(nextObject)))
                        {
                            retval.setErr(SyntaxError.INVALID_CONDITION_FILED, curStartPos, idx + 1);
                            return retval;
                        }
                        retval.valid = false;
                        break;

                    case SyntaxObjectType.CONDITION_FIELD:
                        conditionRelationAC = AdConditionFieldAC.getFieldRelationshipAutoCompleter(curConditionField);
                        if (conditionRelationAC == null)
                        {
                            retval.setErr(SyntaxError.CONDITION_CANT_CREATE_RRELATIONS_AC, curStartPos, idx + 1);
                            return retval;
                        }
                        if (idx + 1 < searchCharArr.Length)
                        {
                            string tryNextObj = searchText.Substring(curStartPos, idx - curStartPos + 2).ToUpper();
                            if (conditionRelationAC.validate(tryNextObj))
                            {
                                break;
                            }
                        }
                        if (!conditionRelationAC.validate(nextObject))
                        {
                            if (!conditionRelationAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.INVALID_CONDITION_RELATION, curStartPos, idx + 1);
                                return retval;
                            }  
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.CONDITION_RELATION, nextObject, curStartPos, idx + 1);
                        }
                        curStartPos = idx + 1;
                        retval.valid = false;
                        
                        break;
                    case SyntaxObjectType.CONDITION_RELATION:
                        bool addObjFlag = false;
                        if (curChar == '"')
                        {
                            betweenDoubleQuotes = (!betweenDoubleQuotes);
                            if (betweenDoubleQuotes)
                            {
                                if (!mFirstDQRegexp.IsMatch(strRealObj))
                                {
                                    retval.setErr(SyntaxError.INVALID_CONDITION_VALUE, curStartPos, idx + 1);
                                    return retval;
                                }
                            }
                            else
                            {
                                strRealObj = strRealObj.Trim(new[] { '\"' });
                                addObjFlag = true;
                            }
                        }
                        //Doing this condition to identify whether this is the last searchObject and no space is predicted !!
                        if (final)
                        {
                            if (((curChar == ' ') || (idx + 1 == searchCharArr.Length)) && (betweenDoubleQuotes == false) && (addObjFlag == false))
                            {
                                strRealObj = strRealObj.Trim();
                                if (mNonSpaceRegexp.IsMatch(strRealObj))
                                {
                                    addObjFlag = true;
                                }
                                else
                                {
                                    curStartPos = idx + 1;
                                }
                            }
                        }
                        else
                        {
                            if ((curChar == ' ') && (betweenDoubleQuotes == false) && (addObjFlag == false))
                            {
                                strRealObj = strRealObj.Trim();
                                if (mNonSpaceRegexp.IsMatch(strRealObj))
                                {
                                    addObjFlag = true;
                                }
                                else
                                {
                                    curStartPos = idx + 1;
                                }
                            }
                        }
                        if (addObjFlag)
                        {
                            if (!AdConditionFieldAC.validateFieldValue(curConditionField, strRealObj))
                            {
                                retval.setErr(SyntaxError.INVALID_CONDITION_VALUE, curStartPos, idx);
                                return retval;
                            }
                            else
                            {
                                retval.addSyntaxObject(SyntaxObjectType.CONDITION_VALUE, strRealObj, curStartPos, idx + 1);
                                curConditionField = string.Empty;
                            }
                            curStartPos = idx + 1;
                            retval.valid = true;
                        }
                        break;
                    default:
                        retval.setErr(SyntaxError.UNIDENTIFIED_STATE, curStartPos, idx);
                        return retval;
                }
            }
            return retval;
            
        }

        public SyntaxContainer getCompletion(string searchText)
        {
            SyntaxContainer retval = analyzeSyntaxState(searchText, false);
            IConditionFieldAutoCompleter AdConditionFieldAC;
            if (retval.Error == SyntaxError.NO_ERROR)
            {
                if (searchText.ToUpper().Contains("ADUSER"))
                {
                    AdConditionFieldAC = new AdUserConditionFieldAutoCompleter();
                }
                else
                {
                    AdConditionFieldAC = new AdGroupConditionFieldAutoCompleter();
                }
                IAutoCompleter conditionRelationAC;
                IConditionValueAutoCompleter conditionValueAC;
                int lastIdx = retval.getLastHandledIndex();
                string curPartialWord = "";
                if (lastIdx < searchText.Length)
                {
                    curPartialWord = searchText.Substring(lastIdx, searchText.Length - lastIdx);
                    curPartialWord = curPartialWord.Trim();
                }
                SyntaxObjectType curState = retval.getState();
                for (int idx = 0; idx < mStateMap[curState].Length; idx++)
                {
                    switch (mStateMap[curState][idx])
                    {
                        case SyntaxObjectType.SEARCH_OBJECT:
                            retval.addToACList( mSearchObjectAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.COLON:
                            retval.addToACList(mColonAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.CONDITION_FIELD:
                            string[] tmpCompletions = AdConditionFieldAC.getCompletion(curPartialWord);
                            List<string> nonDuplicates = new List<string>();
                            for (int itr = 0; itr < tmpCompletions.Length; itr++)
                            {
                                if (!retval.contains(SyntaxObjectType.CONDITION_FIELD, tmpCompletions[itr]))
                                {
                                    nonDuplicates.Add(tmpCompletions[itr]);
                                }
                            }
                            retval.addToACList(nonDuplicates.ToArray());
                            break;
                        case SyntaxObjectType.CONDITION_RELATION:
                            conditionRelationAC = AdConditionFieldAC.getFieldRelationshipAutoCompleter(retval.getPreviousSyntaxObject(1, SyntaxObjectType.CONDITION_FIELD));
                            if (conditionRelationAC != null)
                            {
                                retval.addToACList(conditionRelationAC.getCompletion(curPartialWord));
                            }
                            break;
                        case SyntaxObjectType.CONDITION_VALUE:
                            conditionValueAC = AdConditionFieldAC.getFieldValueAutoCompleter(retval.getPreviousSyntaxObject(2, SyntaxObjectType.CONDITION_FIELD));
                            if (conditionValueAC != null)
                            {
                                retval.addToACList(conditionValueAC.getCompletion(curPartialWord));
                            }
                            break;
                    }
                }
            }
            return retval;
        }

        public string generateQueryFromSyntaxContainer(SyntaxContainer syntax)
        {
            string retval = string.Empty;
            if (syntax.valid)
            {
                retval = generateAdQueryFromSyntaxContainer(syntax);     
            }
            return retval;
        }

        private static string generateAdQueryFromSyntaxContainer(SyntaxContainer syntax)
        {
            StringBuilder retval = new StringBuilder();
            if (syntax.valid)
            {
                IConditionFieldAutoCompleter conditionFieldAC;
                if (syntax.getSearchObjectStr().ToUpper().Contains("ADUSER"))
                {
                    retval.Append("(&(objectCategory=Person)(objectClass=user)");
                    conditionFieldAC = new AdUserConditionFieldAutoCompleter();
                }
                else
                {
                    retval.Append("(&(objectCategory=group)");
                    conditionFieldAC = new AdGroupConditionFieldAutoCompleter();
                }
                StringBuilder phrase = new StringBuilder();
                bool nonEqual = false;
				bool findAll = false;
                foreach (SyntaxObject so in syntax)
                {
                    switch (so.Type)
                    {
                        case SyntaxObjectType.CONDITION_FIELD:
							if (so.Body == "ALLNAMES")
							{
								phrase.Append(" (|(givenname={value})(sn={value})(samaccountname={value})(userPrincipalName={value}))");
								/// mark this search as findAll for later use
								findAll = true;
							}
							else
							{
								phrase.Append(string.Format(" ({0}", conditionFieldAC.getDbFieldName(so.Body)));
							}
                            break;
                        case SyntaxObjectType.CONDITION_RELATION:
							/// append '=' only if not finding all
							if (!findAll)
							{
								phrase.Append("=");
							}
                            if (so.Body == "!=")
                            {
                                nonEqual = true;
                            }
                            break;
                        case SyntaxObjectType.CONDITION_VALUE:
							if (findAll)
							{
								/// replace all {value} occurences with the value searched
								phrase.Replace("{value}", so.Body);
							}
							else
							{
								phrase.Append(string.Format("{0})", so.Body));
							}
                            if (nonEqual)
                            {
                                retval.Append(string.Format("(!{0})", phrase));
                            }
                            else
                            {
                                retval.Append(phrase.ToString());
                            }
							nonEqual = false;
							findAll = false;
                            phrase.Remove(0, phrase.Length);
                            break;
                        default:
                            break;
                    }
                    
                }
                
                
            }
            retval.Append(")");
            return retval.ToString();

        }
    }
}
