﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VdcCommon.BusinessEntities;
using System.Xml;

namespace VdcCommon.Ovf
{
    public class OvfVmReader : OvfReader
    {
        protected VM _vm;

        public OvfVmReader(XmlDocument document, VM vm, List<DiskImage> images)
            :base (document ,images)
        {
            _vm = vm;
        }

        protected override void ReadOsSection(XmlNode section)
        {
            _vm.StaticData.vm_guid = new Guid(section.Attributes["ovf:id"].Value);
            _vm.StaticData.os = (VmOsType)Enum.Parse(typeof(VmOsType), section.SelectSingleNode("Description").InnerText);
        } 

        protected override void ReadHardwareSection(XmlNode section)
        {
            XmlNodeList list = section.SelectNodes("Item");
            foreach (XmlNode node in list)
            {
                string resourceType = node.SelectSingleNode("rasd:ResourceType", _xmlNS).InnerText;

                if (resourceType == OvfHardware.CPU)
                {
                    _vm.StaticData.num_of_sockets = Convert.ToInt32(node.SelectSingleNode("rasd:num_of_sockets", _xmlNS).InnerText);
                    _vm.StaticData.cpu_per_socket = Convert.ToInt32(node.SelectSingleNode("rasd:cpu_per_socket", _xmlNS).InnerText);
                }
                else if (resourceType == OvfHardware.Memory)
                {
                    _vm.StaticData.mem_size_mb = Convert.ToInt32(node.SelectSingleNode("rasd:VirtualQuantity", _xmlNS).InnerText);
                }
                else if (resourceType == OvfHardware.DiskImage)
                {
                    Guid guid = new Guid(node.SelectSingleNode("rasd:InstanceId", _xmlNS).InnerText);
                    DiskImage image = _images.FirstOrDefault(img => img.image_guid == guid);
                    string drive = node.SelectSingleNode("rasd:Caption", _xmlNS).InnerText;
                    if (drive.StartsWith("Drive "))
                    {
                        image.internal_drive_mapping = drive.Substring(6);
                    }
                    image.image_group_id = OvfParser.GetImageGrupIdFromImageFile(node.SelectSingleNode("rasd:HostResource", _xmlNS).InnerText);
                    if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:Parent", _xmlNS).InnerText))
                    {
                        image.ParentId = new Guid(node.SelectSingleNode("rasd:Parent", _xmlNS).InnerText);
                    }
                    if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:Template", _xmlNS).InnerText))
                    {
                        image.it_guid = new Guid(node.SelectSingleNode("rasd:Template", _xmlNS).InnerText);
                    }
                    image.appList = node.SelectSingleNode("rasd:ApplicationList", _xmlNS).InnerText;
                    if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:StorageId", _xmlNS).InnerText))
                    {
                        image.storage_id = new Guid(node.SelectSingleNode("rasd:StorageId", _xmlNS).InnerText);
                    }
                    if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:StoragePoolId", _xmlNS).InnerText))
                    {
                        image.storage_pool_id = new Guid(node.SelectSingleNode("rasd:StoragePoolId", _xmlNS).InnerText);
                    }
                    DateTime creationDate;
                    if (OvfParser.UtcDateStringToLocaDate(node.SelectSingleNode("rasd:CreationDate", _xmlNS).InnerText, out creationDate))
                    {
                        image.creation_date = creationDate;
                    }
                    DateTime lastModified;
                    if (OvfParser.UtcDateStringToLocaDate(node.SelectSingleNode("rasd:LastModified", _xmlNS).InnerText, out lastModified))
                    {
                        image.lastModified = lastModified;
                    }
                    DateTime last_modified_date;
                    if (OvfParser.UtcDateStringToLocaDate(node.SelectSingleNode("rasd:last_modified_date", _xmlNS).InnerText, out last_modified_date))
                    {
                        image.last_modified_date = last_modified_date;
                    }
                }
                else if (resourceType == OvfHardware.Network)
                {
                    Interface iface = new Interface();
                    if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:ResourceSubType", _xmlNS).InnerText))
                    {
                        iface.type = Int32.Parse(node.SelectSingleNode("rasd:ResourceSubType", _xmlNS).InnerText);
                    }
                    iface.network_name = node.SelectSingleNode("rasd:Connection", _xmlNS).InnerText;
                    iface.name = node.SelectSingleNode("rasd:Name", _xmlNS).InnerText;
                    iface.mac_addr = (node.SelectSingleNode("rasd:MACAddress", _xmlNS) != null) ? node.SelectSingleNode("rasd:MACAddress", _xmlNS).InnerText : string.Empty;
                    _vm.Interfaces.Add(iface);
                }
                else if (resourceType == OvfHardware.USB)
                {
                    _vm.StaticData.usb_policy = (UsbPolicy)Enum.Parse(typeof(UsbPolicy), node.SelectSingleNode("rasd:UsbPolicy", _xmlNS).InnerText);
                }
                else if (resourceType == OvfHardware.Monitor)
                {

                    _vm.StaticData.num_of_monitors = Int32.Parse(node.SelectSingleNode("rasd:VirtualQuantity", _xmlNS).InnerText);
                }
            }
        }

        protected override void ReadGeneralData()
        {
            // General Vm
            XmlNode content = _document.SelectSingleNode("//*/Content");

            XmlNode node = content.SelectSingleNode("Name");
            if (node != null)
            {
                _vm.StaticData.vm_name = node.InnerText;
            }
            node = content.SelectSingleNode("TemplateId");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vm.StaticData.vmt_guid = new Guid(node.InnerText);
                }
            }
            node = content.SelectSingleNode("TemplateName");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vm.TemplateName = node.InnerText;
                }
            }
            node = content.SelectSingleNode("Description");
            if (node != null)
            {
                _vm.StaticData.description = node.InnerText;
            }
            node = content.SelectSingleNode("Domain");
            if (node != null)
            {
                _vm.StaticData.domain = node.InnerText;
            }
            node = content.SelectSingleNode("CreationDate");
            DateTime creationDate;
            if (node != null && OvfParser.UtcDateStringToLocaDate(node.InnerText, out creationDate))
            {
                _vm.StaticData.creation_date = creationDate;
            }
            node = content.SelectSingleNode("IsInitilized");
            if (node != null)
            {
                _vm.StaticData.is_initialized = bool.Parse(node.InnerText);
            }
            node = content.SelectSingleNode("IsAutoSuspend");
            if (node != null)
            {
                _vm.StaticData.is_auto_suspend = bool.Parse(node.InnerText);
            }
            node = content.SelectSingleNode("TimeZone");
            if (node != null)
            {
                _vm.StaticData.time_zone = node.InnerText;
            }
			node = content.SelectSingleNode("IsStateless");
			if (node != null)
			{
				_vm.StaticData.is_stateless = bool.Parse(node.InnerText);
			}

            XmlNodeList list = content.SelectNodes("Section");
            foreach (XmlNode section in list)
            {
                string value = section.Attributes["xsi:type"].Value;

                switch (value)
                {
                    case "ovf:OperatingSystemSection_Type":
                        ReadOsSection(section);
                        break;

                    case "ovf:VirtualHardwareSection_Type":
                        ReadHardwareSection(section);
                        break;
                }
            }
            node = content.SelectSingleNode("Origin");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vm.origin = (OriginType)Convert.ToInt32(node.InnerText);
                }
            }
            node = content.SelectSingleNode("VmType");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vm.vm_type = (VmType)Convert.ToInt32(node.InnerText);
                }
            }
            node = content.SelectSingleNode("DefaultDisplayType");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vm.default_display_type = (DisplayType)Convert.ToInt32(node.InnerText);
                }
            }
        }
    }
}
