using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using VdcCommon.Interfaces;
using VdcCommon.VdcQueries;
using VdcFrontend;
using System.ComponentModel;
using System.Collections;
using VdcCommon.BusinessEntities;
using VdcCommon;

namespace org.ovirt.engine.ui.uicommon.models.storage
{
	public class StorageDataCenterListModel : SearchableListModel
	{
		#region Commands

		public UICommand AttachCommand { get; private set; }
		public UICommand DetachCommand { get; private set; }
		public UICommand ActivateCommand { get; private set; }
		public UICommand MaintenanceCommand { get; private set; }

		#endregion

		#region Properties

		public new storage_domains Entity
		{
			get { return (storage_domains)base.Entity; }
			set { base.Entity = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		/// <summary>
		/// Gets the value indicating whether multiple data centers
		/// can be selected to attach storage to.
		/// </summary>
		public bool AttachMultiple { get; private set; }

		#endregion

		public StorageDataCenterListModel()
		{
			Title = "Data Center";

			AttachCommand = new UICommand("Attach", this);
			DetachCommand = new UICommand("Detach", this);
			ActivateCommand = new UICommand("Activate", this);
			MaintenanceCommand = new UICommand("Maintenance", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			SearchCommand.Execute();
			UpdateActionAvailability();
		}

		public override void Search()
		{
			if (Entity != null)
			{
				base.Search();
			}
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
				{
					SearchableListModel searchableListModel = (SearchableListModel)model;
					searchableListModel.Items = (List<storage_domains>)((VdcQueryReturnValue)ReturnValue).ReturnValue;
					IsEmpty = ((IList)searchableListModel.Items).Count == 0;
				};//END_DELEGATE

			Frontend.RunQuery(VdcQueryType.GetStorageDomainListById,
				new StorageDomainQueryParametersBase(Entity.id),
				_asyncQuery);
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetStorageDomainListById, new StorageDomainQueryParametersBase(Entity.id));
			Items = AsyncResult.Data;
		}

		private void Attach()
		{
			if (Window != null)
			{
				return;
			}

			AttachMultiple = Entity.storage_domain_type == StorageDomainType.ISO;

			ListModel model = new ListModel();
			Window = model;
			model.Title = "Attach to Data Center";

			List<EntityModel> items = new List<EntityModel>();

			foreach (storage_pool dataCenter in DataProvider.GetDataCenterList())
			{
				bool add = false;

				switch (Entity.storage_domain_type)
				{
					case StorageDomainType.Master:
					case StorageDomainType.Data:
						{
							if ((dataCenter.status == StoragePoolStatus.Uninitialized || dataCenter.status == StoragePoolStatus.Up)
								&&
								(dataCenter.StoragePoolFormatType == null || dataCenter.StoragePoolFormatType.Value == Entity.StorageStaticData.StorageFormat)
								&& dataCenter.storage_pool_type == Entity.storage_type)
							{
								add = true;
							}
						}
						break;
					case StorageDomainType.ISO:
						{
							if (dataCenter.status == StoragePoolStatus.Up)
							{
								storage_domains isoStorage = DataProvider.GetIsoDomainByDataCenterId(dataCenter.Id);
								if (isoStorage == null)
								{
									add = true;
								}
							}
						}
						break;
					case StorageDomainType.ImportExport:
						{
							if (dataCenter.status == StoragePoolStatus.Up)
							{
								storage_domains exportStorage = DataProvider.GetExportDomainByDataCenterId(dataCenter.Id);
								if (exportStorage == null)
								{
									add = true;
								}
							}
						}
						break;
				}

				if (add)
				{
					items.Add(new EntityModel { Entity = dataCenter });
				}
			}

			model.Items = items;


			if (items.Count == 0)
			{
				model.Message = "There are No Data Centers to which the Storage Domain can be attached";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnAttach", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnAttach()
		{
			ListModel model = (ListModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (Entity == null)
			{
				Cancel();
				return;
			}


			List<storage_pool> items = new List<storage_pool>();
			foreach (EntityModel a in Linq.Cast<EntityModel>(model.Items))
			{
				if (a.IsSelected)
				{
					items.Add((storage_pool)a.Entity);
				}
			}


			if (items.Count > 0)
			{
				model.StartProgress(null);

				List<VdcActionParametersBase> parameters = new List<VdcActionParametersBase>();
				foreach (storage_pool dataCenter in items)
				{
					parameters.Add(new StorageDomainPoolParametersBase(Entity.id, dataCenter.Id));
				}

				Frontend.RunMultipleAction(VdcActionType.AttachStorageDomainToPool, parameters,
					result =>
					{
						ListModel localModel = (ListModel)result.State;

						localModel.StopProgress();
						Cancel();
					},
					model
				);
			}
			else
			{
				Cancel();
			}
		}

		private void Detach()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Detach Storage";
			model.HashName = "detach_storage";
			model.Message = "Are you sure you want to Detach storage from the following Data Center(s)?";

			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				storage_domains a = (storage_domains)item;
				items.Add(a.storage_pool_name);
			}
			model.Items = items;

			if (ContainsLocalStorage(model))
			{
				model.Latch.IsAvailable = true;
				model.Latch.IsChangable = true;

				model.Note = "Note: " + GetLocalStoragesFormattedString() + " will be removed!";
			}

			model.Commands.Add(
				new UICommand("OnDetach", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private string GetLocalStoragesFormattedString()
		{
			string localStorages = "";
			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				if (a.storage_type == StorageType.LOCALFS)
				{
					localStorages += a.storage_name + ", ";
				}
			}
			return localStorages.Remove(localStorages.Length - 2);
		}

		private bool ContainsLocalStorage(ConfirmationModel model)
		{
			foreach (storage_domains a in Linq.Cast<storage_domains>(SelectedItems))
			{
				if (a.storage_type == StorageType.LOCALFS)
				{
					return true;
				}
			}
			return false;
		}

		private void OnDetach()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (!model.Validate())
			{
				return;
			}

			List<VdcActionParametersBase> detachPrms = new List<VdcActionParametersBase>();
			List<VdcActionParametersBase> removePrms = new List<VdcActionParametersBase>();

			foreach (object item in SelectedItems)
			{
				storage_domains a = (storage_domains)item;
				if (a.storage_type != StorageType.LOCALFS)
				{
					DetachStorageDomainFromPoolParameters param = new DetachStorageDomainFromPoolParameters();
					param.StorageDomainId = Entity.id;
					if (a.storage_pool_id != null)
					{
						param.StoragePoolId = a.storage_pool_id.Value;
					}

					detachPrms.Add(param);
				}
				else
				{
					VDS locaVds = DataProvider.GetLocalStorageHost(a.storage_pool_name);
					RemoveStorageDomainParameters removeStorageDomainParameters =
						new RemoveStorageDomainParameters(a.id)
						{
							VdsId = (locaVds != null ? locaVds.vds_id : null),
							DoFormat = true
						};
					removePrms.Add(removeStorageDomainParameters);
				}
			}

			Frontend.RunMultipleAction(VdcActionType.DetachStorageDomainFromPool, detachPrms);
			Frontend.RunMultipleAction(VdcActionType.RemoveStorageDomain, removePrms);

			Cancel();
		}

		private void Maintenance()
		{
			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				storage_domains a = (storage_domains)item;

				StorageDomainPoolParametersBase parameters = new StorageDomainPoolParametersBase();
				parameters.StorageDomainId = Entity.id;
				if (a.storage_pool_id != null)
				{
					parameters.StoragePoolId = a.storage_pool_id.Value;
				}

				list.Add(parameters);
			}

			Frontend.RunMultipleAction(VdcActionType.DeactivateStorageDomain, list,
				result =>
				{
				},
				null
			);
		}

		private void Activate()
		{
			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				storage_domains a = (storage_domains)item;

				StorageDomainPoolParametersBase parameters = new StorageDomainPoolParametersBase();
				parameters.StorageDomainId = Entity.id;
				if (a.storage_pool_id != null)
				{
					parameters.StoragePoolId = a.storage_pool_id.Value;
				}

				list.Add(parameters);
			}

			Frontend.RunMultipleAction(VdcActionType.ActivateStorageDomain, list,
				result =>
				{
				},
				null
			);
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			List<storage_domains> items = SelectedItems != null ? Linq.Cast<storage_domains>(SelectedItems) : new List<storage_domains>();

			ActivateCommand.IsExecutionAllowed = items.Count == 1
				&& VdcActionUtils.CanExecute(items, typeof(storage_domains), VdcActionType.ActivateStorageDomain);

			MaintenanceCommand.IsExecutionAllowed = items.Count == 1
				&& VdcActionUtils.CanExecute(items, typeof(storage_domains), VdcActionType.DeactivateStorageDomain);

			AttachCommand.IsExecutionAllowed = Entity != null
				&& (Entity.storage_domain_shared_status == StorageDomainSharedStatus.Unattached
					|| Entity.storage_domain_type == StorageDomainType.ISO);

			DetachCommand.IsExecutionAllowed = items.Count > 0
				&& VdcActionUtils.CanExecute(items, typeof(storage_domains), VdcActionType.DetachStorageDomainFromPool);
		}


		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == AttachCommand)
			{
				Attach();
			}
			else if (command == DetachCommand)
			{
				Detach();
			}
			else if (command == ActivateCommand)
			{
				Activate();
			}
			else if (command == MaintenanceCommand)
			{
				Maintenance();
			}
			else if (command.Name == "OnAttach")
			{
				OnAttach();
			}
			else if (command.Name == "OnDetach")
			{
				OnDetach();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
