﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VdcCommon.BusinessEntities;
using System.Xml;
using System.IO;
using System.Text.RegularExpressions;

namespace VdcCommon.Ovf
{
    public abstract class OvfWriter: IOvfBuilder, IDisposable
    {
        protected string _fileName;
        protected int _instanceId;
        protected List<DiskImage> _images;
        protected XmlTextWriter _writer;
        protected XmlDocument _document;

        public OvfWriter(out XmlDocument document, List<DiskImage> images)
        {
            _fileName = Path.GetTempFileName();
            document = new XmlDocument();
            _document = document;
            _images = images;
            _writer = new XmlTextWriter(_fileName, Encoding.UTF8);
            WriteHeader();
        }

        private void WriteHeader()
        {
            _instanceId = 0;
            _writer.Formatting = Formatting.Indented; 
            _writer.Indentation = 4;
            _writer.WriteStartDocument(false);
            _writer.WriteStartElement("ovf", "Envelope", "http://schemas.dmtf.org/ovf/envelope/1/");
            _writer.WriteAttributeString("xmlns", "rasd", null, "http://schemas.dmtf.org/wbem/wscim/1/cim-schema/2/CIM_ResourceAllocationSettingData");
            _writer.WriteAttributeString("xmlns", "vssd", null, "http://schemas.dmtf.org/wbem/wscim/1/cim-schema/2/CIM_VirtualSystemSettingData");
            _writer.WriteAttributeString("xmlns", "xsi", null, "http://www.w3.org/2001/XMLSchema-instance");
            _writer.WriteAttributeString("ovf", "version", null, "0.9");
        }

        private void CloseElements()
        {
            _writer.WriteEndElement();
        }

        protected long BytesToGigabyte(long bytes)
        {
            return bytes / 1024 / 1024 / 1024;
        } 
        #region IOvfBuilder Members
        public void BuildReference()
        {
            _writer.WriteStartElement("References");
            foreach (DiskImage image in _images)
            {
                _writer.WriteStartElement("File");
				_writer.WriteAttributeString("ovf", "href", null, OvfParser.CreateImageFile(image));
                _writer.WriteAttributeString("ovf", "id", null, image.image_guid.ToString());
                _writer.WriteAttributeString("ovf", "size", null, image.size.ToString());
                _writer.WriteAttributeString("ovf", "description", null, image.description);
                _writer.WriteEndElement();

            }
            _writer.WriteEndElement();
        }

        public void BuildNetwork()
        {
            _writer.WriteStartElement("Section");
            _writer.WriteAttributeString("xsi", "type", null, "ovf:NetworkSection_Type");
            _writer.WriteStartElement("Info");
            _writer.WriteRaw("List of networks");
            _writer.WriteEndElement();
            _writer.WriteStartElement("Network");
            _writer.WriteAttributeString("ovf", "name", null, "Network 1");
            _writer.WriteEndElement();
            _writer.WriteEndElement(); 
        }

        public void BuildDisk()
        {
            _writer.WriteStartElement("Section");
            _writer.WriteAttributeString("xsi", "type", null, "ovf:DiskSection_Type");
            _writer.WriteStartElement("Info");
            _writer.WriteRaw("List of Virtual Disks");
            _writer.WriteEndElement();
            foreach (DiskImage image in _images)
            {
                _writer.WriteStartElement("Disk");
                _writer.WriteAttributeString("ovf", "diskId", null, image.image_guid.ToString());
                _writer.WriteAttributeString("ovf", "size", null, BytesToGigabyte(image.size).ToString());
                _writer.WriteAttributeString("ovf", "actual_size", null, BytesToGigabyte(image.actual_size).ToString());
                _writer.WriteAttributeString("ovf", "vm_snapshot_id", null, (image.vm_snapshot_id.HasValue) ? image.vm_snapshot_id.Value.ToString() : string.Empty);

                if (image.ParentId == Guid.Empty)
                {
                    _writer.WriteAttributeString("ovf", "parentRef", null, string.Empty);
                }
                else
                {
                    List<DiskImage> res = _images.SkipWhile(img => image.ParentId == img.image_guid).ToList();
                    if (res.Count > 0)
                    {
						_writer.WriteAttributeString("ovf", "parentRef", null, OvfParser.CreateImageFile(res[0]));
                    }
                    else
                    {
                        _writer.WriteAttributeString("ovf", "parentRef", null, string.Empty);
                    } 
                }

				_writer.WriteAttributeString("ovf", "fileRef", null, OvfParser.CreateImageFile(image));

				string format = string.Empty;
                switch (image.volume_format)
                {
                    case VolumeFormat.RAW:
    				    format = @"http://www.vmware.com/specifications/vmdk.html#sparse";
                        break;

                    case VolumeFormat.COW:
				        format = @"http://www.gnome.org/~markmc/qcow-image-format.html";
                        break;

                    case VolumeFormat.Unassigned:
                        break;
                } 
				_writer.WriteAttributeString("ovf", "format", null, format); 
				_writer.WriteAttributeString("ovf", "volume-format", null, image.volume_format.ToString()); 
				_writer.WriteAttributeString("ovf", "volume-type", null, image.volume_type.ToString());
				_writer.WriteAttributeString("ovf", "disk-interface", null, image.disk_interface.ToString());
                _writer.WriteAttributeString("ovf", "disk-type", null, image.disk_type.ToString());
                _writer.WriteAttributeString("ovf", "boot", null, image.boot.ToString());
                _writer.WriteAttributeString("ovf", "wipe-after-delete", null, image.wipe_after_delete.ToString());
                _writer.WriteEndElement();
            }
            _writer.WriteEndElement();
        }

        public void BuildVirtualSystem()
        {
            // General Vm
            _writer.WriteStartElement("Content");
            _writer.WriteAttributeString("ovf", "id", null, "out");
            _writer.WriteAttributeString("xsi", "type", null, "ovf:VirtualSystem_Type");

            // General Data
            WriteGeneralData();
            
            // Application List
            WriteAppList();

            // Content Items
            WriteContentItems();

            _writer.WriteEndElement(); // End Content tag
        }
        #endregion

        protected abstract void WriteGeneralData();

        protected abstract void WriteAppList(); 

        protected abstract void WriteContentItems(); 

        #region IDisposable Members
        public void Dispose()
        {
            if (_writer != null)
            {
                CloseElements(); 
                _writer.Close();
                _document.Load(_fileName);
            }
        }
        #endregion
    }
}