﻿USE [rhevm]
GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

SET NOCOUNT ON
GO

print 'Upgrade script Started...'

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'permissions'))
begin
	CREATE TABLE [permissions]
	(
		[id] uniqueidentifier NOT NULL,
		
		[role_id] uniqueidentifier NOT NULL,
		[is_readonly] bit  NOT NULL ,
		[ad_element_id] uniqueidentifier   NOT NULL ,
	constraint pk_permissions_id primary key ([id])
	) 
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'roles'))
begin
	CREATE TABLE [roles]
	(
		[id] uniqueidentifier not null,
		[name] Nvarchar(126)  NOT NULL,
		[description] Nvarchar(MAX),
		[is_readonly] bit not null,
	constraint pk_roles_id primary key ([id])
	)
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'roles_actions'))
begin

	CREATE TABLE [roles_actions]
	(
	[role_id] uniqueidentifier not null,
	[action_id] int not null,
	constraint pk_roles_actions primary key ([role_id],[action_id])
	)
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='fk_roles_actions_action_id' AND TABLE_NAME='roles_actions')
begin
	ALTER TABLE [roles_actions] ADD CONSTRAINT [fk_roles_actions_action_id] FOREIGN KEY ([role_id]) REFERENCES [roles] (id) ON DELETE CASCADE
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'roles_relations'))
begin
	CREATE TABLE [roles_relations]
	(
	[role_id] uniqueidentifier not null,
	[role_container_id] uniqueidentifier not null,
	constraint [pk_roles_relations] primary key ([role_id],[role_container_id])
	)
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='fk_roles_relations_role_id' AND TABLE_NAME='roles_relations')
begin
	ALTER TABLE [roles_relations] ADD CONSTRAINT [fk_roles_relations_role_id] FOREIGN KEY ([role_id]) REFERENCES [roles] (id)
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='fk_roles_relations_container_id' AND TABLE_NAME='roles_relations')
begin
	ALTER TABLE [roles_relations] ADD CONSTRAINT [fk_roles_relations_container_id] FOREIGN KEY ([role_container_id]) REFERENCES [roles] (id)
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'tags_permissions_map'))
begin
	CREATE TABLE [tags_permissions_map]
	(
		[tag_id] int not null,
		[permission_id] uniqueidentifier NOT NULL,
		constraint [pk_tags_permissions_map] primary key ([tag_id],[permission_id])
	)
end
GO 

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='fk_tags_permissions_tag_id' AND TABLE_NAME='tags_permissions_map')
begin
	ALTER TABLE [tags_permissions_map] ADD CONSTRAINT [fk_tags_permissions_tag_id] FOREIGN KEY ([tag_id]) REFERENCES [tags] (tag_id)
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='fk_roles_relations_permission_id' AND TABLE_NAME='tags_permissions_map')
begin
	ALTER TABLE [tags_permissions_map] ADD CONSTRAINT [fk_roles_relations_permission_id] FOREIGN KEY ([permission_id]) REFERENCES [permissions] (id)
end
GO

-- Upgrade audit_log primary key to be bigint instead of int
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'audit_log' and COLUMN_NAME = 'audit_log_id' and DATA_TYPE ='bigint'))
begin
	CREATE TABLE [dbo].[audit_log_temp](
	[audit_log_id] [bigint] IDENTITY(0,1) NOT NULL,
	[user_id] [uniqueidentifier] NULL,
	[user_name] [nvarchar](40)  NULL,
	[vm_id] [uniqueidentifier] NULL,
	[vm_name] [nvarchar](40)  NULL,
	[vm_template_id] [uniqueidentifier] NULL,
	[vm_template_name] [nvarchar](40)  NULL,
	[vds_id] [int] NULL,
	[vds_name] [nvarchar](40)  NULL,
	[log_time] [datetime] NOT NULL,
	[log_type] [int] NOT NULL,
	[severity] [int] NOT NULL,
	[message] [nvarchar](max)  NOT NULL,
	[storage_pool_id] UNIQUEIDENTIFIER,
	[storage_pool_name] NVARCHAR(40),
	[storage_domain_id] UNIQUEIDENTIFIER,
	[storage_domain_name] NVARCHAR(250),
	Primary Key ([audit_log_id])
	)
end
go

if (exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'audit_log_temp'))
begin
	insert into audit_log_temp (user_id,user_name,vm_id,vm_name,vm_template_id,vm_template_name,vds_id,vds_name,log_time,log_type,severity,message) 
	(select user_id,user_name,vm_id,vm_name,vm_template_id,vm_template_name,vds_id,vds_name,log_time,log_type,severity,message from audit_log)
end
go

if (exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'audit_log_temp'))
begin
	drop table [dbo].[audit_log]
end
go

if (exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'audit_log_temp'))
begin
	exec sp_rename 'audit_log_temp' , 'audit_log'
end
go

-- add non clustered index on log_time (desc)
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='audit_log' and sys.indexes.name = 'IDX_audit_log_log_time'))
begin
CREATE NONCLUSTERED INDEX [IDX_audit_log_log_time] ON [dbo].[audit_log] 
(
	[log_time] DESC
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO


-- add non clustered index on user_name 
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='audit_log' and sys.indexes.name = 'IDX_audit_log_user_name'))
begin
CREATE NONCLUSTERED INDEX [IDX_audit_log_user_name] ON [dbo].[audit_log] 
(
	[user_name] 
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

-- add non clustered index on vm_name (
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='audit_log' and sys.indexes.name = 'IDX_audit_log_vm_name'))
begin
CREATE NONCLUSTERED INDEX [IDX_audit_log_vm_name] ON [dbo].[audit_log] 
(
	[vm_name] 
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

-- add non clustered index on vm_template_name 
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='audit_log' and sys.indexes.name = 'IDX_audit_log_vm_template_name'))
begin
CREATE NONCLUSTERED INDEX [IDX_audit_log_vm_template_name] ON [dbo].[audit_log] 
(
	[vm_template_name] 
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

-- add non clustered index on vds_name
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='audit_log' and sys.indexes.name = 'IDX_audit_log_vds_name'))
begin
CREATE NONCLUSTERED INDEX [IDX_audit_log_vds_name] ON [dbo].[audit_log] 
(
	[vds_name] 
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

-- add non clustered index on storage_pool_name 
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='audit_log' and sys.indexes.name = 'IDX_audit_log_storage_pool_name'))
begin
CREATE NONCLUSTERED INDEX [IDX_audit_log_storage_pool_name] ON [dbo].[audit_log] 
(
	[storage_pool_name] 
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

-- add non clustered index on storage_domain_name 
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='audit_log' and sys.indexes.name = 'IDX_audit_log_storage_domain_name'))
begin
CREATE NONCLUSTERED INDEX [IDX_audit_log_storage_domain_name] ON [dbo].[audit_log] 
(
	[storage_domain_name] 
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO


-- End of Upgrade audit_log

-- Power Management
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_static' and COLUMN_NAME = 'pm_type'))
begin
	ALTER TABLE vds_static ADD [pm_type] nvarchar(10) NULL,
 							   [pm_user] nvarchar(50) NULL,
							   [pm_password] nvarchar(50) NULL,
							   [pm_port] integer null,
							   [pm_secured] bit null,
							   [pm_enabled] bit default 0 not null					

end
go

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_static' and COLUMN_NAME = 'pm_type' and CHARACTER_MAXIMUM_LENGTH = 20))
begin
	alter table vds_static alter column pm_type nvarchar(20)
end
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_static' and COLUMN_NAME = 'pm_type' and CHARACTER_MAXIMUM_LENGTH = 20))
begin
	alter table vds_static alter column pm_type nvarchar(20)
end
GO


-- Add Creation & modification dates to vm/vds/groups for history usage
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_static' and COLUMN_NAME = '_create_date'))
begin

	ALTER TABLE vm_static ADD  [_create_date] datetime default getdate(),
 							   [_update_date] datetime default NULL
end
go

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_static' and COLUMN_NAME = '_create_date'))
begin

	ALTER TABLE vds_static ADD [_create_date] datetime default getdate(),
 							  [_update_date] datetime default NULL
end
go

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_groups' and COLUMN_NAME = '_create_date'))
begin
	ALTER TABLE vds_groups ADD [_create_date] datetime default getdate(),
 							   [_update_date] datetime default NULL
end
GO

update vm_static set [_create_date] = getdate() where [_create_date] is null
update vds_static set [_create_date] = getdate() where [_create_date] is null
update vds_groups set [_create_date] = getdate() where [_create_date] is null
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'tags_vm_map' and COLUMN_NAME = 'DefaultDisplayType'))
begin
	ALTER TABLE [tags_vm_map] ADD [DefaultDisplayType] INT CONSTRAINT [DEF_tags_vm_map_DefaultDisplayType] DEFAULT 0 
end
GO

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_static' and COLUMN_NAME = 'nic_type'))
begin
	ALTER TABLE vm_static DROP COLUMN [nic_type],[mac_addr_1],[mac_addr_2]
end
go

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_templates' and COLUMN_NAME = 'nic_type'))
begin
	ALTER TABLE vm_templates DROP COLUMN nic_type 
end
go

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'users' and COLUMN_NAME = 'vdc_role'))
begin
	ALTER TABLE [users] Drop column [vdc_role]
end
go

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'users' and COLUMN_NAME = 'groups' and CHARACTER_MAXIMUM_LENGTH != -1))
begin
	ALTER TABLE [users] ALTER COLUMN groups nvarchar(max) NOT NULL
end
go

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'ad_groups' and COLUMN_NAME = 'vdc_role'))
begin
	ALTER TABLE [ad_groups] Drop column [vdc_role]
end
go

IF EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='DEF_vm_statistics_rx_dropped_1' AND TABLE_NAME='vm_statistics')
begin

	ALTER TABLE [vm_statistics]  Drop CONSTRAINT [DEF_vm_statistics_rx_dropped_1],[DEF_vm_statistics_rx_rate_1],
												 [DEF_vm_statistics_tx_dropped_1],[DEF_vm_statistics_tx_rate_1],
												 [DEF_vm_statistics_rx_dropped_2],[DEF_vm_statistics_rx_rate_2],
												 [DEF_vm_statistics_tx_dropped_2],[DEF_vm_statistics_tx_rate_2]
end
go

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_statistics' and COLUMN_NAME = 'vm_if_id_1'))
begin

	ALTER TABLE  [vm_statistics]  DROP COLUMN [vm_if_id_1],[vm_if_name_1],[vm_line_rate_1],
											  [rx_dropped_1],[rx_rate_1],[tx_dropped_1],
											  [tx_rate_1],[vm_if_id_2],[vm_if_name_2],
											  [vm_line_rate_2],[rx_dropped_2],[rx_rate_2],
											  [tx_dropped_2], [tx_rate_2] 
end
GO

---StoragePool handling	
if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'storage_pool'))
begin

	CREATE TABLE [storage_pool]
	(
		[id] [uniqueidentifier] not null,
		[name] [nvarchar](40) NOT NULL,
		[description] [nvarchar](250) NOT NULL,
		[storage_pool_type] [int] NOT NULL,
		[status] [int] NOT NULL,
		[master_domain_version] [int] NOT NULL,
		[spm_vds_id] [int] NULL,
		constraint [pk_storage_pool] primary key ([id])
	)
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'storage_domain_static'))
begin
	CREATE TABLE [storage_domain_static]
	(
		[id] [uniqueidentifier] not null,
		[storage] [nvarchar](250) NOT NULL,
		[storage_name] [nvarchar](250) NOT NULL,
		[storage_pool_id] [uniqueidentifier] null,
		[storage_type] [int] NOT NULL,
		[storage_domain_type] [int] NOT NULL,
		constraint [pk_storage] primary key ([id])
	)
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='fk_storage_domain_static_storage_pool_id' AND TABLE_NAME='storage_domain_static')
begin
	ALTER TABLE [storage_domain_static] ADD CONSTRAINT [fk_storage_domain_static_storage_pool_id] FOREIGN KEY ([storage_pool_id]) REFERENCES [storage_pool] (id)
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'storage_server_connections'))
begin
	CREATE TABLE [dbo].[storage_server_connections](
		[id] [nvarchar](50)  NOT NULL,
		[connection] [nvarchar](250) NOT NULL,
		[user_name] [nvarchar](50)  NULL,
		[password] [nvarchar](50)  NULL,
		[iqn] [nvarchar](50)  NULL,
		[port] [nvarchar](50)  NULL,
		[portal] [nvarchar](50)  NULL,
		[storage_type] [int] NOT NULL,

	 CONSTRAINT [PK_storage_server] PRIMARY KEY
	(
		[id] 
	)
	) 
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'storage_domain_dynamic'))
begin
	CREATE TABLE [dbo].[storage_domain_dynamic](
		[id] [uniqueidentifier] NOT NULL,
		[available_disk_size] [int] NULL,
		[used_disk_size] [int] NULL,
		[status] [int] NULL,
	CONSTRAINT [PK_storage_domain_dynamic] PRIMARY KEY
	(
		[id] 
	)
	) 
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_storage_domain_dynamic_storage_domain_static' AND TABLE_NAME='storage_domain_dynamic')
begin

	ALTER TABLE [dbo].[storage_domain_dynamic]  ADD  CONSTRAINT [FK_storage_domain_dynamic_storage_domain_static] FOREIGN KEY([id])
	REFERENCES [dbo].[storage_domain_static] ([id])
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'LUNs'))
begin
	CREATE TABLE [dbo].[LUNs](
		[phisical_volume_id] [nvarchar](50) NULL,
		[LUN_id] [nvarchar](50) NOT NULL,
		[volume_group_id] [nvarchar](50) NULL,
	CONSTRAINT [PK_LUNs] PRIMARY KEY 
	(
		[LUN_id] ASC
	)
	) 
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'LUN_storage_server_connection_map'))
begin

	CREATE TABLE [dbo].[LUN_storage_server_connection_map](
		[LUN_id] [nvarchar](50) NOT NULL,
		[storage_server_connection] [nvarchar](50) NOT NULL,
	 CONSTRAINT [PK_LUN_storage_server_connection_map] PRIMARY KEY 
	(
		[LUN_id] ASC,
		[storage_server_connection] ASC
	)
	) 
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_LUN_storage_server_connection_map_LUNs' AND TABLE_NAME='LUN_storage_server_connection_map')
begin
	ALTER TABLE [dbo].[LUN_storage_server_connection_map]  ADD  CONSTRAINT [FK_LUN_storage_server_connection_map_LUNs] FOREIGN KEY([LUN_id])
    REFERENCES [dbo].[LUNs] ([LUN_id]) ON DELETE CASCADE

end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_LUN_storage_server_connection_map_storage_server_connections' AND TABLE_NAME='LUN_storage_server_connection_map')
begin

	ALTER TABLE [dbo].[LUN_storage_server_connection_map]   ADD  CONSTRAINT [FK_LUN_storage_server_connection_map_storage_server_connections] FOREIGN KEY([storage_server_connection])
	REFERENCES [dbo].[storage_server_connections] ([id]) ON DELETE CASCADE
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'storage_pool_iso_map'))
begin
	CREATE TABLE [dbo].[storage_pool_iso_map](
		[storage_id] [uniqueidentifier] NOT NULL,
		[storage_pool_id] [uniqueidentifier] NOT NULL,
	 CONSTRAINT [PK_storage_domain_pool_map] PRIMARY KEY  
	(
		[storage_id] ASC,
		[storage_pool_id] ASC
	)
	)
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_storage_domain_pool_map_storage_pool' AND TABLE_NAME='storage_pool_iso_map')
begin
	ALTER TABLE [dbo].[storage_pool_iso_map]  ADD  CONSTRAINT [FK_storage_domain_pool_map_storage_pool] FOREIGN KEY([storage_pool_id])
	REFERENCES [dbo].[storage_pool] ([id]) ON DELETE CASCADE
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_storage_domain_pool_map_storage_domain_static' AND TABLE_NAME='storage_pool_iso_map')
begin
	ALTER TABLE [dbo].[storage_pool_iso_map]  ADD  CONSTRAINT [FK_storage_domain_pool_map_storage_domain_static] FOREIGN KEY([storage_id])
	REFERENCES [dbo].[storage_domain_static] ([id]) ON DELETE CASCADE
end
GO

/* ---------------------------------------------------------------------- */
/* ---------------------------------------------------------------------- */
                             -- NETWORS --
/* ---------------------------------------------------------------------- */
/* ---------------------------------------------------------------------- */

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'vm_interface'))
begin
	CREATE TABLE [dbo].[vm_interface](
		[id] [uniqueidentifier] NOT NULL,
		[network_name] [nvarchar](50)  NULL,
		[vm_guid] [uniqueidentifier] NULL,
		[vmt_guid] [uniqueidentifier] NULL,
		[mac_addr] [nvarchar](20)  NOT NULL,
		[name] [nvarchar](50)  NOT NULL,
		[speed] [int] NULL,
		[type] [int] NULL,
	 CONSTRAINT [PK_vm_interface] PRIMARY KEY CLUSTERED 
	(
		[id] ASC
	)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'interface_statistics'))
begin
	CREATE TABLE [dbo].[interface_statistics](
		[id] [uniqueidentifier] NOT NULL,
		[vm_id] [uniqueidentifier] NULL,
		[vds_id] [int] NULL,
		[rx_rate] [decimal](18, 0) NULL,
		[tx_rate] [decimal](18, 0) NULL,
		[rx_drop] [decimal](18, 0) NULL,
		[tx_drop] [decimal](18, 0) NULL,
	 CONSTRAINT [PK_vm_interface_statistics] PRIMARY KEY CLUSTERED 
	(
		[id] ASC
	)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'vds_interface'))
begin
	CREATE TABLE [dbo].[vds_interface](
		[id] [uniqueidentifier] NOT NULL,
		[name] [nvarchar](50)  NOT NULL,
		[network_name] [nvarchar](50)  NULL,
		[vds_id] [int] NULL,
		[mac_addr] [nvarchar](20)  NULL,
		[is_bond] [bit] NULL CONSTRAINT [DF_vds_interface_is_bond]  DEFAULT ((0)),
		[bond_name] [nvarchar](50)  NULL,
		[bond_type] [int] NULL,
		[vlan_id] [int] NULL,
		[speed] [int] NULL,
		[addr] [nvarchar](20)  NULL,
		[subnet] [nvarchar](20)  NULL,
		[gateway] [nvarchar](20)  NULL,
		[type] [int] NULL,
	 CONSTRAINT [PK_vds_interface] PRIMARY KEY CLUSTERED 
	(
		[id] ASC
	)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'network'))
begin
	CREATE TABLE [dbo].[network](
		[id] [uniqueidentifier] NOT NULL,
		[name] [nvarchar](50)  NOT NULL,
		[description] [nvarchar](200)  NULL,
		[type] [int] NULL,
		[addr] [nvarchar](50)  NULL,
		[subnet] [nvarchar](20)  NULL,
		[vlan_id] [int] NULL,
		[stp] [bit] NOT NULL DEFAULT 0, 
		[storage_pool_id] [uniqueidentifier] NULL
	 CONSTRAINT [PK_network] PRIMARY KEY CLUSTERED 
	(
		[id] ASC
	)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'network_cluster'))
begin
	CREATE TABLE [dbo].[network_cluster](
		[network_id] [uniqueidentifier] NOT NULL,
		[cluster_id] [int] NOT NULL,
	 CONSTRAINT [PK_network_cluster] PRIMARY KEY CLUSTERED 
	(
		[network_id] ASC,
		[cluster_id] ASC
	)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_interface' and COLUMN_NAME = '_create_date'))
begin
	ALTER TABLE vm_interface ADD [_create_date] datetime default getdate(),
 							   [_update_date] datetime default NULL
end
go

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_interface' and COLUMN_NAME = '_create_date'))
begin
	ALTER TABLE vds_interface ADD [_create_date] datetime default getdate(),
 							   [_update_date] datetime default NULL
end
GO

update vm_interface set [_create_date] = getdate() where [_create_date] is null
update vds_interface set [_create_date] = getdate() where [_create_date] is null
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'network_cluster' and COLUMN_NAME = 'status'))
begin
	ALTER TABLE  [network_cluster] ADD [status] [int] default 0 NOT NULL,
 				 [is_display] [bit]default 0 NOT NULL
end
GO


-- network constraints --
IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_vm_interface_vm_static' AND TABLE_NAME='vm_interface')
begin
	ALTER TABLE [dbo].[vm_interface]   ADD  CONSTRAINT [FK_vm_interface_vm_static] FOREIGN KEY([vm_guid])
	REFERENCES [dbo].[vm_static] ([vm_guid]) ON DELETE CASCADE
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_vm_interface_vm_templates' AND TABLE_NAME='vm_interface')
begin
	ALTER TABLE [dbo].[vm_interface]   ADD  CONSTRAINT [FK_vm_interface_vm_templates] FOREIGN KEY([vmt_guid])
	REFERENCES [dbo].[vm_templates] ([vmt_guid])
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_vds_interface_vds_interface' AND TABLE_NAME='vds_interface')
begin
	ALTER TABLE [dbo].[vds_interface]   ADD  CONSTRAINT [FK_vds_interface_vds_interface] FOREIGN KEY([vds_id])
	REFERENCES [dbo].[vds_static] ([vds_id]) ON DELETE CASCADE
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_network_cluster_network' AND TABLE_NAME='network_cluster')
begin
	ALTER TABLE [dbo].[network_cluster]   ADD  CONSTRAINT [FK_network_cluster_network] FOREIGN KEY([network_id])
	REFERENCES [dbo].[network] ([id])
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_network_cluster_vds_groups' AND TABLE_NAME='network_cluster')
begin
	ALTER TABLE [dbo].[network_cluster]   ADD  CONSTRAINT [FK_network_cluster_vds_groups] FOREIGN KEY([cluster_id])
	REFERENCES [dbo].[vds_groups] ([vds_group_id])
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_interface_statistics_vds_static' AND TABLE_NAME='interface_statistics')
begin
	ALTER TABLE [dbo].[interface_statistics]  ADD  CONSTRAINT [FK_interface_statistics_vds_static] FOREIGN KEY([vds_id])
	REFERENCES [dbo].[vds_static] ([vds_id])
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_interface_statistics_vm_static' AND TABLE_NAME='interface_statistics')
begin
	ALTER TABLE [dbo].[interface_statistics] ADD  CONSTRAINT [FK_interface_statistics_vm_static] FOREIGN KEY([id])
	REFERENCES [dbo].[vm_static] ([vm_guid]) ON DELETE CASCADE
end
GO

-- extend vdc_option fields size 
if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vdc_options' and COLUMN_NAME = 'option_value' and CHARACTER_MAXIMUM_LENGTH != -1))
begin
	ALTER TABLE vdc_options ALTER COLUMN option_name nvarchar(100) NOT NULL
	ALTER TABLE vdc_options ALTER COLUMN option_value nvarchar(max)  NOT NULL
end

GO

-- VM Static --
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_static' and COLUMN_NAME = 'fail_back'))
begin
	ALTER TABLE [vm_static] ADD [fail_back] bit NOT NULL CONSTRAINT DF_vm_static_fail_back DEFAULT 0,
								[default_boot_device] int NOT NULL CONSTRAINT DF_vm_static_default_boot_device DEFAULT 0,
								[vm_type] int NOT NULL CONSTRAINT DF_vm_static_vm_type DEFAULT 0,
								[hypervisor_type] int NOT NULL CONSTRAINT DF_vm_static_hypervisor_type DEFAULT 0,
								[operation_mode] int NOT NULL CONSTRAINT DF_vm_static_operation_mode DEFAULT 0,
								[nice_level] int NOT NULL CONSTRAINT DF_vm_static_nice_level DEFAULT 0
end
GO

-- add non clustered index on vm_name
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='vm_static' and sys.indexes.name = 'IDX_vm_static_vm_name'))
begin
CREATE NONCLUSTERED INDEX [IDX_vm_static_vm_name] ON [dbo].[vm_static] 
(
	[vm_name] ASC
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

-- VM Dynamic --
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_dynamic' and COLUMN_NAME = 'hibernation_vol_handle'))
begin
	ALTER TABLE [vm_dynamic] ADD [hibernation_vol_handle] varchar(255)
end
GO

-- VM Templates --
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_templates' and COLUMN_NAME = 'fail_back'))
begin
	ALTER TABLE [vm_templates] ADD  [fail_back] bit NOT NULL CONSTRAINT DF_vm_templates_fail_back DEFAULT 0,
									[default_boot_device] int NOT NULL CONSTRAINT DF_vm_templates_default_boot_device DEFAULT 0,
									[vm_type] int NOT NULL CONSTRAINT DF_vm_templates_vm_type DEFAULT 0,
									[hypervisor_type] int NOT NULL CONSTRAINT DF_vm_templates_hypervisor_type DEFAULT 0,
									[operation_mode] int NOT NULL CONSTRAINT DF_vm_templates_operation_mode DEFAULT 0,
									[nice_level] int NOT NULL CONSTRAINT DF_vm_templates_nice_level DEFAULT 0
end
GO

-- VDS Static --
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_static' and COLUMN_NAME = 'vds_strength'))
begin
	ALTER TABLE [vds_static] ADD [vds_strength] int NOT NULL CONSTRAINT DF_vds_static_vds_strength DEFAULT 100
end
GO

-- VDS Dynamic --
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_dynamic' and COLUMN_NAME = 'cpu_over_commit_time_stamp'))
begin
	ALTER TABLE [vds_dynamic] ADD [cpu_over_commit_time_stamp] DATETIME NULL,[hypervisor_type] int NULL,
								  [vms_cores_count] int NULL,[pending_vcpus_count] int NULL,[cpu_sockets] int NULL, [net_config_dirty] bit NULL
end
GO
-- VDS Groups --
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_groups' and COLUMN_NAME = 'selection_algorithm'))
begin
	ALTER TABLE [vds_groups] ADD [selection_algorithm] int NOT NULL CONSTRAINT DF_vds_groups_selection_algorithm DEFAULT 0,
								 [high_utilization] int NOT NULL CONSTRAINT DF_vds_groups_high_utilization DEFAULT 75,
								 [low_utilization] int NOT NULL CONSTRAINT DF_vds_groups_low_utilization DEFAULT 0,
								 [cpu_over_commit_duration_minutes] int NOT NULL CONSTRAINT DF_vds_groups_cpu_over_commit_duration_minutes DEFAULT 2,
								 [hypervisor_type] int NOT NULL CONSTRAINT DF_vds_groups_hypervisor_type DEFAULT 0,
								 [is_initialized] bit NOT NULL CONSTRAINT DF_vds_groups_is_initialized DEFAULT 0,
								 [max_vds_memory_over_commit] int NOT NULL CONSTRAINT DF_vds_groups_mem_over_commit DEFAULT 0,
								 [storage_pool_id] uniqueidentifier NULL CONSTRAINT [Fk_vds_groups_storage_pool_id] FOREIGN KEY([storage_pool_id]) 
									REFERENCES [dbo].[storage_pool] ([id]) ON DELETE SET NULL
end
GO

-- update values from user config
if exists (select option_value from vdc_options where option_name = 'MaxVdsMemOverCommit')
begin
	update vds_groups set [max_vds_memory_over_commit] = 
	(select option_value from vdc_options where option_name = 'MaxVdsMemOverCommit')
end
GO

/* Commented - problematic in 4.4 to 4.4 upgrade
update vds_groups
set [selection_algorithm]=CASE WHEN exists(select option_value 
                            from vdc_options
                            where option_name = 'VdsSelectionAlgorithm'
							and option_value = 'EvenlyDistribute')
        THEN 1 -- set algorithm to be EvenlyDistribute
        ELSE 2 -- set algorithm to be PowerSave
        END
GO

update vds_groups set [high_utilization] = 75
GO

update vds_groups
set [low_utilization] = CASE WHEN exists(select option_value 
                            from vdc_options
                            where option_name = 'VdsSelectionAlgorithm'
							and option_value = 'EvenlyDistribute')
        THEN 0
        ELSE 20
        END
GO

-- overwrite configuration value
update vdc_options set option_value = 75 where option_name in ('HighUtilizationForEvenlyDistribute','HighUtilizationForPowerSave')
update vdc_options set option_value = 0 where option_name = 'LowUtilizationForEvenlyDistribute'
update vdc_options set option_value = 20 where option_name = 'LowUtilizationForPowerSave'
*/

-- Images --
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'images' and COLUMN_NAME = 'storage_id'))
begin
	ALTER TABLE [images] ADD [storage_id] uniqueidentifier NULL CONSTRAINT [Fk_images_storage_id] FOREIGN KEY([storage_id]) 
		REFERENCES [dbo].[storage_domain_static] ([id])
end
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'images' and COLUMN_NAME = 'vm_snapshot_id'))
begin
	ALTER TABLE [images] ADD [vm_snapshot_id] uniqueidentifier NULL,
							 [volume_type] int NOT NULL CONSTRAINT [DEF_images_volume_type] DEFAULT 2,
							 [volume_format] int NOT NULL CONSTRAINT [DEF_images_volume_format] DEFAULT 4,
							 [disk_type] int NOT NULL CONSTRAINT [DEF_images_disk_type] DEFAULT 1,
							 [image_group_id] uniqueidentifier NULL,
							 [disk_interface] int NOT NULL CONSTRAINT [DEF_images_disk_interface] DEFAULT 0,
							 [boot] bit NOT NULL CONSTRAINT [DEF_images_boot] DEFAULT 0,
							 [wipe_after_delete] bit NOT NULL CONSTRAINT [DEF_images_wipe_after_delete] DEFAULT 0,
							 [propagate_errors] int NOT NULL CONSTRAINT [DEF_images_propagate_errors] DEFAULT 0
end
GO

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'images' and COLUMN_NAME = 'image_file'))
begin
	ALTER TABLE [images] Drop column [image_file], [actual_num_of_sectors]
end
GO

-- disk_image_dynamic table
if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'disk_image_dynamic'))
begin
	CREATE TABLE [disk_image_dynamic] (
		[image_group_id] UNIQUEIDENTIFIER NOT NULL,
		[read_rate] INTEGER NULL,
		[write_rate] INTEGER NULL,
		[actual_num_of_sectors] INTEGER NOT NULL,
		CONSTRAINT [PK_disk_image_dynamic] PRIMARY KEY ([image_group_id])
	)
end
GO

-- make internal_drive_mapping column size 50
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'images' and COLUMN_NAME = 'internal_drive_mapping' and CHARACTER_MAXIMUM_LENGTH = 50))
begin
	alter table images alter column internal_drive_mapping varchar(50)
end
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vm_template_image_map' and COLUMN_NAME = 'internal_drive_mapping' and CHARACTER_MAXIMUM_LENGTH = 50))
begin
	alter table vm_template_image_map alter column internal_drive_mapping varchar(50)
end
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'image_vm_pool_map' and COLUMN_NAME = 'internal_drive_mapping' and CHARACTER_MAXIMUM_LENGTH = 50))
begin
	alter table image_vm_pool_map alter column internal_drive_mapping varchar(50)
end
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'stateless_vm_image_map' and COLUMN_NAME = 'internal_drive_mapping' and CHARACTER_MAXIMUM_LENGTH = 50))
begin
	alter table stateless_vm_image_map alter column internal_drive_mapping varchar(50)
end
GO

-- change drive letter C to 1 , D to 2 etc (C=66, Z=90). because disk names are now numbers
update images  set internal_drive_mapping = convert(varchar,(convert(int, ascii(internal_drive_mapping)) - 66)) where convert(int, ascii(internal_drive_mapping)) > 66 and convert(int, ascii(internal_drive_mapping)) < 91
go
update vm_template_image_map  set internal_drive_mapping = convert(varchar,(convert(int, ascii(internal_drive_mapping)) - 66)) where convert(int, ascii(internal_drive_mapping)) > 66 and convert(int, ascii(internal_drive_mapping)) < 91
go
update image_vm_pool_map  set internal_drive_mapping = convert(varchar,(convert(int, ascii(internal_drive_mapping)) - 66)) where convert(int, ascii(internal_drive_mapping)) > 66 and convert(int, ascii(internal_drive_mapping)) < 91
go
update stateless_vm_image_map  set internal_drive_mapping = convert(varchar,(convert(int, ascii(internal_drive_mapping)) - 66)) where convert(int, ascii(internal_drive_mapping)) > 66 and convert(int, ascii(internal_drive_mapping)) < 91
go

--  vds_statistics --
IF  EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='DEF_vds_statistics_rx_dropped' AND TABLE_NAME='vds_statistics')
begin
	ALTER TABLE [vds_statistics]  Drop CONSTRAINT [DEF_vds_statistics_rx_dropped],[DEF_vds_statistics_rx_rate],
												  [DEF_vds_statistics_tx_dropped],[DEF_vds_statistics_tx_rate],
												  [DEF_vds_statistics_elapsed_time]
end
GO

if (exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_statistics' and COLUMN_NAME = 'rx_dropped'))
begin
	ALTER TABLE [vds_statistics] drop COLUMN rx_dropped, rx_rate,tx_dropped,tx_rate,elapsed_time
end
GO 

-- delete old options from vdc_options
delete vdc_options where option_name in ('IrsHost', 'IrsImagesPath', 'IrsPort')
GO



/*--------------------------------------------------
		Event Notification 
--------------------------------------------------*/
-- add to audit_log a processed column that indicates
-- if the message was already processed for event notifications and
-- the log type name 
if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'audit_log' and COLUMN_NAME = 'processed'))
begin
	alter table [dbo].[audit_log] add [processed] bit default 0 NOT NULL,
									  [log_type_name] nvarchar(100) default '' NULL
end
go

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'event_map'))
begin
	CREATE TABLE [dbo].[event_map](
		[event_up_name] [nvarchar](100) NOT NULL,
		[event_down_name] [nvarchar](100) NOT NULL,
	 CONSTRAINT [PK_EVENT_MAP] PRIMARY KEY CLUSTERED 
	(
		[event_up_name] ASC
	)WITH (PAD_INDEX  = OFF, IGNORE_DUP_KEY = OFF) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'event_notification_methods'))
begin
	CREATE TABLE [dbo].[event_notification_methods](
		[method_id] [int] NOT NULL,
		[method_type] [nchar](10) NOT NULL,
	 CONSTRAINT [PK_EVENT_NOTIFICATION_METHODS] PRIMARY KEY CLUSTERED 
	(
		[method_id] ASC
	)WITH (PAD_INDEX  = OFF, IGNORE_DUP_KEY = OFF) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'event_subscriber'))
begin
	CREATE TABLE [dbo].[event_subscriber](
		[subscriber_id] [uniqueidentifier] NOT NULL,
		[event_up_name] [nvarchar](100) NOT NULL,
		[method_id] [int] NOT NULL,
		[method_address] nvarchar(255) NULL,
		[tag_name] [nvarchar](50) default '' NOT NULL,
	 CONSTRAINT [PK_EVENT_SUBSCRIBER] PRIMARY KEY CLUSTERED 
	(
		[subscriber_id],[event_up_name],[method_id],[tag_name]
	)WITH (PAD_INDEX  = OFF, IGNORE_DUP_KEY = OFF) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'event_notification_hist'))
begin
	CREATE TABLE [dbo].[event_notification_hist](
		[subscriber_id] [uniqueidentifier] NOT NULL,
		[event_name] [nvarchar](100) NOT NULL,
		[audit_log_id] [bigint] NOT NULL,
		[method_type] [nchar](10) NOT NULL,
		[sent_at] [datetime] NOT NULL,
		[status] [bit] NOT NULL,
		[reason] [nchar](255)  NULL
	) ON [PRIMARY]
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_EVENT_SUBSCRIBER_EVENT_MAP' AND TABLE_NAME='event_subscriber')
begin
	ALTER TABLE [dbo].[event_subscriber]   ADD  CONSTRAINT [FK_EVENT_SUBSCRIBER_EVENT_MAP] FOREIGN KEY([event_up_name])
	REFERENCES [dbo].[event_map] ([event_up_name])  ON DELETE CASCADE
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_EVENT_SUBSCRIBER_EVENT_MAP' AND TABLE_NAME='event_subscriber')
begin
	ALTER TABLE [dbo].[event_subscriber]   ADD  CONSTRAINT [FK_EVENT_SUBSCRIBER_EVENT_NOTIFICATION_METHODS] FOREIGN KEY([method_id])
	REFERENCES [dbo].[event_notification_methods] ([method_id])  ON DELETE CASCADE
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_EVENT_SUBSCRIBER_EVENT_MAP' AND TABLE_NAME='event_subscriber')
begin
	ALTER TABLE [dbo].[event_subscriber]   ADD  CONSTRAINT [FK_EVENT_SUBSCRIBER_users] FOREIGN KEY([subscriber_id])
	REFERENCES [dbo].[users] ([user_id])  ON DELETE CASCADE
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_EVENT_NOTIFICATION_HIST_audit_log' AND TABLE_NAME='event_notification_hist')
begin
	ALTER TABLE [dbo].[event_notification_hist]   ADD  CONSTRAINT [FK_EVENT_NOTIFICATION_HIST_audit_log] FOREIGN KEY([audit_log_id])
	REFERENCES [dbo].[audit_log] ([audit_log_id])  ON DELETE CASCADE
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_EVENT_NOTIFICATION_users' AND TABLE_NAME='event_notification_hist')
begin
	ALTER TABLE [dbo].[event_notification_hist]   ADD  CONSTRAINT [FK_EVENT_NOTIFICATION_users] FOREIGN KEY([subscriber_id])
	REFERENCES [dbo].[users] ([user_id])  ON DELETE CASCADE
end
GO

-- [vds_spm_id_map]
if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'vds_spm_id_map'))
begin
	CREATE TABLE [dbo].[vds_spm_id_map](
		[storage_pool_id] [uniqueidentifier] NOT NULL,
		[vds_spm_id] [int] NOT NULL,
		[vds_id] [int] NOT NULL,
	 CONSTRAINT [PK_vds_spm_id_map] PRIMARY KEY CLUSTERED 
	(
		[storage_pool_id] ASC,
		[vds_spm_id] ASC
	))
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_vds_spm_id_map_storage_pool' AND TABLE_NAME='vds_spm_id_map')
begin
	ALTER TABLE [dbo].[vds_spm_id_map]   ADD  CONSTRAINT [FK_vds_spm_id_map_storage_pool] FOREIGN KEY([storage_pool_id])
	REFERENCES [dbo].[storage_pool] ([id])
	ON DELETE CASCADE
end
GO

-- vds_tools_version_map
if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'vds_tools_version_map'))
begin
	CREATE TABLE [dbo].[vds_tools_version_map](
		[storage_pool_id] [uniqueidentifier] NOT NULL,
		[vds_minimal_version] [int] NOT NULL,
		[tools_version] [int] NOT NULL,
	 CONSTRAINT [PK_vds_tools_version_map] PRIMARY KEY CLUSTERED 
	(
		[storage_pool_id] ASC,
		[vds_minimal_version] ASC,
		[tools_version] ASC
	)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
	) ON [PRIMARY]
end
GO

-----------------------------------
-- multilevel administrator updates
----------------------------------


BEGIN TRY
	if ((exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'user_vms') or
        exists (select TABLE_NAME from INFORMATION_SCHEMA.VIEWS where TABLE_NAME = 'user_vms')) and 
	    (exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'ad_vm_group_map') or
        exists (select TABLE_NAME from INFORMATION_SCHEMA.VIEWS where TABLE_NAME = 'ad_vm_group_map')))
	begin
		BEGIN TRAN
		declare @permission_id uniqueidentifier
		declare @tag_id int
		declare @group_id uniqueidentifier
		declare @vm_id uniqueidentifier
		declare @name nvarchar(40)
		declare @user_id uniqueidentifier
		declare @DefaultDisplayType int
		declare @username nvarchar(40)

		-- multileveladmin users tag map

		DECLARE c1 CURSOR READ_ONLY
		FOR
		select user_vms.user_id, user_vms.vm_guid, user_vms.DefaultDisplayType, users.username  from user_vms
		inner join users
		on user_vms.[user_id] = users.[user_id]

		OPEN c1

		FETCH NEXT FROM c1
		INTO @user_id, @vm_id, @DefaultDisplayType, @username

		WHILE @@FETCH_STATUS = 0
		BEGIN
			set @permission_id = newid()
			set @tag_id = (select max(tag_id) from tags) + 1 

			-- create default permission 
			if (not exists (select id from permissions where id = @permission_id))
			begin
				insert into [permissions] (id, role_id, is_readonly, ad_element_id)
					   values (@permission_id, '00000000-0000-0000-0001-000000000001', 0, @user_id)
			end
			-- create default tag
			if (not exists (select tag_id from tags where tag_id = @tag_id))
			begin
				insert into tags (tag_id, tag_name, description, parent_id, readonly, [type])
					   values (@tag_id, @username, @user_id, -1, 0, 1)
			end

			-- map permisssion to tag
			if (not exists (select tag_id from tags_permissions_map where tag_id = @tag_id and permission_id = @permission_id))
			begin
				insert into tags_permissions_map (tag_id, permission_id)
					   values (@tag_id, @permission_id)
			end
			-- map vm to tag
			if (not exists (select tag_id from tags_vm_map where tag_id = @tag_id and vm_id = @vm_id))
			begin
				insert into tags_vm_map (tag_id, vm_id, DefaultDisplayType)
					   values (@tag_id, @vm_id, @DefaultDisplayType)
			end

			FETCH NEXT FROM c1
			INTO @user_id, @vm_id, @DefaultDisplayType, @username

		END
		CLOSE c1
		DEALLOCATE c1

		-- multileveladmin ad_groups tag map
	            
		DECLARE c2 CURSOR READ_ONLY
		FOR

		select ad_vm_group_map.vm_id, ad_groups.id, ad_groups.[name]  from ad_groups
		inner join ad_vm_group_map
		on ad_groups.[id] = ad_vm_group_map.[group_id]

		OPEN c2

		FETCH NEXT FROM c2
		INTO @vm_id, @group_id, @name

		WHILE @@FETCH_STATUS = 0
		BEGIN
			set @permission_id = newid()
			set @tag_id = (select max(tag_id) from tags) + 1 

			-- create default permission 
			if (not exists (select id from permissions where id = @permission_id))
			begin
				insert into [permissions] (id, role_id, is_readonly, ad_element_id)
					   values (@permission_id, '00000000-0000-0000-0001-000000000001', 0, @group_id)
			end
			-- create default tag
			if (not exists (select tag_id from tags where tag_id = @tag_id))
			begin
				insert into tags (tag_id, tag_name, description, parent_id, readonly, [type])
					   values (@tag_id, @name, @group_id, -1, 0, 1)
			end
			-- map permisssion to tag
			if (not exists (select tag_id from tags_permissions_map where tag_id = @tag_id and permission_id = @permission_id))
			begin
				insert into tags_permissions_map (tag_id, permission_id)
					   values (@tag_id, @permission_id)
			end
			-- map vm to tag
			if (not exists (select tag_id from tags_vm_map where tag_id = @tag_id and vm_id = @vm_id))
			begin
				insert into tags_vm_map (tag_id, vm_id)
					   values (@tag_id, @vm_id)
			end
			FETCH NEXT FROM c2
			INTO @vm_id, @group_id, @name

		END
		CLOSE c2
		DEALLOCATE c2
		-- The following tables must be manipulated within the transaction
		IF EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='ad_vm_group_map_group' AND TABLE_NAME='ad_vm_group_map')
		begin
			ALTER TABLE [ad_vm_group_map]  Drop CONSTRAINT [ad_vm_group_map_group],[ad_vm_group_map_vm]
		end
		if (exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'ad_vm_group_map'))
		begin
			drop table [dbo].[ad_vm_group_map]
		end
		if (exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'user_vms'))
		begin
			drop table [dbo].[user_vms]
		end
		COMMIT TRAN
	end
END TRY

BEGIN CATCH
    ROLLBACK TRAN
    EXEC RethrowError;
END CATCH


if (not exists (select TABLE_NAME from INFORMATION_SCHEMA.TABLES where TABLE_NAME = 'async_tasks'))
begin
	CREATE TABLE [dbo].[async_tasks](
		[task_id] [uniqueidentifier] NOT NULL,
		[action_type] [int] NOT NULL,
		[status] [int] NOT NULL,
		[result] [int] NOT NULL,
		[action_parameters] [image] NULL,
	 CONSTRAINT [PK_async_tasks] PRIMARY KEY  ([task_id])
	)
end
GO

IF NOT EXISTS(SELECT * FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS WHERE CONSTRAINT_SCHEMA='dbo' AND CONSTRAINT_NAME='FK_image_vm_map_vm_static' AND TABLE_NAME='image_vm_map')
begin
	ALTER TABLE [image_vm_map] ADD  CONSTRAINT [FK_image_vm_map_vm_static] 
		FOREIGN KEY ([vm_id]) REFERENCES [vm_static] ([vm_guid]) ON DELETE CASCADE
end
GO

-- add non clustered index on run_on_vds
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='vm_dynamic' and sys.indexes.name = 'IDX_vm_dynamic_run_on_vds'))
begin
CREATE NONCLUSTERED INDEX [IDX_vm_dynamic_run_on_vds] ON [dbo].[vm_dynamic] 
(
	[run_on_vds] ASC
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]

end
GO

-- add non clustered index on [vds_id]
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='vds_interface' and sys.indexes.name = 'IDX_vds_interface_vds_id'))
begin
CREATE NONCLUSTERED INDEX [IDX_vds_interface_vds_id] ON [dbo].[vds_interface] 
(
	[vds_id] ASC
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

-- add non clustered index on [vm_guid],[vmt_guid]
if (not exists (select sys.indexes.object_id from sys.indexes, sys.tables
				where sys.indexes.object_id = sys.tables.object_id and
				sys.tables.name='vm_interface' and sys.indexes.name = 'IDX_vm_interface_vm_vmt_guid'))
begin
CREATE NONCLUSTERED INDEX [IDX_vm_interface_vm_vmt_guid] ON [dbo].[vm_interface] 
(
	[vm_guid],[vmt_guid]
)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, IGNORE_DUP_KEY = OFF, ONLINE = OFF) ON [PRIMARY]
end
GO

if (not exists (select COLUMN_NAME from INFORMATION_SCHEMA.COLUMNS where TABLE_NAME = 'vds_interface' and COLUMN_NAME = 'boot_protocol'))
begin
        ALTER TABLE [vds_interface] ADD [boot_protocol] int NULL
end
GO

-- Change audit_log_id cluster index (PK) to be DESC instead of ASC
declare @flag bit

select @flag = is_descending_key from sys.index_columns where   index_id = 1
and object_id in (select object_id from sys.objects where name = 'audit_log' and type = 'U')

if (@flag = 0)
begin
	ALTER TABLE [dbo].[event_notification_hist]   DROP  CONSTRAINT [FK_EVENT_NOTIFICATION_HIST_audit_log]

	/****** Object:  Index [PK_audit_log]    Script Date: 08/19/2009 13:11:10 ******/
	ALTER TABLE [dbo].[audit_log] DROP  CONSTRAINT [PK_audit_log] 

	/****** Object:  Index [PK_audit_log]    Script Date: 08/19/2009 13:13:04 ******/
	ALTER TABLE [dbo].[audit_log] ADD  CONSTRAINT [PK_audit_log] PRIMARY KEY CLUSTERED 
	(
		[audit_log_id] DESC
	)WITH (PAD_INDEX  = OFF, SORT_IN_TEMPDB = OFF, IGNORE_DUP_KEY = OFF, FILLFACTOR = 80, ONLINE = OFF) ON [PRIMARY]

	ALTER TABLE [dbo].[event_notification_hist]   ADD  CONSTRAINT [FK_EVENT_NOTIFICATION_HIST_audit_log] FOREIGN KEY([audit_log_id])
	REFERENCES [dbo].[audit_log] ([audit_log_id])  ON DELETE CASCADE
end

GO

-- upgrade propagate errors for all images to false
update images set propagate_errors = 0
go


print 'Upgrade script completed.'
go

