using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Text;
using System.Windows;
using VdcCommon.Interfaces;
using VdcCommon.VdcQueries;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Collections;

namespace org.ovirt.engine.ui.uicommon.models.storage
{
	public abstract class ManageBackupModel : SearchableListModel
	{
		#region Commands

		public UICommand RestoreCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		public new storage_domains Entity
		{
			get { return (storage_domains)base.Entity; }
			set { base.Entity = value; }
		}

		private bool isRefreshing;
		public bool IsRefreshing
		{
			get { return isRefreshing; }
			set
			{
				if (isRefreshing != value)
				{
					isRefreshing = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsRefreshing"));
				}
			}
		}

		#endregion

		protected ManageBackupModel()
		{
			RestoreCommand = new UICommand("Restore", this);
			RemoveCommand = new UICommand("Remove", this);
		}

		protected virtual void Remove()
		{
		}

		protected virtual void Restore()
		{
		}

		protected void Cancel()
		{
			CancelConfirm();
			Window = null;
		}

		protected void CancelConfirm()
		{
			ConfirmWindow = null;
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "storage_domain_shared_status")
			{
				CheckStorageStatus();
			}
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			CheckStorageStatus();
			UpdateActionAvailability();

			SearchCommand.Execute();
		}

		private void CheckStorageStatus()
		{
			if (Entity != null)
			{
				if (Entity.storage_domain_shared_status == StorageDomainSharedStatus.InActive || Entity.storage_domain_shared_status == StorageDomainSharedStatus.Mixed)
				{
					Message = "The Export Domain is inactive. Data can be retrieved only when the Domain is activated";
				}
				else if (Entity.storage_domain_shared_status == StorageDomainSharedStatus.Unattached)
				{
					Message = "Export Domain is not attached to any Data Center. Data can be retrieved only when the Domain is attached to a Data Center and is active";
				}
				else
				{
					Message = null;
				}
			}
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected virtual void UpdateItems()
		{
		}

		private void UpdateActionAvailability()
		{
			RestoreCommand.IsExecutionAllowed = Entity != null
				&& SelectedItems != null
				&& SelectedItems.Count > 0
				&& Entity.storage_domain_shared_status == StorageDomainSharedStatus.Active;

			RemoveCommand.IsExecutionAllowed = Entity != null
				&& SelectedItems != null
				&& SelectedItems.Count > 0
				&& Entity.storage_domain_shared_status == StorageDomainSharedStatus.Active;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == RestoreCommand)
			{
				Restore();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "CancelConfirm")
			{
				CancelConfirm();
			}
		}
	}
}
