using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Threading;
using System.Timers;
using System.Web;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using System.Linq;
using VdcUtils;
using System.Configuration;
using System.IO;
using Timer = System.Timers.Timer;

namespace VdcFrontend
{
	internal enum ClientConfigValues
	{
		RemoteInterface,
		AppErrorsFileName,
		VdsErrorsFileName
	}


	public sealed class RegistrationResult
	{
		/// <summary>
		/// Raised once when a first result retrievement occurs.
		/// </summary>
		public Event RetrievedEvent { get; private set; }
		public static EventDefinition RetrievedEventDefinition;

		public Guid Id { get; private set; }
		public ObservableCollection<IVdcQueryable> Data { get; private set; }
		internal int RetrievementCount { get; set; }

		static RegistrationResult()
		{
			RetrievedEventDefinition = new EventDefinition("RetrievedEvent", typeof(RegistrationResult));
		}

		public RegistrationResult(Guid id, ObservableCollection<IVdcQueryable> data)
		{
			RetrievedEvent = new Event(RetrievedEventDefinition);

			Id = id;
			Data = data;
		}

		internal void NotifyRetrieved()
		{
			RetrievedEvent.raise(this, EventArgs.Empty);
		}
	}



	public interface IFrontendMultipleQueryAsyncCallback
	{
		void Executed(FrontendMultipleQueryAsyncResult result);
	}


	public interface IFrontendActionAsyncCallback
	{
		void Executed(FrontendActionAsyncResult result);
	}


	public class AsyncQuery
	{
		public delegate void AsyncDelelgate(Object model, Object ReturnValue);
		public delegate object ConverterDelelgate(valueObject ReturnValue, AsyncQuery _asyncQuery);
		public AsyncDelelgate asyncCallback { get; set; }
		public ConverterDelelgate converterCallback { get; set; }
		public object Model { get; set; }
		public bool HandleFailure { get; set; }
		public VdcQueryReturnValue OriginalReturnValue { get; set; }
		public Object[] Data { get; set; }
		public string Context { get; set; }

		public AsyncQuery()
		{
		}

		public AsyncQuery(object target, AsyncDelelgate asyncCallback)
		{
			this.Model = target;
			this.asyncCallback = asyncCallback;
		}

		public AsyncQuery(object target, AsyncDelelgate asyncCallback, bool handleFailure)
			:this(target, asyncCallback)
		{
			HandleFailure = handleFailure;
		}

		public AsyncQuery(object target, AsyncDelelgate asyncCallback, string context)
			: this(target, asyncCallback)
		{
			Context = context;
		}
	}

	public static class Frontend
	{
		public const string BackendProxy = "BackendProxyKey";
		public const string BackendProxyFactory = "BackendProxyFactory";
		private static ErrorTranslator _canDoActionErrorsTranslator;
		private static readonly ErrorTranslator _vdsErrorsTranslator;
		private static IFrontendEventsHandler _eventsHandler;
		private static bool isBackendInitialized;
		private static bool _autoLogin;

		private static readonly Dictionary<Guid, RegistrationResult> queries = new Dictionary<Guid, RegistrationResult>();
		private static readonly Timer asyncQueryTimer;

		private static VdcQueryType[] subscribedQueryTypes;

		public static EventDefinition QueryStartedEventDefinition;
		public static Event QueryStartedEvent { get; private set; }

		public static EventDefinition QueryCompleteEventDefinition;
		public static Event QueryCompleteEvent { get; private set; }

		public static VdcUser LoggedInUser { get; private set; }


		static Frontend()
		{
			_canDoActionErrorsTranslator = new ErrorTranslator(GetFullPath(ConfigurationManager.AppSettings[ClientConfigValues.AppErrorsFileName.ToString()]));
			_vdsErrorsTranslator = new ErrorTranslator(GetFullPath(ConfigurationManager.AppSettings[ClientConfigValues.VdsErrorsFileName.ToString()]));

			asyncQueryTimer = new Timer(1000);
			asyncQueryTimer.Elapsed += AsyncQueryTimer_Elapsed;


			QueryStartedEventDefinition = new EventDefinition("QueryStarted", typeof(Frontend));
			QueryStartedEvent = new Event(QueryStartedEventDefinition);

			QueryCompleteEventDefinition = new EventDefinition("QueryComplete", typeof(Frontend));
			QueryCompleteEvent = new Event(QueryCompleteEventDefinition);
		}

		public static void Subscribe(VdcQueryType[] queryTypes)
		{
			subscribedQueryTypes = queryTypes;
		}

		public static void Unsubscribe()
		{
			subscribedQueryTypes = null;
			CurrentContext = null;

			QueryStartedEvent.removeAll();
			QueryCompleteEvent.removeAll();
		}

		/// <summary>
		/// Returns a full-path for a specified file-path. If path is relative - absolute path is returned. If path
		/// is already absolute - it is returned as is.
		/// </summary>
		/// <param name="path">the file-path</param>
		/// <returns>the matching absolute file-path</returns>
		private static string GetFullPath(string path)
		{
			string fullPath = path;
			if (!Path.IsPathRooted(fullPath))
			{
				fullPath = Path.Combine(Path.GetDirectoryName(AppDomain.CurrentDomain.BaseDirectory), fullPath);
			}
			return fullPath;
		}

		/// <summary>
		/// Gets the login action parameters.
		/// </summary>
		/// <param name="actionType">Type of the action.</param>
		/// <param name="userName">Name of the user.</param>
		/// <param name="password">The password.</param>
		/// <param name="domain">The domain.</param>
		/// <returns></returns>
		private static LoginUserParameters GetLoginParameters(VdcActionType actionType, string userName, string password, string domain)
		{
			string browser = string.Empty;
			string os = Environment.OSVersion.Platform + Environment.OSVersion.Version.ToString();
			string clientType = string.Empty;
			string sessionId = string.Empty;
			
			return new LoginUserParameters(userName, password, domain, os, browser, clientType) { IsAdmin = true, SessionId = sessionId, ActionType = actionType };
		}

		private static bool IsBackendInitialized
		{
			get
			{
				return isBackendInitialized;
			}
		}

		private static IBackend Instance
		{
			get { return WCFFactory.GetBackend(); }
		}

		private static IBackend PublicServices
		{
			get { return WCFFactory.PublicServices; }
		}

		public static bool AutoLogIn
		{
			get
			{
				return _autoLogin;
			}
			internal set
			{
				_autoLogin = value;
				isBackendInitialized = true;
			}
		}


		public static bool IsUserLoggedIn
		{
			get
			{
				return LoggedInUser != null;
			}
		}

		public static void Logoff(LogoutUserParameters parameters)
		{
			if (QLogger.Instance.IsDebugEnabled)
			{
				QLogger.Instance.DebugFormat("Logoff: parameters=\n{1}", parameters.Dump());
			}

			try
			{
				if (IsUserLoggedIn)
				{
					LoggedInUser = null;

					UnregisterAll();

					VdcReturnValueBase returnValue = Instance.Logoff(parameters);
					WCFFactory.Backend = null;

					HandleActionResult(VdcActionType.LogoutUser, returnValue);
					if (returnValue != null && returnValue.Succeeded)
					{
						SessionContainer.Instance.SetData("VdcUser", null);
					}
				}
			}
			catch (BackendException ex)
			{
				QLogger.Instance.ErrorFormat("Problem with Connection to Backend. Error: {0}", ex);
				if (_eventsHandler != null)
				{
					LoggedInUser = null;
					_eventsHandler.ConnectionClosed(ex);
				}
			}
			catch (InvalidOperationException ex)
			{
				QLogger.Instance.ErrorFormat("Problem with Backend remote configuration. Error: {0}", ex);
				if (_eventsHandler != null)
				{
					LoggedInUser = null;
					_eventsHandler.ConnectionClosed(ex);
				}
			}
			catch (Exception ex)
			{
				QLogger.Instance.ErrorFormat("Unhandled exception. Exception {0}", ex);
				if (_eventsHandler != null)
				{
					LoggedInUser = null;
					_eventsHandler.ConnectionClosed(ex);
				}
			}
		}

		public static VdcUser Login(string userName, string password, string domain)
		{
			VdcUser user = null;
			AutoLogIn = false;

			WCFFactory.CreateBackend();

			try
			{
				VdcReturnValueBase returnValue = Instance.Login(
					GetLoginParameters(VdcActionType.LoginAdminUser, userName, password, domain));
				HandleActionResult(VdcActionType.LoginUser, returnValue);
				user = (VdcUser)returnValue.ActionReturnValue;
				SessionContainer.Instance.SetData("VdcUser", user);
			}
			catch (Exception ex)
			{
				QLogger.Instance.ErrorFormat("Error during login: {0}", ex);
			}

			LoggedInUser = user;

			return user;
		}

		public static RegistrationResult RegisterSearch(string pattern, SearchType searchType, int maxCount)
		{
			if (QLogger.Instance.IsDebugEnabled)
			{
				QLogger.Instance.DebugFormat("RegisterSearch: pattern={0}, searchType={1}, maxCount={2}", pattern, searchType, maxCount);
			}

			return RegisterQuery(VdcQueryType.Search,
				new SearchParameters(pattern, searchType)
				{
					MaxCount = maxCount
				});
		}

		public static RegistrationResult RegisterQuery(VdcQueryType queryType, VdcQueryParametersBase parameters)
		{
			if (QLogger.Instance.IsDebugEnabled)
			{
				QLogger.Instance.DebugFormat("RegisterQuery: queryType={0}, parameters={1}", queryType, parameters);
			}

			RegisterableQueryReturnDataType type;
			CheckTypeAndThrowIfNeeded(queryType, parameters, out type);

			if (type != RegisterableQueryReturnDataType.LIST_IQUERYABLE && type != RegisterableQueryReturnDataType.SEARCH)
			{
				throw new ArgumentException(string.Format("BackendCallback::RegisterQuery: Failed to register queryType '{0}' - registerable query type ({1}) don't match queryData type (ObservableCollection<IVdcQueryable>)!", queryType, type), "queryType");
			}

			Guid id = Guid.NewGuid();
			var result = new RegistrationResult(id, new QueryListIVdcQueryableUpdatableData());
			queries.Add(id, result);

			RunAsyncQuery(VdcQueryType.RegisterQuery, new RegisterQueryParameters(id, queryType, parameters));

			//Ensure starting timer.
			if (!asyncQueryTimer.Enabled)
			{
				asyncQueryTimer.Start();
			}

			return result;
		}


		public static void UnregisterSearch(Guid searchId)
		{
			UnregisterQuery(searchId);
		}

		public static void UnregisterQuery(Guid queryId)
		{
			QLogger.Instance.DebugFormat("UnregisterQuery: queryId={0}", queryId);

			RunAsyncQuery(VdcQueryType.UnregisterQuery, new UnregisterQueryParameters(queryId));
			queries.Remove(queryId);
		}

		public static void InitEventsHandler(IFrontendEventsHandler eventsHandler)
		{
			_eventsHandler = eventsHandler;
		}

		public static void InitErrorTranslator(ErrorTranslator errorTranslator)
		{
			_canDoActionErrorsTranslator = errorTranslator;
		}

		public static VdcReturnValueBase RunActionAsyncroniousely(VdcActionType actionType, VdcActionParametersBase parameter)
		{
			List<VdcActionParametersBase> parametersForRunMultipleActions = new List<VdcActionParametersBase> { parameter };
			List<VdcReturnValueBase> returnValuesFromRunMultipleActions = RunMultipleAction(actionType, parametersForRunMultipleActions);
			return returnValuesFromRunMultipleActions.Count > 0 ? returnValuesFromRunMultipleActions[0] : null;
		}

		public static List<VdcReturnValueBase> RunMultipleAction(VdcActionType actionType, List<VdcActionParametersBase> parameters)
		{
			if (QLogger.Instance.IsDebugEnabled)
			{
				QLogger.Instance.DebugFormat("RunMultipleActions: actionType={0}, parameters=\n{1}", actionType, parameters.Dump());
			}

			if (parameters == null || parameters.Count == 0)
			{
				QLogger.Instance.Info("Frontend::RunMultipleActions: Parameters list is empty -> skipping Backend access");
				return new List<VdcReturnValueBase>();
			}

			var failed = new List<VdcReturnValueBase>();
			if (!IsUserLoggedIn)
			{
				failed.Add(new VdcReturnValueBase());
				QLogger.Instance.WarnFormat("Attempting to execute RunMultipleAction for action: {0} while user not logged in", actionType.ToString());
			}
			else
			{
				try
				{
					List<VdcReturnValueBase> returnValues = Instance.RunMultipleActions(new RunMultipleActions1(actionType, parameters.ToArray())); //ITAMAR-USER UserBackend.RunUserMultipleActions(actionType, parameters);

					foreach (VdcReturnValueBase returnValue in returnValues)
					{
						if (returnValue.CanDoAction == false)
						{
							failed.Add(returnValue);
						}
					}
					if (failed.Count != 0 && _eventsHandler != null)
					{
						TranslateErrors(failed);
						_eventsHandler.RunActionFailed(failed);
					}
				}
				catch (BackendException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Connection to Backend. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (InvalidOperationException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Backend remote configuration. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (Exception ex)
				{
					QLogger.Instance.ErrorFormat("Unhandled exception. Exception {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
			}
			return failed;
		}

		public static void RunMultipleAction(VdcActionType actionType, List<VdcActionParametersBase> parameters, Action<FrontendMultipleActionAsyncResult> callback, object state)
		{
			ThreadPool.QueueUserWorkItem(
				delegate
				{
					var returnValues = RunMultipleAction(actionType, parameters);

					_eventsHandler.GuiSynchronizationContext.Post(
						delegate
						{
							FrontendMultipleActionAsyncResult result = new FrontendMultipleActionAsyncResult(actionType, parameters, returnValues, state);
							callback(result);
						}, null);
				});
		}

		public static VdcReturnValueBase RunAction(VdcActionType actionType, VdcActionParametersBase parameters)
		{
			if (QLogger.Instance.IsDebugEnabled)
			{
				QLogger.Instance.DebugFormat("RunAction: actionType={0}, parameters=\n{1}", actionType, parameters.Dump());
			}

			VdcReturnValueBase returnValue = null;
			if (!IsUserLoggedIn)
			{
				returnValue = new VdcReturnValueBase();
				QLogger.Instance.WarnFormat("Attempting to execute RunAction for action: {0} while user not logged in", actionType.ToString());
			}
			else
			{
				try
				{
					if (IsBackendInitialized)
					{
						returnValue = Instance.RunAction(new RunAction1(actionType, parameters));
					}
					else
					{
						throw new BackendFaultException();
					}

					HandleActionResult(actionType, returnValue);
				}
				catch (BackendException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Connection to Backend. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (InvalidOperationException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Backend remote configuration. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (Exception ex)
				{
					QLogger.Instance.ErrorFormat("Unhandled exception. Exception {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
			}
			return returnValue;
		}

		[Obsolete]
		public static void RunAction(VdcActionType actionType, VdcActionParametersBase parameters, IFrontendActionAsyncCallback callback)
		{
			ThreadPool.QueueUserWorkItem(
				delegate
				{
					VdcReturnValueBase returnValue = RunAction(actionType, parameters);

					_eventsHandler.GuiSynchronizationContext.Post(
						delegate
						{
							FrontendActionAsyncResult result = new FrontendActionAsyncResult(actionType, parameters, returnValue);
							callback.Executed(result);
						}, null);
				});
		}

		public static void RunAction(VdcActionType actionType, VdcActionParametersBase parameters, Action<FrontendActionAsyncResult> callback, object state)
		{
			ThreadPool.QueueUserWorkItem(
				delegate
				{
					VdcReturnValueBase returnValue = RunAction(actionType, parameters);

					_eventsHandler.GuiSynchronizationContext.Post(
						delegate
						{
							FrontendActionAsyncResult result = new FrontendActionAsyncResult(actionType, parameters, returnValue, state);
							callback(result);
						}, null);
				});
		}

		private const string UserIsNotLoggedInKey = "USER_IS_NOT_LOGGED_IN";

		private static void HandleActionResult(VdcActionType actionType, VdcReturnValueBase returnValue)
		{
			bool success = true;
			if (!returnValue.CanDoAction)
			{
				success = false;
				var failed = new List<VdcReturnValueBase> { returnValue };
				if (returnValue.CanDoActionMessages.Contains(UserIsNotLoggedInKey))
				{
					LoggedInUser = null;
					_eventsHandler.ConnectionClosed(new BackendUserNotLoggedInException());
				}
				else
				{
					TranslateErrors(failed);
					_eventsHandler.RunActionFailed(failed);
				}
			}
			else if (returnValue.IsSynchronous && returnValue.Succeeded == false)
			{
				success = false;
				RunActionExecutionFailed(actionType, returnValue.Fault);
			}

			//Check if a login command succeeded  
			if (!success)
			{
				switch (actionType)
				{
					case VdcActionType.LoginUser:
						WCFFactory.Backend = null;
						break;
				}
			}
		}

		private static void TranslateErrors(IEnumerable<VdcReturnValueBase> errors)
		{
			foreach (VdcReturnValueBase retVal in errors)
			{
				retVal.CanDoActionMessages = _canDoActionErrorsTranslator.TranslateErrorText(retVal.CanDoActionMessages.ToList()).ToArray();
			}
		}

		public static VdcQueryReturnValue RunQuery(VdcQueryType queryType, VdcQueryParametersBase parameters)
		{
			if (QLogger.Instance.IsDebugEnabled)
			{
				QLogger.Instance.DebugFormat("RunQuery: queryType={0}, parameters=\n{1}", queryType, parameters.Dump());
			}

			VdcQueryReturnValue returnValue = null;
			if (!IsUserLoggedIn)
			{
				returnValue = new VdcQueryReturnValue();
				QLogger.Instance.WarnFormat("Attempting to execute RunQuery for query: {0} while user not logged in", queryType.ToString());
			}
			else
			{
				DateTime time = DateTime.Now;
				try
				{
					if (IsBackendInitialized)
					{
						returnValue = Instance.RunQuery(new RunQuery1(queryType, parameters));
					}
					else
					{
						throw new BackendFaultException();
					}

					QLogger.Instance.DebugFormat("Query {0}. Took Time : {1}", queryType, DateTime.Now.Subtract(time).Milliseconds);

					if (!returnValue.Succeeded && _eventsHandler != null)
					{
						if (returnValue.ExceptionString == UserIsNotLoggedInKey)
						{
							LoggedInUser = null;
							_eventsHandler.ConnectionClosed(new BackendUserNotLoggedInException());
						}
						else
						{
							returnValue.ExceptionString =
								_canDoActionErrorsTranslator.TranslateErrorTextSingle(returnValue.ExceptionString);

							_eventsHandler.RunQueryFailed(new List<VdcQueryReturnValue> { returnValue });
						}
					}
				}
				catch (BackendException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Connection to Backend. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (InvalidOperationException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Backend remote configuration. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (Exception ex)
				{
					QLogger.Instance.ErrorFormat("Unhandled exception. Exception {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
			}
			return returnValue;

		}

		public static List<VdcQueryReturnValue> RunMultipleQueries(List<VdcQueryType> queryTypeList, List<VdcQueryParametersBase> queryParamList)
		{
			List<VdcQueryReturnValue> retValue = new List<VdcQueryReturnValue>();

			if (!IsUserLoggedIn)
			{
				QLogger.Instance.Warn("Attempting to execute RunMultipleQueries while user not logged in");
			}

			else
			{
				if (queryTypeList == null || queryParamList == null)
				{
					QLogger.Instance.Warn("Frontend::RunMultipleQueries: queryTypeList and/or queryParamList is null");
					return null;
				}
				if (queryTypeList.Count != queryParamList.Count)
				{
					QLogger.Instance.WarnFormat("Frontend::RunMultipleQueries: queryTypeList and queryParamList don't have the same amount of items (queryTypeList: {0}, queryParamList: {1})", queryTypeList.Count, queryParamList.Count);
					return null;
				}

				if (queryTypeList.Count > 0)
				{
					DateTime time = DateTime.Now;
					try
					{
						if (IsBackendInitialized)
						{
							retValue = Instance.RunMultipleQueries(new RunMultipleQueries1(queryTypeList.ToArray(), queryParamList.ToArray()));
						}
						else
						{
							throw new BackendFaultException();
						}

						QLogger.Instance.DebugFormat(
							"Frontend::RunMultipleQueries: MultipleQueries (count: {0}). Took Time : {1}",
							queryTypeList.Count,
							DateTime.Now.Subtract(time).Milliseconds);

						foreach (VdcQueryReturnValue singleReturnValue in retValue)
						{
							if (!singleReturnValue.Succeeded && _eventsHandler != null)
							{
								singleReturnValue.ExceptionString =
									_canDoActionErrorsTranslator.TranslateErrorTextSingle(singleReturnValue.ExceptionString);
							}
						}

						if (_eventsHandler != null && retValue.Any(a => a.Succeeded == false))
						{
							_eventsHandler.RunQueryFailed(retValue.Where(a => a.Succeeded == false).ToList());
						}
					}

					catch (BackendException ex)
					{
						QLogger.Instance.ErrorFormat("Frontend::RunMultipleQueries: Problem with Connection to Backend. Error: {0}", ex);
						if (_eventsHandler != null)
						{
							LoggedInUser = null;
							_eventsHandler.ConnectionClosed(ex);
						}
					}
					catch (InvalidOperationException ex)
					{
						QLogger.Instance.ErrorFormat("Frontend::RunMultipleQueries: Problem with Backend remote configuration. Error: {0}", ex);
						if (_eventsHandler != null)
						{
							LoggedInUser = null;
							_eventsHandler.ConnectionClosed(ex);
						}
					}
					catch (Exception ex)
					{
						QLogger.Instance.ErrorFormat("Frontend::RunMultipleQueries: Unhandled exception. Exception {0}", ex);
						if (_eventsHandler != null)
						{
							LoggedInUser = null;
							_eventsHandler.ConnectionClosed(ex);
						}
					}
				}
			}

			return retValue;
		}

		public static UIQueryReturnValue RunUIQuery(UIQueryType queryType, UIQueryParametersBase queryParams)
		{
			UIQueryReturnValue returnValue = null;

			if (!IsUserLoggedIn)
			{
				QLogger.Instance.WarnFormat("Attempting to execute RunUIQuery for query: {0} while user not logged in", queryType.ToString());
			}

			else
			{
				DateTime time = DateTime.Now;
				try
				{
					if (IsBackendInitialized)
					{
						returnValue = Instance.RunUIQuery(new RunUIQuery1(queryType, queryParams));
					}
					else
					{
						throw new BackendFaultException();
					}

					QLogger.Instance.DebugFormat(
						"Frontend::RunUIQuery: Query {0}. Took Time : {1}",
						queryType,
						DateTime.Now.Subtract(time).Milliseconds);
				}
				catch (BackendException ex)
				{
					QLogger.Instance.ErrorFormat("Frontend::RunUIQuery: Problem with Connection to Backend. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (InvalidOperationException ex)
				{
					QLogger.Instance.ErrorFormat("Frontend::RunUIQuery: Problem with Backend remote configuration. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (Exception ex)
				{
					QLogger.Instance.ErrorFormat("Frontend::RunUIQuery: Unhandled exception. Exception {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
			}
			return returnValue;
		}

		private static bool AsyncQueriesActivated;
		internal static void RunAsyncQuery(VdcQueryType queryType, VdcQueryParametersBase parameters)
		{
			if (!IsUserLoggedIn)
			{
				QLogger.Instance.WarnFormat("Attempting to execute RunAsyncQuery for query: {0} while user not logged in", queryType);
			}
			else
			{
				try
				{
					Instance.RunAsyncQuery(new RunAsyncQuery1(queryType, parameters));
					if (!AsyncQueriesActivated)
						AsyncQueriesActivated = true;
				}
				catch (BackendException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Connection to Backend. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (InvalidOperationException ex)
				{
					QLogger.Instance.ErrorFormat("Problem with Backend remote configuration. Error: {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
				catch (Exception ex)
				{
					QLogger.Instance.ErrorFormat("Unhandled exception. Exception {0}", ex);
					if (_eventsHandler != null)
					{
						LoggedInUser = null;
						_eventsHandler.ConnectionClosed(ex);
					}
				}
			}
		}

		public static void BackendFaulted()
		{
			if (_eventsHandler != null)
			{
				LoggedInUser = null;
				_eventsHandler.ConnectionClosed(new BackendCommunicationException());
			}
		}

		public static void RunActionExecutionFailed(VdcActionType actionType, VdcFault fault)
		{
			if (_eventsHandler != null)
			{
				fault.Message = _vdsErrorsTranslator.TranslateErrorTextSingle(fault.Message);
				_eventsHandler.RunActionExecutionFailed(actionType, fault);
			}
		}

		public static VdcQueryReturnValue RunPublicQuery(VdcQueryType queryType, VdcQueryParametersBase parameters)
		{
			VdcQueryReturnValue returnValue = null;
			try
			{
				returnValue = PublicServices.RunPublicQuery(new RunPublicQuery1(queryType, parameters));

				if (!returnValue.Succeeded && _eventsHandler != null)
				{
					_eventsHandler.RunQueryFailed(new List<VdcQueryReturnValue> { returnValue });
				}
			}
			catch (BackendException ex)
			{
				QLogger.Instance.ErrorFormat("Problem with Connection to Backend. Error: {0}", ex);
				if (_eventsHandler != null)
				{
					_eventsHandler.PublicConnectionClosed(ex);
				}
			}
			catch (InvalidOperationException ex)
			{
				QLogger.Instance.ErrorFormat("Problem with Backend remote configuration. Error: {0}", ex);
				if (_eventsHandler != null)
				{
					_eventsHandler.PublicConnectionClosed(ex);
				}
			}
			catch (Exception ex)
			{
				QLogger.Instance.ErrorFormat("Unhandled exception. Exception {0}", ex);
				if (_eventsHandler != null)
				{
					_eventsHandler.PublicConnectionClosed(ex);
				}
			}
			return returnValue;
		}

		public static void PublicServicesFaulted()
		{
			if (_eventsHandler != null)
			{
				_eventsHandler.PublicConnectionClosed(new PublicServicesCommunicationException());
			}
		}

		public static List<KeyValuePair<Guid, List<ListIVdcQueryableUpdatedData>>> GetAsyncQueryResults()
		{
			if (QLogger.Instance.IsDebugEnabled)
			{
				QLogger.Instance.Debug("GetAsyncQueryResults");
			}

			try
			{
				if (!AsyncQueriesActivated) //don't poll until queries registered
					return new List<KeyValuePair<Guid, List<ListIVdcQueryableUpdatedData>>>();
			}
			catch (Exception ex)
			{
				QLogger.Instance.ErrorFormat("Frontend.GetAsyncQueryResults.IsUserLoggedIn - {0} - {1}", ex.Message, ex.StackTrace);
			}

			List<KeyValuePair<Guid, List<ListIVdcQueryableUpdatedData>>> returnValue;
			try
			{
				returnValue = Instance.GetAsyncQueryResults(new GetAsyncQueryResults1());
			}
			catch (Exception ex)
			{
				QLogger.Instance.ErrorFormat("Frontend.GetAsyncQueryResults.GetAsyncQueryResults - {0} - {1}", ex.Message, ex.StackTrace);
				returnValue = new List<KeyValuePair<Guid, List<ListIVdcQueryableUpdatedData>>>();
			}
			return returnValue;
		}

		public static void RunQuery(VdcQueryType vdcQueryType, VdcQueryParametersBase vdcQueryParametersBase, AsyncQuery asyncQuery)
		{
			RaiseQueryStartedEvent(vdcQueryType, asyncQuery.Context);

			ThreadPool.QueueUserWorkItem(
				delegate
				{
					VdcQueryReturnValue returnValue = RunQuery(vdcQueryType, vdcQueryParametersBase);
					_eventsHandler.GuiSynchronizationContext.Post(
						delegate
						{
							asyncQuery.OriginalReturnValue = returnValue;
							if (returnValue != null && returnValue.Succeeded)
							{
								if (asyncQuery.converterCallback != null)
								{
									asyncQuery.asyncCallback(asyncQuery.Model, asyncQuery.converterCallback(returnValue.ReturnValue, asyncQuery));
								}
								else
								{
									asyncQuery.asyncCallback(asyncQuery.Model, returnValue);
								}
							}
							else
							{
								if (asyncQuery.HandleFailure)
								{
									asyncQuery.asyncCallback(asyncQuery.Model, returnValue);
								}
							}
							
							RaiseQueryCompleteEvent(vdcQueryType, asyncQuery.Context);

						}, null);
				});
		}

		public static void RunPublicQuery(VdcQueryType queryType, VdcQueryParametersBase parameters, AsyncQuery asyncQuery)
		{
			ThreadPool.QueueUserWorkItem(
				delegate
				{
					VdcQueryReturnValue returnValue = RunPublicQuery(queryType, parameters);

					_eventsHandler.GuiSynchronizationContext.Post(
						delegate
						{
							if (returnValue != null && returnValue.Succeeded)
							{
								if (asyncQuery.converterCallback != null)
								{
									asyncQuery.asyncCallback(asyncQuery.Model, asyncQuery.converterCallback(returnValue.ReturnValue, asyncQuery));
								}
								else
								{
									asyncQuery.asyncCallback(asyncQuery.Model, returnValue);
								}
							}
							else
							{
								if (asyncQuery.HandleFailure)
								{
									asyncQuery.asyncCallback(asyncQuery.Model, returnValue);
								}
							}
						}, null);
				});
		}

		public static void RunMultipleQueries(List<VdcQueryType> queryTypeList, List<VdcQueryParametersBase> queryParamList, IFrontendMultipleQueryAsyncCallback callback)
		{
			RunMultipleQueries(queryTypeList, queryParamList, callback, null);
		}

		public static void RunMultipleQueries(List<VdcQueryType> queryTypeList, List<VdcQueryParametersBase> queryParamList, IFrontendMultipleQueryAsyncCallback callback, string context)
		{
			RaiseQueryStartedEvent(queryTypeList, context);

			ThreadPool.QueueUserWorkItem(
				delegate
				{
					List<VdcQueryReturnValue> returnValues = RunMultipleQueries(queryTypeList, queryParamList);

					_eventsHandler.GuiSynchronizationContext.Post(
						delegate
						{
							FrontendMultipleQueryAsyncResult result = new FrontendMultipleQueryAsyncResult(queryTypeList, queryParamList, returnValues);
							callback.Executed(result);

							RaiseQueryCompleteEvent(queryTypeList, context);
						}, null);
				});
		}

		public static string CurrentContext { get; set; }

		private static void RaiseQueryEvent(Event queryEvent, VdcQueryType queryType, string context) {
			if (context != null && subscribedQueryTypes != null)
			{
				foreach (VdcQueryType vdcQueryType in subscribedQueryTypes)
				{
					if (queryType == vdcQueryType)
					{
						CurrentContext = context;
						queryEvent.raise(typeof (Frontend), EventArgs.Empty);
					}
				}
			}
		}	

		private static void RaiseQueryStartedEvent(VdcQueryType queryType, string context) {
			RaiseQueryEvent(QueryStartedEvent, queryType, context);       
		}
    
		private static void RaiseQueryCompleteEvent(VdcQueryType queryType, string context) {
			RaiseQueryEvent(QueryCompleteEvent, queryType, context);        
		}
    
		private static void RaiseQueryStartedEvent(List<VdcQueryType> queryTypeList, string context) {
			foreach (VdcQueryType queryType in queryTypeList) {
				RaiseQueryStartedEvent(queryType, context);
			}
		}
    
		private static void RaiseQueryCompleteEvent(List<VdcQueryType> queryTypeList, string context) {
			foreach (VdcQueryType queryType in queryTypeList) {
				RaiseQueryCompleteEvent(queryType, context);
			}
		} 

		private static void CheckTypeAndThrowIfNeeded(VdcQueryType queryType, VdcQueryParametersBase queryParams, out RegisterableQueryReturnDataType type)
		{
			type = RegisterableQueryTypes.GetReturnedDataTypeByQueryType(queryType, queryParams);

			if (type == RegisterableQueryReturnDataType.UNDEFINED)
			{
				throw new ArgumentException(string.Format("BackendCallback::CheckTypeAndThrowIfNeeded: Failed to register queryType '{0}' - it is not a registerable query type!", queryType), "queryType");
			}
		}

		private static void UnregisterAll()
		{
			asyncQueryTimer.Stop();

			foreach (var query in queries.ToList())
			{
				UnregisterQuery(query.Key);
			}
		}

		private static void AsyncQueryTimer_Elapsed(object sender, ElapsedEventArgs e)
		{
			if (!IsUserLoggedIn)
			{
				asyncQueryTimer.Stop();
			}

			//If timer was disabled before this call occured, don't proceed with the method.
			if (!asyncQueryTimer.Enabled)
			{
				return;
			}

			asyncQueryTimer.Stop();
			try
			{
				var queryResults = GetAsyncQueryResults();

				foreach (var queryResult in queryResults)
				{
					if (queryResult.Value == null)
					{
						continue;
					}

					if (queryResult.Value.Count == 1 && queryResult.Value[0].faulted != null)
					{
						BackendException(queryResult.Value[0].Faulted.Key, queryResult.Value[0].Faulted.Value);
					}
					else
					{
						foreach (var updatedData in queryResult.Value)
						{
							QueryDataChanged(queryResult.Key, updatedData);
						}
					}
				}
				if (queryResults.Count > 0)
				{
					//Notify for the retrievement other queries that has no updates reported.
					List<RegistrationResult> list = new List<RegistrationResult>();
					foreach (KeyValuePair<Guid, List<ListIVdcQueryableUpdatedData>> keyValuePair in (List<KeyValuePair<Guid, List<ListIVdcQueryableUpdatedData>>>)queryResults)
					{
						if (queries.ContainsKey(keyValuePair.Key))
						{
							RegistrationResult result = queries[keyValuePair.Key];
							Frontend._eventsHandler.GuiSynchronizationContext.Post(
							a =>
							{
								if (result.RetrievementCount == 0)
								{
									result.NotifyRetrieved();
								}
								result.RetrievementCount++;
							},
							null);
						}
					}
				}
			}
			catch (Exception ex)
			{
				QLogger.Instance.ErrorFormat("Frontend.BackendCallbackClient.GetAsyncQueries - {0} - {1}", ex.Message, ex.StackTrace);
				throw;
			}

			asyncQueryTimer.Start();
		}

		private static void QueryDataChanged(Guid queryId, IRegisterQueryUpdatedData updatedData)
		{
			if (queries.ContainsKey(queryId))
			{
				var result = queries[queryId];
				var data = (QueryListIVdcQueryableUpdatableData)result.Data;

				SendOrPostCallback callback =
					a =>
					{
						data.QueryDataChanged(updatedData);

						//Notify about retrieval for the first time.
						if (result.RetrievementCount == 0)
						{
							result.NotifyRetrieved();
						}
						result.RetrievementCount++;
					};

				if (_eventsHandler != null && _eventsHandler.GuiSynchronizationContext != null)
				{
					_eventsHandler.GuiSynchronizationContext.Post(callback, null);
				}
				else
				{
					callback(null);
				}
			}
		}

		private static void BackendException(VdcActionType actionType, VdcFault fault)
		{
			// should compare to VdcBllErrors.SESSION_ERROR (which indicates backend does not recognize the session).
			if (fault.Error == VdcBllErrors.SESSION_ERROR)
			{
				asyncQueryTimer.Stop();
				BackendFaulted();
			}
			else
				RunActionExecutionFailed(actionType, fault);
		}
	}


	public sealed class FrontendQueryAsyncResult
	{
		public VdcQueryType QueryType { get; private set; }
		public VdcQueryParametersBase Parameters { get; set; }
		public VdcQueryReturnValue ReturnValue { get; private set; }

		public FrontendQueryAsyncResult(VdcQueryType queryType, VdcQueryParametersBase parameters, VdcQueryReturnValue returnValue)
		{
			QueryType = queryType;
			Parameters = parameters;
			ReturnValue = returnValue;
		}
	}


	public sealed class FrontendMultipleQueryAsyncResult
	{
		public List<VdcQueryType> QueryTypes { get; private set; }
		public List<VdcQueryParametersBase> Parameters { get; set; }
		public List<VdcQueryReturnValue> ReturnValues { get; private set; }

		public FrontendMultipleQueryAsyncResult(List<VdcQueryType> queryTypes, List<VdcQueryParametersBase> parameters, List<VdcQueryReturnValue> returnValues)
		{
			QueryTypes = queryTypes;
			Parameters = parameters;
			ReturnValues = returnValues;
		}
	}


	public sealed class FrontendActionAsyncResult
	{
		public VdcActionType ActionType { get; private set; }
		public VdcActionParametersBase Parameters { get; set; }
		public VdcReturnValueBase ReturnValue { get; private set; }
		public object State { get; private set; }

		public FrontendActionAsyncResult(VdcActionType actionType, VdcActionParametersBase parameters, VdcReturnValueBase returnValue)
		{
			ActionType = actionType;
			Parameters = parameters;
			ReturnValue = returnValue;
		}

		public FrontendActionAsyncResult(VdcActionType actionType, VdcActionParametersBase parameters, VdcReturnValueBase returnValue, object state)
			: this(actionType, parameters, returnValue)
		{
			State = state;
		}
	}


	public sealed class FrontendMultipleActionAsyncResult
	{
		public VdcActionType ActionType { get; private set; }
		public IList<VdcActionParametersBase> Parameters { get; set; }
		public IList<VdcReturnValueBase> ReturnValue { get; private set; }
		public object State { get; private set; }

		public FrontendMultipleActionAsyncResult(VdcActionType actionType, IList<VdcActionParametersBase> parameters, IList<VdcReturnValueBase> returnValue)
		{
			ActionType = actionType;
			Parameters = parameters;
			ReturnValue = returnValue;
		}

		public FrontendMultipleActionAsyncResult(VdcActionType actionType, IList<VdcActionParametersBase> parameters, IList<VdcReturnValueBase> returnValue, object state)
			: this(actionType, parameters, returnValue)
		{
			State = state;
		}
	}
}
