# encoding: UTF-8
require 'fileutils'
require 'logging'

module Kafo
  class Logger
    class << self
      attr_writer :loggers

      def loggers
        @loggers ||= []
      end

      def buffer
        @buffer ||= []
      end

      def error_buffer
        @error_buffer ||= []
      end
    end

    PATTERN = "[%5l %d %c] %m\n"
    Logging.color_scheme('bright',
                         :levels => {
                             :info  => :green,
                             :warn  => :yellow,
                             :error => :red,
                             :fatal => [:white, :on_red]
                         },
                         :date   => :blue,
                         :logger => :cyan,
                         :line   => :yellow,
                         :file   => :yellow,
                         :method => :yellow
    )
    COLOR_LAYOUT   = Logging::Layouts::Pattern.new(:pattern => PATTERN, :color_scheme => 'bright')
    NOCOLOR_LAYOUT = Logging::Layouts::Pattern.new(:pattern => PATTERN, :color_scheme => nil)

    def self.setup
      begin
        FileUtils.mkdir_p(KafoConfigure.config.app[:log_dir], :mode => 0750)
      rescue Errno::EACCES => e
        puts "No permissions to create log dir #{KafoConfigure.config.app[:log_dir]}"
      end

      logger   = Logging.logger['main']
      filename = "#{KafoConfigure.config.app[:log_dir]}/#{KafoConfigure.config.app[:log_name] || 'configure.log'}"
      begin
        logger.appenders = ::Logging.appenders.rolling_file('configure',
                                                            :filename => filename,
                                                            :layout   => NOCOLOR_LAYOUT,
                                                            :truncate => true
        )
        # set owner and group (it's ignored if attribute is nil)
        FileUtils.chown KafoConfigure.config.app[:log_owner], KafoConfigure.config.app[:log_group], filename
      rescue ArgumentError => e
        puts "File #{filename} not writeable, won't log anything to file!"
      end

      logger.level = KafoConfigure.config.app[:log_level]

      fatal_logger           = Logging.logger['fatal']
      fatal_logger.level     = 'fatal'
      layout                 = KafoConfigure.config.app[:colors] ? COLOR_LAYOUT : NOCOLOR_LAYOUT
      fatal_logger.appenders = [::Logging.appenders.stderr(:layout => layout)]
      
      self.loggers = [logger, fatal_logger]
    end

    def self.setup_verbose
      logger           = Logging.logger['verbose']
      logger.level     = KafoConfigure.config.app[:verbose_log_level]
      layout           = KafoConfigure.config.app[:colors] ? COLOR_LAYOUT : NOCOLOR_LAYOUT
      logger.appenders = [::Logging.appenders.stdout(:layout => layout)]
      self.loggers<< logger
    end

    def self.buffering?
      KafoConfigure.verbose.nil? || ((KafoConfigure.verbose && !loggers.detect {|l| l.name == 'verbose'}) || self.loggers.empty?)
    end

    def self.dump_needed?
      !self.buffer.empty?
    end

    def self.to_buffer(buffer, *args)
      buffer << args
    end

    def self.dump_errors
      unless self.error_buffer.empty?
        loggers.each { |logger| logger.error 'Repeating errors encountered during run:' }
        self.dump_buffer(self.error_buffer)
      end
    end

    def dump_errors
      self.class.dump_errors
    end

    def self.dump_buffer(buffer)
      buffer.each do |log|
        self.loggers.each { |logger| logger.send log[0], *log[1], &log[2] }
      end
      buffer.clear
    end

    def log(name, *args, &block)
      if self.class.buffering?
        self.class.to_buffer(self.class.buffer, name, args, &block)
      else
        self.class.dump_buffer(self.class.buffer) if self.class.dump_needed?
        self.class.loggers.each { |logger| logger.send name, *args, &block }
      end
    end

    %w(warn info debug).each do |name|
      define_method(name) do |*args, &block|
        self.log(name, *args, &block)
      end
    end

    %w(fatal error).each do |name|
      define_method(name) do |*args, &block|
        self.class.to_buffer(self.class.error_buffer, name, *args, &block)
        self.log(name, *args, &block)
      end
    end
  end
end
