/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.yesmail.gwt.rolodex.client;

import com.google.gwt.user.client.ui.AbsolutePanel;
import com.google.gwt.user.client.ui.Composite;

import java.util.ArrayList;
import java.util.List;

/**
 * This widget displays a stack of RolodexCards taken from a RolodexCardBundle.
 *
 * @see RolodexCardBundle
 */
public class RolodexPanel extends Composite {

  final protected List cards = new ArrayList();

  final protected AbsolutePanel panel;

  protected RolodexCard previousCard, selectedCard;

  protected int leftAdjustment;

  protected int suggestedWidth = 0;

  protected int collapsedEntryWidth, collapsedPrecedingEntryWidth, expandedEntryWidth;

  protected boolean animate = true;

  /**
   * @param cardBundle Cards to place in the rolodex
   */
  public RolodexPanel(RolodexCardBundle cardBundle) {
    this(cardBundle, 0);
  }

  /**
   * @param cardBundle     Cards to place in the rolodex
   * @param suggestedWidth If the suggestedWidth is greater than the minimum width of the widget, then the suggestedWidth will be used
   */
  public RolodexPanel(RolodexCardBundle cardBundle, int suggestedWidth) {
    this(cardBundle, suggestedWidth, null, true);
  }

  /**
   * @param cardBundle      Cards to place in the rolodex
   * @param defaultSelected This card will be selected by default when the widget is shown
   */
  public RolodexPanel(
      RolodexCardBundle cardBundle, RolodexCard defaultSelected) {
    this(cardBundle, 0, defaultSelected, true);
  }

  /**
   * @param cardBundle      Cards to place in the rolodex
   * @param suggestedWidth  If the suggestedWidth is greater than the minimum width of the widget, then the suggestedWidth will be used
   * @param defaultSelected This card will be selected by default when the widget is shown
   * @param animate         If true, this widget will not animate the transitions between cards
   */
  public RolodexPanel(
      RolodexCardBundle cardBundle, int suggestedWidth,
      RolodexCard defaultSelected, boolean animate) {
    panel = new AbsolutePanel();
    initWidget(panel);

    this.animate = animate;

    this.suggestedWidth = suggestedWidth;

    int totalCollapsedWidth = 0, totalExpandedWidth = 0;

    RolodexCard.Slider primarySlider =
        new RolodexCard.Slider(600);
    RolodexCard.Slider secondarySlider =
        new RolodexCard.Slider(400);

    // todo: would be nice to not have to xlate here
    RolodexCard[] rolodexCards = cardBundle.getRolodexCards();
    for (int i = 0; i < rolodexCards.length; i++) {
      RolodexCard card = rolodexCards[i];
      card.setPanel(this);
      add(card);
      card.setTop(0); // should only need to be done once
      card.setPrimarySlider(primarySlider);
      card.setSecondarySlider(secondarySlider);
      totalCollapsedWidth += card.collapsedWidth;
      totalExpandedWidth += card.expandedWidth;
    }

    // todo: possibly do this up front stuff in the generator
    // 30% of avg. card collapsed width
    this.collapsedEntryWidth =
        (int) ((totalCollapsedWidth / rolodexCards.length) * 0.3);
    // 70% of avg. card collapsed width
    this.collapsedPrecedingEntryWidth =
        (int) ((totalCollapsedWidth / rolodexCards.length) * 0.7);
    // 80% of avg. card expanded width
    this.expandedEntryWidth =
        (int) ((totalExpandedWidth / rolodexCards.length) * 0.8);

    int totalMinWidth = ((cards.size() - 2) * collapsedEntryWidth) +
        (collapsedPrecedingEntryWidth * 2) + expandedEntryWidth + 1;
    leftAdjustment = (suggestedWidth - totalMinWidth > 0) ?
        (suggestedWidth - totalMinWidth) / 2 : 0;

    panel.setWidth(totalMinWidth + "px");
    panel.setHeight(cardBundle.getMaxHeight() + "px");

    // default card
    if (defaultSelected == null) defaultSelected = (RolodexCard) cards.get(0);
    setSelectedCard(defaultSelected);

    placeEntries(false);
  }

  protected RolodexPanel(RolodexCardBundle cardBundle, boolean animate) {
    this(cardBundle, 0, null, animate);
  }

  /**
   * Add a new card at the end of the stack (used to dynamically add cards
   * since all cards in a cardbundle are added during construction.
   * @param card new card
   */
  public void add(RolodexCard card) {
    cards.add(card);
    panel.add(card);
  }

  /**
   * Fire the click event on the currently selected card.
   */
  public void clickCurrentCard() {
    if (selectedCard == null) return;
    selectedCard.fireClickListeners();
  }

  /**
   * @return The currently selected card
   */
  public RolodexCard getSelectedCard() {
    return selectedCard;
  }

  /**
   * Set this as the currently selected card
   * @param card Card to select
   */
  public void select(RolodexCard card) {
    card.onMouseOver();
  }

  /**
   * Select the next card in the list
   */
  public void selectNextCard() {
    if (selectedCard == null) select((RolodexCard) cards.get(0));
    final int currentIndex = cards.indexOf(selectedCard);
    if (currentIndex + 1 >= cards.size()) return;
    select((RolodexCard) cards.get(currentIndex + 1));
  }

  /**
   * Select the previous card in the list
   */
  public void selectPreviousCard() {
    if (selectedCard == null) select((RolodexCard) cards.get(0));
    final int currentIndex = cards.indexOf(selectedCard);
    if (currentIndex - 1 < 0) return;
    select((RolodexCard) cards.get(currentIndex - 1));
  }

  public void setPreviousCard(RolodexCard previousCard) {
    this.previousCard = previousCard;
  }

  public void setSelectedCard(RolodexCard selectedCard) {
    this.selectedCard = selectedCard;
  }

  protected void placeEntries(boolean animate) {
    int left = leftAdjustment;
    int selectedEntryIndex = cards.indexOf(selectedCard);
    int previousEntryIndex = cards.indexOf(previousCard);
    boolean leftExpand = previousEntryIndex >= selectedEntryIndex;

    int numberOfEntries = cards.size();
    for (int i = 0; i < numberOfEntries; i++) {
      RolodexCard card = (RolodexCard) cards.get(i);

      int zIndex = numberOfEntries + 2 - Math.abs(selectedEntryIndex - i);
      // give a small buffer to the beginning so the overall offset will be consistent
      if (selectedEntryIndex == 0 && i == 0) {
        left += collapsedPrecedingEntryWidth - collapsedEntryWidth;
      }

      boolean secondaryEntry = false;
      if (leftExpand && selectedEntryIndex - i == -1) {
        secondaryEntry = true;
      } else if (!leftExpand && selectedEntryIndex - i == 1) {
        secondaryEntry = true;
      }

      // place the entry
      if (i == selectedEntryIndex) {
        if (leftExpand) {
          card.expandLeft(left, zIndex, animate);
        } else {
          card.expandRight(left, zIndex, animate);
        }
      } else if (i < selectedEntryIndex) {
        card.collapseLeft(left, zIndex, secondaryEntry & animate);
      } else {
        card.collapseRight(left, zIndex, secondaryEntry & animate);
      }

      // find width for positioning
      if (i == selectedEntryIndex) {
        left += expandedEntryWidth;
      } else if (selectedEntryIndex - i == 1) {
        left += collapsedPrecedingEntryWidth;
      } else {
        left += collapsedEntryWidth;
      }
    }
  }
}
