package org.infinispan.configuration.cache;

import java.util.concurrent.TimeUnit;

import org.infinispan.commons.configuration.Builder;

/**
 * Configuration for the async cache store. If enabled, this provides you with asynchronous writes
 * to the cache store, giving you 'write-behind' caching.
 *
 * @author pmuir
 *
 */
public class AsyncStoreConfigurationBuilder<S> extends AbstractStoreConfigurationChildBuilder<S> implements Builder<AsyncStoreConfiguration> {

   private boolean enabled = false;
   private int modificationQueueSize = 1024;
   private int threadPoolSize = 1;

   AsyncStoreConfigurationBuilder(AbstractStoreConfigurationBuilder<? extends AbstractStoreConfiguration, ?> builder) {
      super(builder);
   }

   /**
    * If true, all modifications to this cache store happen asynchronously, on a separate thread.
    */
   public AsyncStoreConfigurationBuilder<S> enable() {
      this.enabled = true;
      return this;
   }

   public AsyncStoreConfigurationBuilder<S> disable() {
      this.enabled = false;
      return this;
   }

   public AsyncStoreConfigurationBuilder<S> enabled(boolean enabled) {
      this.enabled = enabled;
      return this;
   }

   /**
    * @deprecated Unused.
    */
   @Deprecated
   public AsyncStoreConfigurationBuilder<S> flushLockTimeout(long l) {
      return this;
   }

   /**
    * @deprecated Unused.
    */
   @Deprecated
   public AsyncStoreConfigurationBuilder<S> flushLockTimeout(long l, TimeUnit unit) {
      return this;
   }

   /**
    * Sets the size of the modification queue for the async store. If updates are made at a rate
    * that is faster than the underlying cache store can process this queue, then the async store
    * behaves like a synchronous store for that period, blocking until the queue can accept more
    * elements.
    */
   public AsyncStoreConfigurationBuilder<S> modificationQueueSize(int i) {
      this.modificationQueueSize = i;
      return this;
   }

   /**
    * @deprecated Unused.
    */
   @Deprecated
   public AsyncStoreConfigurationBuilder<S> shutdownTimeout(long l) {
      return this;
   }

   /**
    * @deprecated Unused.
    */
   @Deprecated
   public AsyncStoreConfigurationBuilder<S> shutdownTimeout(long l, TimeUnit unit) {
      return this;
   }

   /**
    * Size of the thread pool whose threads are responsible for applying the modifications.
    */
   public AsyncStoreConfigurationBuilder<S> threadPoolSize(int i) {
      this.threadPoolSize = i;
      return this;
   }

   @Override
   public
   void validate() {
   }

   @Override
   public AsyncStoreConfiguration create() {
      return new AsyncStoreConfiguration(enabled, modificationQueueSize, threadPoolSize);
   }

   @Override
   public AsyncStoreConfigurationBuilder<S> read(AsyncStoreConfiguration template) {
      this.enabled = template.enabled();
      this.modificationQueueSize = template.modificationQueueSize();
      this.threadPoolSize = template.threadPoolSize();

      return this;
   }

   @Override
   public String toString() {
      return "AsyncLoaderConfigurationBuilder{" +
            "enabled=" + enabled +
            ", modificationQueueSize=" + modificationQueueSize +
            ", threadPoolSize=" + threadPoolSize +
            '}';
   }

}
