package org.infinispan.client.hotrod.xsite;

import org.infinispan.Cache;
import org.infinispan.client.hotrod.HitsAwareCacheManagersTest.HitCountInterceptor;
import org.infinispan.client.hotrod.RemoteCacheManager;
import org.infinispan.client.hotrod.test.HotRodClientTestingUtil;
import org.infinispan.client.hotrod.test.InternalRemoteCacheManager;
import org.infinispan.configuration.cache.BackupConfiguration.BackupStrategy;
import org.infinispan.configuration.cache.BackupConfigurationBuilder;
import org.infinispan.configuration.cache.CacheMode;
import org.infinispan.configuration.cache.ConfigurationBuilder;
import org.infinispan.configuration.global.GlobalConfigurationBuilder;
import org.infinispan.interceptors.base.CommandInterceptor;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.server.hotrod.HotRodServer;
import org.infinispan.server.hotrod.configuration.HotRodServerConfigurationBuilder;
import org.infinispan.test.TestingUtil;
import org.infinispan.xsite.AbstractXSiteTest;
import org.testng.annotations.AfterClass;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.infinispan.server.hotrod.test.HotRodTestingUtil.hotRodCacheConfiguration;
import static org.testng.AssertJUnit.assertEquals;

abstract class AbstractHotRodSiteFailoverTest extends AbstractXSiteTest {

   static String SITE_A = "LON";
   static String SITE_B = "NYC";
   static int NODES_PER_SITE = 2;

   Map<String, List<HotRodServer>> siteServers = new HashMap<String, List<HotRodServer>>();

   RemoteCacheManager client(String siteName, String backupSiteName) {
      HotRodServer server = siteServers.get(siteName).get(0);
      org.infinispan.client.hotrod.configuration.ConfigurationBuilder clientBuilder =
         new org.infinispan.client.hotrod.configuration.ConfigurationBuilder();
      clientBuilder
         .addServer().host("localhost").port(server.getPort())
         .maxRetries(3); // Some retries so that shutdown nodes can be skipped

      Integer backupPort = null;
      if (backupSiteName != null) {
         HotRodServer backupServer = siteServers.get(backupSiteName).get(0);
         clientBuilder.addCluster(backupSiteName).addClusterNode("localhost", backupServer.getPort());
         backupPort = backupServer.getPort();
      }

      if (backupPort != null)
         log.debugf("Client for site '%s' connecting to main server in port %d, and backup cluster node port is %d",
            siteName, server.getPort(), backupPort);
      else
         log.debugf("Client for site '%s' connecting to main server in port %d",
            siteName, server.getPort());

      return new InternalRemoteCacheManager(clientBuilder.build());
   }

   int findServerPort(String siteName) {
      return siteServers.get(siteName).get(0).getPort();
   }

   void killSite(String siteName) {
      List<Integer> serverPorts = new ArrayList<Integer>();
      for (HotRodServer server : siteServers.get(siteName))
         serverPorts.add(server.getPort());

      log.debugf("Kill site '%s' with ports: %s", siteName, serverPorts);

      for (HotRodServer server : siteServers.get(siteName))
         HotRodClientTestingUtil.killServers(server);

      for (EmbeddedCacheManager cacheManager : site(siteName).cacheManagers())
         TestingUtil.killCacheManagers(cacheManager);
   }

   @Override
   protected void createSites() {
      createHotRodSite(SITE_A, SITE_B, null);
      createHotRodSite(SITE_B, SITE_A, null);
   }

   @AfterClass(alwaysRun = true) // run even if the test failed
   protected void destroy() {
      try {
         for (List<HotRodServer> servers : siteServers.values())
            HotRodClientTestingUtil.killServers(servers.toArray(new HotRodServer[servers.size()]));
      } finally {
         super.destroy();
      }
   }

   protected void createHotRodSite(String siteName, String backupSiteName, Integer serverPort) {
      ConfigurationBuilder builder = hotRodCacheConfiguration(
         getDefaultClusteredCacheConfig(CacheMode.DIST_SYNC, false));
      BackupConfigurationBuilder backup = builder.sites().addBackup();
      backup.site(backupSiteName).strategy(BackupStrategy.SYNC);

      GlobalConfigurationBuilder globalBuilder = GlobalConfigurationBuilder.defaultClusteredBuilder();
      globalBuilder.globalJmxStatistics().allowDuplicateDomains(true);
      globalBuilder.site().localSite(siteName);
      TestSite site = createSite(siteName, NODES_PER_SITE, globalBuilder, builder);
      Collection<EmbeddedCacheManager> cacheManagers = site.cacheManagers();

      List<HotRodServer> servers = new ArrayList<HotRodServer>();
      for (EmbeddedCacheManager cm : cacheManagers) {
         if (serverPort != null)
            servers.add(HotRodClientTestingUtil.startHotRodServer(cm, serverPort,
               new HotRodServerConfigurationBuilder()));
         else
            servers.add(HotRodClientTestingUtil.startHotRodServer(cm));
      }
      siteServers.put(siteName, servers);

      if (log.isDebugEnabled()) {
         List<Integer> serverPorts = new ArrayList<Integer>();
         for (HotRodServer server : servers)
            serverPorts.add(server.getPort());

         log.debugf("Create site '%s' with ports: %s", siteName, serverPorts);
      }
   }

   protected void addHitCountInterceptors() {
      for (Map.Entry<String, List<HotRodServer>> entry : siteServers.entrySet()) {
         for (HotRodServer server : entry.getValue()) {
            HitCountInterceptor interceptor = new HitCountInterceptor();
            server.getCacheManager().getCache().getAdvancedCache().addInterceptor(interceptor, 1);
         }
      }
   }

   protected void assertNoHits() {
      for (Map.Entry<String, List<HotRodServer>> entry : siteServers.entrySet()) {
         for (HotRodServer server : entry.getValue()) {
            Cache<?, ?> cache = server.getCacheManager().getCache();
            HitCountInterceptor interceptor = getHitCountInterceptor(cache);
            assertEquals(0, interceptor.getHits());
         }
      }
   }

   protected void resetHitCounters() {
      for (Map.Entry<String, List<HotRodServer>> entry : siteServers.entrySet()) {
         for (HotRodServer server : entry.getValue()) {
            Cache<?, ?> cache = server.getCacheManager().getCache();
            HitCountInterceptor interceptor = getHitCountInterceptor(cache);
            interceptor.reset();
         }
      }
   }

   protected void assertSiteHit(String siteName, int expectedHits) {
      int hit = 0;
      for (HotRodServer server : siteServers.get(siteName)) {
         Cache<?, ?> cache = server.getCacheManager().getCache();
         HitCountInterceptor interceptor = getHitCountInterceptor(cache);
         if (interceptor.getHits() == expectedHits) hit++;
      }

      assertEquals(1, hit);
      resetHitCounters();
   }

   protected void assertSiteNotHit(String siteName) {
      for (HotRodServer server : siteServers.get(siteName)) {
         Cache<?, ?> cache = server.getCacheManager().getCache();
         HitCountInterceptor interceptor = getHitCountInterceptor(cache);
         assertEquals(0, interceptor.getHits());
      }
   }

   protected HitCountInterceptor getHitCountInterceptor(Cache<?, ?> cache) {
      HitCountInterceptor hitCountInterceptor = null;
      List<CommandInterceptor> interceptorChain = cache.getAdvancedCache().getInterceptorChain();
      for (CommandInterceptor interceptor : interceptorChain) {
         boolean isHitCountInterceptor = interceptor instanceof HitCountInterceptor;
         if (hitCountInterceptor != null && isHitCountInterceptor) {
            throw new IllegalStateException("Two HitCountInterceptors! " + interceptorChain);
         }
         if (isHitCountInterceptor) {
            hitCountInterceptor = (HitCountInterceptor) interceptor;
         }
      }
      return hitCountInterceptor;
   }

}
