/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009 Red Hat Inc. and/or its affiliates and other
 * contributors as indicated by the @author tags. All rights reserved.
 * See the copyright.txt in the distribution for a full listing of
 * individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.infinispan.loaders;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;

import org.infinispan.commons.util.TypedProperties;

/**
 * Configures {@link AbstractCacheStore}. This allows you to tune a number of characteristics of the
 * {@link AbstractCacheStore}.
 * <p/>
 * <ul>
 * <li><tt>purgeSynchronously</tt> - whether
 * {@link org.infinispan.loaders.CacheStore#purgeExpired()} calls happen synchronously or not. By
 * default, this is set to <tt>false</tt>.</li>
 * <li><tt>purgerThreads</tt> - number of threads to use when purging. Defaults to <tt>1</tt> if
 * <tt>purgeSynchronously</tt> is <tt>true</tt>, ignored if <tt>false</tt>.</li>
 * </ul>
 * 
 * 
 * 
 * @author Mircea.Markus@jboss.com
 * @author Vladimir Blagojevic
 * @since 4.0
 * 
 * @see <a href="../../../config.html#ce_loaders_loader">Configuration reference</a>
 */

public class AbstractCacheStoreConfig extends AbstractCacheLoaderConfig implements CacheStoreConfig {

   /** The serialVersionUID */
   private static final long serialVersionUID = 4607371052771122893L;
   
   protected Boolean ignoreModifications = false;
   protected Boolean fetchPersistentState = false;
   protected Boolean purgeOnStartup = false;
   protected Boolean purgeSynchronously = false;

   protected Integer purgerThreads = 1;

   @Override
   public Boolean isPurgeSynchronously() {
      return purgeSynchronously;
   }

   @Override
   public Integer getPurgerThreads() {
      return purgerThreads;
   }
   
   @XmlElement(name="properties")
   public TypedProperties getTypedProperties(){
      return properties;
   }
   
   public void setTypedProperties (TypedProperties tp){
      this.properties = tp;
   }

   /**
    * If true, CacheStore#purgeExpired() call will be done synchronously
    *
    * @param purgeSynchronously
    */
   @Override
   public void setPurgeSynchronously(Boolean purgeSynchronously) {
      this.purgeSynchronously = purgeSynchronously;
   }
   
   /**
    * If true, CacheStore#purgeExpired() call will be done synchronously
    * 
    * @param purgeSynchronously
    */
   @Override
   public CacheStoreConfig purgeSynchronously(Boolean purgeSynchronously) {
      this.purgeSynchronously = purgeSynchronously;
      return this;
   }

   /**
    * The number of threads to use when purging asynchronously.
    * 
    * @param purgerThreads
    * @deprecated use {@link #purgerThreads(Integer)} instead
   */
   @Deprecated
   public void setPurgerThreads(Integer purgerThreads) {
      this.purgerThreads = purgerThreads;
   }

   @Override
   public CacheStoreConfig purgerThreads(Integer purgerThreads) {
      setPurgerThreads(purgerThreads);
      return this;
   }

   @Override
   @XmlAttribute
   public Boolean isFetchPersistentState() {
      return fetchPersistentState;
   }

   /**
    * If true, fetch persistent state when joining a cluster. If multiple cache stores are chained,
    * only one of them can have this property enabled. Persistent state transfer with a shared cache
    * store does not make sense, as the same persistent store that provides the data will just end
    * up receiving it. Therefore, if a shared cache store is used, the cache will not allow a
    * persistent state transfer even if a cache store has this property set to true. Finally,
    * setting it to true only makes sense if in a clustered environment, and only 'replication' and
    * 'invalidation' cluster modes are supported.
    * 
    * 
    * @param fetchPersistentState
    */
   @Override
   public void setFetchPersistentState(Boolean fetchPersistentState) {
      this.fetchPersistentState = fetchPersistentState;
   }
   
   /**
    * If true, fetch persistent state when joining a cluster. If multiple cache stores are chained,
    * only one of them can have this property enabled. Persistent state transfer with a shared cache
    * store does not make sense, as the same persistent store that provides the data will just end
    * up receiving it. Therefore, if a shared cache store is used, the cache will not allow a
    * persistent state transfer even if a cache store has this property set to true. Finally,
    * setting it to true only makes sense if in a clustered environment, and only 'replication' and
    * 'invalidation' cluster modes are supported.
    * 
    * 
    * @param fetchPersistentState
    */
   @Override
   public CacheStoreConfig fetchPersistentState(Boolean fetchPersistentState) {
      this.fetchPersistentState = fetchPersistentState;
      return this;
   }

   /**
    * If true, any operation that modifies the cache (put, remove, clear, store...etc) won't be
    * applied to the cache store. This means that the cache store could become out of sync with the
    * cache.
    * 
    * @param ignoreModifications
    */
   @Override
   public void setIgnoreModifications(Boolean ignoreModifications) {
      this.ignoreModifications = ignoreModifications;
   }
   
   /**
    * If true, any operation that modifies the cache (put, remove, clear, store...etc) won't be
    * applied to the cache store. This means that the cache store could become out of sync with the
    * cache.
    * 
    * @param ignoreModifications
    */
   @Override
   public CacheStoreConfig ignoreModifications(Boolean ignoreModifications) {
      this.ignoreModifications = ignoreModifications;
      return this;
   }

   @Override
   public Boolean isIgnoreModifications() {
      return ignoreModifications;
   }
   
   @Override
   public Boolean isPurgeOnStartup() {
      return purgeOnStartup;
   }

   /**
    * 
    * If true, purges this cache store when it starts up.
    * 
    * @param purgeOnStartup
    */
   @Override
   public CacheStoreConfig purgeOnStartup(Boolean purgeOnStartup) {
      this.purgeOnStartup = purgeOnStartup;
      return this;
   }
   
   /**
    * 
    * If true, purges this cache store when it starts up.
    * 
    * @param purgeOnStartup
    */
   @Override
   public void setPurgeOnStartup(Boolean purgeOnStartup) {
      this.purgeOnStartup = purgeOnStartup;
   }
}
