module ThreeScale
  module Backend
    module TimeHacks
      ONE_MINUTE = 60
      ONE_HOUR   = 60 * ONE_MINUTE
      ONE_DAY    = 24 * ONE_HOUR

      def beginning_of_cycle(cycle)
        case cycle
        when :eternity then self.class.utc(1970, 1, 1)
        when :year     then self.class.utc(year, 1, 1)
        when :month    then self.class.utc(year, month, 1)
        when :week     then beginning_of_week_hack
        when :day      then beginning_of_day_hack
        when :hour     then self.class.utc(year, month, day, hour)
        when :minute   then self.class.utc(year, month, day, hour, min)
        when Numeric   then beginning_of_numeric_cycle(cycle)
        else
          raise_invalid_period(cycle)
        end
      end

      def end_of_cycle(cycle)
        case cycle
        ## a WTF take-away for future generations
        when :eternity then self.class.utc(9999, 12, 31)
        when :year     then self.class.utc(year + 1, 1, 1)
        when :month    then end_of_month_hack
        when :week     then end_of_week_hack
        when :day      then beginning_of_day_hack + ONE_DAY
        when :hour     then beginning_of_cycle(:hour) + ONE_HOUR
        when :minute   then beginning_of_cycle(:minute) + ONE_MINUTE
        else
          raise_invalid_period(cycle)
        end
      end

      def end_of_month_hack
        if month == 12
          end_of_cycle(:year)
        else
          self.class.utc(year, month + 1, 1)
        end
      end

      def beginning_of_week_hack
        # This is stolen from active support and slightly modified
        days_to_monday = wday != 0 ? wday - 1 : 6
        (self - days_to_monday * ONE_DAY).beginning_of_day_hack
      end

      def end_of_week_hack
        days_to_next_monday = wday != 0 ? 8 - wday : 1
        (self + days_to_next_monday * ONE_DAY).beginning_of_day_hack
      end

      def beginning_of_day_hack
        self.class.utc(year, month, day)
      end

      def beginning_of_bucket(seconds_in_bucket)
        if seconds_in_bucket > 30 || seconds_in_bucket < 1 || !seconds_in_bucket.is_a?(Fixnum)
          raise Exception, "seconds_in_bucket cannot be larger than 30 seconds or smaller than 1"
        end
        norm_sec = (sec/seconds_in_bucket)*seconds_in_bucket
        self.class.utc(year, month, day, hour, min, norm_sec)
      end

      # Formats the time using as little characters as possible, but still keeping
      # readability.
      #
      # == Examples
      #
      # Time.utc(2010, 5, 6, 17, 24, 22).to_compact_s # "20100506172422"
      # Time.utc(2010, 5, 6, 17, 24, 00).to_compact_s # "201005061724"
      # Time.utc(2010, 5, 6, 17, 00, 00).to_compact_s # "2010050617"
      # Time.utc(2010, 5, 6, 00, 00, 00).to_compact_s # "20100506"
      #
      # Careful with cases where hours, minutes or seconds have 2 digits and
      # the second one is a 0. You might find them a bit counter-intuitive
      # (notice the missing 0 at the end of the resulting string):
      # Time.utc(2016, 1, 2, 10, 11, 10).to_compact_s # "2016010210111"
      # Time.utc(2016, 1, 2, 18, 10, 0).to_compact_s # "20160102181"
      # Time.utc(2016, 1, 2, 10, 0, 0).to_compact_s # "201601021"
      #
      # That behavior does not happen with days ending with a 0:
      # Time.utc(2016, 1, 20, 0, 0, 0).to_compact_s # "20160120"
      def to_compact_s
        strftime('%Y%m%d%H%M%S').sub(/0{0,6}$/, '')
      end

      def to_not_compact_s
        strftime('%Y%m%d%H%M%S')
      end

      private

      def beginning_of_numeric_cycle(cycle)
        base = cycle_base(cycle)

        cycles_count = ((self - base) / cycle).floor
        base + cycles_count * cycle
      end

      def cycle_base(cycle)
        case cycle
        when 0..ONE_MINUTE        then ::Time.utc(year, month, day, hour, min)
        when ONE_MINUTE..ONE_HOUR then ::Time.utc(year, month, day, hour)
        when ONE_HOUR..ONE_DAY    then ::Time.utc(year, month, day)
        else raise ArgumentError, "Argument must be duration from 0 seconds to 1 day."
        end
      end

      def raise_invalid_period(period)
        raise ArgumentError, "Argument must be a number or one of :minute, :hour, :day, :week, :month, or :year, not #{period.inspect}"
      end

      module ClassMethods
        def parse_to_utc(input)
          parts = nil

          begin
            parts = Date._parse(input.to_s)
          rescue TypeError => e
          end

          return if parts.nil? || parts.empty? || !parts.has_key?(:year) || !parts.has_key?(:mon) || !parts.has_key?(:mday)

          time = nil
          begin
            time = Time.utc(parts[:year],
                            parts[:mon],
                            parts[:mday],
                            parts[:hour],
                            parts[:min],
                            parts[:sec],
                            parts[:sec_fraction])
            time -= parts[:offset] if parts[:offset]

          rescue ArgumentError => e
          end

          return time
        end
      end
    end
  end
end

Time.send(:include, ThreeScale::Backend::TimeHacks)
Time.extend(ThreeScale::Backend::TimeHacks::ClassMethods)
