#include <glib.h>
#include <linux/fs.h>
#include <stdio.h>
#include <string.h>
#include <utils.h>
#include <asm/dasd.h>
#include <s390utils/vtoc.h>



typedef enum {
    BD_S390_ERROR_DEVICE,
    BD_S390_ERROR_FORMAT_FAILED,
    BD_S390_ERROR_DASDFMT,
} BDS390Error;


/**
 * bd_s390_dasd_format:
 * @dasd: dasd to format
 * @error: (out): place to store error (if any)
 *
 * Returns: whether dasdfmt was successful or not
 */
gboolean  bd_s390_dasd_format (const gchar *dasd, GError **error);


/**
 * bd_s390_dasd_needs_format:
 * @dasd: dasd to check, whether it needs dasdfmt run on it
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a dasd needs dasdfmt run against it
 */
gboolean  bd_s390_dasd_needs_format (const gchar *dasd, GError **error);


/**
 * bd_s390_dasd_online:
 * @dasd: dasd to switch online
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a dasd was successfully switched online
 */
gboolean  bd_s390_dasd_online (gchar *dasd, GError **error);


/**
 * bd_s390_dasd_is_Ldl:
 * @dasd: dasd to check, whether it is LDL formatted
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a dasd is LDL formatted
 */
gboolean  bd_s390_dasd_is_ldl (const gchar *dasd, GError **error);


/**
 * bd_s390_sanitize_dev_input:
 * @dev a DASD or zFCP device number
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): a synthesized dasd or zfcp device number
 */
gchar* bd_s390_sanitize_dev_input (const gchar *dev, GError **error);


/**
 * bd_s390_zfcp_sanitize_wwpn_input:
 * @wwpn a zFCP WWPN identifier
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): a synthesized zFCP WWPN
 */
gchar* bd_s390_zfcp_sanitize_wwpn_input (const gchar *wwpn, GError **error);


/**
 * bd_s390_zfcp_sanitize_lun_input:
 * @lun a zFCP LUN identifier
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): a synthesized zFCP LUN
 */
gchar* bd_s390_zfcp_sanitize_lun_input (const gchar *lun, GError **error);

