package org.infinispan.cdi.test.interceptor;

import org.infinispan.Cache;
import org.infinispan.cdi.interceptor.DefaultCacheKey;
import org.infinispan.cdi.test.interceptor.config.Config;
import org.infinispan.cdi.test.interceptor.config.Custom;
import org.infinispan.cdi.test.interceptor.service.CacheRemoveEntryService;
import org.infinispan.cdi.test.interceptor.service.CustomCacheKey;
import org.infinispan.cdi.test.testutil.Deployments;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.testng.Arquillian;
import org.jboss.shrinkwrap.api.Archive;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.cache.CacheException;
import javax.cache.annotation.GeneratedCacheKey;
import javax.inject.Inject;
import java.lang.reflect.Method;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertTrue;

/**
 * @author Kevin Pollet <kevin.pollet@serli.com> (C) 2011 SERLI
 * @see javax.cache.annotation.CacheRemove
 */
@Test(groups = "functional", testName = "cdi.test.interceptor.CacheRemoveEntryInterceptorTest")
public class CacheRemoveEntryInterceptorTest extends Arquillian{

   @Deployment
   public static Archive<?> deployment() {
      return Deployments.baseDeployment()
            .addClass(CacheRemoveEntryInterceptorTest.class)
            .addPackage(CacheRemoveEntryService.class.getPackage())
            .addPackage(Config.class.getPackage());
   }

   @Inject
   private CacheRemoveEntryService service;

   @Inject
   @Custom
   private Cache<GeneratedCacheKey, String> customCache;

   @BeforeMethod
   public void beforeMethod() {
      customCache.clear();
      assertTrue(customCache.isEmpty());
   }

   @Test(expectedExceptions = CacheException.class)
   public void testCacheRemoveEntry() {
      service.removeEntry("Kevin");
   }

   @Test
   public void testCacheRemoveEntryWithCacheName() {
      final GeneratedCacheKey cacheKey = new DefaultCacheKey(new Object[]{"Kevin"});

      customCache.put(cacheKey, "Hello Kevin");

      assertEquals(customCache.size(), 1);
      assertTrue(customCache.containsKey(cacheKey));

      service.removeEntryWithCacheName("Kevin");

      assertEquals(customCache.size(), 0);
   }

   @Test
   public void testCacheRemoveEntryWithCacheKeyParam() {
      final GeneratedCacheKey cacheKey = new DefaultCacheKey(new Object[]{"Kevin"});

      customCache.put(cacheKey, "Hello Kevin");

      assertEquals(customCache.size(), 1);
      assertTrue(customCache.containsKey(cacheKey));

      service.removeEntryWithCacheKeyParam("Kevin", "foo");

      assertEquals(customCache.size(), 0);
   }

   @Test
   public void testCacheRemoveEntryAfterInvocationWithException() {
      final GeneratedCacheKey cacheKey = new DefaultCacheKey(new Object[]{"Kevin"});

      customCache.put(cacheKey, "Hello Kevin");

      assertEquals(customCache.size(), 1);
      assertTrue(customCache.containsKey(cacheKey));

      try {

         service.removeEntryWithCacheName(null);

      } catch (NullPointerException e) {
         assertEquals(customCache.size(), 1);
      }
   }

   @Test
   public void testCacheRemoveEntryWithCacheKeyGenerator() throws NoSuchMethodException {
      final Method method = CacheRemoveEntryService.class.getMethod("removeEntryWithCacheKeyGenerator", String.class);
      final GeneratedCacheKey cacheKey = new CustomCacheKey("remove", "Kevin");

      customCache.put(cacheKey, "Hello Kevin");

      assertEquals(customCache.size(), 1);
      assertTrue(customCache.containsKey(cacheKey));

      service.removeEntryWithCacheKeyGenerator("Kevin");

      assertEquals(customCache.size(), 0);
   }

   @Test
   public void testCacheRemoveEntryBeforeInvocationWithException() {
      final GeneratedCacheKey cacheKey = new DefaultCacheKey(new Object[]{"Kevin"});

      customCache.put(cacheKey, "Hello Kevin");

      assertEquals(customCache.size(), 1);
      assertTrue(customCache.containsKey(cacheKey));

      try {

         service.removeEntryBeforeInvocationWithException("Kevin");

      } catch (NullPointerException e) {
         assertEquals(customCache.size(), 0);
      }
   }
}
