/* pam_wheel module */

/*
 * Written by Cristian Gafton <gafton@redhat.com> 1996/09/10
 * See the end of the file for Copyright Information
 *
 *
 * 1.2 - added 'deny' and 'group=' options
 * 1.1 - added 'trust' option
 * 1.0 - the code is working for at least another person, so... :-)
 * 0.1 - use vsyslog instead of vfprintf/syslog in _pam_log
 *     - return PAM_IGNORE on success (take care of sloppy sysadmins..)
 *     - use pam_get_user instead of pam_get_item(...,PAM_USER,...)
 *     - a new arg use_uid to auth the current uid instead of the
 *       initial (logged in) one.
 * 0.0 - first release
 *
 * TODO:
 *  - try to use make_remark from pam_unix/support.c
 *  - consider returning on failure PAM_FAIL_NOW if the user is not
 *    a wheel member.
 */

#include "../../_pam_aconf.h"
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <limits.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <utmp.h>

/*
 * here, we make a definition for the externally accessible function
 * in this file (this definition is required for static a module
 * but strongly encouraged generally) it is used to instruct the
 * modules include file to define the function prototypes.
 */

#define PAM_SM_AUTH
#define PAM_SM_ACCOUNT
#include "../../libpam/include/security/pam_modules.h"

#define PAM_GETPWNAM_R
#define PAM_GETPWUID_R
#define PAM_GETGRNAM_R
#define PAM_GETGRGID_R
#include "../../libpam/include/security/_pam_macros.h"

/* some syslogging */

static void _pam_log(int err, const char *format, ...)
{
    va_list args;

    va_start(args, format);
    openlog("PAM-Wheel", LOG_CONS|LOG_PID, LOG_AUTH);
    vsyslog(err, format, args);
    va_end(args);
    closelog();
}

/* checks if a user is on a list of members of the GID 0 group */

static int is_on_list(char * const *list, const char *member)
{
    while (list && *list) {
        if (strcmp(*list, member) == 0)
            return 1;
        list++;
    }
    return 0;
}

/* argument parsing */

#define PAM_DEBUG_ARG       0x0001
#define PAM_USE_UID_ARG     0x0002
#define PAM_TRUST_ARG       0x0004
#define PAM_DENY_ARG        0x0010  

static int _pam_parse(int argc, const char **argv, char *use_group,
		      size_t group_length)
{
     int ctrl=0;

     memset(use_group, '\0', group_length);

     /* step through arguments */
     for (ctrl=0; argc-- > 0; ++argv) {

          /* generic options */

          if (!strcmp(*argv,"debug"))
               ctrl |= PAM_DEBUG_ARG;
          else if (!strcmp(*argv,"use_uid"))
               ctrl |= PAM_USE_UID_ARG;
          else if (!strcmp(*argv,"trust"))
               ctrl |= PAM_TRUST_ARG;
          else if (!strcmp(*argv,"deny"))
               ctrl |= PAM_DENY_ARG;
          else if (!strncmp(*argv,"group=",6))
                strncpy(use_group,*argv+6,group_length-1);
          else {
               _pam_log(LOG_ERR,"pam_parse: unknown option; %s",*argv);
          }
     }

     return ctrl;
}

  /*
   * Because getlogin() is braindead and sometimes it just
   * doesn't work, we reimplement it here.  (Copied in from pam_unix.)
   */
static char *PAM_getlogin(void)
{
	struct utmp *ut, line;
	char *curr_tty, *retval;
	static char curr_user[sizeof(ut->ut_user) + 4];

	retval = NULL;

	curr_tty = ttyname(0);
	if (curr_tty != NULL) {
		D(("PAM_getlogin ttyname: %s", curr_tty));
		/* utmp/utmpx only lists ttys under /dev. */
		if ((strlen(curr_tty) > 5) &&
		    (strncmp(curr_tty, "/dev/", 5) == 0)) {
			curr_tty += 5;
			setutent();
			strncpy(line.ut_line, curr_tty, sizeof(line.ut_line));
			line.ut_line[sizeof(line.ut_line) - 1] = '\0';
			if ((ut = getutline(&line)) != NULL) {
				strncpy(curr_user, ut->ut_user, sizeof(ut->ut_user));
				curr_user[sizeof(curr_user) - 1] = '\0';
				retval = curr_user;
			}
			endutent();
		}
	}
	D(("PAM_getlogin retval: %s", retval));

	return retval;
}

/* --- authentication management functions --- */

PAM_EXTERN
int pam_sm_authenticate(pam_handle_t *pamh,int flags,int argc
			,const char **argv)
{
     int ctrl = 0;
     const char *username = NULL;
     char *fromsu = NULL;
     struct passwd passwd, *pwd = NULL, *tpwd = NULL;
     struct group group, *grp = NULL;
     int retval = PAM_AUTH_ERR;
     char use_group[BUFSIZ];
     char *ubuf = NULL, *gbuf = NULL, *tbuf = NULL;
     size_t ubuflen, gbuflen, tbuflen;
    
     ctrl = _pam_parse(argc, argv, use_group, sizeof(use_group));
     retval = pam_get_user(pamh,&username,"login: ");
     if ((retval != PAM_SUCCESS) || (!username)) {
        if (ctrl & PAM_DEBUG_ARG)
            _pam_log(LOG_DEBUG,"can not get the username");
        return PAM_SERVICE_ERR;
     }

     /* su to a uid 0 account ? */
     if (_pam_getpwnam_r(username, &passwd, &ubuf, &ubuflen, &pwd) != 0)
	pwd = NULL;
     if (!pwd) {
        if (ctrl & PAM_DEBUG_ARG)
            _pam_log(LOG_NOTICE,"unknown user %s",username);
        return PAM_USER_UNKNOWN;
     }
     
     /* Now we know that the username exists, pass on to other modules...
      * the call to pam_get_user made this obsolete, so is commented out
      *
      * pam_set_item(pamh,PAM_USER,(const void *)username);
      */

     if (ctrl & PAM_USE_UID_ARG) {
         if (_pam_getpwuid_r(getuid(), &passwd, &tbuf, &tbuflen, &tpwd) != 0)
	    tpwd = NULL;
         if (!tpwd) {
            if (ctrl & PAM_DEBUG_ARG)
                _pam_log(LOG_NOTICE,"who is running me ?!");
            return PAM_SERVICE_ERR;
         }
         fromsu = tpwd->pw_name;
     } else {
         fromsu = PAM_getlogin();
	 if ((fromsu == NULL) ||
	     (_pam_getpwnam_r(fromsu, &passwd, &tbuf, &tbuflen, &tpwd) != 0))
	     tpwd = NULL;
	 if((!fromsu) || (!tpwd)) {
             if (ctrl & PAM_DEBUG_ARG)
                _pam_log(LOG_NOTICE,"who is running me ?!");
             return PAM_SERVICE_ERR;
         }
     }
     
     if (!use_group[0]) {
	 if (_pam_getgrnam_r("wheel", &group, &gbuf, &gbuflen, &grp) != 0)
	     grp = NULL;
	 if (grp == NULL) {
	 	if(_pam_getgrgid_r(0, &group, &gbuf, &gbuflen, &grp) != 0)
			grp = NULL;
	 }
     } else
	 if(_pam_getgrnam_r(use_group, &group, &gbuf, &gbuflen, &grp) != 0)
		grp = NULL;
        
     if (!grp || (!grp->gr_mem && (tpwd->pw_gid != grp->gr_gid))) {
        if (ctrl & PAM_DEBUG_ARG) {
            if (!use_group[0])
                _pam_log(LOG_NOTICE,"no members in a GID 0 group");
            else
                _pam_log(LOG_NOTICE,"no members in '%s' group",use_group);
        }
        if (ctrl & PAM_DENY_ARG) {
            /* if this was meant to deny access to the members
             * of this group and the group does not exist, allow
             * access
             */
            if (ubuf) {
                free(ubuf);
	    }
            if (gbuf) {
                free(gbuf);
	    }
            if (tbuf) {
                free(tbuf);
	    }
            return PAM_IGNORE;
	} else {
            if (ubuf) {
                free(ubuf);
	    }
            if (gbuf) {
                free(gbuf);
	    }
            if (tbuf) {
                free(tbuf);
	    }
            return PAM_AUTH_ERR;
	}
     }
        
     if (is_on_list(grp->gr_mem, fromsu) || (tpwd->pw_gid == grp->gr_gid)) {
        if (ctrl & PAM_DEBUG_ARG)
            _pam_log(LOG_NOTICE,"Access %s to '%s' for '%s'",
                     (ctrl & PAM_DENY_ARG)?"denied":"granted",
                     fromsu,username);
        if (ubuf) {
            free(ubuf);
	}
        if (gbuf) {
            free(gbuf);
	}
        if (tbuf) {
            free(tbuf);
	}
        if (ctrl & PAM_DENY_ARG)
            return PAM_PERM_DENIED;
        else
            if (ctrl & PAM_TRUST_ARG)
                return PAM_SUCCESS;
            else
                return PAM_IGNORE;
     }

     if (ctrl & PAM_DEBUG_ARG)
        _pam_log(LOG_NOTICE,"Access %s for '%s' to '%s'",
        (ctrl & PAM_DENY_ARG)?"granted":"denied",fromsu,username);
     if (ubuf) {
        free(ubuf);
     }
     if (gbuf) {
        free(gbuf);
     }
     if (tbuf) {
        free(tbuf);
     }
     if (ctrl & PAM_DENY_ARG)
        if (ctrl & PAM_TRUST_ARG)
            return PAM_SUCCESS;
        else
            return PAM_IGNORE;
     else
        return PAM_PERM_DENIED;
}

PAM_EXTERN
int pam_sm_setcred(pam_handle_t *pamh,int flags,int argc
		   ,const char **argv)
{
     return PAM_SUCCESS;
}

/* --- account management function --- */

PAM_EXTERN
int pam_sm_acct_mgmt(pam_handle_t *pamh,int flags,int argc
		     ,const char **argv)
{
    int ctrl;
    char use_group[BUFSIZ];

    /* parse arguments */
    ctrl = _pam_parse(argc, argv, use_group, sizeof(use_group));

    if (ctrl & PAM_DEBUG_ARG)
        _pam_log(LOG_DEBUG,"pam_wheel called for account management, calling "
		 "authentication function");
    return pam_sm_authenticate(pamh, 0, argc, argv);
}

#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_wheel_modstruct = {
     "pam_wheel",
     pam_sm_authenticate,
     pam_sm_setcred,
     pam_sm_acct_mgmt,
     NULL,
     NULL,
     NULL,
};

#endif

/*
 * Copyright (c) Cristian Gafton <gafton@redhat.com>, 1996, 1997
 *                                              All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED `AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
