
/*
 * Mesa 3-D graphics library
 * Version:  4.0.1
 *
 * Copyright (C) 1999-2001  Brian Paul   All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * BRIAN PAUL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


/*
 * This file manages the OpenGL API dispatch layer.
 * The dispatch table (struct _glapi_table) is basically just a list
 * of function pointers.
 * There are functions to set/get the current dispatch table for the
 * current thread and to manage registration/dispatch of dynamically
 * added extension functions.
 *
 * It's intended that this file and the other glapi*.[ch] files are
 * flexible enough to be reused in several places:  XFree86, DRI-
 * based libGL.so, and perhaps the SGI SI.
 *
 * There are no dependencies on Mesa in this code.
 *
 * Versions (API changes):
 *   2000/02/23  - original version for Mesa 3.3 and XFree86 4.0
 *   2001/01/16  - added dispatch override feature for Mesa 3.5
 */



#include "glheader.h"
#include "glapi.h"
#include "glapioffsets.h"
#include "glapitable.h"
#include "glthread.h"

extern hidden void *__glapi_noop_table[];

/* if we support thread-safety, build a special dispatch table for use
 * in thread-safety mode (ThreadSafe == GL_TRUE).  Each entry in the
 * dispatch table will call _glthread_GetTSD() to get the actual dispatch
 * table bound to the current thread, then jump through that table.
 */

#if defined(GLX_USE_TLS)

__thread struct _glapi_table *_gl_tls_Dispatch
  __attribute__((tls_model("initial-exec")))
  = (struct _glapi_table *) __glapi_noop_table;
__thread struct _glapi_table *_gl_tls_RealDispatch
  __attribute__((tls_model("initial-exec")))
  = (struct _glapi_table *) __glapi_noop_table;
__thread void *_gl_tls_Context
  __attribute__((tls_model("initial-exec")));

#elif defined(THREADS)

static GLboolean ThreadSafe = GL_FALSE;  /* In thread-safe mode? */
hidden _glthread_TSD _gl_DispatchTSD;	 /* Per-thread dispatch pointer */
static _glthread_TSD RealDispatchTSD;    /* only when using override */
static _glthread_TSD ContextTSD;         /* Per-thread context pointer */


#ifdef USE_ASM_PTHREAD_GETSPECIFIC
#define __glapi_threadsafe_table NULL
/* _glapi_Dispatch will be NULL if the per-thread dispatch table needs
 * to be fetched.  Avoid anyone outside of libGL.so to play with it.  */
extern hidden struct _glapi_table *_glapi_Dispatch;
#else
extern hidden void *__glapi_threadsafe_table[];
#endif
#endif

#ifndef GLX_USE_TLS
struct _glapi_table *_glapi_Dispatch = (struct _glapi_table *) __glapi_noop_table;
struct _glapi_table *_glapi_RealDispatch = (struct _glapi_table *) __glapi_noop_table;
#endif

/* Used when thread safety disabled */
void *_glapi_Context = NULL;

static GLuint MaxDispatchOffset = sizeof(struct _glapi_table) / sizeof(void *) - 1;
static GLboolean GetSizeCalled = GL_FALSE;

static GLboolean DispatchOverride = GL_FALSE;


/* strdup() is actually not a standard ANSI C or POSIX routine.
 * Irix will not define it if ANSI mode is in effect.
 */
static corefn char *
str_dup(const char *str)
{
   char *copy;
   copy = (char*) malloc(strlen(str) + 1);
   if (!copy)
      return NULL;
   strcpy(copy, str);
   return copy;
}



/*
 * We should call this periodically from a function such as glXMakeCurrent
 * in order to test if multiple threads are being used.  When we detect
 * that situation we should then call _glapi_enable_thread_safety()
 */
corefn void
_glapi_check_multithread(void)
{
#if defined(THREADS) && !defined(GLX_USE_TLS)
   if (!ThreadSafe) {
      static unsigned long knownID;
      static GLboolean firstCall = GL_TRUE;
      if (firstCall) {
         knownID = _glthread_GetID();
         firstCall = GL_FALSE;
      }
      else if (knownID != _glthread_GetID()) {
         ThreadSafe = GL_TRUE;
      }
   }
   if (ThreadSafe) {
      /* make sure that this thread's dispatch pointer isn't null */
      if (!_glapi_get_dispatch()) {
         _glapi_set_dispatch(NULL);
      }
   }
#endif
}



/*
 * Set the current context pointer for this thread.
 * The context pointer is an opaque type which should be cast to
 * void from the real context pointer type.
 */
corefn void
_glapi_set_context(void *context)
{
#if defined(GLX_USE_TLS)
   _gl_tls_Context = context;
#elif defined(THREADS)
   _glthread_SetTSD(&ContextTSD, context);
   if (ThreadSafe)
      _glapi_Context = NULL;
   else
      _glapi_Context = context;
#else
   _glapi_Context = context;
#endif
}



/*
 * Get the current context pointer for this thread.
 * The context pointer is an opaque type which should be cast from
 * void to the real context pointer type.
 */
corefn void *
_glapi_get_context(void)
{
#if defined(GLX_USE_TLS)
   return _gl_tls_Context;
#elif defined(THREADS)
   if (ThreadSafe) {
      return _glthread_GetTSD(&ContextTSD);
   }
   else {
      return _glapi_Context;
   }
#else
   return _glapi_Context;
#endif
}



/*
 * Set the global or per-thread dispatch table pointer.
 */
corefn void
_glapi_set_dispatch(struct _glapi_table *dispatch)
{
#if defined(GLX_USE_TLS) && defined (USE_X86_ASM)
   static int entrypoint_relocated;
   if (__builtin_expect (entrypoint_relocated, 1) == 0) {
      /* We have to update the instructions atomically, so that other thread
       * executing one of the gl* functions sees always either a call followed
       * by nop, or movl %gs:something, %eax.  */
      extern hidden int _gl_entrypoint_templ[];
      extern hidden int _gl_end_table[];
      int *p, tmp1, tmp2, c = _gl_entrypoint_templ[1] & 0xffff;
      int use_cmpxchg8b = 0;
      int fl1, fl2;

      /* Not all IA-32 CPUs have cmpxchg8b instruction.
	 Check for it.  This is an optimization only, so if CPU lacks
	 cmpxchg8b, libGL.so will be just tiny bit slower than it could.  */

      /* See if we can use cpuid.  */
      __asm ("pushfl; pushfl; popl %0; movl %0,%1; xorl %2,%0;"
	     "pushl %0; popfl; pushfl; popl %0; popfl"
	     : "=&r" (fl1), "=&r" (fl2)
	     : "i" (0x00200000));
      if (((fl1 ^ fl2) & 0x00200000) != 0) {
	/* See if cpuid gives capabilities.  */
	__asm ("cpuid" : "=a" (fl1) : "0" (0) : "ebx", "ecx", "edx", "cc");
	if (fl1 != 0) {
	   /* See if capabilities include CX8 (8th bit).  */
	   __asm ("cpuid" : "=a" (fl1), "=d" (fl2) : "0" (1)
		  : "ebx", "ecx", "cc");
	   if ((fl2 & (1 << 8)) != 0)
	      use_cmpxchg8b = 1;
	}
      }

      if (use_cmpxchg8b)
	for (p = _gl_entrypoint_templ + 4; p < _gl_end_table; p += 4) {
	   __asm __volatile ("lock; cmpxchg8b %0"
			     : "=m" (p[0]), "=m" (p[1]), "=a" (tmp1), "=d" (tmp2)
			     : "m" (p[0]), "m" (p[1]), "a" (p[0]), "d" (p[1]),
			       "b" (_gl_entrypoint_templ[0]),
			       "c" (c | (p[1] & 0xffff0000))
			     : "cc");
	}
      entrypoint_relocated = 1;
   }
#endif
   if (!dispatch) {
      /* use the no-op functions */
      dispatch = (struct _glapi_table *) __glapi_noop_table;
   }
#ifdef DEBUG
   else {
      _glapi_check_table(dispatch);
   }
#endif

#if defined(GLX_USE_TLS)
   /* Set RealDispatch unconditionally, so that get_dispatch
    * can always return _gl_tls_RealDispatch.  */
   _gl_tls_RealDispatch = dispatch;
   if (!DispatchOverride)
      _gl_tls_Dispatch = dispatch;
#elif defined(THREADS)
   if (DispatchOverride) {
      _glthread_SetTSD(&RealDispatchTSD, (void *) dispatch);
      if (ThreadSafe)
         _glapi_RealDispatch = (struct _glapi_table*) __glapi_threadsafe_table;
      else
         _glapi_RealDispatch = dispatch;
   }
   else {
      /* normal operation */
      _glthread_SetTSD(&_gl_DispatchTSD, (void *) dispatch);
      if (ThreadSafe)
         _glapi_Dispatch = (struct _glapi_table *) __glapi_threadsafe_table;
      else
         _glapi_Dispatch = dispatch;
   }
#else /*THREADS*/
   if (DispatchOverride) {
      _glapi_RealDispatch = dispatch;
   }
   else {
      _glapi_Dispatch = dispatch;
   }
#endif /*THREADS*/
}



/*
 * Return pointer to current dispatch table for calling thread.
 */
corefn struct _glapi_table *
_glapi_get_dispatch(void)
{
#if defined(GLX_USE_TLS)
   return _gl_tls_RealDispatch;
#elif defined(THREADS)
   if (ThreadSafe) {
      if (DispatchOverride) {
         return (struct _glapi_table *) _glthread_GetTSD(&RealDispatchTSD);
      }
      else {
         return (struct _glapi_table *) _glthread_GetTSD(&_gl_DispatchTSD);
      }
   }
   else {
      if (DispatchOverride) {
         assert(_glapi_RealDispatch);
         return _glapi_RealDispatch;
      }
      else {
         assert(_glapi_Dispatch);
         return _glapi_Dispatch;
      }
   }
#else
   /* FIXME: Shouldn't this be
    * if (DispatchOverride) return _glapi_RealDispatch;
    * else return _glapi_Dispatch; ?  */
   return _glapi_Dispatch;
#endif
}


/*
 * Notes on dispatch overrride:
 *
 * Dispatch override allows an external agent to hook into the GL dispatch
 * mechanism before execution goes into the core rendering library.  For
 * example, a trace mechanism would insert itself as an overrider, print
 * logging info for each GL function, then dispatch to the real GL function.
 *
 * libGLS (GL Stream library) is another agent that might use override.
 *
 * We don't allow more than one layer of overriding at this time.
 * In the future we may allow nested/layered override.  In that case
 * _glapi_begin_dispatch_override() will return an override layer,
 * _glapi_end_dispatch_override(layer) will remove an override layer
 * and _glapi_get_override_dispatch(layer) will return the dispatch
 * table for a given override layer.  layer = 0 will be the "real"
 * dispatch table.
 */

/*
 * Return: dispatch override layer number.
 */
corefn int
_glapi_begin_dispatch_override(struct _glapi_table *override)
{
   struct _glapi_table *real = _glapi_get_dispatch();

   assert(!DispatchOverride);  /* can't nest at this time */
   DispatchOverride = GL_TRUE;

   _glapi_set_dispatch(real);

#if defined(GLX_USE_TLS)
   _gl_tls_Dispatch = override;
#elif defined(THREADS)
   _glthread_SetTSD(&_gl_DispatchTSD, (void *) override);
   if (ThreadSafe)
      _glapi_Dispatch = (struct _glapi_table *) __glapi_threadsafe_table;
   else
      _glapi_Dispatch = override;
#else
   _glapi_Dispatch = override;
#endif
   return 1;
}


corefn void
_glapi_end_dispatch_override(int layer)
{
   struct _glapi_table *real = _glapi_get_dispatch();
   (void) layer;
   DispatchOverride = GL_FALSE;
   _glapi_set_dispatch(real);
   /* the rest of this isn't needed, just play it safe */
#if defined(THREADS) && !defined(GLX_USE_TLS)
   _glthread_SetTSD(&RealDispatchTSD, NULL);
#endif
#if !defined(GLX_USE_TLS)
   _glapi_RealDispatch = NULL;
#endif
}


corefn struct _glapi_table *
_glapi_get_override_dispatch(int layer)
{
   if (layer == 0) {
      return _glapi_get_dispatch();
   }
   else {
      if (DispatchOverride) {
#if defined(GLX_USE_TLS)
	 return _gl_tls_Dispatch;
#elif defined(THREADS)
         return (struct _glapi_table *) _glthread_GetTSD(&_gl_DispatchTSD);
#else
         return _glapi_Dispatch;
#endif
      }
      else {
         return NULL;
      }
   }
}



/*
 * Return size of dispatch table struct as number of functions (or
 * slots).
 */
corefn GLuint
_glapi_get_dispatch_table_size(void)
{
   /*   return sizeof(struct _glapi_table) / sizeof(void *);*/
   GetSizeCalled = GL_TRUE;
   return MaxDispatchOffset + 1;
}



/*
 * Get API dispatcher version string.
 */
corefn const char *
_glapi_get_version(void)
{
   return "20010116";  /* YYYYMMDD */
}


/*
 * For each entry in static_functions[] which use this function
 * we should implement a dispatch function in glapitemp.h and
 * in glapinoop.c
 */
corefn static int
#if __GNUC__ >= 3
__attribute__((used))
#endif
NotImplemented(void)
{
   return 0;
}


struct name_address_offset {
   const char *Name;
   GLvoid *Address;
   GLuint Offset;
};

#if defined(CORE_WITHOUT_PIC) && defined(USE_HIDDEN_VISIBILITY) && defined(__i386__)
#define GOT_RELATIVE_FUNCTIONS
#endif

#ifdef GOT_RELATIVE_FUNCTIONS
extern hidden const long gotdiff[];
struct name_address_offset2 {
   GLuint Name, Address, Offset;
};
extern struct name_address_offset2 static_functions[];
__asm (".section .rodata.static_functions,\"a\",@progbits; .balign 4; static_functions:; .previous");
__asm (".data; .balign 4; gotdiff: .long gotdiff@GOTOFF, gotdiff; .previous");
#define FN(x,y,z) FN1(x,y,z)
#define FN1(x,y,z) __asm (".section .rodata.str1.1,\"aMS\",@progbits,1;" \
			  "1: .string " #x "; .previous;" \
			  ".section .rodata.static_functions;" \
			  ".long 1b@GOTOFF; .long " #y "@GOTOFF, " #z "; .previous");
#define FN_NAME(x) (char *) gotdiff[1] - gotdiff[0] + (long) static_functions[x].Name
#define FN_ADDRESS(x) (char *) gotdiff[1] - gotdiff[0] + (long) static_functions[x].Address
#else
static struct name_address_offset static_functions[] = {
#define FN(x,y,z) { x, (GLvoid *) y, z },
#define FN_NAME(x) static_functions[x].Name
#define FN_ADDRESS(x) static_functions[x].Address
#endif
	/* GL 1.1 */
FN ("glNewList", glNewList, _gloffset_NewList)
FN ("glEndList", glEndList, _gloffset_EndList)
FN ("glCallList", glCallList, _gloffset_CallList)
FN ("glCallLists", glCallLists, _gloffset_CallLists)
FN ("glDeleteLists", glDeleteLists, _gloffset_DeleteLists)
FN ("glGenLists", glGenLists, _gloffset_GenLists)
FN ("glListBase", glListBase, _gloffset_ListBase)
FN ("glBegin", glBegin, _gloffset_Begin)
FN ("glBitmap", glBitmap, _gloffset_Bitmap)
FN ("glColor3b", glColor3b, _gloffset_Color3b)
FN ("glColor3bv", glColor3bv, _gloffset_Color3bv)
FN ("glColor3d", glColor3d, _gloffset_Color3d)
FN ("glColor3dv", glColor3dv, _gloffset_Color3dv)
FN ("glColor3f", glColor3f, _gloffset_Color3f)
FN ("glColor3fv", glColor3fv, _gloffset_Color3fv)
FN ("glColor3i", glColor3i, _gloffset_Color3i)
FN ("glColor3iv", glColor3iv, _gloffset_Color3iv)
FN ("glColor3s", glColor3s, _gloffset_Color3s)
FN ("glColor3sv", glColor3sv, _gloffset_Color3sv)
FN ("glColor3ub", glColor3ub, _gloffset_Color3ub)
FN ("glColor3ubv", glColor3ubv, _gloffset_Color3ubv)
FN ("glColor3ui", glColor3ui, _gloffset_Color3ui)
FN ("glColor3uiv", glColor3uiv, _gloffset_Color3uiv)
FN ("glColor3us", glColor3us, _gloffset_Color3us)
FN ("glColor3usv", glColor3usv, _gloffset_Color3usv)
FN ("glColor4b", glColor4b, _gloffset_Color4b)
FN ("glColor4bv", glColor4bv, _gloffset_Color4bv)
FN ("glColor4d", glColor4d, _gloffset_Color4d)
FN ("glColor4dv", glColor4dv, _gloffset_Color4dv)
FN ("glColor4f", glColor4f, _gloffset_Color4f)
FN ("glColor4fv", glColor4fv, _gloffset_Color4fv)
FN ("glColor4i", glColor4i, _gloffset_Color4i)
FN ("glColor4iv", glColor4iv, _gloffset_Color4iv)
FN ("glColor4s", glColor4s, _gloffset_Color4s)
FN ("glColor4sv", glColor4sv, _gloffset_Color4sv)
FN ("glColor4ub", glColor4ub, _gloffset_Color4ub)
FN ("glColor4ubv", glColor4ubv, _gloffset_Color4ubv)
FN ("glColor4ui", glColor4ui, _gloffset_Color4ui)
FN ("glColor4uiv", glColor4uiv, _gloffset_Color4uiv)
FN ("glColor4us", glColor4us, _gloffset_Color4us)
FN ("glColor4usv", glColor4usv, _gloffset_Color4usv)
FN ("glEdgeFlag", glEdgeFlag, _gloffset_EdgeFlag)
FN ("glEdgeFlagv", glEdgeFlagv, _gloffset_EdgeFlagv)
FN ("glEnd", glEnd, _gloffset_End)
FN ("glIndexd", glIndexd, _gloffset_Indexd)
FN ("glIndexdv", glIndexdv, _gloffset_Indexdv)
FN ("glIndexf", glIndexf, _gloffset_Indexf)
FN ("glIndexfv", glIndexfv, _gloffset_Indexfv)
FN ("glIndexi", glIndexi, _gloffset_Indexi)
FN ("glIndexiv", glIndexiv, _gloffset_Indexiv)
FN ("glIndexs", glIndexs, _gloffset_Indexs)
FN ("glIndexsv", glIndexsv, _gloffset_Indexsv)
FN ("glNormal3b", glNormal3b, _gloffset_Normal3b)
FN ("glNormal3bv", glNormal3bv, _gloffset_Normal3bv)
FN ("glNormal3d", glNormal3d, _gloffset_Normal3d)
FN ("glNormal3dv", glNormal3dv, _gloffset_Normal3dv)
FN ("glNormal3f", glNormal3f, _gloffset_Normal3f)
FN ("glNormal3fv", glNormal3fv, _gloffset_Normal3fv)
FN ("glNormal3i", glNormal3i, _gloffset_Normal3i)
FN ("glNormal3iv", glNormal3iv, _gloffset_Normal3iv)
FN ("glNormal3s", glNormal3s, _gloffset_Normal3s)
FN ("glNormal3sv", glNormal3sv, _gloffset_Normal3sv)
FN ("glRasterPos2d", glRasterPos2d, _gloffset_RasterPos2d)
FN ("glRasterPos2dv", glRasterPos2dv, _gloffset_RasterPos2dv)
FN ("glRasterPos2f", glRasterPos2f, _gloffset_RasterPos2f)
FN ("glRasterPos2fv", glRasterPos2fv, _gloffset_RasterPos2fv)
FN ("glRasterPos2i", glRasterPos2i, _gloffset_RasterPos2i)
FN ("glRasterPos2iv", glRasterPos2iv, _gloffset_RasterPos2iv)
FN ("glRasterPos2s", glRasterPos2s, _gloffset_RasterPos2s)
FN ("glRasterPos2sv", glRasterPos2sv, _gloffset_RasterPos2sv)
FN ("glRasterPos3d", glRasterPos3d, _gloffset_RasterPos3d)
FN ("glRasterPos3dv", glRasterPos3dv, _gloffset_RasterPos3dv)
FN ("glRasterPos3f", glRasterPos3f, _gloffset_RasterPos3f)
FN ("glRasterPos3fv", glRasterPos3fv, _gloffset_RasterPos3fv)
FN ("glRasterPos3i", glRasterPos3i, _gloffset_RasterPos3i)
FN ("glRasterPos3iv", glRasterPos3iv, _gloffset_RasterPos3iv)
FN ("glRasterPos3s", glRasterPos3s, _gloffset_RasterPos3s)
FN ("glRasterPos3sv", glRasterPos3sv, _gloffset_RasterPos3sv)
FN ("glRasterPos4d", glRasterPos4d, _gloffset_RasterPos4d)
FN ("glRasterPos4dv", glRasterPos4dv, _gloffset_RasterPos4dv)
FN ("glRasterPos4f", glRasterPos4f, _gloffset_RasterPos4f)
FN ("glRasterPos4fv", glRasterPos4fv, _gloffset_RasterPos4fv)
FN ("glRasterPos4i", glRasterPos4i, _gloffset_RasterPos4i)
FN ("glRasterPos4iv", glRasterPos4iv, _gloffset_RasterPos4iv)
FN ("glRasterPos4s", glRasterPos4s, _gloffset_RasterPos4s)
FN ("glRasterPos4sv", glRasterPos4sv, _gloffset_RasterPos4sv)
FN ("glRectd", glRectd, _gloffset_Rectd)
FN ("glRectdv", glRectdv, _gloffset_Rectdv)
FN ("glRectf", glRectf, _gloffset_Rectf)
FN ("glRectfv", glRectfv, _gloffset_Rectfv)
FN ("glRecti", glRecti, _gloffset_Recti)
FN ("glRectiv", glRectiv, _gloffset_Rectiv)
FN ("glRects", glRects, _gloffset_Rects)
FN ("glRectsv", glRectsv, _gloffset_Rectsv)
FN ("glTexCoord1d", glTexCoord1d, _gloffset_TexCoord1d)
FN ("glTexCoord1dv", glTexCoord1dv, _gloffset_TexCoord1dv)
FN ("glTexCoord1f", glTexCoord1f, _gloffset_TexCoord1f)
FN ("glTexCoord1fv", glTexCoord1fv, _gloffset_TexCoord1fv)
FN ("glTexCoord1i", glTexCoord1i, _gloffset_TexCoord1i)
FN ("glTexCoord1iv", glTexCoord1iv, _gloffset_TexCoord1iv)
FN ("glTexCoord1s", glTexCoord1s, _gloffset_TexCoord1s)
FN ("glTexCoord1sv", glTexCoord1sv, _gloffset_TexCoord1sv)
FN ("glTexCoord2d", glTexCoord2d, _gloffset_TexCoord2d)
FN ("glTexCoord2dv", glTexCoord2dv, _gloffset_TexCoord2dv)
FN ("glTexCoord2f", glTexCoord2f, _gloffset_TexCoord2f)
FN ("glTexCoord2fv", glTexCoord2fv, _gloffset_TexCoord2fv)
FN ("glTexCoord2i", glTexCoord2i, _gloffset_TexCoord2i)
FN ("glTexCoord2iv", glTexCoord2iv, _gloffset_TexCoord2iv)
FN ("glTexCoord2s", glTexCoord2s, _gloffset_TexCoord2s)
FN ("glTexCoord2sv", glTexCoord2sv, _gloffset_TexCoord2sv)
FN ("glTexCoord3d", glTexCoord3d, _gloffset_TexCoord3d)
FN ("glTexCoord3dv", glTexCoord3dv, _gloffset_TexCoord3dv)
FN ("glTexCoord3f", glTexCoord3f, _gloffset_TexCoord3f)
FN ("glTexCoord3fv", glTexCoord3fv, _gloffset_TexCoord3fv)
FN ("glTexCoord3i", glTexCoord3i, _gloffset_TexCoord3i)
FN ("glTexCoord3iv", glTexCoord3iv, _gloffset_TexCoord3iv)
FN ("glTexCoord3s", glTexCoord3s, _gloffset_TexCoord3s)
FN ("glTexCoord3sv", glTexCoord3sv, _gloffset_TexCoord3sv)
FN ("glTexCoord4d", glTexCoord4d, _gloffset_TexCoord4d)
FN ("glTexCoord4dv", glTexCoord4dv, _gloffset_TexCoord4dv)
FN ("glTexCoord4f", glTexCoord4f, _gloffset_TexCoord4f)
FN ("glTexCoord4fv", glTexCoord4fv, _gloffset_TexCoord4fv)
FN ("glTexCoord4i", glTexCoord4i, _gloffset_TexCoord4i)
FN ("glTexCoord4iv", glTexCoord4iv, _gloffset_TexCoord4iv)
FN ("glTexCoord4s", glTexCoord4s, _gloffset_TexCoord4s)
FN ("glTexCoord4sv", glTexCoord4sv, _gloffset_TexCoord4sv)
FN ("glVertex2d", glVertex2d, _gloffset_Vertex2d)
FN ("glVertex2dv", glVertex2dv, _gloffset_Vertex2dv)
FN ("glVertex2f", glVertex2f, _gloffset_Vertex2f)
FN ("glVertex2fv", glVertex2fv, _gloffset_Vertex2fv)
FN ("glVertex2i", glVertex2i, _gloffset_Vertex2i)
FN ("glVertex2iv", glVertex2iv, _gloffset_Vertex2iv)
FN ("glVertex2s", glVertex2s, _gloffset_Vertex2s)
FN ("glVertex2sv", glVertex2sv, _gloffset_Vertex2sv)
FN ("glVertex3d", glVertex3d, _gloffset_Vertex3d)
FN ("glVertex3dv", glVertex3dv, _gloffset_Vertex3dv)
FN ("glVertex3f", glVertex3f, _gloffset_Vertex3f)
FN ("glVertex3fv", glVertex3fv, _gloffset_Vertex3fv)
FN ("glVertex3i", glVertex3i, _gloffset_Vertex3i)
FN ("glVertex3iv", glVertex3iv, _gloffset_Vertex3iv)
FN ("glVertex3s", glVertex3s, _gloffset_Vertex3s)
FN ("glVertex3sv", glVertex3sv, _gloffset_Vertex3sv)
FN ("glVertex4d", glVertex4d, _gloffset_Vertex4d)
FN ("glVertex4dv", glVertex4dv, _gloffset_Vertex4dv)
FN ("glVertex4f", glVertex4f, _gloffset_Vertex4f)
FN ("glVertex4fv", glVertex4fv, _gloffset_Vertex4fv)
FN ("glVertex4i", glVertex4i, _gloffset_Vertex4i)
FN ("glVertex4iv", glVertex4iv, _gloffset_Vertex4iv)
FN ("glVertex4s", glVertex4s, _gloffset_Vertex4s)
FN ("glVertex4sv", glVertex4sv, _gloffset_Vertex4sv)
FN ("glClipPlane", glClipPlane, _gloffset_ClipPlane)
FN ("glColorMaterial", glColorMaterial, _gloffset_ColorMaterial)
FN ("glCullFace", glCullFace, _gloffset_CullFace)
FN ("glFogf", glFogf, _gloffset_Fogf)
FN ("glFogfv", glFogfv, _gloffset_Fogfv)
FN ("glFogi", glFogi, _gloffset_Fogi)
FN ("glFogiv", glFogiv, _gloffset_Fogiv)
FN ("glFrontFace", glFrontFace, _gloffset_FrontFace)
FN ("glHint", glHint, _gloffset_Hint)
FN ("glLightf", glLightf, _gloffset_Lightf)
FN ("glLightfv", glLightfv, _gloffset_Lightfv)
FN ("glLighti", glLighti, _gloffset_Lighti)
FN ("glLightiv", glLightiv, _gloffset_Lightiv)
FN ("glLightModelf", glLightModelf, _gloffset_LightModelf)
FN ("glLightModelfv", glLightModelfv, _gloffset_LightModelfv)
FN ("glLightModeli", glLightModeli, _gloffset_LightModeli)
FN ("glLightModeliv", glLightModeliv, _gloffset_LightModeliv)
FN ("glLineStipple", glLineStipple, _gloffset_LineStipple)
FN ("glLineWidth", glLineWidth, _gloffset_LineWidth)
FN ("glMaterialf", glMaterialf, _gloffset_Materialf)
FN ("glMaterialfv", glMaterialfv, _gloffset_Materialfv)
FN ("glMateriali", glMateriali, _gloffset_Materiali)
FN ("glMaterialiv", glMaterialiv, _gloffset_Materialiv)
FN ("glPointSize", glPointSize, _gloffset_PointSize)
FN ("glPolygonMode", glPolygonMode, _gloffset_PolygonMode)
FN ("glPolygonStipple", glPolygonStipple, _gloffset_PolygonStipple)
FN ("glScissor", glScissor, _gloffset_Scissor)
FN ("glShadeModel", glShadeModel, _gloffset_ShadeModel)
FN ("glTexParameterf", glTexParameterf, _gloffset_TexParameterf)
FN ("glTexParameterfv", glTexParameterfv, _gloffset_TexParameterfv)
FN ("glTexParameteri", glTexParameteri, _gloffset_TexParameteri)
FN ("glTexParameteriv", glTexParameteriv, _gloffset_TexParameteriv)
FN ("glTexImage1D", glTexImage1D, _gloffset_TexImage1D)
FN ("glTexImage2D", glTexImage2D, _gloffset_TexImage2D)
FN ("glTexEnvf", glTexEnvf, _gloffset_TexEnvf)
FN ("glTexEnvfv", glTexEnvfv, _gloffset_TexEnvfv)
FN ("glTexEnvi", glTexEnvi, _gloffset_TexEnvi)
FN ("glTexEnviv", glTexEnviv, _gloffset_TexEnviv)
FN ("glTexGend", glTexGend, _gloffset_TexGend)
FN ("glTexGendv", glTexGendv, _gloffset_TexGendv)
FN ("glTexGenf", glTexGenf, _gloffset_TexGenf)
FN ("glTexGenfv", glTexGenfv, _gloffset_TexGenfv)
FN ("glTexGeni", glTexGeni, _gloffset_TexGeni)
FN ("glTexGeniv", glTexGeniv, _gloffset_TexGeniv)
FN ("glFeedbackBuffer", glFeedbackBuffer, _gloffset_FeedbackBuffer)
FN ("glSelectBuffer", glSelectBuffer, _gloffset_SelectBuffer)
FN ("glRenderMode", glRenderMode, _gloffset_RenderMode)
FN ("glInitNames", glInitNames, _gloffset_InitNames)
FN ("glLoadName", glLoadName, _gloffset_LoadName)
FN ("glPassThrough", glPassThrough, _gloffset_PassThrough)
FN ("glPopName", glPopName, _gloffset_PopName)
FN ("glPushName", glPushName, _gloffset_PushName)
FN ("glDrawBuffer", glDrawBuffer, _gloffset_DrawBuffer)
FN ("glClear", glClear, _gloffset_Clear)
FN ("glClearAccum", glClearAccum, _gloffset_ClearAccum)
FN ("glClearIndex", glClearIndex, _gloffset_ClearIndex)
FN ("glClearColor", glClearColor, _gloffset_ClearColor)
FN ("glClearStencil", glClearStencil, _gloffset_ClearStencil)
FN ("glClearDepth", glClearDepth, _gloffset_ClearDepth)
FN ("glStencilMask", glStencilMask, _gloffset_StencilMask)
FN ("glColorMask", glColorMask, _gloffset_ColorMask)
FN ("glDepthMask", glDepthMask, _gloffset_DepthMask)
FN ("glIndexMask", glIndexMask, _gloffset_IndexMask)
FN ("glAccum", glAccum, _gloffset_Accum)
FN ("glDisable", glDisable, _gloffset_Disable)
FN ("glEnable", glEnable, _gloffset_Enable)
FN ("glFinish", glFinish, _gloffset_Finish)
FN ("glFlush", glFlush, _gloffset_Flush)
FN ("glPopAttrib", glPopAttrib, _gloffset_PopAttrib)
FN ("glPushAttrib", glPushAttrib, _gloffset_PushAttrib)
FN ("glMap1d", glMap1d, _gloffset_Map1d)
FN ("glMap1f", glMap1f, _gloffset_Map1f)
FN ("glMap2d", glMap2d, _gloffset_Map2d)
FN ("glMap2f", glMap2f, _gloffset_Map2f)
FN ("glMapGrid1d", glMapGrid1d, _gloffset_MapGrid1d)
FN ("glMapGrid1f", glMapGrid1f, _gloffset_MapGrid1f)
FN ("glMapGrid2d", glMapGrid2d, _gloffset_MapGrid2d)
FN ("glMapGrid2f", glMapGrid2f, _gloffset_MapGrid2f)
FN ("glEvalCoord1d", glEvalCoord1d, _gloffset_EvalCoord1d)
FN ("glEvalCoord1dv", glEvalCoord1dv, _gloffset_EvalCoord1dv)
FN ("glEvalCoord1f", glEvalCoord1f, _gloffset_EvalCoord1f)
FN ("glEvalCoord1fv", glEvalCoord1fv, _gloffset_EvalCoord1fv)
FN ("glEvalCoord2d", glEvalCoord2d, _gloffset_EvalCoord2d)
FN ("glEvalCoord2dv", glEvalCoord2dv, _gloffset_EvalCoord2dv)
FN ("glEvalCoord2f", glEvalCoord2f, _gloffset_EvalCoord2f)
FN ("glEvalCoord2fv", glEvalCoord2fv, _gloffset_EvalCoord2fv)
FN ("glEvalMesh1", glEvalMesh1, _gloffset_EvalMesh1)
FN ("glEvalPoint1", glEvalPoint1, _gloffset_EvalPoint1)
FN ("glEvalMesh2", glEvalMesh2, _gloffset_EvalMesh2)
FN ("glEvalPoint2", glEvalPoint2, _gloffset_EvalPoint2)
FN ("glAlphaFunc", glAlphaFunc, _gloffset_AlphaFunc)
FN ("glBlendFunc", glBlendFunc, _gloffset_BlendFunc)
FN ("glLogicOp", glLogicOp, _gloffset_LogicOp)
FN ("glStencilFunc", glStencilFunc, _gloffset_StencilFunc)
FN ("glStencilOp", glStencilOp, _gloffset_StencilOp)
FN ("glDepthFunc", glDepthFunc, _gloffset_DepthFunc)
FN ("glPixelZoom", glPixelZoom, _gloffset_PixelZoom)
FN ("glPixelTransferf", glPixelTransferf, _gloffset_PixelTransferf)
FN ("glPixelTransferi", glPixelTransferi, _gloffset_PixelTransferi)
FN ("glPixelStoref", glPixelStoref, _gloffset_PixelStoref)
FN ("glPixelStorei", glPixelStorei, _gloffset_PixelStorei)
FN ("glPixelMapfv", glPixelMapfv, _gloffset_PixelMapfv)
FN ("glPixelMapuiv", glPixelMapuiv, _gloffset_PixelMapuiv)
FN ("glPixelMapusv", glPixelMapusv, _gloffset_PixelMapusv)
FN ("glReadBuffer", glReadBuffer, _gloffset_ReadBuffer)
FN ("glCopyPixels", glCopyPixels, _gloffset_CopyPixels)
FN ("glReadPixels", glReadPixels, _gloffset_ReadPixels)
FN ("glDrawPixels", glDrawPixels, _gloffset_DrawPixels)
FN ("glGetBooleanv", glGetBooleanv, _gloffset_GetBooleanv)
FN ("glGetClipPlane", glGetClipPlane, _gloffset_GetClipPlane)
FN ("glGetDoublev", glGetDoublev, _gloffset_GetDoublev)
FN ("glGetError", glGetError, _gloffset_GetError)
FN ("glGetFloatv", glGetFloatv, _gloffset_GetFloatv)
FN ("glGetIntegerv", glGetIntegerv, _gloffset_GetIntegerv)
FN ("glGetLightfv", glGetLightfv, _gloffset_GetLightfv)
FN ("glGetLightiv", glGetLightiv, _gloffset_GetLightiv)
FN ("glGetMapdv", glGetMapdv, _gloffset_GetMapdv)
FN ("glGetMapfv", glGetMapfv, _gloffset_GetMapfv)
FN ("glGetMapiv", glGetMapiv, _gloffset_GetMapiv)
FN ("glGetMaterialfv", glGetMaterialfv, _gloffset_GetMaterialfv)
FN ("glGetMaterialiv", glGetMaterialiv, _gloffset_GetMaterialiv)
FN ("glGetPixelMapfv", glGetPixelMapfv, _gloffset_GetPixelMapfv)
FN ("glGetPixelMapuiv", glGetPixelMapuiv, _gloffset_GetPixelMapuiv)
FN ("glGetPixelMapusv", glGetPixelMapusv, _gloffset_GetPixelMapusv)
FN ("glGetPolygonStipple", glGetPolygonStipple, _gloffset_GetPolygonStipple)
FN ("glGetString", glGetString, _gloffset_GetString)
FN ("glGetTexEnvfv", glGetTexEnvfv, _gloffset_GetTexEnvfv)
FN ("glGetTexEnviv", glGetTexEnviv, _gloffset_GetTexEnviv)
FN ("glGetTexGendv", glGetTexGendv, _gloffset_GetTexGendv)
FN ("glGetTexGenfv", glGetTexGenfv, _gloffset_GetTexGenfv)
FN ("glGetTexGeniv", glGetTexGeniv, _gloffset_GetTexGeniv)
FN ("glGetTexImage", glGetTexImage, _gloffset_GetTexImage)
FN ("glGetTexParameterfv", glGetTexParameterfv, _gloffset_GetTexParameterfv)
FN ("glGetTexParameteriv", glGetTexParameteriv, _gloffset_GetTexParameteriv)
FN ("glGetTexLevelParameterfv", glGetTexLevelParameterfv, _gloffset_GetTexLevelParameterfv)
FN ("glGetTexLevelParameteriv", glGetTexLevelParameteriv, _gloffset_GetTexLevelParameteriv)
FN ("glIsEnabled", glIsEnabled, _gloffset_IsEnabled)
FN ("glIsList", glIsList, _gloffset_IsList)
FN ("glDepthRange", glDepthRange, _gloffset_DepthRange)
FN ("glFrustum", glFrustum, _gloffset_Frustum)
FN ("glLoadIdentity", glLoadIdentity, _gloffset_LoadIdentity)
FN ("glLoadMatrixf", glLoadMatrixf, _gloffset_LoadMatrixf)
FN ("glLoadMatrixd", glLoadMatrixd, _gloffset_LoadMatrixd)
FN ("glMatrixMode", glMatrixMode, _gloffset_MatrixMode)
FN ("glMultMatrixf", glMultMatrixf, _gloffset_MultMatrixf)
FN ("glMultMatrixd", glMultMatrixd, _gloffset_MultMatrixd)
FN ("glOrtho", glOrtho, _gloffset_Ortho)
FN ("glPopMatrix", glPopMatrix, _gloffset_PopMatrix)
FN ("glPushMatrix", glPushMatrix, _gloffset_PushMatrix)
FN ("glRotated", glRotated, _gloffset_Rotated)
FN ("glRotatef", glRotatef, _gloffset_Rotatef)
FN ("glScaled", glScaled, _gloffset_Scaled)
FN ("glScalef", glScalef, _gloffset_Scalef)
FN ("glTranslated", glTranslated, _gloffset_Translated)
FN ("glTranslatef", glTranslatef, _gloffset_Translatef)
FN ("glViewport", glViewport, _gloffset_Viewport)
        /* 1.1 */
FN ("glArrayElement", glArrayElement, _gloffset_ArrayElement)
FN ("glColorPointer", glColorPointer, _gloffset_ColorPointer)
FN ("glDisableClientState", glDisableClientState, _gloffset_DisableClientState)
FN ("glDrawArrays", glDrawArrays, _gloffset_DrawArrays)
FN ("glDrawElements", glDrawElements, _gloffset_DrawElements)
FN ("glEdgeFlagPointer", glEdgeFlagPointer, _gloffset_EdgeFlagPointer)
FN ("glEnableClientState", glEnableClientState, _gloffset_EnableClientState)
FN ("glGetPointerv", glGetPointerv, _gloffset_GetPointerv)
FN ("glIndexPointer", glIndexPointer, _gloffset_IndexPointer)
FN ("glInterleavedArrays", glInterleavedArrays, _gloffset_InterleavedArrays)
FN ("glNormalPointer", glNormalPointer, _gloffset_NormalPointer)
FN ("glTexCoordPointer", glTexCoordPointer, _gloffset_TexCoordPointer)
FN ("glVertexPointer", glVertexPointer, _gloffset_VertexPointer)
FN ("glPolygonOffset", glPolygonOffset, _gloffset_PolygonOffset)
FN ("glCopyTexImage1D", glCopyTexImage1D, _gloffset_CopyTexImage1D)
FN ("glCopyTexImage2D", glCopyTexImage2D, _gloffset_CopyTexImage2D)
FN ("glCopyTexSubImage1D", glCopyTexSubImage1D, _gloffset_CopyTexSubImage1D)
FN ("glCopyTexSubImage2D", glCopyTexSubImage2D, _gloffset_CopyTexSubImage2D)
FN ("glTexSubImage1D", glTexSubImage1D, _gloffset_TexSubImage1D)
FN ("glTexSubImage2D", glTexSubImage2D, _gloffset_TexSubImage2D)
FN ("glAreTexturesResident", glAreTexturesResident, _gloffset_AreTexturesResident)
FN ("glBindTexture", glBindTexture, _gloffset_BindTexture)
FN ("glDeleteTextures", glDeleteTextures, _gloffset_DeleteTextures)
FN ("glGenTextures", glGenTextures, _gloffset_GenTextures)
FN ("glIsTexture", glIsTexture, _gloffset_IsTexture)
FN ("glPrioritizeTextures", glPrioritizeTextures, _gloffset_PrioritizeTextures)
FN ("glIndexub", glIndexub, _gloffset_Indexub)
FN ("glIndexubv", glIndexubv, _gloffset_Indexubv)
FN ("glPopClientAttrib", glPopClientAttrib, _gloffset_PopClientAttrib)
FN ("glPushClientAttrib", glPushClientAttrib, _gloffset_PushClientAttrib)
	/* 1.2 */
#ifdef GL_VERSION_1_2
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glBlendColor", NAME(glBlendColor), _gloffset_BlendColor)
FN ("glBlendEquation", NAME(glBlendEquation), _gloffset_BlendEquation)
FN ("glDrawRangeElements", NAME(glDrawRangeElements), _gloffset_DrawRangeElements)
FN ("glColorTable", NAME(glColorTable), _gloffset_ColorTable)
FN ("glColorTableParameterfv", NAME(glColorTableParameterfv), _gloffset_ColorTableParameterfv)
FN ("glColorTableParameteriv", NAME(glColorTableParameteriv), _gloffset_ColorTableParameteriv)
FN ("glCopyColorTable", NAME(glCopyColorTable), _gloffset_CopyColorTable)
FN ("glGetColorTable", NAME(glGetColorTable), _gloffset_GetColorTable)
FN ("glGetColorTableParameterfv", NAME(glGetColorTableParameterfv), _gloffset_GetColorTableParameterfv)
FN ("glGetColorTableParameteriv", NAME(glGetColorTableParameteriv), _gloffset_GetColorTableParameteriv)
FN ("glColorSubTable", NAME(glColorSubTable), _gloffset_ColorSubTable)
FN ("glCopyColorSubTable", NAME(glCopyColorSubTable), _gloffset_CopyColorSubTable)
FN ("glConvolutionFilter1D", NAME(glConvolutionFilter1D), _gloffset_ConvolutionFilter1D)
FN ("glConvolutionFilter2D", NAME(glConvolutionFilter2D), _gloffset_ConvolutionFilter2D)
FN ("glConvolutionParameterf", NAME(glConvolutionParameterf), _gloffset_ConvolutionParameterf)
FN ("glConvolutionParameterfv", NAME(glConvolutionParameterfv), _gloffset_ConvolutionParameterfv)
FN ("glConvolutionParameteri", NAME(glConvolutionParameteri), _gloffset_ConvolutionParameteri)
FN ("glConvolutionParameteriv", NAME(glConvolutionParameteriv), _gloffset_ConvolutionParameteriv)
FN ("glCopyConvolutionFilter1D", NAME(glCopyConvolutionFilter1D), _gloffset_CopyConvolutionFilter1D)
FN ("glCopyConvolutionFilter2D", NAME(glCopyConvolutionFilter2D), _gloffset_CopyConvolutionFilter2D)
FN ("glGetConvolutionFilter", NAME(glGetConvolutionFilter), _gloffset_GetConvolutionFilter)
FN ("glGetConvolutionParameterfv", NAME(glGetConvolutionParameterfv), _gloffset_GetConvolutionParameterfv)
FN ("glGetConvolutionParameteriv", NAME(glGetConvolutionParameteriv), _gloffset_GetConvolutionParameteriv)
FN ("glGetSeparableFilter", NAME(glGetSeparableFilter), _gloffset_GetSeparableFilter)
FN ("glSeparableFilter2D", NAME(glSeparableFilter2D), _gloffset_SeparableFilter2D)
FN ("glGetHistogram", NAME(glGetHistogram), _gloffset_GetHistogram)
FN ("glGetHistogramParameterfv", NAME(glGetHistogramParameterfv), _gloffset_GetHistogramParameterfv)
FN ("glGetHistogramParameteriv", NAME(glGetHistogramParameteriv), _gloffset_GetHistogramParameteriv)
FN ("glGetMinmax", NAME(glGetMinmax), _gloffset_GetMinmax)
FN ("glGetMinmaxParameterfv", NAME(glGetMinmaxParameterfv), _gloffset_GetMinmaxParameterfv)
FN ("glGetMinmaxParameteriv", NAME(glGetMinmaxParameteriv), _gloffset_GetMinmaxParameteriv)
FN ("glHistogram", NAME(glHistogram), _gloffset_Histogram)
FN ("glMinmax", NAME(glMinmax), _gloffset_Minmax)
FN ("glResetHistogram", NAME(glResetHistogram), _gloffset_ResetHistogram)
FN ("glResetMinmax", NAME(glResetMinmax), _gloffset_ResetMinmax)
FN ("glTexImage3D", NAME(glTexImage3D), _gloffset_TexImage3D)
FN ("glTexSubImage3D", NAME(glTexSubImage3D), _gloffset_TexSubImage3D)
FN ("glCopyTexSubImage3D", NAME(glCopyTexSubImage3D), _gloffset_CopyTexSubImage3D)
#undef NAME

	/* 1.3 */
#ifdef GL_VERSION_1_3
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glActiveTexture", NAME(glActiveTexture), _gloffset_ActiveTextureARB)
FN ("glClientActiveTexture", NAME(glClientActiveTexture), _gloffset_ClientActiveTextureARB)
FN ("glCompressedTexImage1D", NAME(glCompressedTexImage1D), _gloffset_CompressedTexImage1DARB)
FN ("glCompressedTexImage2D", NAME(glCompressedTexImage2D), _gloffset_CompressedTexImage2DARB)
FN ("glCompressedTexImage3D", NAME(glCompressedTexImage3D), _gloffset_CompressedTexImage3DARB)
FN ("glCompressedTexSubImage1D", NAME(glCompressedTexSubImage1D), _gloffset_CompressedTexSubImage1DARB)
FN ("glCompressedTexSubImage2D", NAME(glCompressedTexSubImage2D), _gloffset_CompressedTexSubImage2DARB)
FN ("glCompressedTexSubImage3D", NAME(glCompressedTexSubImage3D), _gloffset_CompressedTexSubImage3DARB)
FN ("glGetCompressedTexImage", NAME(glGetCompressedTexImage), _gloffset_GetCompressedTexImageARB)
FN ("glMultiTexCoord1d", NAME(glMultiTexCoord1d), _gloffset_MultiTexCoord1dARB)
FN ("glMultiTexCoord1dv", NAME(glMultiTexCoord1dv), _gloffset_MultiTexCoord1dvARB)
FN ("glMultiTexCoord1f", NAME(glMultiTexCoord1f), _gloffset_MultiTexCoord1fARB)
FN ("glMultiTexCoord1fv", NAME(glMultiTexCoord1fv), _gloffset_MultiTexCoord1fvARB)
FN ("glMultiTexCoord1i", NAME(glMultiTexCoord1i), _gloffset_MultiTexCoord1iARB)
FN ("glMultiTexCoord1iv", NAME(glMultiTexCoord1iv), _gloffset_MultiTexCoord1ivARB)
FN ("glMultiTexCoord1s", NAME(glMultiTexCoord1s), _gloffset_MultiTexCoord1sARB)
FN ("glMultiTexCoord1sv", NAME(glMultiTexCoord1sv), _gloffset_MultiTexCoord1svARB)
FN ("glMultiTexCoord2d", NAME(glMultiTexCoord2d), _gloffset_MultiTexCoord2dARB)
FN ("glMultiTexCoord2dv", NAME(glMultiTexCoord2dv), _gloffset_MultiTexCoord2dvARB)
FN ("glMultiTexCoord2f", NAME(glMultiTexCoord2f), _gloffset_MultiTexCoord2fARB)
FN ("glMultiTexCoord2fv", NAME(glMultiTexCoord2fv), _gloffset_MultiTexCoord2fvARB)
FN ("glMultiTexCoord2i", NAME(glMultiTexCoord2i), _gloffset_MultiTexCoord2iARB)
FN ("glMultiTexCoord2iv", NAME(glMultiTexCoord2iv), _gloffset_MultiTexCoord2ivARB)
FN ("glMultiTexCoord2s", NAME(glMultiTexCoord2s), _gloffset_MultiTexCoord2sARB)
FN ("glMultiTexCoord2sv", NAME(glMultiTexCoord2sv), _gloffset_MultiTexCoord2svARB)
FN ("glMultiTexCoord3d", NAME(glMultiTexCoord3d), _gloffset_MultiTexCoord3dARB)
FN ("glMultiTexCoord3dv", NAME(glMultiTexCoord3dv), _gloffset_MultiTexCoord3dvARB)
FN ("glMultiTexCoord3f", NAME(glMultiTexCoord3f), _gloffset_MultiTexCoord3fARB)
FN ("glMultiTexCoord3fv", NAME(glMultiTexCoord3fv), _gloffset_MultiTexCoord3fvARB)
FN ("glMultiTexCoord3i", NAME(glMultiTexCoord3i), _gloffset_MultiTexCoord3iARB)
FN ("glMultiTexCoord3iv", NAME(glMultiTexCoord3iv), _gloffset_MultiTexCoord3ivARB)
FN ("glMultiTexCoord3s", NAME(glMultiTexCoord3s), _gloffset_MultiTexCoord3sARB)
FN ("glMultiTexCoord3sv", NAME(glMultiTexCoord3sv), _gloffset_MultiTexCoord3svARB)
FN ("glMultiTexCoord4d", NAME(glMultiTexCoord4d), _gloffset_MultiTexCoord4dARB)
FN ("glMultiTexCoord4dv", NAME(glMultiTexCoord4dv), _gloffset_MultiTexCoord4dvARB)
FN ("glMultiTexCoord4f", NAME(glMultiTexCoord4f), _gloffset_MultiTexCoord4fARB)
FN ("glMultiTexCoord4fv", NAME(glMultiTexCoord4fv), _gloffset_MultiTexCoord4fvARB)
FN ("glMultiTexCoord4i", NAME(glMultiTexCoord4i), _gloffset_MultiTexCoord4iARB)
FN ("glMultiTexCoord4iv", NAME(glMultiTexCoord4iv), _gloffset_MultiTexCoord4ivARB)
FN ("glMultiTexCoord4s", NAME(glMultiTexCoord4s), _gloffset_MultiTexCoord4sARB)
FN ("glMultiTexCoord4sv", NAME(glMultiTexCoord4sv), _gloffset_MultiTexCoord4svARB)
FN ("glLoadTransposeMatrixd", NAME(glLoadTransposeMatrixd), _gloffset_LoadTransposeMatrixdARB)
FN ("glLoadTransposeMatrixf", NAME(glLoadTransposeMatrixf), _gloffset_LoadTransposeMatrixfARB)
FN ("glMultTransposeMatrixd", NAME(glMultTransposeMatrixd), _gloffset_MultTransposeMatrixdARB)
FN ("glMultTransposeMatrixf", NAME(glMultTransposeMatrixf), _gloffset_MultTransposeMatrixfARB)
FN ("glSampleCoverage", NAME(glSampleCoverage), _gloffset_SampleCoverageARB)
#if 0
FN ("glSamplePass", NAME(glSamplePass), _gloffset_SamplePassARB)
#endif
#undef NAME

	/* ARB 1. GL_ARB_multitexture */
#ifdef GL_ARB_multitexture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glActiveTextureARB", NAME(glActiveTextureARB), _gloffset_ActiveTextureARB)
FN ("glClientActiveTextureARB", NAME(glClientActiveTextureARB), _gloffset_ClientActiveTextureARB)
FN ("glMultiTexCoord1dARB", NAME(glMultiTexCoord1dARB), _gloffset_MultiTexCoord1dARB)
FN ("glMultiTexCoord1dvARB", NAME(glMultiTexCoord1dvARB), _gloffset_MultiTexCoord1dvARB)
FN ("glMultiTexCoord1fARB", NAME(glMultiTexCoord1fARB), _gloffset_MultiTexCoord1fARB)
FN ("glMultiTexCoord1fvARB", NAME(glMultiTexCoord1fvARB), _gloffset_MultiTexCoord1fvARB)
FN ("glMultiTexCoord1iARB", NAME(glMultiTexCoord1iARB), _gloffset_MultiTexCoord1iARB)
FN ("glMultiTexCoord1ivARB", NAME(glMultiTexCoord1ivARB), _gloffset_MultiTexCoord1ivARB)
FN ("glMultiTexCoord1sARB", NAME(glMultiTexCoord1sARB), _gloffset_MultiTexCoord1sARB)
FN ("glMultiTexCoord1svARB", NAME(glMultiTexCoord1svARB), _gloffset_MultiTexCoord1svARB)
FN ("glMultiTexCoord2dARB", NAME(glMultiTexCoord2dARB), _gloffset_MultiTexCoord2dARB)
FN ("glMultiTexCoord2dvARB", NAME(glMultiTexCoord2dvARB), _gloffset_MultiTexCoord2dvARB)
FN ("glMultiTexCoord2fARB", NAME(glMultiTexCoord2fARB), _gloffset_MultiTexCoord2fARB)
FN ("glMultiTexCoord2fvARB", NAME(glMultiTexCoord2fvARB), _gloffset_MultiTexCoord2fvARB)
FN ("glMultiTexCoord2iARB", NAME(glMultiTexCoord2iARB), _gloffset_MultiTexCoord2iARB)
FN ("glMultiTexCoord2ivARB", NAME(glMultiTexCoord2ivARB), _gloffset_MultiTexCoord2ivARB)
FN ("glMultiTexCoord2sARB", NAME(glMultiTexCoord2sARB), _gloffset_MultiTexCoord2sARB)
FN ("glMultiTexCoord2svARB", NAME(glMultiTexCoord2svARB), _gloffset_MultiTexCoord2svARB)
FN ("glMultiTexCoord3dARB", NAME(glMultiTexCoord3dARB), _gloffset_MultiTexCoord3dARB)
FN ("glMultiTexCoord3dvARB", NAME(glMultiTexCoord3dvARB), _gloffset_MultiTexCoord3dvARB)
FN ("glMultiTexCoord3fARB", NAME(glMultiTexCoord3fARB), _gloffset_MultiTexCoord3fARB)
FN ("glMultiTexCoord3fvARB", NAME(glMultiTexCoord3fvARB), _gloffset_MultiTexCoord3fvARB)
FN ("glMultiTexCoord3iARB", NAME(glMultiTexCoord3iARB), _gloffset_MultiTexCoord3iARB)
FN ("glMultiTexCoord3ivARB", NAME(glMultiTexCoord3ivARB), _gloffset_MultiTexCoord3ivARB)
FN ("glMultiTexCoord3sARB", NAME(glMultiTexCoord3sARB), _gloffset_MultiTexCoord3sARB)
FN ("glMultiTexCoord3svARB", NAME(glMultiTexCoord3svARB), _gloffset_MultiTexCoord3svARB)
FN ("glMultiTexCoord4dARB", NAME(glMultiTexCoord4dARB), _gloffset_MultiTexCoord4dARB)
FN ("glMultiTexCoord4dvARB", NAME(glMultiTexCoord4dvARB), _gloffset_MultiTexCoord4dvARB)
FN ("glMultiTexCoord4fARB", NAME(glMultiTexCoord4fARB), _gloffset_MultiTexCoord4fARB)
FN ("glMultiTexCoord4fvARB", NAME(glMultiTexCoord4fvARB), _gloffset_MultiTexCoord4fvARB)
FN ("glMultiTexCoord4iARB", NAME(glMultiTexCoord4iARB), _gloffset_MultiTexCoord4iARB)
FN ("glMultiTexCoord4ivARB", NAME(glMultiTexCoord4ivARB), _gloffset_MultiTexCoord4ivARB)
FN ("glMultiTexCoord4sARB", NAME(glMultiTexCoord4sARB), _gloffset_MultiTexCoord4sARB)
FN ("glMultiTexCoord4svARB", NAME(glMultiTexCoord4svARB), _gloffset_MultiTexCoord4svARB)
#undef NAME

	/* ARB 3. GL_ARB_transpose_matrix */
#ifdef GL_ARB_transpose_matrix
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glLoadTransposeMatrixdARB", NAME(glLoadTransposeMatrixdARB), _gloffset_LoadTransposeMatrixdARB)
FN ("glLoadTransposeMatrixfARB", NAME(glLoadTransposeMatrixfARB), _gloffset_LoadTransposeMatrixfARB)
FN ("glMultTransposeMatrixdARB", NAME(glMultTransposeMatrixdARB), _gloffset_MultTransposeMatrixdARB)
FN ("glMultTransposeMatrixfARB", NAME(glMultTransposeMatrixfARB), _gloffset_MultTransposeMatrixfARB)
#undef NAME

	/* ARB 5. GL_ARB_multisample */
#ifdef GL_ARB_multisample
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glSampleCoverageARB", NAME(glSampleCoverageARB), _gloffset_SampleCoverageARB)
#undef NAME

	/* ARB 12. GL_ARB_texture_compression */
#ifdef GL_ARB_texture_compression
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glCompressedTexImage3DARB", NAME(glCompressedTexImage3DARB), _gloffset_CompressedTexImage3DARB)
FN ("glCompressedTexImage2DARB", NAME(glCompressedTexImage2DARB), _gloffset_CompressedTexImage2DARB)
FN ("glCompressedTexImage1DARB", NAME(glCompressedTexImage1DARB), _gloffset_CompressedTexImage1DARB)
FN ("glCompressedTexSubImage3DARB", NAME(glCompressedTexSubImage3DARB), _gloffset_CompressedTexSubImage3DARB)
FN ("glCompressedTexSubImage2DARB", NAME(glCompressedTexSubImage2DARB), _gloffset_CompressedTexSubImage2DARB)
FN ("glCompressedTexSubImage1DARB", NAME(glCompressedTexSubImage1DARB), _gloffset_CompressedTexSubImage1DARB)
FN ("glGetCompressedTexImageARB", NAME(glGetCompressedTexImageARB), _gloffset_GetCompressedTexImageARB)
#undef NAME

	/* 2. GL_EXT_blend_color */
#ifdef GL_EXT_blend_color
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glBlendColorEXT", NAME(glBlendColorEXT), _gloffset_BlendColor)
#undef NAME

	/* 3. GL_EXT_polygon_offset */
#ifdef GL_EXT_polygon_offset
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glPolygonOffsetEXT", NAME(glPolygonOffsetEXT), _gloffset_PolygonOffsetEXT)
#undef NAME

	/* 6. GL_EXT_texture3D */
#ifdef GL_EXT_texture3D
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glCopyTexSubImage3DEXT", NAME(glCopyTexSubImage3DEXT), _gloffset_CopyTexSubImage3D)
FN ("glTexImage3DEXT", NAME(glTexImage3DEXT), _gloffset_TexImage3D)
FN ("glTexSubImage3DEXT", NAME(glTexSubImage3DEXT), _gloffset_TexSubImage3D)
#undef NAME

	/* 7. GL_SGI_texture_filter4 */
#ifdef GL_SGI_texture_filter4
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glGetTexFilterFuncSGIS", NAME(glGetTexFilterFuncSGIS), _gloffset_GetTexFilterFuncSGIS)
FN ("glTexFilterFuncSGIS", NAME(glTexFilterFuncSGIS), _gloffset_TexFilterFuncSGIS)
#undef NAME

	/* 9. GL_EXT_subtexture */
#ifdef GL_EXT_subtexture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glTexSubImage1DEXT", NAME(glTexSubImage1DEXT), _gloffset_TexSubImage1D)
FN ("glTexSubImage2DEXT", NAME(glTexSubImage2DEXT), _gloffset_TexSubImage2D)
#undef NAME

	/* 10. GL_EXT_copy_texture */
#ifdef GL_EXT_copy_texture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glCopyTexImage1DEXT", NAME(glCopyTexImage1DEXT), _gloffset_CopyTexImage1D)
FN ("glCopyTexImage2DEXT", NAME(glCopyTexImage2DEXT), _gloffset_CopyTexImage2D)
FN ("glCopyTexSubImage1DEXT", NAME(glCopyTexSubImage1DEXT), _gloffset_CopyTexSubImage1D)
FN ("glCopyTexSubImage2DEXT", NAME(glCopyTexSubImage2DEXT), _gloffset_CopyTexSubImage2D)
#undef NAME

	/* 11. GL_EXT_histogram */
#ifdef GL_EXT_histogram
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glGetHistogramEXT", NAME(glGetHistogramEXT), _gloffset_GetHistogramEXT)
FN ("glGetHistogramParameterfvEXT", NAME(glGetHistogramParameterfvEXT), _gloffset_GetHistogramParameterfvEXT)
FN ("glGetHistogramParameterivEXT", NAME(glGetHistogramParameterivEXT), _gloffset_GetHistogramParameterivEXT)
FN ("glGetMinmaxEXT", NAME(glGetMinmaxEXT), _gloffset_GetMinmaxEXT)
FN ("glGetMinmaxParameterfvEXT", NAME(glGetMinmaxParameterfvEXT), _gloffset_GetMinmaxParameterfvEXT)
FN ("glGetMinmaxParameterivEXT", NAME(glGetMinmaxParameterivEXT), _gloffset_GetMinmaxParameterivEXT)
FN ("glHistogramEXT", NAME(glHistogramEXT), _gloffset_Histogram)
FN ("glMinmaxEXT", NAME(glMinmaxEXT), _gloffset_Minmax)
FN ("glResetHistogramEXT", NAME(glResetHistogramEXT), _gloffset_ResetHistogram)
FN ("glResetMinmaxEXT", NAME(glResetMinmaxEXT), _gloffset_ResetMinmax)
#undef NAME

	/* 12. GL_EXT_convolution */
#ifdef GL_EXT_convolution
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glConvolutionFilter1DEXT", NAME(glConvolutionFilter1DEXT), _gloffset_ConvolutionFilter1D)
FN ("glConvolutionFilter2DEXT", NAME(glConvolutionFilter2DEXT), _gloffset_ConvolutionFilter2D)
FN ("glConvolutionParameterfEXT", NAME(glConvolutionParameterfEXT), _gloffset_ConvolutionParameterf)
FN ("glConvolutionParameterfvEXT", NAME(glConvolutionParameterfvEXT), _gloffset_ConvolutionParameterfv)
FN ("glConvolutionParameteriEXT", NAME(glConvolutionParameteriEXT), _gloffset_ConvolutionParameteri)
FN ("glConvolutionParameterivEXT", NAME(glConvolutionParameterivEXT), _gloffset_ConvolutionParameteriv)
FN ("glCopyConvolutionFilter1DEXT", NAME(glCopyConvolutionFilter1DEXT), _gloffset_CopyConvolutionFilter1D)
FN ("glCopyConvolutionFilter2DEXT", NAME(glCopyConvolutionFilter2DEXT), _gloffset_CopyConvolutionFilter2D)
FN ("glGetConvolutionFilterEXT", NAME(glGetConvolutionFilterEXT), _gloffset_GetConvolutionFilterEXT)
FN ("glGetConvolutionParameterivEXT", NAME(glGetConvolutionParameterivEXT), _gloffset_GetConvolutionParameterivEXT)
FN ("glGetConvolutionParameterfvEXT", NAME(glGetConvolutionParameterfvEXT), _gloffset_GetConvolutionParameterfvEXT)
FN ("glGetSeparableFilterEXT", NAME(glGetSeparableFilterEXT), _gloffset_GetSeparableFilterEXT)
FN ("glSeparableFilter2DEXT", NAME(glSeparableFilter2DEXT), _gloffset_SeparableFilter2D)
#undef NAME

	/* 14. GL_SGI_color_table */
#ifdef GL_SGI_color_table
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glColorTableSGI", NAME(glColorTableSGI), _gloffset_ColorTable)
FN ("glColorTableParameterfvSGI", NAME(glColorTableParameterfvSGI), _gloffset_ColorTableParameterfv)
FN ("glColorTableParameterivSGI", NAME(glColorTableParameterivSGI), _gloffset_ColorTableParameteriv)
FN ("glCopyColorTableSGI", NAME(glCopyColorTableSGI), _gloffset_CopyColorTable)
FN ("glGetColorTableSGI", NAME(glGetColorTableSGI), _gloffset_GetColorTableSGI)
FN ("glGetColorTableParameterfvSGI", NAME(glGetColorTableParameterfvSGI), _gloffset_GetColorTableParameterfvSGI)
FN ("glGetColorTableParameterivSGI", NAME(glGetColorTableParameterivSGI), _gloffset_GetColorTableParameterivSGI)
#undef NAME

	/* 15. GL_SGIS_pixel_texture */
#ifdef GL_SGIS_pixel_texture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glPixelTexGenParameterfSGIS", NAME(glPixelTexGenParameterfSGIS), _gloffset_PixelTexGenParameterfSGIS)
FN ("glPixelTexGenParameterfvSGIS", NAME(glPixelTexGenParameterfvSGIS), _gloffset_PixelTexGenParameterfvSGIS)
FN ("glPixelTexGenParameteriSGIS", NAME(glPixelTexGenParameteriSGIS), _gloffset_PixelTexGenParameteriSGIS)
FN ("glPixelTexGenParameterivSGIS", NAME(glPixelTexGenParameterivSGIS), _gloffset_PixelTexGenParameterivSGIS)
FN ("glGetPixelTexGenParameterfvSGIS", NAME(glGetPixelTexGenParameterfvSGIS), _gloffset_GetPixelTexGenParameterfvSGIS)
FN ("glGetPixelTexGenParameterivSGIS", NAME(glGetPixelTexGenParameterivSGIS), _gloffset_GetPixelTexGenParameterivSGIS)
#undef NAME

	/* 15a. GL_SGIX_pixel_texture */
#ifdef GL_SGIX_pixel_texture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glPixelTexGenSGIX", NAME(glPixelTexGenSGIX), _gloffset_PixelTexGenSGIX)
#undef NAME

	/* 16. GL_SGIS_texture4D */
#ifdef  GL_SGIS_texture4D
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glTexImage4DSGIS", NAME(glTexImage4DSGIS), _gloffset_TexImage4DSGIS)
FN ("glTexSubImage4DSGIS", NAME(glTexSubImage4DSGIS), _gloffset_TexSubImage4DSGIS)
#undef NAME

	/* 20. GL_EXT_texture_object */
#ifdef GL_EXT_texture_object
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glAreTexturesResidentEXT", NAME(glAreTexturesResidentEXT), _gloffset_AreTexturesResidentEXT)
FN ("glBindTextureEXT", NAME(glBindTextureEXT), _gloffset_BindTexture)
FN ("glDeleteTexturesEXT", NAME(glDeleteTexturesEXT), _gloffset_DeleteTextures)
FN ("glGenTexturesEXT", NAME(glGenTexturesEXT), _gloffset_GenTexturesEXT)
FN ("glIsTextureEXT", NAME(glIsTextureEXT), _gloffset_IsTextureEXT)
FN ("glPrioritizeTexturesEXT", NAME(glPrioritizeTexturesEXT), _gloffset_PrioritizeTextures)
#undef NAME

	/* 21. GL_SGIS_detail_texture */
#ifdef GL_SGIS_detail_texture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glDetailTexFuncSGIS", NAME(glDetailTexFuncSGIS), _gloffset_DetailTexFuncSGIS)
FN ("glGetDetailTexFuncSGIS", NAME(glGetDetailTexFuncSGIS), _gloffset_GetDetailTexFuncSGIS)
#undef NAME

	/* 22. GL_SGIS_sharpen_texture */
#ifdef GL_SGIS_sharpen_texture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glGetSharpenTexFuncSGIS", NAME(glGetSharpenTexFuncSGIS), _gloffset_GetSharpenTexFuncSGIS)
FN ("glSharpenTexFuncSGIS", NAME(glSharpenTexFuncSGIS), _gloffset_SharpenTexFuncSGIS)
#undef NAME

	/* 25. GL_SGIS_multisample */
#ifdef GL_SGIS_multisample
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glSampleMaskSGIS", NAME(glSampleMaskSGIS), _gloffset_SampleMaskSGIS)
FN ("glSamplePatternSGIS", NAME(glSamplePatternSGIS), _gloffset_SamplePatternSGIS)
#undef NAME

	/* 30. GL_EXT_vertex_array */
#ifdef GL_EXT_vertex_array
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glArrayElementEXT", NAME(glArrayElementEXT), _gloffset_ArrayElement)
FN ("glColorPointerEXT", NAME(glColorPointerEXT), _gloffset_ColorPointerEXT)
FN ("glDrawArraysEXT", NAME(glDrawArraysEXT), _gloffset_DrawArrays)
FN ("glEdgeFlagPointerEXT", NAME(glEdgeFlagPointerEXT), _gloffset_EdgeFlagPointerEXT)
FN ("glGetPointervEXT", NAME(glGetPointervEXT), _gloffset_GetPointerv)
FN ("glIndexPointerEXT", NAME(glIndexPointerEXT), _gloffset_IndexPointerEXT)
FN ("glNormalPointerEXT", NAME(glNormalPointerEXT), _gloffset_NormalPointerEXT)
FN ("glTexCoordPointerEXT", NAME(glTexCoordPointerEXT), _gloffset_TexCoordPointerEXT)
FN ("glVertexPointerEXT", NAME(glVertexPointerEXT), _gloffset_VertexPointerEXT)
#undef NAME

	/* 37. GL_EXT_blend_minmax */
#ifdef GL_EXT_blend_minmax
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glBlendEquationEXT", NAME(glBlendEquationEXT), _gloffset_BlendEquation)
#undef NAME

	/* 52. GL_SGIX_sprite */
#ifdef GL_SGIX_sprite
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glSpriteParameterfSGIX", NAME(glSpriteParameterfSGIX), _gloffset_SpriteParameterfSGIX)
FN ("glSpriteParameterfvSGIX", NAME(glSpriteParameterfvSGIX), _gloffset_SpriteParameterfvSGIX)
FN ("glSpriteParameteriSGIX", NAME(glSpriteParameteriSGIX), _gloffset_SpriteParameteriSGIX)
FN ("glSpriteParameterivSGIX", NAME(glSpriteParameterivSGIX), _gloffset_SpriteParameterivSGIX)
#undef NAME

	/* 54. GL_EXT_point_parameters */
#ifdef GL_EXT_point_parameters
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glPointParameterfEXT", NAME(glPointParameterfEXT), _gloffset_PointParameterfEXT)
FN ("glPointParameterfvEXT", NAME(glPointParameterfvEXT), _gloffset_PointParameterfvEXT)
FN ("glPointParameterfSGIS", NAME(glPointParameterfSGIS), _gloffset_PointParameterfEXT)
FN ("glPointParameterfvSGIS", NAME(glPointParameterfvSGIS), _gloffset_PointParameterfvEXT)
#undef NAME

	/* 55. GL_SGIX_instruments */
#ifdef GL_SGIX_instruments
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glInstrumentsBufferSGIX", NAME(glInstrumentsBufferSGIX), _gloffset_InstrumentsBufferSGIX)
FN ("glStartInstrumentsSGIX", NAME(glStartInstrumentsSGIX), _gloffset_StartInstrumentsSGIX)
FN ("glStopInstrumentsSGIX", NAME(glStopInstrumentsSGIX), _gloffset_StopInstrumentsSGIX)
FN ("glReadInstrumentsSGIX", NAME(glReadInstrumentsSGIX), _gloffset_ReadInstrumentsSGIX)
FN ("glPollInstrumentsSGIX", NAME(glPollInstrumentsSGIX), _gloffset_PollInstrumentsSGIX)
FN ("glGetInstrumentsSGIX", NAME(glGetInstrumentsSGIX), _gloffset_GetInstrumentsSGIX)
#undef NAME

	/* 57. GL_SGIX_framezoom */
#ifdef GL_SGIX_framezoom
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glFrameZoomSGIX", NAME(glFrameZoomSGIX), _gloffset_FrameZoomSGIX)
#undef NAME

	/* 58. GL_SGIX_tag_sample_buffer */
#ifdef GL_SGIX_tag_sample_buffer
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glTagSampleBufferSGIX", NAME(glTagSampleBufferSGIX), _gloffset_TagSampleBufferSGIX)
#undef NAME

	/* 60. GL_SGIX_reference_plane */
#ifdef GL_SGIX_reference_plane
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glReferencePlaneSGIX", NAME(glReferencePlaneSGIX), _gloffset_ReferencePlaneSGIX)
#undef NAME

	/* 61. GL_SGIX_flush_raster */
#ifdef GL_SGIX_flush_raster
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glFlushRasterSGIX", NAME(glFlushRasterSGIX), _gloffset_FlushRasterSGIX)
#undef NAME

	/* 66. GL_HP_image_transform */
#if 0
#ifdef GL_HP_image_transform
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glGetImageTransformParameterfvHP", NAME(glGetImageTransformParameterfvHP), _gloffset_GetImageTransformParameterfvHP)
FN ("glGetImageTransformParameterivHP", NAME(glGetImageTransformParameterivHP), _gloffset_GetImageTransformParameterivHP)
FN ("glImageTransformParameterfHP", NAME(glImageTransformParameterfHP), _gloffset_ImageTransformParameterfHP)
FN ("glImageTransformParameterfvHP", NAME(glImageTransformParameterfvHP), _gloffset_ImageTransformParameterfvHP)
FN ("glImageTransformParameteriHP", NAME(glImageTransformParameteriHP), _gloffset_ImageTransformParameteriHP)
FN ("glImageTransformParameterivHP", NAME(glImageTransformParameterivHP), _gloffset_ImageTransformParameterivHP)
#undef NAME
#endif

	/* 74. GL_EXT_color_subtable */
#ifdef GL_EXT_color_subtable
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glColorSubTableEXT", NAME(glColorSubTableEXT), _gloffset_ColorSubTable)
FN ("glCopyColorSubTableEXT", NAME(glCopyColorSubTableEXT), _gloffset_CopyColorSubTable)
#undef NAME

	/* 77. GL_PGI_misc_hints */
#ifdef GL_PGI_misc_hints
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glHintPGI", NAME(glHintPGI), _gloffset_HintPGI)
#undef NAME

	/* 78. GL_EXT_paletted_texture */
#ifdef GL_EXT_paletted_texture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glColorTableEXT", NAME(glColorTableEXT), _gloffset_ColorTable)
FN ("glGetColorTableEXT", NAME(glGetColorTableEXT), _gloffset_GetColorTable)
FN ("glGetColorTableParameterfvEXT", NAME(glGetColorTableParameterfvEXT), _gloffset_GetColorTableParameterfv)
FN ("glGetColorTableParameterivEXT", NAME(glGetColorTableParameterivEXT), _gloffset_GetColorTableParameteriv)
#undef NAME

	/* 80. GL_SGIX_list_priority */
#ifdef GL_SGIX_list_priority
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glGetListParameterfvSGIX", NAME(glGetListParameterfvSGIX), _gloffset_GetListParameterfvSGIX)
FN ("glGetListParameterivSGIX", NAME(glGetListParameterivSGIX), _gloffset_GetListParameterivSGIX)
FN ("glListParameterfSGIX", NAME(glListParameterfSGIX), _gloffset_ListParameterfSGIX)
FN ("glListParameterfvSGIX", NAME(glListParameterfvSGIX), _gloffset_ListParameterfvSGIX)
FN ("glListParameteriSGIX", NAME(glListParameteriSGIX), _gloffset_ListParameteriSGIX)
FN ("glListParameterivSGIX", NAME(glListParameterivSGIX), _gloffset_ListParameterivSGIX)
#undef NAME

	/* 94. GL_EXT_index_material */
#ifdef GL_EXT_index_material
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glIndexMaterialEXT", NAME(glIndexMaterialEXT), _gloffset_IndexMaterialEXT)
#undef NAME

	/* 95. GL_EXT_index_func */
#ifdef GL_EXT_index_func
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glIndexFuncEXT", NAME(glIndexFuncEXT), _gloffset_IndexFuncEXT)
#undef NAME

	/* 97. GL_EXT_compiled_vertex_array */
#ifdef GL_EXT_compiled_vertex_array
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glLockArraysEXT", NAME(glLockArraysEXT), _gloffset_LockArraysEXT)
FN ("glUnlockArraysEXT", NAME(glUnlockArraysEXT), _gloffset_UnlockArraysEXT)
#undef NAME

	/* 98. GL_EXT_cull_vertex */
#ifdef GL_EXT_cull_vertex
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glCullParameterfvEXT", NAME(glCullParameterfvEXT), _gloffset_CullParameterfvEXT)
FN ("glCullParameterdvEXT", NAME(glCullParameterdvEXT), _gloffset_CullParameterdvEXT)
#undef NAME

	/* 102. GL_SGIX_fragment_lighting */
#ifdef GL_SGIX_fragment_lighting
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glFragmentColorMaterialSGIX", NAME(glFragmentColorMaterialSGIX), _gloffset_FragmentColorMaterialSGIX)
FN ("glFragmentLightfSGIX", NAME(glFragmentLightfSGIX), _gloffset_FragmentLightfSGIX)
FN ("glFragmentLightfvSGIX", NAME(glFragmentLightfvSGIX), _gloffset_FragmentLightfvSGIX)
FN ("glFragmentLightiSGIX", NAME(glFragmentLightiSGIX), _gloffset_FragmentLightiSGIX)
FN ("glFragmentLightivSGIX", NAME(glFragmentLightivSGIX), _gloffset_FragmentLightivSGIX)
FN ("glFragmentLightModelfSGIX", NAME(glFragmentLightModelfSGIX), _gloffset_FragmentLightModelfSGIX)
FN ("glFragmentLightModelfvSGIX", NAME(glFragmentLightModelfvSGIX), _gloffset_FragmentLightModelfvSGIX)
FN ("glFragmentLightModeliSGIX", NAME(glFragmentLightModeliSGIX), _gloffset_FragmentLightModeliSGIX)
FN ("glFragmentLightModelivSGIX", NAME(glFragmentLightModelivSGIX), _gloffset_FragmentLightModelivSGIX)
FN ("glFragmentMaterialfSGIX", NAME(glFragmentMaterialfSGIX), _gloffset_FragmentMaterialfSGIX)
FN ("glFragmentMaterialfvSGIX", NAME(glFragmentMaterialfvSGIX), _gloffset_FragmentMaterialfvSGIX)
FN ("glFragmentMaterialiSGIX", NAME(glFragmentMaterialiSGIX), _gloffset_FragmentMaterialiSGIX)
FN ("glFragmentMaterialivSGIX", NAME(glFragmentMaterialivSGIX), _gloffset_FragmentMaterialivSGIX)
FN ("glGetFragmentLightfvSGIX", NAME(glGetFragmentLightfvSGIX), _gloffset_GetFragmentLightfvSGIX)
FN ("glGetFragmentLightivSGIX", NAME(glGetFragmentLightivSGIX), _gloffset_GetFragmentLightivSGIX)
FN ("glGetFragmentMaterialfvSGIX", NAME(glGetFragmentMaterialfvSGIX), _gloffset_GetFragmentMaterialfvSGIX)
FN ("glGetFragmentMaterialivSGIX", NAME(glGetFragmentMaterialivSGIX), _gloffset_GetFragmentMaterialivSGIX)
FN ("glLightEnviSGIX", NAME(glLightEnviSGIX), _gloffset_LightEnviSGIX)
#undef NAME

	/* 112. GL_EXT_draw_range_elements */
#if 000
#ifdef GL_EXT_draw_range_elements
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glDrawRangeElementsEXT", NAME(glDrawRangeElementsEXT), _gloffset_DrawRangeElementsEXT)
#undef NAME
#endif

	/* 117. GL_EXT_light_texture */
#if 000
#ifdef GL_EXT_light_texture
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glApplyTextureEXT", NAME(glApplyTextureEXT), _gloffset_ApplyTextureEXT)
FN ("glTextureLightEXT", NAME(glTextureLightEXT), _gloffset_TextureLightEXT)
FN ("glTextureMaterialEXT", NAME(glTextureMaterialEXT), _gloffset_TextureMaterialEXT)
#undef NAME

	/* 135. GL_INTEL_texture_scissor */
#ifdef GL_INTEL_texture_scissor
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glTexScissorINTEL", NAME(glTexScissorINTEL), _gloffset_TexScissorINTEL)
FN ("glTexScissorFuncINTEL", NAME(glTexScissorFuncINTEL), _gloffset_glTexScissorFuncINTEL)
#undef NAME

	/* 136. GL_INTEL_parallel_arrays */
#ifdef GL_INTEL_parallel_arrays
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glVertexPointervINTEL", NAME(glVertexPointervINTEL), _gloffset_VertexPointervINTEL)
FN ("glNormalPointervINTEL", NAME(glNormalPointervINTEL), _gloffset_NormalPointervINTEL)
FN ("glColorPointervINTEL", NAME(glColorPointervINTEL), _gloffset_ColorPointervINTEL)
FN ("glTexCoordPointervINTEL", NAME(glTexCoordPointervINTEL), _gloffset_glxCoordPointervINTEL)
#undef NAME
#endif

	/* 138. GL_EXT_pixel_transform */
#if 000
#ifdef GL_EXT_pixel_transform
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glPixelTransformParameteriEXT", NAME(glPixelTransformParameteriEXT), _gloffset_PixelTransformParameteriEXT)
FN ("glPixelTransformParameterfEXT", NAME(glPixelTransformParameterfEXT), _gloffset_PixelTransformParameterfEXT)
FN ("glPixelTransformParameterivEXT", NAME(glPixelTransformParameterivEXT), _gloffset_PixelTransformParameterivEXT)
FN ("glPixelTransformParameterfvEXT", NAME(glPixelTransformParameterfvEXT), _gloffset_PixelTransformParameterfvEXT)
FN ("glGetPixelTransformParameterivEXT", NAME(glGetPixelTransformParameterivEXT), _gloffset_GetPixelTransformParameterivEXT)
FN ("glGetPixelTransformParameterfvEXT", NAME(glGetPixelTransformParameterfvEXT), _gloffset_GetPixelTransformParameterfvEXT)
#undef NAME
#endif

	/* 145. GL_EXT_secondary_color */
#ifdef GL_EXT_secondary_color
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glSecondaryColor3bEXT", NAME(glSecondaryColor3bEXT), _gloffset_SecondaryColor3bEXT)
FN ("glSecondaryColor3dEXT", NAME(glSecondaryColor3dEXT), _gloffset_SecondaryColor3dEXT)
FN ("glSecondaryColor3fEXT", NAME(glSecondaryColor3fEXT), _gloffset_SecondaryColor3fEXT)
FN ("glSecondaryColor3iEXT", NAME(glSecondaryColor3iEXT), _gloffset_SecondaryColor3iEXT)
FN ("glSecondaryColor3sEXT", NAME(glSecondaryColor3sEXT), _gloffset_SecondaryColor3sEXT)
FN ("glSecondaryColor3ubEXT", NAME(glSecondaryColor3ubEXT), _gloffset_SecondaryColor3ubEXT)
FN ("glSecondaryColor3uiEXT", NAME(glSecondaryColor3uiEXT), _gloffset_SecondaryColor3uiEXT)
FN ("glSecondaryColor3usEXT", NAME(glSecondaryColor3usEXT), _gloffset_SecondaryColor3usEXT)
FN ("glSecondaryColor3bvEXT", NAME(glSecondaryColor3bvEXT), _gloffset_SecondaryColor3bvEXT)
FN ("glSecondaryColor3dvEXT", NAME(glSecondaryColor3dvEXT), _gloffset_SecondaryColor3dvEXT)
FN ("glSecondaryColor3fvEXT", NAME(glSecondaryColor3fvEXT), _gloffset_SecondaryColor3fvEXT)
FN ("glSecondaryColor3ivEXT", NAME(glSecondaryColor3ivEXT), _gloffset_SecondaryColor3ivEXT)
FN ("glSecondaryColor3svEXT", NAME(glSecondaryColor3svEXT), _gloffset_SecondaryColor3svEXT)
FN ("glSecondaryColor3ubvEXT", NAME(glSecondaryColor3ubvEXT), _gloffset_SecondaryColor3ubvEXT)
FN ("glSecondaryColor3uivEXT", NAME(glSecondaryColor3uivEXT), _gloffset_SecondaryColor3uivEXT)
FN ("glSecondaryColor3usvEXT", NAME(glSecondaryColor3usvEXT), _gloffset_SecondaryColor3usvEXT)
FN ("glSecondaryColorPointerEXT", NAME(glSecondaryColorPointerEXT), _gloffset_SecondaryColorPointerEXT)
#undef NAME

	/* 147. GL_EXT_texture_perturb_normal */
#if 000
#ifdef GL_EXT_texture_perturb_normal
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glTextureNormalEXT", NAME(glTextureNormalEXT), _gloffset_TextureNormalEXT)
#undef NAME
#endif

	/* 148. GL_EXT_multi_draw_arrays */
#if 000
#ifdef GL_EXT_multi_draw_arrays
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glMultiDrawArraysEXT", NAME(glMultiDrawArraysEXT), _gloffset_MultiDrawArraysEXT)
#undef NAME
#endif

	/* 149. GL_EXT_fog_coord */
#ifdef GL_EXT_fog_coord
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glFogCoordfEXT", NAME(glFogCoordfEXT), _gloffset_FogCoordfEXT)
FN ("glFogCoordfvEXT", NAME(glFogCoordfvEXT), _gloffset_FogCoordfvEXT)
FN ("glFogCoorddEXT", NAME(glFogCoorddEXT), _gloffset_FogCoorddEXT)
FN ("glFogCoorddvEXT", NAME(glFogCoorddvEXT), _gloffset_FogCoorddvEXT)
FN ("glFogCoordPointerEXT", NAME(glFogCoordPointerEXT), _gloffset_FogCoordPointerEXT)
#undef NAME

	/* 156. GL_EXT_coordinate_frame */
#if 000
#ifdef GL_EXT_coordinate_frame
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glTangent3bEXT", NAME(glTangent3bEXT), _gloffset_Tangent3bEXT)
FN ("glTangent3dEXT", NAME(glTangent3dEXT), _gloffset_Tangent3dEXT)
FN ("glTangent3fEXT", NAME(glTangent3fEXT), _gloffset_Tangent3fEXT)
FN ("glTangent3iEXT", NAME(glTangent3iEXT), _gloffset_Tangent3iEXT)
FN ("glTangent3sEXT", NAME(glTangent3sEXT), _gloffset_Tangent3sEXT)
FN ("glTangent3bvEXT", NAME(glTangent3bvEXT), _gloffset_Tangent3bvEXT)
FN ("glTangent3dvEXT", NAME(glTangent3dvEXT), _gloffset_Tangent3dvEXT)
FN ("glTangent3fvEXT", NAME(glTangent3fvEXT), _gloffset_Tangent3fvEXT)
FN ("glTangent3ivEXT", NAME(glTangent3ivEXT), _gloffset_Tangent3ivEXT)
FN ("glTangent3svEXT", NAME(glTangent3svEXT), _gloffset_Tangent3svEXT)
FN ("glBinormal3bEXT", NAME(glBinormal3bEXT), _gloffset_Binormal3bEXT)
FN ("glBinormal3dEXT", NAME(glBinormal3dEXT), _gloffset_Binormal3dEXT)
FN ("glBinormal3fEXT", NAME(glBinormal3fEXT), _gloffset_Binormal3fEXT)
FN ("glBinormal3iEXT", NAME(glBinormal3iEXT), _gloffset_Binormal3iEXT)
FN ("glBinormal3sEXT", NAME(glBinormal3sEXT), _gloffset_Binormal3sEXT)
FN ("glBinormal3bvEXT", NAME(glBinormal3bvEXT), _gloffset_Binormal3bvEXT)
FN ("glBinormal3dvEXT", NAME(glBinormal3dvEXT), _gloffset_Binormal3dvEXT)
FN ("glBinormal3fvEXT", NAME(glBinormal3fvEXT), _gloffset_Binormal3fvEXT)
FN ("glBinormal3ivEXT", NAME(glBinormal3ivEXT), _gloffset_Binormal3ivEXT)
FN ("glBinormal3svEXT", NAME(glBinormal3svEXT), _gloffset_Binormal3svEXT)
FN ("glTangentPointerEXT", NAME(glTangentPointerEXT), _gloffset_TangentPointerEXT)
FN ("glBinormalPointerEXT", NAME(glBinormalPointerEXT), _gloffset_BinormalPointerEXT)
#undef NAME
#endif

	/* 164. GL_SUN_global_alpha */
#if 000
#ifdef GL_SUN_global_alpha
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glGlobalAlphaFactorbSUN", NAME(glGlobalAlphaFactorbSUN), _gloffset_GlobalAlphaFactorbSUN)
FN ("glGlobalAlphaFactorsSUN", NAME(glGlobalAlphaFactorsSUN), _gloffset_GlobalAlphaFactorsSUN)
FN ("glGlobalAlphaFactoriSUN", NAME(glGlobalAlphaFactoriSUN), _gloffset_GlobalAlphaFactoriSUN)
FN ("glGlobalAlphaFactorfSUN", NAME(glGlobalAlphaFactorfSUN), _gloffset_GlobalAlphaFactorfSUN)
FN ("glGlobalAlphaFactordSUN", NAME(glGlobalAlphaFactordSUN), _gloffset_GlobalAlphaFactordSUN)
FN ("glGlobalAlphaFactorubSUN", NAME(glGlobalAlphaFactorubSUN), _gloffset_GlobalAlphaFactorubSUN)
FN ("glGlobalAlphaFactorusSUN", NAME(glGlobalAlphaFactorusSUN), _gloffset_GlobalAlphaFactorusSUN)
FN ("glGlobalAlphaFactoruiSUN", NAME(glGlobalAlphaFactoruiSUN), _gloffset_GlobalAlphaFactoruiSUN)
#undef NAME
#endif

	/* 165. GL_SUN_triangle_list */
#if 000
#ifdef GL_SUN_triangle_list
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glReplacementCodeuiSUN", NAME(glReplacementCodeuiSUN), _gloffset_ReplacementCodeuiSUN)
FN ("glReplacementCodeusSUN", NAME(glReplacementCodeusSUN), _gloffset_ReplacementCodeusSUN)
FN ("glReplacementCodeubSUN", NAME(glReplacementCodeubSUN), _gloffset_ReplacementCodeubSUN)
FN ("glReplacementCodeuivSUN", NAME(glReplacementCodeuivSUN), _gloffset_ReplacementCodeuivSUN)
FN ("glReplacementCodeusvSUN", NAME(glReplacementCodeusvSUN), _gloffset_ReplacementCodeusvSUN)
FN ("glReplacementCodeubvSUN", NAME(glReplacementCodeubvSUN), _gloffset_ReplacementCodeubvSUN)
FN ("glReplacementCodePointerSUN", NAME(glReplacementCodePointerSUN), _gloffset_ReplacementCodePointerSUN)
#undef NAME
#endif

	/* 166. GL_SUN_vertex */
#if 000
#ifdef GL_SUN_vertex
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glColor4ubVertex2fSUN", NAME(glColor4ubVertex2fSUN), _gloffset_Color4ubVertex2fSUN)
FN ("glColor4ubVertex2fvSUN", NAME(glColor4ubVertex2fvSUN), _gloffset_Color4ubVertex2fvSUN)
FN ("glColor4ubVertex3fSUN", NAME(glColor4ubVertex3fSUN), _gloffset_Color4ubVertex3fSUN)
FN ("glColor4ubVertex3fvSUN", NAME(glColor4ubVertex3fvSUN), _gloffset_Color4ubVertex3fvSUN)
FN ("glColor3fVertex3fSUN", NAME(glColor3fVertex3fSUN), _gloffset_Color3fVertex3fSUN)
FN ("glColor3fVertex3fvSUN", NAME(glColor3fVertex3fvSUN), _gloffset_Color3fVertex3fvSUN)
FN ("glNormal3fVertex3fSUN", NAME(glNormal3fVertex3fSUN), _gloffset_Normal3fVertex3fSUN)
FN ("glNormal3fVertex3fvSUN", NAME(glNormal3fVertex3fvSUN), _gloffset_Normal3fVertex3fvSUN)
FN ("glColor4fNormal3fVertex3fSUN", NAME(glColor4fNormal3fVertex3fSUN), _gloffset_Color4fNormal3fVertex3fSUN)
FN ("glColor4fNormal3fVertex3fvSUN", NAME(glColor4fNormal3fVertex3fvSUN), _gloffset_Color4fNormal3fVertex3fvSUN)
FN ("glTexCoord2fVertex3fSUN", NAME(glTexCoord2fVertex3fSUN), _gloffset_TexCoord2fVertex3fSUN)
FN ("glTexCoord2fVertex3fvSUN", NAME(glTexCoord2fVertex3fvSUN), _gloffset_TexCoord2fVertex3fvSUN)
FN ("glTexCoord4fVertex4fSUN", NAME(glTexCoord4fVertex4fSUN), _gloffset_TexCoord4fVertex4fSUN)
FN ("glTexCoord4fVertex4fvSUN", NAME(glTexCoord4fVertex4fvSUN), _gloffset_TexCoord4fVertex4fvSUN)
FN ("glTexCoord2fColor4ubVertex3fSUN", NAME(glTexCoord2fColor4ubVertex3fSUN), _gloffset_TexCoord2fColor4ubVertex3fSUN)
FN ("glTexCoord2fColor4ubVertex3fvSUN", NAME(glTexCoord2fColor4ubVertex3fvSUN), _gloffset_TexCoord2fColor4ubVertex3fvSUN)
FN ("glTexCoord2fColor3fVertex3fSUN", NAME(glTexCoord2fColor3fVertex3fSUN), _gloffset_TexCoord2fColor3fVertex3fSUN)
FN ("glTexCoord2fColor3fVertex3fvSUN", NAME(glTexCoord2fColor3fVertex3fvSUN), _gloffset_TexCoord2fColor3fVertex3fvSUN)
FN ("glTexCoord2fNormal3fVertex3fSUN", NAME(glTexCoord2fNormal3fVertex3fSUN), _gloffset_TexCoord2fNormal3fVertex3fSUN)
FN ("glTexCoord2fNormal3fVertex3fvSUN", NAME(glTexCoord2fNormal3fVertex3fvSUN), _gloffset_TexCoord2fNormal3fVertex3fvSUN)
FN ("glTexCoord2fColor4fNormal3fVertex3fSUN", NAME(glTexCoord2fColor4fNormal3fVertex3fSUN), _gloffset_TexCoord2fColor4fNormal3fVertex3fSUN)
FN ("glTexCoord2fColor4fNormal3fVertex3fvSUN", NAME(glTexCoord2fColor4fNormal3fVertex3fvSUN), _gloffset_TexCoord2fColor4fNormal3fVertex3fvSUN)
FN ("glTexCoord4fColor4fNormal3fVertex4fSUN", NAME(glTexCoord4fColor4fNormal3fVertex4fSUN), _gloffset_TexCoord4fColor4fNormal3fVertex4fSUN)
FN ("glTexCoord4fColor4fNormal3fVertex4fvSUN", NAME(glTexCoord4fColor4fNormal3fVertex4fvSUN), _gloffset_TexCoord4fColor4fNormal3fVertex4fvSUN)
FN ("glReplacementCodeuiVertex3fSUN", NAME(glReplacementCodeuiVertex3fSUN), _gloffset_ReplacementCodeuiVertex3fSUN)
FN ("glReplacementCodeuiVertex3fvSUN", NAME(glReplacementCodeuiVertex3fvSUN), _gloffset_ReplacementCodeuiVertex3fvSUN)
FN ("glReplacementCodeuiColor4ubVertex3fSUN", NAME(glReplacementCodeuiColor4ubVertex3fSUN), _gloffset_ReplacementCodeuiColor4ubVertex3fSUN)
FN ("glReplacementCodeuiColor4ubVertex3fvSUN", NAME(glReplacementCodeuiColor4ubVertex3fvSUN), _gloffset_ReplacementCodeuiColor4ubVertex3fvSUN)
FN ("glReplacementCodeuiColor3fVertex3fSUN", NAME(glReplacementCodeuiColor3fVertex3fSUN), _gloffset_ReplacementCodeuiColor3fVertex3fSUN)
FN ("glReplacementCodeuiColor3fVertex3fvSUN", NAME(glReplacementCodeuiColor3fVertex3fvSUN), _gloffset_ReplacementCodeuiColor3fVertex3fvSUN)
FN ("glReplacementCodeuiNormal3fVertex3fSUN", NAME(glReplacementCodeuiNormal3fVertex3fSUN), _gloffset_ReplacementCodeuiNormal3fVertex3fSUN)
FN ("glReplacementCodeuiNormal3fVertex3fvSUN", NAME(glReplacementCodeuiNormal3fVertex3fvSUN), _gloffset_ReplacementCodeuiNormal3fVertex3fvSUN)
FN ("glReplacementCodeuiColor4fNormal3fVertex3fSUN", NAME(glReplacementCodeuiColor4fNormal3fVertex3fSUN), _gloffset_ReplacementCodeuiColor4fNormal3fVertex3fSUN)
FN ("glReplacementCodeuiColor4fNormal3fVertex3fvSUN", NAME(glReplacementCodeuiColor4fNormal3fVertex3fvSUN), _gloffset_ReplacementCodeuiColor4fNormal3fVertex3fvSUN)
FN ("glReplacementCodeuiTexCoord2fVertex3fSUN", NAME(glReplacementCodeuiTexCoord2fVertex3fSUN), _gloffset_ReplacementCodeuiTexCoord2fVertex3fSUN)
FN ("glReplacementCodeuiTexCoord2fVertex3fvSUN", NAME(glReplacementCodeuiTexCoord2fVertex3fvSUN), _gloffset_ReplacementCodeuiTexCoord2fVertex3fvSUN)
FN ("glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN", NAME(glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN), _gloffset_ReplacementCodeuiTexCoord2fNormal3fVertex3fSUN)
FN ("glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN", NAME(glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN), _gloffset_ReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN)
FN ("glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN", NAME(glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN), _gloffset_ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN)
FN ("glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN", NAME(glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN), _gloffset_ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN)
#undef NAME
#endif

	/* 173. GL_EXT/INGR_blend_func_separate */
#ifdef GL_EXT_blend_func_separate
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glBlendFuncSeparateEXT", NAME(glBlendFuncSeparateEXT), _gloffset_BlendFuncSeparateEXT)
FN ("glBlendFuncSeparateINGR", NAME(glBlendFuncSeparateEXT), _gloffset_BlendFuncSeparateEXT)
#undef NAME

	/* 188. GL_EXT_vertex_weighting */
#ifdef GL_EXT_vertex_weighting
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glVertexWeightfEXT", NAME(glVertexWeightfEXT), _gloffset_VertexWeightfEXT)
FN ("glVertexWeightfvEXT", NAME(glVertexWeightfvEXT), _gloffset_VertexWeightfvEXT)
FN ("glVertexWeightPointerEXT", NAME(glVertexWeightPointerEXT), _gloffset_VertexWeightPointerEXT)
#undef NAME

	/* 190. GL_NV_vertex_array_range */
#ifdef GL_NV_vertex_array_range
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glFlushVertexArrayRangeNV", NAME(glFlushVertexArrayRangeNV), _gloffset_FlushVertexArrayRangeNV)
FN ("glVertexArrayRangeNV", NAME(glVertexArrayRangeNV), _gloffset_VertexArrayRangeNV)
#undef NAME

	/* 191. GL_NV_register_combiners */
#ifdef GL_NV_register_combiners
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glCombinerParameterfvNV", NAME(glCombinerParameterfvNV), _gloffset_CombinerParameterfvNV)
FN ("glCombinerParameterfNV", NAME(glCombinerParameterfNV), _gloffset_CombinerParameterfNV)
FN ("glCombinerParameterivNV", NAME(glCombinerParameterivNV), _gloffset_CombinerParameterivNV)
FN ("glCombinerParameteriNV", NAME(glCombinerParameteriNV), _gloffset_CombinerParameteriNV)
FN ("glCombinerInputNV", NAME(glCombinerInputNV), _gloffset_CombinerInputNV)
FN ("glCombinerOutputNV", NAME(glCombinerOutputNV), _gloffset_CombinerOutputNV)
FN ("glFinalCombinerInputNV", NAME(glFinalCombinerInputNV), _gloffset_FinalCombinerInputNV)
FN ("glGetCombinerInputParameterfvNV", NAME(glGetCombinerInputParameterfvNV), _gloffset_GetCombinerInputParameterfvNV)
FN ("glGetCombinerInputParameterivNV", NAME(glGetCombinerInputParameterivNV), _gloffset_GetCombinerInputParameterivNV)
FN ("glGetCombinerOutputParameterfvNV", NAME(glGetCombinerOutputParameterfvNV), _gloffset_GetCombinerOutputParameterfvNV)
FN ("glGetCombinerOutputParameterivNV", NAME(glGetCombinerOutputParameterivNV), _gloffset_GetCombinerOutputParameterivNV)
FN ("glGetFinalCombinerInputParameterfvNV", NAME(glGetFinalCombinerInputParameterfvNV), _gloffset_GetFinalCombinerInputParameterfvNV)
FN ("glGetFinalCombinerInputParameterivNV", NAME(glGetFinalCombinerInputParameterivNV), _gloffset_GetFinalCombinerInputParameterivNV)
#undef NAME

	/* 196. GL_MESA_resize_buffers */
#ifdef MESA_resize_buffers
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glResizeBuffersMESA", NAME(glResizeBuffersMESA), _gloffset_ResizeBuffersMESA)
#undef NAME

	/* 197. GL_MESA_window_pos */
#ifdef MESA_window_pos
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glWindowPos2iMESA", NAME(glWindowPos2iMESA), _gloffset_WindowPos2iMESA)
FN ("glWindowPos2sMESA", NAME(glWindowPos2sMESA), _gloffset_WindowPos2sMESA)
FN ("glWindowPos2fMESA", NAME(glWindowPos2fMESA), _gloffset_WindowPos2fMESA)
FN ("glWindowPos2dMESA", NAME(glWindowPos2dMESA), _gloffset_WindowPos2dMESA)
FN ("glWindowPos2ivMESA", NAME(glWindowPos2ivMESA), _gloffset_WindowPos2ivMESA)
FN ("glWindowPos2svMESA", NAME(glWindowPos2svMESA), _gloffset_WindowPos2svMESA)
FN ("glWindowPos2fvMESA", NAME(glWindowPos2fvMESA), _gloffset_WindowPos2fvMESA)
FN ("glWindowPos2dvMESA", NAME(glWindowPos2dvMESA), _gloffset_WindowPos2dvMESA)
FN ("glWindowPos3iMESA", NAME(glWindowPos3iMESA), _gloffset_WindowPos3iMESA)
FN ("glWindowPos3sMESA", NAME(glWindowPos3sMESA), _gloffset_WindowPos3sMESA)
FN ("glWindowPos3fMESA", NAME(glWindowPos3fMESA), _gloffset_WindowPos3fMESA)
FN ("glWindowPos3dMESA", NAME(glWindowPos3dMESA), _gloffset_WindowPos3dMESA)
FN ("glWindowPos3ivMESA", NAME(glWindowPos3ivMESA), _gloffset_WindowPos3ivMESA)
FN ("glWindowPos3svMESA", NAME(glWindowPos3svMESA), _gloffset_WindowPos3svMESA)
FN ("glWindowPos3fvMESA", NAME(glWindowPos3fvMESA), _gloffset_WindowPos3fvMESA)
FN ("glWindowPos3dvMESA", NAME(glWindowPos3dvMESA), _gloffset_WindowPos3dvMESA)
FN ("glWindowPos4iMESA", NAME(glWindowPos4iMESA), _gloffset_WindowPos4iMESA)
FN ("glWindowPos4sMESA", NAME(glWindowPos4sMESA), _gloffset_WindowPos4sMESA)
FN ("glWindowPos4fMESA", NAME(glWindowPos4fMESA), _gloffset_WindowPos4fMESA)
FN ("glWindowPos4dMESA", NAME(glWindowPos4dMESA), _gloffset_WindowPos4dMESA)
FN ("glWindowPos4ivMESA", NAME(glWindowPos4ivMESA), _gloffset_WindowPos4ivMESA)
FN ("glWindowPos4svMESA", NAME(glWindowPos4svMESA), _gloffset_WindowPos4svMESA)
FN ("glWindowPos4fvMESA", NAME(glWindowPos4fvMESA), _gloffset_WindowPos4fvMESA)
FN ("glWindowPos4dvMESA", NAME(glWindowPos4dvMESA), _gloffset_WindowPos4dvMESA)
#undef NAME

	/* 209. WGL_EXT_multisample */
#ifdef WGL_EXT_multisample
#define NAME(X) X
#else
#define NAME(X) NotImplemented
#endif
FN ("glSampleMaskEXT", NAME(glSampleMaskEXT), _gloffset_SampleMaskSGIS)
FN ("glSamplePatternEXT", NAME(glSamplePatternEXT), _gloffset_SamplePatternSGIS)
#undef NAME

#ifdef GOT_RELATIVE_FUNCTIONS
__asm (".section .rodata.string_functions; .long 0, 0, 0; .previous");
#else
	{ NULL, NULL }  /* end of list marker */
};
#endif


/*
 * Return dispatch table offset of the named static (built-in) function.
 * Return -1 if function not found.
 */
static corefn GLint
get_static_proc_offset(const char *funcName)
{
   GLuint i;
   for (i = 0; static_functions[i].Name; i++)
      if (strcmp(FN_NAME(i), funcName) == 0)
	 return static_functions[i].Offset;
   return -1;
}


/*
 * Return dispatch function address the named static (built-in) function.
 * Return NULL if function not found.
 */
static corefn GLvoid *
get_static_proc_address(const char *funcName)
{
   GLint i;
   for (i = 0; static_functions[i].Name; i++)
      if (strcmp(FN_NAME(i), funcName) == 0)
         return FN_ADDRESS(i);
   return NULL;
}



/**********************************************************************
 * Extension function management.
 */


#define MAX_EXTENSION_FUNCS 1000

static struct name_address_offset ExtEntryTable[MAX_EXTENSION_FUNCS];
static GLuint NumExtEntryPoints = 0;

#ifdef USE_SPARC_ASM
extern void __glapi_sparc_icache_flush(unsigned int *);
#endif

/*
 * Generate a dispatch function (entrypoint) which jumps through
 * the given slot number (offset) in the current dispatch table.
 * We need assembly language in order to accomplish this.
 */
static corefn void *
generate_entrypoint(GLuint functionOffset)
{
#if defined(USE_X86_ASM)
#if defined(GLX_USE_TLS)
   /*
    *  0:   65 a1 10 32 54 76       movl   %gs:_gl_tls_Dispatch@NTPOFF,%eax
    *       00 01 02 03 04 05
    *  7:   ff a0 10 32 54 76       jmp    *0x76543210(%eax)
    *       06 07 08 09 0a 0b
    */
   extern hidden int _gl_entrypoint_templ[4];
   unsigned char *code = malloc(sizeof(_gl_entrypoint_templ));
   if (code) {
      memcpy(code, _gl_entrypoint_templ, sizeof(_gl_entrypoint_templ));   
      *(unsigned short *)(code + 0x06) = 0xa0ff;
      *(unsigned int *)(code + 0x08) = (unsigned int)functionOffset * 4;
   }
   return code;
#else
   /*
    * This x86 code contributed by Josh Vanderhoof.
    *
    *  0:   a1 10 32 54 76          movl   __glapi_Dispatch,%eax
    *       00 01 02 03 04
    *  5:   85 c0                   testl  %eax,%eax
    *       05 06
    *  7:   74 06                   je     f <entrypoint+0xf>
    *       07 08
    *  9:   ff a0 10 32 54 76       jmp    *0x76543210(%eax)
    *       09 0a 0b 0c 0d 0e
    *  f:   e8 fc ff ff ff          call   __glapi_get_dispatch
    *       0f 10 11 12 13
    * 14:   ff a0 10 32 54 76       jmp    *0x76543210(%eax)
    *       14 15 16 17 18 19
    */
   static const unsigned char temp[] = {
      0xa1, 0x00, 0x00, 0x00, 0x00,
      0x85, 0xc0,
      0x74, 0x06,
      0xff, 0xa0, 0x00, 0x00, 0x00, 0x00,
      0xe8, 0x00, 0x00, 0x00, 0x00,
      0xff, 0xa0, 0x00, 0x00, 0x00, 0x00
   };
#ifdef USE_ASM_PTHREAD_GETSPECIFIC
   extern hidden struct _glapi_table * _gl_get_dispatch(void);
#define get_dispatch _gl_get_dispatch
#else
#define get_dispatch _glapi_get_dispatch
#endif
   unsigned char *code = malloc(sizeof(temp));
   unsigned int next_insn;
   if (code) {
      memcpy(code, temp, sizeof(temp));

      *(unsigned int *)(code + 0x01) = (unsigned int)&_glapi_Dispatch;
      *(unsigned int *)(code + 0x0b) = (unsigned int)functionOffset * 4;
      next_insn = (unsigned int)(code + 0x14);
      *(unsigned int *)(code + 0x10) = (unsigned int)get_dispatch - next_insn;
      *(unsigned int *)(code + 0x16) = (unsigned int)functionOffset * 4;
   }
   return code;
#endif
#elif defined(USE_SPARC_ASM)

#ifdef __sparc_v9__
   static const unsigned int insn_template[] = {
	   0x05000000,	/* sethi	%uhi(_glapi_Dispatch), %g2	*/
	   0x03000000,	/* sethi	%hi(_glapi_Dispatch), %g1	*/
	   0x8410a000,	/* or		%g2, %ulo(_glapi_Dispatch), %g2	*/
	   0x82106000,	/* or		%g1, %lo(_glapi_Dispatch), %g1	*/
	   0x8528b020,	/* sllx		%g2, 32, %g2			*/
	   0xc2584002,	/* ldx		[%g1 + %g2], %g1		*/
	   0x05000000,	/* sethi	%hi(8 * glapioffset), %g2	*/
	   0x8410a000,	/* or		%g2, %lo(8 * glapioffset), %g2	*/
	   0xc6584002,	/* ldx		[%g1 + %g2], %g3		*/
	   0x81c0c000,	/* jmpl		%g3, %g0			*/
	   0x01000000	/*  nop						*/
   };
#else
   static const unsigned int insn_template[] = {
	   0x03000000,	/* sethi	%hi(_glapi_Dispatch), %g1	  */
	   0xc2006000,	/* ld		[%g1 + %lo(_glapi_Dispatch)], %g1 */
	   0xc6006000,	/* ld		[%g1 + %lo(4*glapioffset)], %g3	  */
	   0x81c0c000,	/* jmpl		%g3, %g0			  */
	   0x01000000	/*  nop						  */
   };
#endif
   unsigned int *code = malloc(sizeof(insn_template));
   unsigned long glapi_addr = (unsigned long) &_glapi_Dispatch;
   if (code) {
      memcpy(code, insn_template, sizeof(insn_template));

#ifdef __sparc_v9__
      code[0] |= (glapi_addr >> (32 + 10));
      code[1] |= ((glapi_addr & 0xffffffff) >> 10);
      __glapi_sparc_icache_flush(&code[0]);
      code[2] |= ((glapi_addr >> 32) & ((1 << 10) - 1));
      code[3] |= (glapi_addr & ((1 << 10) - 1));
      __glapi_sparc_icache_flush(&code[2]);
      code[6] |= ((functionOffset * 8) >> 10);
      code[7] |= ((functionOffset * 8) & ((1 << 10) - 1));
      __glapi_sparc_icache_flush(&code[6]);
#else
      code[0] |= (glapi_addr >> 10);
      code[1] |= (glapi_addr & ((1 << 10) - 1));
      __glapi_sparc_icache_flush(&code[0]);
      code[2] |= (functionOffset * 4);
      __glapi_sparc_icache_flush(&code[2]);
#endif
   }
   return code;
#else
   return NULL;
#endif
}



/*
 * Add a new extension function entrypoint.
 * Return: GL_TRUE = success or GL_FALSE = failure
 */
corefn GLboolean
_glapi_add_entrypoint(const char *funcName, GLuint offset)
{
   /* first check if the named function is already statically present */
   {
      GLint index = get_static_proc_offset(funcName);
      if (index >= 0) {
         return (GLboolean) ((GLuint) index == offset);  /* bad offset! */
      }
   }

   {
      /* be sure index and name match known data */
      GLuint i;
      for (i = 0; i < NumExtEntryPoints; i++) {
         if (strcmp(ExtEntryTable[i].Name, funcName) == 0) {
            /* function already registered with api */
            if (ExtEntryTable[i].Offset == offset) {
               return GL_TRUE;  /* offsets match */
            }
            else {
               return GL_FALSE;  /* bad offset! */
            }
         }
      }

      /* Make sure we don't try to add a new entrypoint after someone
       * has already called _glapi_get_dispatch_table_size()!  If that's
       * happened the caller's information would become out of date.
       */
      if (GetSizeCalled)
         return GL_FALSE;

      /* make sure we have space */
      if (NumExtEntryPoints >= MAX_EXTENSION_FUNCS) {
         return GL_FALSE;
      }
      else {
         void *entrypoint = generate_entrypoint(offset);
         if (!entrypoint)
            return GL_FALSE;

         ExtEntryTable[NumExtEntryPoints].Name = str_dup(funcName);
         ExtEntryTable[NumExtEntryPoints].Offset = offset;
         ExtEntryTable[NumExtEntryPoints].Address = entrypoint;
         NumExtEntryPoints++;

         if (offset > MaxDispatchOffset)
            MaxDispatchOffset = offset;

         return GL_TRUE;  /* success */
      }
   }

   /* should never get here, but play it safe */
   return GL_FALSE;
}



#if 0000 /* prototype code for dynamic extension slot allocation */

static int NextFreeOffset = 409; /*XXX*/
#define MAX_DISPATCH_TABLE_SIZE 1000

/*
 * Dynamically allocate a dispatch slot for an extension entrypoint
 * and generate the assembly language dispatch stub.
 * Return the dispatch offset for the function or -1 if no room or error.
 */
corefn GLint
_glapi_add_entrypoint2(const char *funcName)
{
   int offset;

   /* first see if extension func is already known */
   offset = _glapi_get_proc_offset(funcName);
   if (offset >= 0)
      return offset;

   if (NumExtEntryPoints < MAX_EXTENSION_FUNCS
       && NextFreeOffset < MAX_DISPATCH_TABLE_SIZE) {
      void *entryPoint;
      offset = NextFreeOffset;
      entryPoint = generate_entrypoint(offset);
      if (entryPoint) {
         NextFreeOffset++;
         ExtEntryTable[NumExtEntryPoints].Name = str_dup(funcName);
         ExtEntryTable[NumExtEntryPoints].Offset = offset;
         ExtEntryTable[NumExtEntryPoints].Address = entryPoint;
         NumExtEntryPoints++;
         return offset;
      }
   }
   return -1;
}

#endif



/*
 * Return offset of entrypoint for named function within dispatch table.
 */
corefn GLint
_glapi_get_proc_offset(const char *funcName)
{
   /* search extension functions first */
   GLuint i;
   for (i = 0; i < NumExtEntryPoints; i++) {
      if (strcmp(ExtEntryTable[i].Name, funcName) == 0) {
         return ExtEntryTable[i].Offset;
      }
   }

   /* search static functions */
   return get_static_proc_offset(funcName);
}



/*
 * Return entrypoint for named function.
 */
corefn const GLvoid *
_glapi_get_proc_address(const char *funcName)
{
   /* search extension functions first */
   GLuint i;
   for (i = 0; i < NumExtEntryPoints; i++) {
      if (strcmp(ExtEntryTable[i].Name, funcName) == 0) {
         return ExtEntryTable[i].Address;
      }
   }

   /* search static functions */
   return get_static_proc_address(funcName);
}




/*
 * Return the name of the function at the given dispatch offset.
 * This is only intended for debugging.
 */
corefn const char *
_glapi_get_proc_name(GLuint offset)
{
   GLuint i;
#ifdef GOT_RELATIVE_FUNCTIONS
   for (i = 0; static_functions[i].Name; i++)
#else
   const GLuint n = sizeof(static_functions) / sizeof(struct name_address_offset);
   for (i = 0; i < n; i++)
#endif
      return FN_NAME(i);

   /* search added extension functions */
   for (i = 0; i < NumExtEntryPoints; i++) {
      if (ExtEntryTable[i].Offset == offset) {
         return ExtEntryTable[i].Name;
      }
   }
   return NULL;
}



/*
 * Make sure there are no NULL pointers in the given dispatch table.
 * Intended for debugging purposes.
 */
corefn void
_glapi_check_table(const struct _glapi_table *table)
{
#ifdef DEBUG
   const GLuint entries = _glapi_get_dispatch_table_size();
   const void **tab = (const void **) table;
   GLuint i;
   for (i = 1; i < entries; i++) {
      assert(tab[i]);
   }

   /* Do some spot checks to be sure that the dispatch table
    * slots are assigned correctly.
    */
   {
      GLuint BeginOffset = _glapi_get_proc_offset("glBegin");
      char *BeginFunc = (char*) &table->Begin;
      GLuint offset = (BeginFunc - (char *) table) / sizeof(void *);
      assert(BeginOffset == _gloffset_Begin);
      assert(BeginOffset == offset);
   }
   {
      GLuint viewportOffset = _glapi_get_proc_offset("glViewport");
      char *viewportFunc = (char*) &table->Viewport;
      GLuint offset = (viewportFunc - (char *) table) / sizeof(void *);
      assert(viewportOffset == _gloffset_Viewport);
      assert(viewportOffset == offset);
   }
   {
      GLuint VertexPointerOffset = _glapi_get_proc_offset("glVertexPointer");
      char *VertexPointerFunc = (char*) &table->VertexPointer;
      GLuint offset = (VertexPointerFunc - (char *) table) / sizeof(void *);
      assert(VertexPointerOffset == _gloffset_VertexPointer);
      assert(VertexPointerOffset == offset);
   }
   {
      GLuint ResetMinMaxOffset = _glapi_get_proc_offset("glResetMinmax");
      char *ResetMinMaxFunc = (char*) &table->ResetMinmax;
      GLuint offset = (ResetMinMaxFunc - (char *) table) / sizeof(void *);
      assert(ResetMinMaxOffset == _gloffset_ResetMinmax);
      assert(ResetMinMaxOffset == offset);
   }
   {
      GLuint blendColorOffset = _glapi_get_proc_offset("glBlendColor");
      char *blendColorFunc = (char*) &table->BlendColor;
      GLuint offset = (blendColorFunc - (char *) table) / sizeof(void *);
      assert(blendColorOffset == _gloffset_BlendColor);
      assert(blendColorOffset == offset);
   }
   {
      GLuint istextureOffset = _glapi_get_proc_offset("glIsTextureEXT");
      char *istextureFunc = (char*) &table->IsTextureEXT;
      GLuint offset = (istextureFunc - (char *) table) / sizeof(void *);
      assert(istextureOffset == _gloffset_IsTextureEXT);
      assert(istextureOffset == offset);
   }
   {
      GLuint secondaryColor3fOffset = _glapi_get_proc_offset("glSecondaryColor3fEXT");
      char *secondaryColor3fFunc = (char*) &table->SecondaryColor3fEXT;
      GLuint offset = (secondaryColor3fFunc - (char *) table) / sizeof(void *);
      assert(secondaryColor3fOffset == _gloffset_SecondaryColor3fEXT);
      assert(secondaryColor3fOffset == offset);
      assert(_glapi_get_proc_address("glSecondaryColor3fEXT") == (void *) &glSecondaryColor3fEXT);
   }
#endif
}
