require 'test_helper'

class Admin::LogRequestsControllerTest < ActionController::TestCase
  setup do
    CubertRequestLogsService.stubs(enabled?: true)
    @request.host = master_account.domain
    login_provider Account.master
    @service = master_account.first_service!
    Admin::LogRequestsController.any_instance.expects(:authorize_log_requests).returns(true)
  end

  test "should render index and no_log_requests" do
    backend_object = stub(:backend_object)
    backend_object.expects(:log_requests).returns([])
    Service.any_instance.expects(:backend_object).returns(backend_object)
    get :index, service_id: @service.id

    assert_template "index"
    assert_template partial: "_no_log_requests"
  end


  test "should render index if provider can't view cubert logs" do
    @service.create_cubert_info
    @controller.expects(:provider_can_use_cubert_request_logs?).returns(false).once

    backend_object = stub(:backend_object)
    backend_object.expects(:log_requests).returns([])
    Service.any_instance.expects(:backend_object).returns(backend_object)
    get :index, service_id: @service.id

    assert_template "index"
    assert_template partial: "_no_log_requests"
  end

  test "should render cubert_logs if provider can view cubert logs" do
    application_id = 20
    @service.create_cubert_info
    @controller.expects(:provider_can_use_cubert_request_logs?).returns(true).once
    CubertRequestLogsService.any_instance.expects(:get_list)
      .with do |search_params, _|
      search_params == { "application_id" => application_id.to_s }
    end.returns([])

    get :index, service_id: @service.id, "search" => { "application_id" => application_id }

    assert_template "admin/log_requests/cubert_logs"
    assert_template "application/_cubert_logs"
  end

  test "should render cubert_logs and no_log_requests" do
    @service.create_cubert_info
    CubertRequestLogsService.any_instance.expects(:get_list).returns([])
    get :index, service_id: @service.id

    assert_template "cubert_logs"
    assert_template partial: "_no_log_requests"
  end

  test 'renders json' do
    res = [{"id"=>"beb6d24e-4904-4c70-93cc-454046a60b96",
      "body"=>
          {"service_id"=>"6",
           "application_id"=>"245dfc03",
           "timestamp"=>"2015-09-04T14:22:07.000Z",
           "log"=>
               {"request"=>
                    {"path"=>"/admin/api/applications/find.xml?provider_key=xxx&application_id=xxx", "method"=>"GET", "remote_ip"=>"54.214.51.166", "headers"=>{"HTTP_VERSION"=>"HTTP/1.0"}},
                "response"=>{"code"=>"200", "length"=>"935", "body"=>"discarded"},
                "code"=>200},
           "usage"=>"account: 1, ",
           "user_id"=>nil},
      "bucket"=>"92a71f62-ecba-416f-82c2-e4f573aed2c8",
      "collection"=>"request_logs"}]

    @service.create_cubert_info(bucket_id: 'stubbed-bucket')

    stub_request(:get, %r{/v0/documents/stubbed-bucket/request_logs\?filter%5Bservice_id%5D=#{@service.id}&page=1&per_page=20&sort%5Btimestamp%5D=desc})
        .to_return(status: 200, body: res.to_json)

    get :index, service_id: @service.id, format: :json

    response = JSON.parse(@response.body)

    assert_equal(
      {
        "logs" => [{
          "code" => 200,
          "application_id" => "245dfc03",
          "usage" => "account: 1, ",
          "request" => {
              "path" => "/admin/api/applications/find.xml?provider_key=xxx&application_id=xxx",
              "method" => "GET",
              "remote_ip" => "54.214.51.166",
              "headers" => {"HTTP_VERSION" => "HTTP/1.0"}},
          "response" => {
              "code" => "200",
              "length" => "935",
              "body" => "discarded"
          },
          "timestamp" => "04 Sep 2015 14:22:07 UTC",
          "app" => nil
                   }],
          "pagination" => {
            "total" => 1,
            "page" => 1,
            "per_page" => 20
        }
      }, response)
  end

  class Admin::LogRequestsControllerImpersonationTest < ActionController::TestCase
    self.controller_class = Admin::LogRequestsController
    setup do
      @service = FactoryGirl.create(:simple_service)
      @provider = @service.account
      user = FactoryGirl.create :active_admin, :username => "3scaleadmin", :account => @provider
      @request.host = @provider.domain
      @controller.stubs(current_account: @provider)
      login_as user
      @controller.stubs(:service_cubert_info_present?).returns(true)
      Admin::LogRequestsController.any_instance.expects(:authorize_log_requests).returns(true)
    end

    test "should not render cubert logs when impersonating and user has them disabled" do
      @provider.stubs(:provider_can_use?).with(:cubert_request_logs).returns(false)
      backend_object = mock
      backend_object.expects(:log_requests).returns([])
      Service.any_instance.expects(:backend_object).returns(backend_object)
      CubertRequestLogsService.any_instance.expects(:get_list).never

      get :index, service_id: @service.id

      assert_template "index"
    end

    test "should render cubert logs when impersonating and user has them enabled" do
      @provider.stubs(:provider_can_use?).with(:cubert_request_logs).returns(true)
      CubertRequestLogsService.any_instance.expects(:get_list).returns([])

      get :index, service_id: @service.id

      assert_template "cubert_logs"
    end
  end
end
