import $ from 'jquery'
import createElement from 'virtual-dom/create-element'

import {StatsLogsUI} from 'stats/lib/logs_ui'

describe('StatsLogsUI', () => {
  let fakeResponse = {
    logs: [{
      code: 200,
      application_id: '0d108fd6',
      usage: 'analytics: 1, ',
      timestamp: '09 Dec 2015 09:30:16 PST',
      request: '{"path"=>"/stats/applications/1409612420750/usage.json?provider_key=0abb459337c7418c7b0c8f69ef876293&metric_name=Kii_dm_file_decreased_bytes&since=2012-10-10&granularity=day&until=2016-07-08", "method"=>"GET", "remote_ip"=>"54.214.51.166", "headers"=>[["HTTP_VERSION", "HTTP/1.0"], ["HTTP_X_FORWARDED_FOR", "54.214.51.166, 54.214.51.166"], ["HTTP_X_REAL_IP", "54.214.51.166"], ["HTTP_X_REQUEST_START", "t=1467988737.129"], ["HTTP_HOST", "kii-production-admin.3scale.net"], ["HTTP_CONNECTION", "close"], ["HTTP_ACCEPT", "application/json"], ["HTTP_ACCEPT_ENCODING", "gzip, deflate"], ["HTTP_USER_AGENT", "Ruby"], ["HTTP_X_FORWARDED_PORT", "443"], ["HTTP_X_FORWARDED_PROTO", "https"]]}',
      app: {
        id: 1409607043251,
        name: 'Default',
        account: {
          id: 2445579851281,
          org_name: 'Hand Inc'
        }
      }
    }],
    pagination: { pagination: { total: 1 } }
  }

  class FakeLogs {
    logRequests (_state) {
      return new Promise((resolve) => {
        resolve(fakeResponse)
      })
    }
  }

  fixture.set('<div id="logs"></div>')

  describe('Logs list', () => {
    let userSelectedState = {
      code: '2XX, 4XX, 5XX'
    }
    let logsSource = new FakeLogs()
    let logsUI = new StatsLogsUI({statsState: {state: userSelectedState}, source: logsSource, container: '#logs'})

    describe('data', () => {
      beforeEach(done => {
        logsUI.update(userSelectedState, () => { done() })
      })

      it('should get the right data', (done) => {
        let expectedData = fakeResponse
        expect(logsUI.data).toEqual(expectedData)
        done()
      })
    })

    describe('no data', () => {
      let userSelectedState = {
        code: ''
      }
      beforeEach(done => {
        logsUI.update(userSelectedState, () => { done() })
      })

      it('should return stubbed response and not make source.logRequest call', (done) => {
        let expectedData = {
          logs: [],
          pagination: {
            total: 1,
            page: '1',
            per_page: 20
          }
        }
        spyOn(logsSource, 'logRequests')
        expect(logsUI.data).toEqual(expectedData)
        expect(logsSource.logRequests).not.toHaveBeenCalled()

        done()
      })
    })

    describe('template', () => {
      beforeEach(done => {
        logsUI.render()
        setTimeout(() => {
          done()
        })
      })

      it('should display right template', (done) => {
        let expectedElement = 'table.data td a.show-trace-lr.success'
        expect(logsUI.element).toContainElement(expectedElement)
        done()
      })

      it('should render the corrent app and account links', (done) => {
        expect(logsUI.element).toContainHtml(`<a href="/buyers/applications/1409607043251">Default</a>`)
        expect(logsUI.element).toContainHtml(`<a href="/buyers/accounts/2445579851281">Hand Inc</a>`)
        done()
      })

      it('should display the correct date format', (done) => {
        expect(logsUI.element).toContainHtml(`<td>09 Dec 2015 09:30:16 PST</td>`)
        done()
      })
    })
  })

  describe('Pagination', () => {
    let fakeState = {state: {}}
    let fakeLogsSource = {}
    let paginationData = {
      logs: [],
      pagination: {
        page: 2,
        per_page: 20,
        total: 60
      }
    }
    let logsUI, fixture, pagination, rootNode

    beforeEach(() => {
      logsUI = new StatsLogsUI({statsState: fakeState, source: fakeLogsSource, container: '#logs'})
      logsUI.data = paginationData
      fixture = $('<div id="logs"></div>')
      pagination = logsUI.paginate()
      rootNode = createElement(pagination)
      fixture.append(rootNode)
    })

    it('should render', () => {
      expect(fixture).toContainElement('.StatsLogs-pagination')
    })

    it('should display the right amount of pages', () => {
      let pages = fixture.find('.StatsLogs-pagination').children()
      expect(pages.length).toEqual(5)
    })
  })
})
