class Buyers::AccountsController < Buyers::BaseController
  include SearchSupport
  include ThreeScale::Search::Helpers

  before_action :set_plans, :only => [:new, :create]

  activate_menu :buyers, :accounts

  def index
    @countries = Country.all
    @account_plans = current_account.account_plans.stock
    @search = ThreeScale::Search.new(params[:search] || params)
    @accounts = end_of_association_chain.scope_search(@search).
                  # this preloading collides with joins for sorting by country and plan
                  includes(:bought_account_plan, :country)
                  .order_by(params[:sort], params[:direction])
                  .paginate(pagination_params)
  end

  def new
    @buyer = current_account.buyers.build
    @user  = @buyer.users.build_with_fields :role => :admin
  end

  def update
    source = begin_of_association_chain.send(method_for_association_chain)
    @account = source.find(params[:id])

    # vat_rate is protected attribute
    if vat = account_params[:vat_rate]
      @account.vat_rate = vat
    end

    if @account.update_attributes(account_params)
      redirect_to admin_buyers_account_path(@account)
    else
      render :edit
    end
  end

  def create
    user_params    = account_params.try!(:delete, :user)

    signup = current_account.signup_with_plans([], false) do |buyer, user|
      @buyer, @user = buyer, user

      buyer.attributes = account_params
      buyer.provider_account = current_account
      user.attributes = user_params
      user.account = buyer

      user.signup_type = :created_by_provider
    end

    if signup
      @buyer.approve!
      @user.activate!
      flash[:notice] = 'Developer account was successfully created.'
      redirect_to admin_buyers_account_path(@buyer)
    else
      render :new
    end
  end

  def destroy
    if resource.destroy
      redirect_to admin_buyers_accounts_path, notice: 'Account deleted.'
    else
      flash[:error] = resource.errors.full_messages.join(' ')
      redirect_to admin_buyers_account_path(resource)
    end
  end

  def approve
    if resource.approve
      flash[:notice] = 'Developer account was approved.'
    else
      flash[:error] = 'Cannot approve developer account.'
    end

    redirect_to admin_buyers_account_path(resource)
  end

  def reject
    if resource.reject
      flash[:notice] = 'Developer account was rejected.'
    else
      flash[:error] = 'Cannot reject developer account.'
    end

    redirect_to admin_buyers_account_path(resource)
  end

  def suspend
    if resource.suspend
      flash[:notice] = 'Developer account was suspended.'
    else
      flash[:error] = 'Cannot suspend developer account.'
    end

    redirect_to admin_buyers_account_path(resource)
  end

  def resume
    if resource.resume
      flash[:notice] = 'Developer account was resumed.'
    else
      flash[:error] = 'Cannot resume developer account.'
    end

    redirect_to admin_buyers_account_path(resource)
  end

  def toggle_monthly_charging
    resource.settings.toggle!(:monthly_charging_enabled)
    redirect_to(:back)
  end

  def toggle_monthly_billing
    resource.settings.toggle!(:monthly_billing_enabled)
    redirect_to(:back)
  end

  def show
    @account = find_account
    @available_account_plans = current_account.account_plans.stock
  end

  protected

  def account_params
    params.require(:account)
  end

  def find_account
    @account = current_account.buyers.find(params[:id])
  end

  def begin_of_association_chain
    current_account
  end

  def method_for_association_chain
    :buyer_accounts
  end

  def set_plans
    unless current_account.create_buyer_possible?
      redirect_to admin_buyers_account_plans_path, alert: 'Please, create an Account Plan first'
    end

    @plans = [] # this is here only to make new_signups/form happy
  end

end
