/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb;

import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoCredential;
import com.mongodb.MongoInternalException;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import javax.net.ssl.SSLSocketFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class MongoClientURI {
    private static final String PREFIX = "mongodb://";
    private static final String UTF_8 = "UTF-8";
    static Set<String> generalOptionsKeys = new HashSet<String>();
    static Set<String> authKeys = new HashSet<String>();
    static Set<String> readPreferenceKeys = new HashSet<String>();
    static Set<String> writeConcernKeys = new HashSet<String>();
    static Set<String> allKeys = new HashSet<String>();
    private final MongoClientOptions options;
    private final MongoCredential credentials;
    private final List<String> hosts;
    private final String database;
    private final String collection;
    private final String uri;
    static final Logger LOGGER;

    public MongoClientURI(String uri) {
        this(uri, new MongoClientOptions.Builder());
    }

    public MongoClientURI(String uri, MongoClientOptions.Builder builder) {
        try {
            String optionsPart;
            String nsPart;
            String serverPart;
            this.uri = uri;
            if (!uri.startsWith(PREFIX)) {
                throw new IllegalArgumentException("uri needs to start with mongodb://");
            }
            uri = uri.substring(PREFIX.length());
            String userName = null;
            char[] password = null;
            int idx = uri.lastIndexOf("/");
            if (idx < 0) {
                if (uri.contains("?")) {
                    throw new IllegalArgumentException("URI contains options without trailing slash");
                }
                serverPart = uri;
                nsPart = null;
                optionsPart = "";
            } else {
                serverPart = uri.substring(0, idx);
                nsPart = uri.substring(idx + 1);
                if ((idx = nsPart.indexOf("?")) >= 0) {
                    optionsPart = nsPart.substring(idx + 1);
                    nsPart = nsPart.substring(0, idx);
                } else {
                    optionsPart = "";
                }
            }
            LinkedList all = new LinkedList();
            int idx2 = serverPart.indexOf("@");
            if (idx2 > 0) {
                String authPart = serverPart.substring(0, idx2);
                serverPart = serverPart.substring(idx2 + 1);
                idx2 = authPart.indexOf(":");
                if (idx2 == -1) {
                    userName = URLDecoder.decode(authPart, UTF_8);
                } else {
                    userName = URLDecoder.decode(authPart.substring(0, idx2), UTF_8);
                    password = URLDecoder.decode(authPart.substring(idx2 + 1), UTF_8).toCharArray();
                }
            }
            Collections.addAll(all, serverPart.split(","));
            this.hosts = Collections.unmodifiableList(all);
            if (nsPart != null && !nsPart.isEmpty()) {
                idx = nsPart.indexOf(".");
                if (idx < 0) {
                    this.database = nsPart;
                    this.collection = null;
                } else {
                    this.database = nsPart.substring(0, idx);
                    this.collection = nsPart.substring(idx + 1);
                }
            } else {
                this.database = null;
                this.collection = null;
            }
            Map<String, List<String>> optionsMap = this.parseOptions(optionsPart);
            this.options = this.createOptions(optionsMap, builder);
            this.credentials = this.createCredentials(optionsMap, userName, password, this.database);
            this.warnOnUnsupportedOptions(optionsMap);
        }
        catch (UnsupportedEncodingException e) {
            throw new MongoInternalException("This should not happen", e);
        }
    }

    private void warnOnUnsupportedOptions(Map<String, List<String>> optionsMap) {
        for (String key : optionsMap.keySet()) {
            if (allKeys.contains(key)) continue;
            LOGGER.warning("Unknown or Unsupported Option '" + key + "'");
        }
    }

    private MongoClientOptions createOptions(Map<String, List<String>> optionsMap, MongoClientOptions.Builder builder) {
        for (String key : generalOptionsKeys) {
            String value = this.getLastValue(optionsMap, key);
            if (value == null) continue;
            if (key.equals("maxpoolsize")) {
                builder.connectionsPerHost(Integer.parseInt(value));
                continue;
            }
            if (key.equals("waitqueuemultiple")) {
                builder.threadsAllowedToBlockForConnectionMultiplier(Integer.parseInt(value));
                continue;
            }
            if (key.equals("waitqueuetimeoutms")) {
                builder.maxWaitTime(Integer.parseInt(value));
                continue;
            }
            if (key.equals("connecttimeoutms")) {
                builder.connectTimeout(Integer.parseInt(value));
                continue;
            }
            if (key.equals("sockettimeoutms")) {
                builder.socketTimeout(Integer.parseInt(value));
                continue;
            }
            if (key.equals("autoconnectretry")) {
                builder.autoConnectRetry(this._parseBoolean(value));
                continue;
            }
            if (!key.equals("ssl") || !this._parseBoolean(value)) continue;
            builder.socketFactory(SSLSocketFactory.getDefault());
        }
        WriteConcern writeConcern = this.createWriteConcern(optionsMap);
        ReadPreference readPreference = this.createReadPreference(optionsMap);
        if (writeConcern != null) {
            builder.writeConcern(writeConcern);
        }
        if (readPreference != null) {
            builder.readPreference(readPreference);
        }
        return builder.build();
    }

    private WriteConcern createWriteConcern(Map<String, List<String>> optionsMap) {
        Boolean safe = null;
        String w = null;
        int wTimeout = 0;
        boolean fsync = false;
        boolean journal = false;
        for (String key : writeConcernKeys) {
            String value = this.getLastValue(optionsMap, key);
            if (value == null) continue;
            if (key.equals("safe")) {
                safe = this._parseBoolean(value);
                continue;
            }
            if (key.equals("w")) {
                w = value;
                continue;
            }
            if (key.equals("wtimeout")) {
                wTimeout = Integer.parseInt(value);
                continue;
            }
            if (key.equals("fsync")) {
                fsync = this._parseBoolean(value);
                continue;
            }
            if (!key.equals("j")) continue;
            journal = this._parseBoolean(value);
        }
        return this.buildWriteConcern(safe, w, wTimeout, fsync, journal);
    }

    private ReadPreference createReadPreference(Map<String, List<String>> optionsMap) {
        Boolean slaveOk = null;
        String readPreferenceType = null;
        DBObject firstTagSet = null;
        ArrayList<DBObject> remainingTagSets = new ArrayList<DBObject>();
        for (String key : readPreferenceKeys) {
            String value = this.getLastValue(optionsMap, key);
            if (value == null) continue;
            if (key.equals("slaveok")) {
                slaveOk = this._parseBoolean(value);
                continue;
            }
            if (key.equals("readpreference")) {
                readPreferenceType = value;
                continue;
            }
            if (!key.equals("readpreferencetags")) continue;
            for (String cur : optionsMap.get(key)) {
                DBObject tagSet = this.getTagSet(cur.trim());
                if (firstTagSet == null) {
                    firstTagSet = tagSet;
                    continue;
                }
                remainingTagSets.add(tagSet);
            }
        }
        return this.buildReadPreference(readPreferenceType, firstTagSet, remainingTagSets, slaveOk);
    }

    private MongoCredential createCredentials(Map<String, List<String>> optionsMap, String userName, char[] password, String database) {
        if (userName == null) {
            return null;
        }
        if (database == null) {
            database = "admin";
        }
        String mechanism = "MONGODB-CR";
        String authSource = database;
        for (String key : authKeys) {
            String value = this.getLastValue(optionsMap, key);
            if (value == null) continue;
            if (key.equals("authmechanism")) {
                mechanism = value;
                continue;
            }
            if (!key.equals("authsource")) continue;
            authSource = value;
        }
        if (mechanism.equals("GSSAPI")) {
            return MongoCredential.createGSSAPICredential(userName);
        }
        if (mechanism.equals("MONGODB-CR")) {
            return MongoCredential.createMongoCRCredential(userName, authSource, password);
        }
        throw new IllegalArgumentException("Unsupported authMechanism: " + mechanism);
    }

    private String getLastValue(Map<String, List<String>> optionsMap, String key) {
        List<String> valueList = optionsMap.get(key);
        if (valueList == null) {
            return null;
        }
        return valueList.get(valueList.size() - 1);
    }

    private Map<String, List<String>> parseOptions(String optionsPart) {
        HashMap<String, List<String>> optionsMap = new HashMap<String, List<String>>();
        for (String _part : optionsPart.split("&|;")) {
            int idx = _part.indexOf("=");
            if (idx < 0) continue;
            String key = _part.substring(0, idx).toLowerCase();
            String value = _part.substring(idx + 1);
            ArrayList<String> valueList = (ArrayList<String>)optionsMap.get(key);
            if (valueList == null) {
                valueList = new ArrayList<String>(1);
            }
            valueList.add(value);
            optionsMap.put(key, valueList);
        }
        return optionsMap;
    }

    private ReadPreference buildReadPreference(String readPreferenceType, DBObject firstTagSet, List<DBObject> remainingTagSets, Boolean slaveOk) {
        if (readPreferenceType != null) {
            if (firstTagSet == null) {
                return ReadPreference.valueOf(readPreferenceType);
            }
            return ReadPreference.valueOf(readPreferenceType, firstTagSet, remainingTagSets.toArray(new DBObject[remainingTagSets.size()]));
        }
        if (slaveOk != null && slaveOk.equals(Boolean.TRUE)) {
            return ReadPreference.secondaryPreferred();
        }
        return null;
    }

    private WriteConcern buildWriteConcern(Boolean safe, String w, int wTimeout, boolean fsync, boolean journal) {
        if (w != null || wTimeout != 0 || fsync || journal) {
            if (w == null) {
                return new WriteConcern(1, wTimeout, fsync, journal);
            }
            try {
                return new WriteConcern(Integer.parseInt(w), wTimeout, fsync, journal);
            }
            catch (NumberFormatException e) {
                return new WriteConcern(w, wTimeout, fsync, journal);
            }
        }
        if (safe != null) {
            if (safe.booleanValue()) {
                return WriteConcern.ACKNOWLEDGED;
            }
            return WriteConcern.UNACKNOWLEDGED;
        }
        return null;
    }

    private DBObject getTagSet(String tagSetString) {
        BasicDBObject tagSet = new BasicDBObject();
        if (tagSetString.length() > 0) {
            for (String tag : tagSetString.split(",")) {
                String[] tagKeyValuePair = tag.split(":");
                if (tagKeyValuePair.length != 2) {
                    throw new IllegalArgumentException("Bad read preference tags: " + tagSetString);
                }
                tagSet.put(tagKeyValuePair[0].trim(), (Object)tagKeyValuePair[1].trim());
            }
        }
        return tagSet;
    }

    boolean _parseBoolean(String _in) {
        String in = _in.trim();
        return in != null && in.length() > 0 && (in.equals("1") || in.toLowerCase().equals("true") || in.toLowerCase().equals("yes"));
    }

    public String getUsername() {
        return this.credentials != null ? this.credentials.getUserName() : null;
    }

    public char[] getPassword() {
        return this.credentials != null ? this.credentials.getPassword() : null;
    }

    public List<String> getHosts() {
        return this.hosts;
    }

    public String getDatabase() {
        return this.database;
    }

    public String getCollection() {
        return this.collection;
    }

    public String getURI() {
        return this.uri;
    }

    public MongoCredential getCredentials() {
        return this.credentials;
    }

    public MongoClientOptions getOptions() {
        return this.options;
    }

    public String toString() {
        return this.uri;
    }

    static {
        generalOptionsKeys.add("maxpoolsize");
        generalOptionsKeys.add("waitqueuemultiple");
        generalOptionsKeys.add("waitqueuetimeoutms");
        generalOptionsKeys.add("connecttimeoutms");
        generalOptionsKeys.add("sockettimeoutms");
        generalOptionsKeys.add("sockettimeoutms");
        generalOptionsKeys.add("autoconnectretry");
        generalOptionsKeys.add("ssl");
        readPreferenceKeys.add("slaveok");
        readPreferenceKeys.add("readpreference");
        readPreferenceKeys.add("readpreferencetags");
        writeConcernKeys.add("safe");
        writeConcernKeys.add("w");
        writeConcernKeys.add("wtimeout");
        writeConcernKeys.add("fsync");
        writeConcernKeys.add("j");
        authKeys.add("authmechanism");
        authKeys.add("authsource");
        allKeys.addAll(generalOptionsKeys);
        allKeys.addAll(authKeys);
        allKeys.addAll(readPreferenceKeys);
        allKeys.addAll(writeConcernKeys);
        LOGGER = Logger.getLogger("com.mongodb.MongoURI");
    }
}

