package com.amentra.metamatrix.solr.visitor;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import com.metamatrix.data.api.ConnectorLogger;
import com.metamatrix.data.exception.ConnectorException;
import com.metamatrix.data.language.IAggregate;
import com.metamatrix.data.language.ICompareCriteria;
import com.metamatrix.data.language.ICompoundCriteria;
import com.metamatrix.data.language.ICriteria;
import com.metamatrix.data.language.IElement;
import com.metamatrix.data.language.IExpression;
import com.metamatrix.data.language.IInCriteria;
import com.metamatrix.data.language.ILikeCriteria;
import com.metamatrix.data.language.ILimit;
import com.metamatrix.data.language.ILiteral;
import com.metamatrix.data.language.INotCriteria;
import com.metamatrix.data.language.ISelectSymbol;
import com.metamatrix.data.metadata.runtime.Element;
import com.metamatrix.data.metadata.runtime.RuntimeMetadata;
import com.metamatrix.data.visitor.framework.HierarchyVisitor;

/**
 * Translates MetaMatrix queries into Solr search strings, using the Hierarchy Visitor design pattern.
 * @author Michael Walker
 */	
public class SolrHierarchyVisitor extends HierarchyVisitor {
	// Removed * and ? from reserved Chars -- you can include them in search, but can't search from them...
	// Consider removing this and requiring user to double-quote any reserved chars manually, to allow search for *
	//private static String[] reservedChars = new String[]{"+","-", "&&", "||", "!", "(", ")", "{", "}", "[", "]", "^", "\"", "~", ":", "\\"};
	private RuntimeMetadata md;
	private ArrayList<String> fieldList = new ArrayList();
	private HashMap<String, Class> fieldMap = new HashMap<String, Class>();
	private ConnectorLogger logger;
	private Integer limit = null;
	private Integer offset = null;
	private String queryString = "";
	private boolean useLowerCase;
	
	public SolrHierarchyVisitor(RuntimeMetadata md, ConnectorLogger logger, boolean useLowerCase) {
		this.md = md;
		this.logger = logger;
		this.useLowerCase = useLowerCase;
	}

	/**
	 * Extract the select symbols (columns) from the query.
	 */
	public void visit(ISelectSymbol symbol) {
		super.visit(symbol);
		String columnName = null;
		if(symbol.getExpression() instanceof IElement) {			
			try {
				columnName = md.getObject(((IElement)symbol.getExpression()).getMetadataID()).getNameInSource();
				// Doesn't appear to be necessary
				//columnName = URLEncoder.encode(columnName, "UTF-8");
			} catch (ConnectorException e) {
				logger.logError("Unable to extract NameInSource from symbol in select clause. Check the model properties.");
				try {
					throw new ConnectorException(e);
				} catch (ConnectorException e1) {
					e1.printStackTrace();
				}
			}
		} 
		Class columnType = ((IElement)symbol.getExpression()).getType();
		if(columnName==null || columnType==null) {
			logger.logError("Column NameInSource or type cannot be null. Check the model properties.");
		}
		fieldList.add(columnName);
		fieldMap.put(columnName, columnType);
	}
	
	public void visit(ILimit limit) {
		this.limit = limit.getRowLimit();
		offset = limit.getRowOffset();
	}
	
	public void visit(INotCriteria criteria) {
		queryString = queryString + "NOT ";
		super.visitNode(criteria.getCriteria());
	}
	
	public void visit(ILikeCriteria criteria) {
		logger.logTrace("Parsing LIKE criteria. Treating as a compare criteria.");
		super.visit(criteria);
		if(criteria.isNegated()) {
			// NOT can occur here and is apparently not considered to be a separate INotCriteria.
			queryString = queryString + "NOT ";
		}
		// Ignore the LIKE, and treat it just like a typical CompareCriteria. 
		// Wildcards will be substituted appropriately within that visitor method.
		try {
			processCompareCriteria(criteria.getLeftExpression(), criteria.getRightExpression(), ICompareCriteria.EQ);
		} catch (ConnectorException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	public void visit(ICompareCriteria criteria) {
		super.visit(criteria);
		IExpression lhs = criteria.getLeftExpression();
		IExpression rhs = criteria.getRightExpression();
		int op = criteria.getOperator();
		try {
			processCompareCriteria(lhs, rhs, op);
		} catch (ConnectorException e) {
			e.printStackTrace();
		}
	}
	
	public void visit(IInCriteria criteria) {
		super.visit(criteria);

		List<IExpression> rhs = criteria.getRightExpressions();
		IExpression lhs = criteria.getLeftExpression();
		if(rhs.size() == 0) {
			return;
		}
		queryString = queryString + "(";


		Iterator<IExpression> i = rhs.iterator();
		while(i.hasNext()) {
			IExpression rExpression = i.next();
			try {
				processCompareCriteria(lhs, rExpression, ICompareCriteria.EQ);
				if(i.hasNext()) {
					queryString = queryString + " OR ";
				}
			} catch (ConnectorException e) {
				logger.logError("Failed to process criteria in IN clause");
				e.printStackTrace();
			}
		}
		queryString = queryString + ") ";
	}
	
	public String getExpressionString(IExpression e) throws ConnectorException {
		String expressionName = null;
		if(e instanceof IElement) {
			Element el = (Element)md.getObject(((IElement)e).getMetadataID());
			expressionName = el.getNameInSource();
			if(expressionName == null || expressionName.equals("")) {
				expressionName = el.getMetadataID().getName();
			}
		}
		return expressionName;
	}
	
	public void processCompareCriteria(IExpression lhs, IExpression rhs, int op) throws ConnectorException {
		// TODO *******************Clean this part up/enhance to handle more cases. 
		if(!(lhs instanceof IElement && rhs instanceof ILiteral)) {
			throw new ConnectorException("Received criteria not of the form field=value. Not supported by connector.");
		}
		try {
			if(!rhs.getType().equals(Class.forName(String.class.getName()))) {
				// TODO handle each type here
				throw new ConnectorException("Received non-string literal. Not supported by connector.");
			}
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}
		// ***********************End Cleanup
		
		switch(op) {
		case ICompareCriteria.EQ:
			String name = getExpressionString(lhs);
			String value = ((ILiteral)rhs).getValue().toString();
			value = processLiteral(value);
			queryString = queryString + name + ":" + value + " ";
			logger.logTrace("Parsing compare criteria. String is: " + queryString);
			
			//logger.logTrace("DEBUG: created new term with name: " + ((IElement)lhs).getMetadataID().getName() + " and value: " + ((ILiteral)rhs).getValue().toString());
			break;
		default:
			throw new ConnectorException("Received non-EQ criteria. Not supported by connector."); 
		}
	}
	/**
	 * Escape all special characters, replace SQL wildcards with Solr equivalents, and shift to lower case if
	 * connector property requests it.
	 * @param literal
	 * @return Processed literal
	 */
	private String processLiteral(String literal) {
		// Replace wildcard characters with their Solr equivalent
		literal = literal.replace("%", "*");
		literal = literal.replace("_", "?");
		
		/*
		 * Removing this code entirely. If the user wants to use reserved characters in the search, they should double-quote them:
		 * select fieldA from index where fieldB="my\"*\"string"
		 * This will retrieve docs where fieldB is set to the string literal "my*string". 
		 * The * does not have any wildcard meaning in this case.
		 * select fieldA from index where fieldB="my*specialstring"
		 * This retrieves docs where fieldB is set to "mystring", "my123string", etc -- the wildcard is in effect.
		 * This will go in the user docs.
		 * 
		 * 
		// Escape reserved characters by double-quoting them.
		// Originally written to escape all reserved characters with the URL encoding.
		// However, URL encoding did not work in practice (nor did escaping them with \). Double-quoting worked.
		for(int i = 0; i < reservedChars.length; i++) {
			 String reservedChar = reservedChars[i];
			 //try {
				//literal = literal.replace(reservedChar, URLEncoder.encode(reservedChar, "UTF-8"));
				 literal = literal.replace(reservedChar, "\"" + reservedChar + "\"");
			//} catch (UnsupportedEncodingException e) {
				// TODO Auto-generated catch block
				//logger.logError("URL encoding failed.");
				//e.printStackTrace();
			//}
		}
		*/
		if(useLowerCase) {
			literal = literal.toLowerCase();
		}
		return literal;
	}

	
	/**
	 * Add parentheses, process each criterion, and add the appropriate operator to the search string.
	 * 
	 * If we instead chose to build a Solr Query object, we'd create a boolean query here, processing each
	 * criterion and adding it to the boolean query.
	 */
	public void visit(ICompoundCriteria compoundCriteria) {
		queryString = queryString + "(";
		logger.logTrace("Parsing compound criteria. String is: " + queryString);
		Iterator i = compoundCriteria.getCriteria().iterator();
		while(i.hasNext()) {
			ICriteria criterion = (ICriteria)i.next();
			super.visitNode(criterion);
			// Add operator, if another operand is to come
			if(i.hasNext()) {
				switch(compoundCriteria.getOperator()) {
				case ICompoundCriteria.AND:
					queryString = queryString + "AND ";
					break;
				// A space would also be acceptable here. Writing OR is more explicit.
				case ICompoundCriteria.OR:
					queryString = queryString + "OR ";
					break;
				default:
					break;
				}
			} 
		}
		// Add closed parenthesis
		queryString = queryString + ")";
		//logger.logTrace("Exiting parse of compound criteria. String is: " + queryString);
	}
	
	public ArrayList<String> getFieldList() {
		return fieldList;
	}
	
	public HashMap<String, Class> getFieldMap() {
		return fieldMap;
	}
	public Integer getLimit() { 
		return limit;
	}
	public Integer getOffset() {
		return offset;
	}
	
	/**
	 * Search everything if the query was not defined by any where clause. 
	 * @return String Solr queryString
	 */
	public String getQueryString() {
		if(queryString == null || queryString.length()==0) {
			return "*:*";
		} else {
			return queryString;
		}
	}
}
