require File.expand_path(File.dirname(__FILE__) + '/../../../test_helper')

class BackendClient::Provider::TransactionsTest < ActiveSupport::TestCase

  include TestHelpers::FakeWeb

  def setup
    set_backend_host 'example.org'

    @provider_account = Factory(:provider_account)
    @provider_key     = @provider_account.api_key

    @connection = BackendClient::Connection.new(:host => backend_host)
    @provider   = @connection.provider(@provider_account)
  end

  # TODO: #latest_transactions with invalid metric_id
  # TODO: #latest_transactions and timezones

  test '#latest_transactions returns empty array if response contains no entries' do
    fake_backend_transactions @provider_account.default_service, []
    assert_equal [], @provider.latest_transactions
  end

  test '#latest_transactions returns collection of transaction objects' do
    plan          = Factory(:application_plan, :issuer => @provider_account.default_service)

    cinstance_one = Factory(:cinstance, :plan => plan)
    cinstance_two = Factory(:cinstance, :plan => plan)

    metric_one    = Factory(:metric, :service => @provider_account.default_service)
    metric_two    = Factory(:metric, :service => @provider_account.default_service)

    transactions = [
      {:application_id => cinstance_one.application_id, :timestamp => "2010-09-13 14:14:00",
       :metrics => [{:id => metric_one.id, :value => 1}, {:id => metric_two.id, :value => 10000}] },
      {:application_id => cinstance_two.application_id, :timestamp => "2010-09-13 14:10:00",
       :metrics => [{:id => metric_one.id, :value => 2}, {:id => metric_two.id, :value => 7000}] }
    ]
    fake_backend_transactions @provider_account.default_service, transactions

    @provider_account.reload
    @provider_account.default_service.reload
    transactions = @provider.latest_transactions

    assert_equal 2, transactions.size

    assert_equal cinstance_one,                          transactions[0].cinstance
    assert_equal Time.utc(2010, 9, 13, 14, 14),          transactions[0].timestamp
    assert_equal({metric_one => 1, metric_two => 10000}, transactions[0].usage)

    assert_equal cinstance_two,                          transactions[1].cinstance
    assert_equal Time.utc(2010, 9, 13, 14, 10),          transactions[1].timestamp
    assert_equal({metric_one => 2, metric_two => 7000},  transactions[1].usage)
  end

  test '#latest_transactions converts timestamps to the current timezone' do
    plan      = Factory( :application_plan, :issuer => @provider_account.default_service)
    cinstance = Factory(:cinstance, :plan => plan)
    metric    = @provider_account.default_service.metrics.hits

    transactions = [
      {:application_id => cinstance.application_id, :timestamp => "2010-09-23 14:25:00",
       :metrics => [{:id => metric.id, :value => 1}] }
    ]
    fake_backend_transactions @provider_account.default_service, transactions

    Time.use_zone('Alaska') do
      transactions = @provider.latest_transactions
      assert_equal Time.utc(2010, 9, 23, 14, 25).in_time_zone('Alaska'), transactions[0].timestamp
    end
  end

  test '#latest_transactions returns transactions with nil cinstance for invalid application id' do
    metric = @provider_account.default_service.metrics.hits
    transactions = [
      {:application_id => 'invalid', :timestamp => "2010-09-23 14:28:00",
       :metrics => [{:id => metric.id, :value => 1}] }
    ]
    fake_backend_transactions @provider_account.default_service, transactions

    assert_nil @provider.latest_transactions.first.cinstance
  end


end
