/** @jsx StatsUI.dom */
import moment from 'moment'

import {StatsUI} from '../lib/ui'
import {StatsChart} from '../lib/chart'
import {StatsTopAppsSourceCollector} from '../lib/top_apps_source_collector'
import {StatsSourceCollectorChartManager} from '../lib/source_collector_chart_manager'
import {StatsSeries} from '../lib/series'
import {StatsMetricsList} from '../lib/metrics_list'
import {StatsCSVLink} from '../lib/csv_link'
import {Stats} from '../lib/stats'

export class StatsTopAppsSeries extends StatsSeries {
  _getSeriesName (serie) {
    return `${serie.application.name} by ${serie.application.account.name}`
  }

  _totalValues (responses) {
    return responses.map((response) => response.total).reduce((previous, current) => previous + current, 0)
  }

  _customOptions (responses) {
    return {
      _topAppsSelectionPeriod: responses[0].topAppsSelectionPeriod
    }
  }
}

class StatsTopAppsChartManager extends StatsSourceCollectorChartManager {
  static get Series () {
    return StatsTopAppsSeries
  }
}

class StatsTopAppsHelpText extends StatsUI {
  template () {
    if (this.period) {
      let since = moment(this.period.since).utcOffset(this.period.since).subtract(1, 'day').format('L')
      let until = moment(this.period.until).utcOffset(this.period.until).format('L')

      return (
        <p className='Stats-helptext Stats-message--notice'>
          Top Applications are determined from usage data between midnight {since} and midnight {until}
        </p>
      )
    } else {
      return <p></p>
    }
  }

  update (data) {
    this.period = data._topAppsSelectionPeriod
    this.refresh()
  }

  _bindEvents () {

  }

}

class StatsTopAppsChart extends StatsChart {
  _chartCustomOptions () {
    return {
      legend: {
        position: 'right'
      }
    }
  }
}

let statsTopApps = (serviceId, options = {}) => {
  let topAppsMetricsUrl = `/services/${serviceId}/stats/usage/top_applications.json`
  let metricsList = StatsMetricsList.getMetrics(topAppsMetricsUrl)
  let csvLink = new StatsCSVLink({container: options.csvLinkContainer})
  let helpText = new StatsTopAppsHelpText({container: options.helpTextContainer})

  Stats({ChartManager: StatsTopAppsChartManager, Chart: StatsTopAppsChart, Sources: StatsTopAppsSourceCollector}).build({
    id: serviceId,
    metricsList,
    widgets: [csvLink, helpText],
    options: {
      granularities: ['day', 'month'],
      hasGroupedMethods: false
    }
  })
}

export { statsTopApps, StatsTopAppsHelpText }
