class UserSession < ActiveRecord::Base

  belongs_to :user
  has_one :account, through: :user

  validates :user_id, :key, presence: true

  before_validation :set_unique_key, :on => :create


  attr_readonly :key

  scope :active, -> { where('accessed_at >= ? and revoked_at is null', 2.weeks.ago) }

  def self.authenticate(key)
    return nil if key.nil?
    self.active.find_by_key(key)
  end

  def self.null
    new
  end

  def revoke!
    return unless persisted?
    self.revoked_at = Time.zone.now
    save!
  end

  def access(request)
    return unless valid?

    update_attributes!(accessed_at: Time.zone.now,
                       ip: request.ip,
                       user_agent: request.user_agent)
  rescue => error
    System::ErrorReporting.report_error(error)
  end

  def user_agent=(value)
    if value != nil
      value = value[0..254]
    end
    self[:user_agent] = value
  end

  private
    def set_unique_key
      self.key = SecureRandom.urlsafe_base64(32)
    end
end
